;;----------------------------------------------------------------------------
;; Name: VERIFY_IMAGE
;;
;; Purpose: To verify a PDS IMAGE object and its WINDOW subobjects.
;;
;; Calling Sequence:
;;     result = verify_image (label)
;;
;; Input:
;;     label - a string array containing PDS header.
;;
;; Output:
;;     result - the routine returns 1 if there are no errors in label.
;;     The routine outputs an error message if there are any errors
;;     found in the PDS label for IMAGE/WINDOW objects.
;;
;; Optional inputs: none.
;;
;; External routines: Pdspar, Clean, Remove, Extract_keyword,
;;     Test_integer, Get_index
;;
;; Modification history:
;;     Written by Puneet Khetarpal, 30 June 2005
;;
;;----------------------------------------------------------------------------

;- level 1 -------------------------------------------------------------------

; precondition: label contains image object definitions; startindex
;     and endindex are both viable
; postcondition: the required keywords for current image object are
; tested
pro check_image_required, label, startindex, endindex
    ; error check
    on_error, 1

    ; check LINES keyword
    lines = extract_keyword(label, 'LINES', startindex, endindex, 1, 1)

    test_integer, 'LINES', lines, 0       ; test the value for integer
    if (lines eq 0) then return           ; if no data available, then return

    ; check LINE_SAMPLES keyword
    line_samples = extract_keyword(label, 'LINE_SAMPLES', startindex, $
        endindex, 1, 1)
    test_integer, 'LINE_SAMPLES', line_samples, 0  ; test value for integer
    if (line_samples eq 0) then return            ; if no data then return

    ; check for SAMPLE_BITS keyword
    sample_bits = extract_keyword(label, 'SAMPLE_BITS', startindex,endindex, 1)
    test_integer, 'SAMPLE_BITS', sample_bits, 1
    ; check whether the sample bits value is in range
    bitsrange = [1, 2, 4, 8, 16, 32, 64]                ; define range values
    pos = where(fix(sample_bits) eq bitsrange, matches) ; check for value 
    if (matches eq 0) then begin      ; if not in range, then issue error
        message, "Error: SAMPLE_BITS value must be [2^0,2^6] - " + sample_bits 
    endif

    ; check for SAMPLE_TYPE keyword
    sample_type = extract_keyword(label, 'SAMPLE_TYPE', startindex, endindex,1)
    sample_type = remove(sample_type, ['"'])
    types = ["IEEE_REAL","LSB_INTEGER","LSB_UNSIGNED_INTEGER","MSB_INTEGER", $
             "MSB_UNSIGNED_INTEGER","PC_REAL","UNSIGNED_INTEGER","VAX_REAL"]
    test = where(stregex(types, sample_type,/boolean) eq 1, matches)
    ; if not part a standard value set, then issue error
    if (matches eq 0) then begin
        message, "Error: SAMPLE_TYPE does not belong to standard value set "+$
            sample_type
    endif
end

; precondition: the required keywords for image object have been checked
; postcondition: the label is checked for optional image object keywords
pro check_image_optional, label, startindex, endindex
    ; error check
    on_error, 1

    ; check for FIRST_LINE keyword
    first_line = extract_keyword(label, 'FIRST_LINE', startindex, endindex,0,1)
    if (first_line ne '###~') then begin    ; if first line keyword exists
        lines = extract_keyword(label, 'LINES', startindex, endindex, 1, 1)
        if (lines eq 0) then return   ; if no data, then return
        test_integer, 'FIRST_LINE', first_line, 1, fix(lines)
    endif

    ; check FIRST_LINE_SAMPLE keyword
    first_line_sample = extract_keyword(label, 'FIRST_LINE_SAMPLE',startindex,$
        endindex, 0, 1)
    if (first_line_sample ne '###~') then begin ; if first line sample exists
        line_samples = extract_keyword(label, 'LINE_SAMPLES', startindex, $
            endindex, 1, 1)
        if (line_samples eq 0) then return       ; if no data, then return
        test_integer, 'FIRST_LINE_SAMPLE', first_line_sample, 1, $
            fix(line_samples)
    endif

    ; check LINE_PREFIX_BYTES keyword
    line_prefix_bytes = extract_keyword(label, 'LINE_PREFIX_BYTES',startindex,$
        endindex, 0)
    if (line_prefix_bytes ne '###~') then begin ; if line prefix exists
        test_integer, 'LINE_PREFIX_BYTES', line_prefix_bytes, 0
    endif
    
    ; check LINE_SUFFIX_BYTES keyword
    line_suffix_bytes = extract_keyword(label, 'LINE_SUFFIX_BYTES',startindex,$
        endindex, 0)
    if (line_suffix_bytes ne '###~') then begin ; if line suffix exists
        test_integer, 'LINE_SUFFIX_BYTES', line_suffix_bytes, 0
    endif

    ; check OFFSET keyword
    offset = extract_keyword(label, 'OFFSET', startindex, endindex, 0)
    if (offset ne '###~') then begin ; if offset exists
        if (stregex(offset, '[^-\+\.eE0-9]',/boolean)) then begin
            message, "Error: OFFSET value must be numeric - " + offset
        endif
    endif

    ; check SCALING_FACTOR keyword
    scal_fact = extract_keyword(label, 'SCALING_FACTOR',startindex,endindex, 0)
    if (scal_fact ne '###~') then begin ; if scaling factor exists
        if (stregex(scal_fact, '[^-\+\.eE0-9]', /boolean)) then begin
            message, "Error: SCALING_FACTOR value must be numeric - "+scal_fact
        endif
    endif

    ; check SAMPLE_BIT_MASK keyword
    bit_mask = extract_keyword(label, 'SAMPLE_BIT_MASK',startindex,endindex,0)
    if (bit_mask ne '###~') then begin ; if bit mask exists
        bits = extract_keyword(label,'SAMPLE_BITS',startindex,endindex, 1)
        expr = '2#[01]{'+bits+'}#'
        if (~ stregex(bit_mask, expr, /boolean)) then begin
            message, "Error: SAMPLE_BIT_MASK value invalid - " + bit_mask
        endif
    endif

    ; check LINE_DISPLAY_DIRECTION keyword
    dd_range = ["DOWN","LEFT","RIGHT","UP"]
    ldd = extract_keyword(label,'LINE_DISPLAY_DIRECTION',startindex,endindex,0)
    if (ldd ne '###~') then begin   ; if line display direction exists
        ldd = remove(ldd, '"')
        test = where(stregex(dd_range, ldd, /boolean) eq 1, matches)
        if (matches eq 0) then begin   ; if invalid value, then issue error
            message, "Error: invalid LINE_DISPLAY_DIRECTION value - " + ldd
        endif
    endif

    ; check SAMPLE_DISPLAY_DIRECTION keyword
    sdd = extract_keyword(label,'SAMPLE_DISPLAY_DIRECTION',startindex,endindex,0)
    if (sdd ne '###~') then begin ; if sample display direction exists
        sdd = remove(sdd, '"')
        test = where(stregex(dd_range, sdd, /boolean) eq 1, matches)
        if (matches eq 0) then begin ; if invalid value, then issue error
            message, "Error: invalid SAMPLE_DISPLAY_RECTION value - " + sdd
        endif
    endif
end

; precondition: all required and optional keywords have been verified
; postcondition: the object is searched for windows sub objects and
;     they are verified
pro check_window, label, startindex, endindex
    ; error check
    on_error, 1

    ; extract window objects
    objects = pdspar(label, 'OBJECT', COUNT=objcnt, INDEX=objindx) ; get all
    objects = strtrim(objects)
    ;; get position of those objects which are WINDOW and lie between start
    ;; and end index
    win_pos = where(stregex(objects, 'WINDOW$', /boolean) eq 1 and $
                 objindx ge startindex and objindx le endindex, matches)
    if (matches eq 0) then return    ; if none found then return
    windows = objects[win_pos]       ; store name
    win_index = objindx[win_pos]     ; store index
    win_count = matches              ; store found

    ; get image lines and line_samples keyword values
    img_lines = long(extract_keyword(label, 'LINES', startindex, endindex,1,1))
    img_linsmp = long(extract_keyword(label, 'LINE_SAMPLES', startindex, $
        endindex, 1, 1))
    ; if no data present in source image, then return
    if (img_lines eq 0 || img_linsmp eq 0) then return

    ; loop through each window sub object and check
    for j = 0, win_count - 1 do begin
        win_end = get_index(label, win_index[j]) ; get endindex for cur windows

        ; check for LINES keyword
        lines = extract_keyword(label, 'LINES', win_index[j], win_end, 1)
        test_integer, 'LINES', lines, 0, img_lines
        if (lines eq 0) then continue  ; if no data, then goto next iteration

        ; check for FIRST_LINE keyword
        first_line = extract_keyword(label,'FIRST_LINE',win_index[j],win_end,1)
        test_integer, 'FIRST_LINE', first_line, 1, img_lines

        ; check whether first-line and lines keyword values are in range
        line_range = long(first_line) + long(lines) - 1
        if (line_range gt img_lines) then begin
            message, "Error: FIRST_LINE ("+first_line+") and LINES ("+lines+$
                ") values must be within source image's LINES ("+$
                clean(string(img_lines), /space) + ")."
        endif

        ; check for LINE_SAMPLES keyword
        line_smp = extract_keyword(label,'LINE_SAMPLES',win_index[j],win_end,1)
        test_integer, 'LINE_SAMPLES', line_smp, 0, img_linsmp
        if (line_smp eq 0) then continue ; if no data, then goto next iteration

        ; check for FIRST_LINE_SAMPLE keyword
        first_line_smp = extract_keyword(label,'FIRST_LINE_SAMPLE', $
            win_index[j], win_end, 1)
        test_integer, 'FIRST_LINE_SAMPLE', first_line_smp,1,img_linsmp

        ; check whether first line sample and line samples are in range
        line_smp_range = long(first_line_smp) + long(line_smp) - 1
        if (line_smp_range gt img_linsmp) then begin
            message, "Error: FIRST_LINE_SAMPLE ("+first_line_smp+") and "+ $
                "LINE_SAMPLES ("+line_smp+") values must be within source "+$
                "image's LINE_SAMPLES ("+clean(string(img_linsmp),/space)+")."
        endif
    endfor
end

;- level 0 -------------------------------------------------------------------

function verify_image, label
    ; error check
    on_error, 1      ; return to main level

    ; extract all object keywords
    objects = pdspar(label, 'OBJECT', COUNT=objcnt, INDEX=objindx)
    objects = strtrim(objects)
    images = stregex(objects, 'IMAGE$', /boolean) ; look for image objects
    image_pos = where(images eq 1, matches)       ; find matches
    if (matches eq 0) then return, 1              ; if none, then return 1

    ; select image objects from all the objects
    objects = objects[image_pos]                  ; store object names
    index = objindx[image_pos]                    ; store object indices
    count = n_elements(objects)                   ; store number of objects

    ; check each image object
    for i = 0, count - 1 do begin
        ;; determine the end index for current object
        endindex = get_index(label, index[i])
        ; verify required keywords
        check_image_required, label, index[i], endindex
        ; verify optional keywords
        check_image_optional, label, index[i], endindex
        ; verify window object and keywords if present
        check_window, label, index[i], endindex
    endfor

    return, 1
end
