;;----------------------------------------------------------------------------
;; Name: VERIFY_ELEM
;;
;; Purpose: To verify the PDS ELEMENT object in the given label.
;;
;; Calling Sequence:
;;     result = verify_elem (label, startind, endind, byte_count)
;;
;; Input:
;;     label - string array containing the PDS label definitions for
;;             current object.
;;     startind - the index of the label array representing the start
;;                of the current element object
;;     endind - the index of the label array representing the end of
;;              the current element object
;;     byte_count - a long integer variable tracking the bytes from
;;                  start of parent object; this is pased in by reference.
;;
;; Output:
;;     result - The routine returns 1 if there are no errors in the label.
;;     The procedure looks for errors in the current element object
;;     and croaks if it finds any and terminates program.
;;
;; Optional inputs: none.
;;
;; External routines: Extract_keyword, Clean, Remove, Test_integer
;;
;; Modification history:
;;     Written by Puneet Khetarpal, 30 June 2005
;;
;;----------------------------------------------------------------------------

;- level 1 -------------------------------------------------------------------

; precondition: the start and end indices are viable
; postcondition: the required element object keywords are verified
pro check_elem_required, label, startind, endind
    ; error check
    on_error, 1

    ; check for BYTES keyword
    bytes = extract_keyword(label, 'BYTES', startind, endind, 1)
    test_integer, 'BYTES', bytes, 1

    ; check for DATA_TYPE keyword
    data_type = extract_keyword(label, 'DATA_TYPE', startind, endind, 1)
    data_type = remove(data_type, '"')
    datarange = ["ASCII_COMPLEX","ASCII_INTEGER","ASCII_REAL","BIT_STRING", $
     "BINARY_CODED_DECIMAL","BOOLEAN","CHARACTER","COMPLEX","DATE","FLOAT", $
     "EBCDIC_CHARACTER","IBM_COMPLEX","IBM_INTEGER","IBM_REAL","IEEE_REAL", $
     "IBM_UNSIGNED_INTEGER","IEEE_COMPLEX","INTEGER","LSB_BIT_STRING","N/A", $
     "LSB_INTEGER","LSB_UNSIGNED_INTEGER","MAC_INTEGER","MAC_REAL","PC_REAL",$
     "MAC_UNSIGNED_INTEGER","MSB_BIT_STRING","MSB_INTEGER","PC_COMPLEX", $
     "MSB_UNSIGNED_INTEGER","PC_INTEGER","PC_UNSIGNED_INTEGER","SUN_COMPLEX",$
     "REAL","SUN_INTEGER","SUN_REAL","SUN_UNSIGNED_INTEGER","TIME","VAX_REAL",$
     "UNSIGNED_INTEGER","VAXG_COMPLEX","VAXG_REAL","VAX_BIT_STRING", $
     "VAX_COMPLEX","VAX_DOUBLE","VAX_INTEGER","VAX_UNSIGNED_INTEGER"]
    ; check if specified value is viable
    pos = where(stregex(datarange, data_type, /boolean) eq 1, matches)
    if (matches eq 0) then begin    ; if not, then issue error
        message, "Error: invalid DATA_TYPE value found - " + data_type
    endif

    ; check for NAME keyword
    name = extract_keyword(label, 'NAME', startind, endind, 1)
end

; precondition: the required keywords have been verified
; postcondition: the optional keywords are verified
pro check_elem_optional, label, startind, endind, byte_count
    ; error check
    on_error, 1

    ; check for START_BYTE keyword
    start_byte = extract_keyword(label, 'START_BYTE', startind, endind, 0)
    if (start_byte ne '###~') then begin
        test_integer, 'START_BYTE', start_byte, 1
        if (long(start_byte) le byte_count) then begin
            message, "Error: invalid START_BYTE position specified "+start_byte
        endif else if (long(start_byte) gt byte_count + 1) then begin
            byte_count += long(start_byte) - byte_count - 1
        endif
    endif

    ; check for BIT_MASK keyword
    bit_mask = extract_keyword(label, 'BIT_MASK', startind, endind, 0) 
    if (bit_mask ne '###~') then begin
        bytes = extract_keyword(label, 'BYTES', startind, endind, 1)
        expr = '2#[01]{'+clean(string(8 * long(bytes)), /space)+'}#'
        if (~ stregex(bit_mask, expr, /boolean)) then begin
            message, "Error: BIT_MASK value invalid - " + bit_mask
        endif
    endif

    ; check for FORMAT keyword
    format = extract_keyword(label, 'FORMAT', startind, endind, 0)
    if (format ne '###~') then begin
        expr = '[0-9]*[AFDEGI][0-9]+\.?[0-9]*'
        if (~ stregex(format, expr, /boolean)) then begin
            message, "Error: invalid FORMAT value specified - " + format
        endif
    endif

    ; check for OFFSET keyword
    offset = extract_keyword(label, 'OFFSET', startindex, endindex, 0)
    if (offset ne '###~') then begin ; if offset exists
        if (stregex(offset, '[^-\+\.eE0-9]',/boolean)) then begin
            message, "Error: OFFSET value must be numeric - " + offset
        endif
    endif

    ; check SCALING_FACTOR keyword
    scal_fact = extract_keyword(label, 'SCALING_FACTOR',startindex,endindex, 0)
    if (scal_fact ne '###~') then begin ; if scaling factor exists
        if (stregex(scal_fact, '[^-\+\.eE0-9]', /boolean)) then begin
            message, "Error: SCALING_FACTOR value must be numeric - "+scal_fact
        endif
    endif
end

;- level 0 -------------------------------------------------------------------

function verify_elem, label, startind, endind, byte_count
    ; error check
    on_error, 1
    ; check required keywords
    check_elem_required, label, startind, endind
    ; check optional keywords
    check_elem_optional, label, startind, endind, byte_count

    bytes = extract_keyword(label, 'BYTES', startind, endind, 1)
    byte_count += long(bytes)

    return, 1
end

