;;----------------------------------------------------------------------------
;; Name: VERIFY_ARR
;;
;; Purpose: To verify PDS ARRAY objects.
;;
;; Calling Sequence:
;;     result = verify_arr (label, startind, endind, byte_count)
;;
;; Input:
;;     label - string array containing the PDS label definitions for
;;             current object.
;;     startind - the index of the label array representing the start of
;;             the current array object to be processed
;;     endind - the index of the label array representing the end of
;;              the current array object.
;;     byte_count - a long integer variable tracking the bytes from
;;                  start of parent object; this is passed in by
;;                  reference
;;
;; Output:
;;     result - routine returns 1 if there are no errors in the label.
;;     The procedure looks for errors in the current array object and croaks
;;     if it finds any and terminates program.
;;
;; Optional inputs: none.
;;
;; External routines: Get_index, Verify_elem, Verify_col, Pdspar, 
;;     Test_integer, Extract_keyword, Break_string. 
;;
;; Modification history:
;;     Written by Puneet Khetarpal, 30 June 2005.
;;
;;----------------------------------------------------------------------------

;- level 1 -------------------------------------------------------------------

; precondition: the start and end indices are viable; total items is a 
;     integer to keep track of bytes
; postcondition: the required keywords for current array are verified;
;     total items value is determined from axis_items 
pro check_array_required, label, startind, endind, total_items
    ; error check
    on_error, 1

    ; check for AXES keyword
    axes = extract_keyword(label, 'AXES', startind, endind, 1, 1)
    test_integer, 'AXES', axes, 1, 6     ; test for integer, within 6 axes
    
    ; check for AXIS_ITEMS keyword
    axis_items = extract_keyword(label, 'AXIS_ITEMS', startind, endind, 1, 1)
    vals = break_string('AXIS_ITEMS', axis_items, long(axes)) ; separate
    for k = 0, axes - 1 do begin               ; test each one for integer
        test_integer, 'AXIS_ITEMS', vals[k], 1
        total_items *= long(vals[k])           ; multiply to get total items
    endfor

    ; check for NAME keyword
    name = extract_keyword(label, 'NAME', startind, endind, 1, 1)
end

; precondition: the required keywords have been checked, byte count keeps
;    track of total bytes for parent object
; postcondition: the optional keywords are verified; the difference between
;    start of this object and parent object is added to byte count for buffer
pro check_array_optional, label, startind, endind, byte_count
    ; error check
    on_error, 1

    ; check for START_BYTE keyword
    start_byte = extract_keyword(label, 'START_BYTE', startind, endind, 0, 1)
    if (start_byte ne '###~') then begin             ; if defined then
        test_integer, 'START_BYTE', start_byte, 1    ; test for integer
        ; check whether the start byte is interleaved with previous object
        if (long(start_byte) le byte_count) then begin  ; if so then error
            message, "Error: invalid START_BYTE position supplied "+start_byte
        endif else if (long(start_byte) gt byte_count + 1) then begin ; else
            byte_count += long(start_byte) - byte_count - 1  ; add difference
        endif
    endif
end

; precondition: there exists at least one sub object 
; postcondition: the object is checked and returned as a struct
function extract_arr_subobjects, label, startind, endind
    ; error checking
    on_error, 1

    ; extract all objects
    objects = pdspar(label, 'OBJECT', COUNT=objcnt, INDEX=objindx)
    objects = strtrim(objects, 2)
    pos = where(objindx gt startind and objindx lt endind, matches)
    if (matches eq 0) then begin    ; if no subobjects found, then error
        message, "Error: ARRAY object must have at least one subobject"
    endif
    ; store in a structure
    struct = {name:objects[pos], index:objindx[pos], count:matches}

    return, struct
end

;- level 0 -------------------------------------------------------------------

; precondition: label contains an array object, start and end index
;     are viable indices, and INTERCHANGE FORMAT keyword has already been
;     tested for all array objects; byte count is passed by reference to track
;     total bytes
; postcondition: the current level of array is verified, and if another
;     array or collection subobject is present, then it is called again.

function verify_arr, label, startind, endind, byte_count
    ; error check
    on_error, 1

    ; initialize local variable
    total_items = 1           ; total number of items
    local_count = 0           ; local data bytes count variable    
    counter = 0               ; var to track objects in loop

    ; check for required keywords
    check_array_required, label, startind, endind, total_items
    ; check for optional keywords
    check_array_optional, label, startind, endind, byte_count
    ; check for sub object(s)
    objstruct = extract_arr_subobjects(label, startind, endind)
    name = objstruct.name             ; store obj names
    index = objstruct.index           ; store indices
    count = objstruct.count           ; store count
    ; go through each of the sub objects and verify
    while (counter lt count) do begin
        endindex = get_index(label, index[counter])   ; get endindex
        ; find how many objects are contained within this object
        pos = where(index gt index[counter] and index lt endindex, matches)
        
        ; process the current object for element, array, and collection
        if (stregex(name[counter], 'ELEMENT$', /boolean)) then begin
            test = verify_elem(label, index[counter], endindex, local_count)
        endif else if (stregex(name[counter], 'ARRAY$', /boolean)) then begin
            test = verify_arr(label, index[counter], endindex, local_count)
        endif else if (stregex(name[counter],'COLLECTION$',/boolean)) then $
            begin
            test = verify_col(label, index[counter], endindex, local_count)
        endif else begin
            message, "Error: invalid OBJECT found in ARRAY object "+$
                name[counter] 
        endelse

        ; increment counter accordingly
        counter++                   ; increment once for current object
        if (matches gt 0) then begin
            counter += matches      ; skip over the objects within this one
        endif                       ; taken care of by recursion
    endwhile

    ; obtain total bytes for current object
    byte_count += total_items * local_count
 
    return, 1
end
