;;-----------------------------------------------------------------------------
;; Name: GET_INDEX
;;
;; Purpose: To obtain the END_OBJECT index position for a specified PDS object
;;          in a PDS label.
;;
;; Calling Sequence: 
;;    result = get_index (label, start index)
;;
;; Input:
;;    label - string array containing the object information according to
;;           PDS standards. The label must meet all PDS standards.
;;    start index - integer specifying the index of the OBJECT for which
;;                 the END_OBJECT index is desired. Start index must be
;;                 a valid object index in the label string array.
;;
;; Output:
;;    result - a scalar integer specifying the END_OBJECT index in the label
;;            string array for which the start index was specified.
;;
;; Optional Input: none
;;
;; Examples:
;;    To obtain the index of the table array starting at label index 7:
;;    IDL> label = HEADPDS ("TABLE.LBL", /SILENT)
;;    IDL> index = GET_INDEX (label, 7)
;;    IDL> help, index
;;    INDEX          LONG      =           60
;;
;; External routines: Clean, Pdspar
;;
;; Modification history:
;;    Written by Puneet Khetarpal, 24 February 2004
;;    For a complete list of modifications, see changelog.txt file.
;;
;;-----------------------------------------------------------------------------

;-- level 0 ------------------------------------------------------------------

function get_index, label, start_index
    ; error protection:
    on_error, 1
   
    ; check number of arguments:
    if (n_params() lt 2) then begin
        message, "Syntax: result = get_index (label, startindex)"
    endif

    ; extract all objects and their indices in the label
    obj_names = pdspar(label, 'object', COUNT=obj_count, INDEX=obj_index)
    ;; extract all end object keywords values and indices
    eobj_names = pdspar(label, 'end_object', COUNT=eobj_cnt, INDEX=eobj_indx)
   
    ; determine the position of start index in obj_index array
    pos = where(obj_index eq start_index, matches)
    if (matches eq 0) then begin
        print, "Error: invalid start index value provided in label."
        return, -1
    endif
    
    ; initialize loop variables
    count = 0                     ; counter for the object stack
    i = 0                         ; counter for the indices in label 
    found = 0                     ; variable to flag whether endIndex is found
    stack = intarr(obj_count)     ; stack to track the object indices
    end_index = -1                ; var to store endIndex value for object

    ;; go through each index of the label until its end or not found 
    while (i lt n_elements(label) && ~ found) do begin
        ; check whether current counter matches object index value  
        pos = where (obj_index eq i, matchcnt)
        if (matchcnt gt 0) then begin              ; if matched
            stack[count] = obj_index[pos[0]]          ; then push onto stack
            count++                                ; increment counter
        endif else begin                           ; else
            ; check whether current counter matches 
            pos = where (eobj_indx eq i, matchcnt) 
            if (matchcnt gt 0) then begin          ; if matched
                ; check whether the start index matches top of stack value
                if (stack[count-1] eq start_index) then begin
                    end_index = eobj_indx[pos[0]]     ; store endIndex value
                    found++                        ; we have found endIndex
                endif
                count--                            ; decrement count
            endif
        endelse
        i++                                        ; increment label index
    endwhile
    
    return, end_index
end
