;+
; NAME:
;       FILE_TOK
;
; PURPOSE:
;       This function can be used to extract several parts of the given
;       filename. If no keyword is specified, the original filename is
;       returned. Operating system dependencies are taken into
;       consideration. The VMS version number is stripped in any case.
;
; CATEGORY:
;       File Management.
;
; CALLING SEQUENCE:
;       Result = FILE_TOK(File)
;
; INPUTS:
;       File:   A string containing the filepath.
;
; OPTIONAL INPUTS:
;       None
;
; KEYWORD PARAMETERS:
;       PATH:   Set this keyword to extract the path of the file.
;
;       NAME:   Set this keyword to extract the name (without extension)
;               of the given file.
;
;       EXT:    Set this keyword to extract the file extension.
;
; OUTPUTS:
;       This function returns a string containing the selected parts of the
;       given filepath.
;
; EXAMPLE:
;       Remove extension from the given filename:
;
;       sFile = FILE_TOK(sFile, /PATH, /NAME)
;
; MODIFICATION HISTORY:
;       Written by:     Harald Jan Jeszenszky, 12 Dec 96
;       10 Feb 1997     Modified to use multiple keywords.
;-

FUNCTION file_tok, sFile, NAME=isName, PATH=isPath, EXT=isExt

    CATCH, error
    IF (error NE 0) THEN BEGIN
        MESSAGE, !ERROR_STATE.MSG, /INFORMATIONAL
        RETURN, ''
    ENDIF

    IF (N_PARAMS() NE 1) THEN $
        MESSAGE, 'Missing paramter.', /NONAME, /NOPRINT
    asFile = STRING(sFile)

    isPath = KEYWORD_SET(isPath)
    isName = KEYWORD_SET(isName)
    isExt  = KEYWORD_SET(isExt)
    IF (isPath+isName+isExt EQ 0) THEN BEGIN
        isPath = 1B
        isName = 1B
        isExt  = 1B
    ENDIF

CASE (!VERSION.OS_FAMILY) OF
  'Windows' : BEGIN
    sDel = '\'
  END
  'unix' : BEGIN
    sDel = '/'
  END
  ELSE : BEGIN
    sDel = ']'
  END
ENDCASE

    nFiles = N_ELEMENTS(asFile)
    IF (nFiles EQ 1) THEN $
        asFile = [asFile]
    iDel = STRPOS(asFile, sDel, /REVERSE_SEARCH)
    idx = WHERE(iDel EQ -1L, cnt)
    IF (cnt EQ nFiles) THEN $
        iDel = STRPOS(asFile, ':', /REVERSE_SEARCH)
    iDot = STRPOS(asFile, '.', /REVERSE_SEARCH)
    idx = WHERE((iDot NE -1L) AND (iDot LT iDel), cnt)
    IF (cnt EQ nFiles) THEN $
        iDot[*] = -1L

    sPath = STRARR(nFiles)
    IF KEYWORD_SET(isPath) THEN BEGIN
        idx = WHERE(iDel NE -1L, cnt)
        IF (cnt GT 0) THEN BEGIN
            sPath[idx] = STRMID(asFile[idx], 0, TRANSPOSE(iDel[idx] + 1))
        ENDIF
    ENDIF

    sName = STRARR(nFiles)
    IF KEYWORD_SET(isName) THEN BEGIN
        sName = STRMID(asFile, TRANSPOSE(iDel + 1), $
                          TRANSPOSE(STRLEN(asFile)-iDel+1))
        idx = WHERE(iDot NE -1L, cnt)
        IF (cnt GT 0) THEN BEGIN
            sName[idx] = STRMID(sName[idx], 0, $
                                TRANSPOSE(STRPOS(sName[idx], '.', /REVERSE_SEARCH)))
        ENDIF
    ENDIF

    sExt = STRARR(nFiles)
    IF KEYWORD_SET(isExt) THEN BEGIN
        idx = WHERE(iDot NE -1L, cnt)
        IF (cnt GT 0) THEN BEGIN
            sExt[idx] = STRMID(asFile[idx], TRANSPOSE(iDot[idx]), $
                               TRANSPOSE(STRLEN(asFile[idx])-iDot[idx]))
            iPos = STRPOS(sExt, ';', /REVERSE_SEARCH)   ; strip version number (vms only)
            idx = WHERE(iPos NE -1L, cnt)
            IF (cnt GT 0) THEN BEGIN
                sExt[idx] = STRMID_JHJ(sExt[idx], 0, TRANSPOSE(iPos[idx]))
            ENDIF
        ENDIF
    ENDIF

    sToken = sPath + sName + sExt
    IF (nFiles EQ 1) THEN $
        sToken = sToken[0]

    RETURN, sToken

END ; file_tok
