;;----------------------------------------------------------------------------
;; Name: EXTRACT_KEYWORD
;;
;; Purpose: To extract the value of a PDS keyword from PDS label within
;;     a given range of label indices
;;
;; Calling Sequence: 
;;     value = extract_keyword (label, keyword, start, endindex, flag, m_flag) 
;;
;; Input:
;;     label - a string array containing the PDS label definitions
;;     keyword - the PDS keyword name to be searched for in the label
;;     start - the starting index for Label to start the search from
;;     endindex - the end index for Label to end the search for keyword
;;     flag - this is a boolean deciding whether the keyword is required
;;     m_flag - this boolean decides whether multiple instances of keyword
;;         are allowed in the indices range; optional parameter,
;;         default is 0 (i.e., multiple instances not allowed for keyword)
;;
;; Output:
;;     value: a scalar string containing the value of the keyword; if multiple
;;         instances of keyword are allowed, then the first value is returned.
;;         If keyword is optional, and not present in range, then
;;         returns '###~' to indicate not present. If keyword is
;;         required, and none found, then issues an error and halts program.
;; 
;; External routines: Pdspar, Clean
;;
;; Modification history:
;;     Written by Puneet Khetarpal, 30 June 2005
;;
;;-----------------------------------------------------------------------------

; precondition: label has been tested for specified objects, keyword name
;     is a viable one, and start and end index specify the start and
;     end of current object; an integer representing either 1 or 0 to
;     specify whether the keyword is a required one or not for the
;     object; m_flag determines whether the keyword can have multiple values
;     or not and is not required to be in, if not specified, then default
;     is 0.
; postcondition: the label is searched for the keyword and its value
;     is returned, if found.
function extract_keyword, label, keyword, start, endindex, flag, m_flag
    ; error checking
    on_error, 1

    ; initialize variable
    value = '###~'
    default = value

    ; extract from label
    value = pdspar(label, keyword, COUNT=count, INDEX=index)
    if (flag && count eq 0) then begin ; if required and no value found
        message, "Error: required keyword " + keyword + " not found in label."
    endif else if (~ flag && count eq 0) then begin  ; if no value found 
        return, default                     ; return '###~'
    endif else begin                        ; else
        ; find position in index array between start and endindex
        pos = where(start le index and endindex ge index, matches)
        ; if required keyword and no matches found then issue error
        if (flag && matches eq 0) then begin
            message, "Error: required keyword "+keyword+" not found in label."
        endif else if (~ flag && matches eq 0) then begin ; not req and not fnd
            return, default                  ; then return default value
        endif else if (n_params() eq 5 && matches gt 1) then begin
            ;; if multiple flag not present and multiple matches found
            return, "Error: multiple "+keyword+" keywords found." ; issue err
        endif else begin   ; else clean and return the first value in list
            value = clean(value[pos[0]], /space)
        endelse
    endelse

    return, value
end
