;;-----------------------------------------------------------------------------
;; Name: ELEM_STRUCT
;;
;; Purpose: To construct an IDL structure for the PDS ELEMENT object
;;
;; Calling Sequence: 
;;     struct = elem_struct (label, index, endindex, byte_count)
;;
;; Input:
;;     label - string array containing the PDS label definitions for
;;         current object 
;;     index - the index of the label array representing the start of
;;         the current element object to be processed
;;     endindex - the index of the label array representing the end of
;;         the current element object
;;     byte_count - a long integer variable tracking the bytes from
;;         start of the parent object; this is passed in by reference
;;
;; Output:
;;     struct: an IDL structure containing the definition of current
;;         element object; if no buffer needed, then only a scalar
;;         element is returned
;;
;; External routines: Get_idl_type, Remove, Extract_keyword
;;
;; Modification history:
;;     Written by Puneet Khetarpal, 30 June 2005
;;
;;-----------------------------------------------------------------------------

;- level 1 --------------------------------------------------------------------

; precondition: none.
; postcondition: the element keywords are extracted and returned
function obtain_element_keywords, label, index, endindex
    ; obtain bytes keyword
    bytes = extract_keyword(label, 'BYTES', index, endindex, 1)
    bytes = long(bytes)
    ; obtain data type keyword
    data_type = extract_keyword(label, 'DATA_TYPE', index, endindex, 1)
    data_type = remove(data_type, '"')
    ; obtain name keyword
    name = extract_keyword(label, 'NAME', index, endindex, 1)
    name = remove(name, '"')
    name = idl_validname(name, /convert_all)
    ; obtain start byte keyword
    start_byte = extract_keyword(label, 'START_BYTE', index, endindex, 0)
    start_byte = (start_byte eq '###~') ? 1 : long(start_byte)
    ; obtain bit_mask keyword
    bit_mask = extract_keyword(label, 'BIT_MASK', index, endindex, 0)
    bit_mask = (bit_mask eq '###~') ? '0' : bit_mask
    ; obtain offset keyword
    offset = extract_keyword(label, 'OFFSET', index, endindex, 0)
    offset = (offset eq '###~') ? 0.0 : float(offset)
    ; obtain scaling factor
    scal_fact = extract_keyword(label, 'SCALING_FACTOR', index, endindex, 0)
    scal_fact = (scal_fact eq '###~') ? 1.0 : float(scal_fact)
    ; add to structure
    keys = {bytes:bytes,data_type:data_type, name:name, start_byte:start_byte,$
            bit_mask:bit_mask, offset:offset, scal_fact:scal_fact}
    return, keys
end

;- level 0 --------------------------------------------------------------------

; precondition: this routine has been invoked either by collection or array
;     structure, and index and endindex contain range of label indices for
;     current element object.
; postcondition: the element object structure is created and returned

function elem_struct, label, index, endindex, byte_count
    ; error protection
    on_error, 1

    ; obtain element keywords
    keywords = obtain_element_keywords(label, index, endindex)
    ; get idl type for current element
    idltype = get_idl_type(keywords.data_type, keywords.bytes, 'BINARY')
    ; construct element
    element = fix(0, type = idltype)
    ; if start byte > byte count + 1, then add buffer to element structure
    if (keywords.start_byte gt byte_count + 1) then begin
        diff = keywords.start_byte - byte_count - 1   ; compute difference
        ; add buffer to structure
        struct = create_struct('buffer', bytarr(diff), keywords.name, element)
        byte_count += diff             ; increment byte count by difference
    endif else begin                   ; else just store element
        struct = element
    endelse
    ; increment byte count for the number of bytes
    byte_count += keywords.bytes

    return, struct
end
