;;-----------------------------------------------------------------------------
;; Name: COL_STRUCT
;;
;; Purpose: To construct an IDL structure for the PDS COLLECTION object
;;
;; Calling Sequence: 
;;   struct = col_struct (label, index, endindex, byte_count)
;;
;; Input:
;;     label - string array containing the PDS label definitions for
;;         current object
;;     index - the index of label array representing the start of the current
;;         collection object to be processed
;;     endindex - the index of the label array representing the end of
;;         the current collection object.
;;     byte_count - a long integer variable tracking the bytes from
;;         start of the parent object; this is passed in by reference.
;;
;; Output: 
;;     struct: an IDL structure containing the definition of current
;;         collection object
;;
;; External routines: Arr_struct, Elem_struct, Extract_col_subobjects
;;     (from verify_col.pro), Get_index, Extract_keyword, Remove, Break_string
;; 
;; Modification history:
;;     Written by Puneet Khetarpal, 30 June 2005
;;
;;-----------------------------------------------------------------------------

;- level 1 --------------------------------------------------------------------

; precondition: none.
; postcondition: collection keywords are extracted from the label
function obtain_collection_keywords, label, index, endindex
    ; obtain name keyword
    name = extract_keyword(label, 'NAME', index, endindex, 1, 1)
    name = remove(name, '"')                    ; remove quotes
    name = idl_validname(name, /convert_all)    ; convert to idl variable name
    ; obtain bytes keyword
    bytes = extract_keyword(label, 'BYTES', index, endindex, 1, 1)
    bytes = long(bytes)
    ; obtain optional start byte keyword
    start_byte = extract_keyword(label, 'START_BYTE', index, endindex, 1, 1)
    start_byte = (start_byte eq '###~') ? 1L : long(start_byte)
    ; create a structure
    keys = {name:name, start_byte:start_byte, bytes:bytes}
    return, keys
end

;- level 0 --------------------------------------------------------------------

; precondition: label contains collection defintion; index and
;     endindex define range of the current collection object, and byte count
;     tracks the total bytes for the entire object
; postcondition: the collection object structure is created and returned

function col_struct, label, index, endindex, byte_count
    ; error protection
    on_error, 1

    ; initialize local variables
    local_count = 0          ; local data bytes count variable
    counter = 0              ; variable to track objects in loop

    ; obtain collection keywords
    keywords = obtain_collection_keywords(label, index, endindex)

    ; initialize structure with any byte count separation
    if (keywords.start_byte gt byte_count + 1) then begin
        diff = keywords.start_byte - byte_count - 1
        byte_count += diff
        struct = {buffer:bytarr(diff)}
    endif
    localstruct = 0          ; local structure for the while loop

    ; extract sub objects from label
    objstruct = extract_col_subobjects(label, index, endindex)
    name = objstruct.name                ; store obj names
    subindex = objstruct.index           ; store indices
    count = objstruct.count              ; store obj count    

   ; go through each of the subobjects and construct the sub structure
   while (counter lt count) do begin
        subendind = get_index(label, subindex[counter])   ; get endindex
        ; find how many objects are contained within this object
        pos = where(subindex gt subindex[counter] and subindex lt subendind, $
              matches)
        ; obtain name of current sub object
        subname = extract_keyword(label,'NAME',subindex[counter],subendind,1,1)
        subname = remove(subname, '"')
        subname = idl_validname(subname, /convert_all)   ; convert to valid
 
        ; process the current object for element, array, and collection
        if (stregex(name[counter], 'ELEMENT *',/boolean)) then begin
            substruct = elem_struct(label, subindex[counter], subendind, $
                local_count)
        ;; else if array object, then process array structure
        endif else if (stregex(name[counter], 'ARRAY *', /boolean)) then begin
            substruct = arr_struct(label, subindex[counter], subendind, $
                local_count)
        endif else begin    ; process collection object structure  
            substruct = col_struct(label, subindex[counter], subendind, $
                local_count)
        endelse

        ; add to local structure the current structure
        if (size(localstruct, /type) eq 8) then begin  ; if already defined
            localstruct = create_struct(localstruct, subname, substruct)
        endif else begin  ; else create a new local structure variable
            localstruct = create_struct(subname, substruct)
        endelse

        ;; increment counter
        counter++
        if (matches gt 0) then begin     ; if sub objects found then
            counter += matches           ; skip the sub objects
        endif
   endwhile

   ; update byte count
   byte_count += local_count
   ; add to main structure if already defined, else define a new structure
   struct =  (size(struct, /type) eq 8) ? create_struct(struct, keywords.name,$
              localstruct) : create_struct(keywords.name, localstruct)

   return, struct
end
