;;-----------------------------------------------------------------------------
;; Name: ARRCOLPDS
;; 
;; Purpose: To read an ARRAY or COLLECTION object from the given filename
;; 
;; Calling Sequence: 
;;     result = arrcolpds (filename, label, index, [/SILENT])
;; 
;; Input:
;;     filename - the name of the label file
;;     label - string array containing the contents of the PDS label
;;     index - an integer specifying the index in label array where the
;;         array/collection object starts
;; Output:
;;     result - an IDL structure containing the array/collection object read
;;         read from filename.
;;
;; Optional Input:
;;     SILENT - this keyword suppresses all processing information to be
;;         displayed on the screen. 
;;
;; Examples:
;;     Given a PDS header label, file and index for an ARRAY object
;;
;;     IDL> result = arrcolpds ('arr.lbl', label, 5)
;;     Now reading NEW_ARRAY ARRAY/COLLECTION object
;;     ** Structure <821a754>, 2 tags, length=124, data length=97, refs=1:
;;     OBJECTS         INT              1
;;     NEW_ARRAY       STRUCT    -> <Anonymous> Array[1]
;;
;; External routines: Get_index, Verify_label, Extract_Keyword, Pointpds, 
;;     Arr_struct, Col_struct
;;
;; Modification history: 
;;     Written by Puneet Khetarpal, 30 June 2005
;;
;;-----------------------------------------------------------------------------

;- level 1 -------------------------------------------------------------------

; precondition: the label has been verified, and index and endindex
;     are range of indices for start and end of the current object
; postcondition: the structure for the array/collection object is
;     constructed for reading and returned
function obtain_arrcol_struct, label, index, endindex
    ; initialize byte count
    byte_count = 0

    ; branch the arrcol structure creation process
    if (stregex(label[index], 'ARRAY *', /boolean)) then begin
        struct = arr_struct(label, index, endindex, byte_count)
    endif else begin
        struct = col_struct(label, index, endindex, byte_count)
    endelse

    return, struct
end

; precondition: the array/collection data structure has been created and
;     architecture and pointer contain architecture of data file and 
;     pointer information for the data
; postcondition: the array/collection data is read from the file
function read_arrcol_data, pointer, struct, arch
    ; error checking
    on_ioerror, signal

    ; initialize variable and declare flag
    data_read = {flag:1}    ; 0: error, 1: no error
    ; open the file to be read and apply swap endian if needed
    if (arch eq 'MSB') then begin
        openr, unit, pointer.datafile, /get_lun, /swap_if_little_endian
    endif else begin
        openr, unit, pointer.datafile, /get_lun, /swap_if_big_endian
    endelse
    ; set the file pointer to current object to be read
    point_lun, unit, pointer.skip
    ; read the array/collection object into the structure
    readu, unit, struct
    ; close the unit and free it
    close, unit
    free_lun, unit
    ; add to structure
    data_read = create_struct(data_read, 'struct', struct)
    ; return
    return, data_read
    signal:
        on_ioerror, null
        print, "Error: file either corrupted or invalid parameters specified"
        data_read.flag = 0
        return, data_read
end

;- level 0 -------------------------------------------------------------------

; precondition: the filename and label are viable variables for file
;     and label for PDS file, index is a viable index of the label
;     array containing the name of the object to be currently processed
; postcondition: the array or collection object at the given index is
;     read from filename and returned
function arrcolpds, filename, label, index, SILENT=silent
    ; error protection
    on_error, 1

    ; check for number of parameters in function call
    if (n_params() lt 3) then begin
        message, "Syntax Error: arrcol = arrcolpds(filename, label, index "+$
            "[, /SILENT])"
    endif
    silent = keyword_set(silent)

    ; verify label
    res = verify_label(label, filename)

    ; check for current array or collection object index
    if (~ stregex(label[index], '(ARRAY)|(COLLECTION)', /boolean)) then begin
        print, "Error: invalid index specified for ARRAY/COLLECTION object "+$
             string(index)
        return, 0
    endif

    ; get endindex value for current arrcol object
    endindex = get_index(label, index)
    ; obtain structure to be read from file
    struct = obtain_arrcol_struct(label, index, endindex)
    ; obtain a data type keyword from the current object
    data_type = extract_keyword(label, 'DATA_TYPE', index, endindex, 1, 1)
    arch = (stregex(data_type, '(LSB)|(VAX)|(PC)', /boolean)) ? 'LSB' : 'MSB'
    ; obtain name of current object
    name = stregex(label[index], '= +([0-9A-Z_]+)', /extract, /subexpr)
    ; obtain pointer information
    pointer = pointpds(label, filename, name[1])
    ; if not silent then inform user
    if (~ silent) then begin
        print, "Now reading " + name[1] + " ARRAY/COLLECTION object"
    endif
    ; read data from file
    data_read = read_arrcol_data(pointer, struct, arch)
    if (~ data_read.flag) then return, 0

    ; perform data conversion if necessary
    return, data_read.struct
end
