;;----------------------------------------------------------------------------
;; Name: VERIFY_QUBE
;;
;; Purpose: To verify a PDS QUBE object and its WINDOW subobjects.
;;
;; Calling Sequence:
;;     result = verify_qube (label)
;;
;; Input:
;;     label - a string array containing PDS header.
;;
;; Output:
;;     result - the routine returns 1 if there are no errors in label.
;;     The routine outputs an error message if there are any errors
;;     found in the PDS label for QUBE objects.
;;
;; Optional inputs: none.
;;
;; External routines: Pdspar, Clean, Remove, Extract_keyword,
;;     Test_integer, Get_index, Break_string
;;
;; Modification history:
;;     Written by Puneet Khetarpal, 30 June 2005
;;
;;----------------------------------------------------------------------------

;- level 1 -------------------------------------------------------------------

; precondition: the label contains qube objects, and start and end
;     indices are viable
; postcondition: the label is verified for the required qube objects
pro check_qube_required, label, startind, endind
    ; error check
    on_error, 1         ; return to main level

    ; check for AXES keyword
    axes = extract_keyword(label, 'AXES', startind, endind, 1)
    test_integer, 'AXES', axes, 1, 6
    
    ; check for AXIS_NAME keyword
    axis_name = extract_keyword(label, 'AXIS_NAME', startind, endind, 1)
    vals = break_string('AXIS_NAME', axis_name, long(axes)) ; extract values
    ; check whether all values match LINE, BAND, or SAMPLE
    pos = where(stregex(vals, '(LINE)|(BAND)|(SAMPLE)',/boolean) eq 1, matches)
    if (matches lt long(axes)) then begin
        message, "Error: AXIS_NAME values must be BAND, LINE, OR SAMPLE - " + $
            axis_name
    endif

    ; check for CORE_ITEMS
    core_items = extract_keyword(label, 'CORE_ITEMS', startind, endind, 1)
    vals = break_string('CORE_ITEMS', core_items, long(axes)) ; extract values
    for j = 0, long(axes)-1 do begin
        test_integer, 'CORE_ITEMS', vals[j], 1, 5000
    endfor

    ; check for CORE_ITEM_BYTES
    core_item_byt = extract_keyword(label,'CORE_ITEM_BYTES',startind,endind,1)
    test_integer, 'CORE_ITEM_BYTES', core_item_byt, 1, 4
    if (long(core_item_byt) eq 3) then begin ; only values allowed: 1, 2, 4
        message, "Error: invalid CORE_ITEM_BYTES value specified - "+$
            core_item_byt
    endif

    ; check for CORE_ITEM_TYPE
    core_item_typ = extract_keyword(label,'CORE_ITEM_TYPE',startind, endind, 1)
    expr = '(UNSIGNED_INTEGER)|(INTEGER)|(REAL)'
    if (~ stregex(core_item_typ, expr, /boolean)) then begin
        message, "Error: invalid CORE_ITEM_TYPE value - " + core_item_typ
    endif

    ; check for CORE_BASE keyword
    core_base = extract_keyword(label, 'CORE_BASE', startind, endind, 1)
    if (stregex(core_base, '[^-\+\.eE0-9]', /boolean)) then begin
        message, "Error: CORE_BASE value must be numeric - " + core_base
    endif

    ; check for CORE_MULTIPLIER keyword
    core_mul = extract_keyword(label, 'CORE_MULTIPLIER', startind, endind, 1)
    if (stregex(core_mul, '[^-\+\.eE0-9]', /boolean)) then begin
        message, "Error: CORE_MULTIPLIER value must be numeric - " + core_mul
    endif
    
    ; check for SUFFIX_BYTES keyword
    suffix_bytes = extract_keyword(label, 'SUFFIX_BYTES', startind, endind, 1)
    test_integer, 'SUFFIX_BYTES', suffix_bytes, 1
    if (long(suffix_bytes) ne 4) then begin
        message, "Error: SUFFIX_BYTES value must always be 4 - "+suffix_bytes
    endif

    ; check for SUFFIX_ITEMS keyword
    suffix_items = extract_keyword(label, 'SUFFIX_ITEMS', startind, endind, 1)
    vals = break_string('SUFFIX_ITEMS', suffix_items, long(axes)) ; extract
    for j = 0, long(axes) -1 do begin
        test_integer, 'SUFFIX_ITEMS', vals[j], 0, 512
    endfor

end

;- level 0 -------------------------------------------------------------------

function verify_qube, label
    ; error checking
    on_error, 1    ; return to main level

    ; extract all object keywords
    objects = pdspar(label, 'OBJECT', COUNT=objcnt, INDEX=objindx)
    objects = strtrim(objects, 2)
    ; search for qube objects among all objects
    pos = where(stregex(objects, 'QUBE$', /boolean) eq 1, matches)
    if (matches eq 0) then return, 1    

    ; store qube objects
    objects = objects[pos]         ; store object names
    index = objindx[pos]           ; store object indices
    count = matches

    ; go through all qube objects and verify
    for i = 0, count - 1 do begin
        ;; extract end index for current qube object
        endindex = get_index(label, index[i])
        ; verify required objects
        check_qube_required, label, index[i], endindex
    endfor

    return, 1
end
