;;----------------------------------------------------------------------------
;; Name: VERIFY_COL
;;
;; Purpose: To verify the PDS COLLECTION object in the given label.
;;
;; Calling Sequence:
;;     result = verify_col (label, startind, endind, byte_count)
;;
;; Input:
;;     label - string array containing the PDS label definitions for
;;             current object.
;;     startind - the index of the label array representing the start of
;;                the current collection object.
;;     endind - the index of the label array representing the end of
;;              the current collection object.
;;     byte_count - a long integer variable tracking the bytes from
;;                  start of parent object; this is passed in by reference
;; 
;; Output:
;;     result - the routine returns 1 if there are no errors in label.
;;     The procedure looks for errors in the current collection object
;;     and croaks if it finds any and terminates program.
;;
;; Optional inputs: none.
;;
;; External routines: Get_index, Verify_elem, Verify_arr, Pdspar, 
;;     Test_integer, Extract_keyword, Break_string.
;;
;; Modification history:
;;     Written by Puneet Khetarpal, 30 June 2005
;;
;;----------------------------------------------------------------------------

;- level 1 -------------------------------------------------------------------

; precondition: the start and end indices are viable; total bytes is
;     set to 0 to be used later for error checking
; postcondition: the required keywords for current collection are verified;
;     total bytes value is set to value extracted from BYTES keyword.
pro check_col_required, label, startind, endind, total_bytes
    ; error check
    on_error, 1

    ; check NAME keyword
    name = extract_keyword(label, 'NAME', startind, endind, 1, 1)
    
    ; check BYTES keyword
    bytes = extract_keyword(label, 'BYTES', startind, endind, 1, 1)
    test_integer, 'BYTES', bytes, 1          ; test for integer
    total_bytes = long(bytes)                ; store value in total bytes
end

; precondition: the start and end indices are viable; required
;     keywords have been tested; byte count tracks count for parent object
; postcondition: the optional keywords are tested; start byte is
;     checked for buffer between parent object and current object, and
;     the difference is added to byte count
pro check_col_optional, label, startind, endind, byte_count
    ; error check
    on_error, 1

    ; check for START_BYTE keyword
    start_byte = extract_keyword(label, 'START_BYTE', startind, endind, 0, 1)
    if (start_byte ne '###~') then begin         ; if start byte is present
        test_integer, 'START_BYTE', start_byte, 1   ; test for integer
        ; if start byte is interleaved with previous object, then error
        if (long(start_byte) le byte_count) then begin
            message, "Error: invalid START_BYTE position supplied "+start+byte
        endif else if (long(start_byte) - byte_count - 1) then begin
            byte_count += long(start_byte) - byte_count - 1  ; else add buffer
        endif
    endif
end

; precondition: there exists at least one sub object 
; postcondition: the object is checked and returned as a struct
function extract_col_subobjects, label, startind, endind
    ; error checking
    on_error, 1

    ; extract all objects from label
    objects = pdspar(label, 'OBJECT', COUNT=objcnt, INDEX=objindx)
    objects = strtrim(objects, 2)
    pos = where(objindx gt startind and objindx lt endind, matches)
    if (matches eq 0) then begin   ; if no subobject found, then issue err
        message, "Error: COLLECTION object must have at least one subobject"
    endif
    ; store in a structure
    struct = {name:objects[pos], index:objindx[pos], count:matches}

    return, struct
end

;- level 0 -------------------------------------------------------------------

; precondition: label contains a collection object, start and end index
;     are viable indices, and INTERCHANGE_FORMAT keyword has already been
;     tested for all array objects; byte count is passed by reference
;     to track total bytes
; postcondition: the current level of collection is verified, and if
;     another array or collection subobject is present, then it is
;     called again

function verify_col, label, startind, endind, byte_count
    ; error check
    on_error, 1

    ; initialize local variable
    total_bytes = 0       ; collection object bytes as defined by BYTES
    counter = 0           ; counter for loop objects 
    local_count = 0       ; data byte count variable for current object

    ; check required keywords
    check_col_required, label, startind, endind, total_bytes
    ; check optional keywords
    check_col_optional, label, startind, endind, byte_count
    ; extract sub object(s) from label
    objstruct = extract_col_subobjects(label, startind, endind)
    name = objstruct.name           ; store obj names
    index = objstruct.index         ; store indices
    count = objstruct.count         ; store count
    ; go through each of the sub objects and verify
    while (counter lt count) do begin
        endindex = get_index(label, index[counter])    ; get endindex
        ; get number of objects within current object
        pos = where(index gt index[counter] and index lt endindex, matches)
        
        ; process the current object for element, array, or collection
        if (stregex(name[counter], 'ELEMENT$', /boolean)) then begin
            test = verify_elem(label, index[counter], endindex, local_count)
        endif else if (stregex(name[counter], 'ARRAY$', /boolean)) then begin
            test = verify_arr(label, index[counter], endindex, local_count)
        endif else if (stregex(name[counter],'COLLECTION$',/boolean)) then $
            begin
            test = verify_col(label, index[counter], endindex, local_count)
        endif else begin
            message, "Error: invalid OBJECT found in COLLECTION object "+$
                name[counter]
        endelse

        ; increment counter accordingly
        counter++                      ; increment once for current object
        if (matches gt 0) then begin 
            counter += matches         ; skip over objects within current obj
        endif                          ; taken care of during recursion
    endwhile

    ; if the local byte count does not match to that specified by BYTES
    if (local_count ne total_bytes) then begin    ; then issue error
        message, "Error: inconsistent BYTES value for COLLECTION object "+$
            clean(string(total_bytes), /space)
    endif
    ; add total bytes to byte count if everything went fine
    byte_count += total_bytes

    return, 1
end
