;;----------------------------------------------------------------------------
;; Name: VERIFY_ARRCOL
;;
;; Purpose: To verify all ARRAY/COLLECTION PDS objects in the label array
;;
;; Calling Sequence: 
;;     result = verify_arrcol (label)
;;
;; Input:
;;     label - a string array containing PDS header
;;
;; Output:
;;     result - the routine returns a value of 1, if there are no
;;              errors in the label. The routine outputs an error
;;              message if there are any errors found in the PDS label
;;              for ARRAY/COLLECTION objects.
;;
;; Optional Inputs: none.
;;
;; External routines: Pdspar, Get_index, Remove, Verify_arr, Verify_col
;;
;; Modification history:
;;     Written by Puneet Khetarpal, 30 June 2005
;;
;;----------------------------------------------------------------------------

;- level 0 -------------------------------------------------------------------

function verify_arrcol, label
    ; error check
    on_error, 1

    ; get all objects from label
    objects = pdspar(label, 'OBJECT', COUNT=objcnt, INDEX=objindx)
    objects = strtrim(objects, 2)
    pos = where(stregex(objects,'(ARRAY$)|(COLLECTION$)', /boolean) eq 1,match)
    if (match eq 0) then return, 1    ; if no array or collection, then return

    ; store array and or collection objects
    objects = objects[pos]            ; store object names
    index = objindx[pos]              ; store object indices
    count = match                     ; store number of objects

    ; extract the ones that need to be processed and process them
    counter = 0                        ; counter for looping all objects
    while (counter lt count) do begin  ; loop through each object
        ; extract endindex for current object
        endindex = get_index(label, index[counter])
        ; find how many indices are between current and endindex
        pos = where (index gt index[counter] and index lt endindex, matches)

        ; check for current object's INTERCHANGE FORMAT
        interform = extract_keyword(label, 'INTERCHANGE_FORMAT', $
            index[counter], endindex, 1, 1)
        interform = remove(interform, '"')         ; remove quotes
        if (interform ne 'ASCII' && interform ne 'BINARY') then begin
            message, "Error: invalid INTERCHANGE_FORMAT value " + interform
        endif else if (interform eq 'ASCII') then begin
            message, "Error: ASCII ARRAY/COLLECTION object found. Currently"+$
                " not supported by this software."
        endif

        ; start a byte count
        byte_count = 0                 ; variable to keep track of data bytes

        ; start processing the current object by calling arr or col
        temp = objects[counter]
        if (stregex(temp, 'ARRAY$', /boolean)) then begin
            test = verify_arr(label, index[counter], endindex, byte_count)
        endif else if (stregex(temp,'COLLECTION$',/boolean)) then begin
            test = verify_col(label, index[counter], endindex, byte_count)
        endif else begin
            message, "Error: invalid OBJECT found in ARRAY/COLLECTION object"+$
                     " "+temp
        endelse

        counter++                         ; increment counter
        ; if more than one index found between current and endindex
        if (matches gt 0) then begin
            counter += matches            ; add the number found to counter
        endif 
    endwhile

    return, 1
end
