;;-----------------------------------------------------------------------------
;; Name: READPDS
;;
;; Purpose: To read a PDS file into IDL data and structure variables
;;
;; Calling Sequence:
;;     result = readpds (filename [, /SILENT, /NOSCALE, PATH=path])
;;
;; Input:
;;     filename - scalar string containing the name of the PDS file to
;;                read
;;
;; Output:
;;     result - PDS data structure constructed from designated record
;;
;; Optional inputs:
;;     SILENT - suppresses any messages from the procedure
;;     NOSCALE - does not perform scaling and offset of values in
;;               image or qube arrays, default is to scale and offset
;;     PATH	Optional dataset root path for correct expansion of
;;		(relative) label pointers.
;;
;; Examples:
;;     To read a PDS file test.lbl into an IDL image array, img:
;;     IDL> img = readpds ('test.lbl', /SILENT)
;;     IDL> help, /st, img
;;          OBJECTS     INT      1
;;          IMAGE       LONG    [200, 200]
;;     To read a PDS file with multiple objects:
;;     IDL> data = readpds ('multiple.lbl', /SILENT)
;;     IDL> help, /st, data
;;          OBJECTS     INT         2
;;          TABLE       STRUCT      -> ARRAY[1]
;;          IMAGE       STRUCT      -> ARRAY[1]
;;
;; External routines: Headpds, Verify_label, Preplabel, Imagepds,
;;     Qubepds, Tablepds, Arrcolpds, Get_index
;;
;; Modification history:
;;     Written by Puneet Khetarpal, 24 February 2003
;;     12 Feb 08, HJJ:	PATH keyword specifies datset root (used by POINTPDS).
;;			LABEL keyword returns the retrieved label value.
;;     For a complete list of modifications, see changelog.txt file.
;;
;;-----------------------------------------------------------------------------

;- level 1 --------------------------------------------------------------------

; precondition: label has been verified, and prepared for processing
; postcondition: all viable objects are extracted from the label and returned
function get_all_objects, label
    ; error check
    on_error, 1

    ; extract all objects
    objects = objpds(label, 'ALL')
    if (objects.count eq 0) then begin    ; if no viable objects, then error
        message, "Error: no viable PDS objects found in label."
    endif

    ; declare loop variables
    newobj = strarr(objects.count)           ; initialize new obj array
    newind = intarr(objects.count)           ; initialize new ind array
    newcount = 0                             ; initialize new obj counter
    counter = 0                              ; counter to loop through objects

    ; remove any sub objects within each major object if present
    while (counter lt objects.count) do begin
        ; extract endindex for current object element
        endindex = get_index(label, objects.index[counter])
        ; find how many indices are between current and endindex
        pos = where(objects.index gt objects.index[counter] and $
                    objects.index lt endindex, matches)
        ; store current object attributes
        newobj[newcount] = objects.array[counter]    ; store name
        newind[newcount] = objects.index[counter]   ; store index
        newcount++                                  ; increment object count
        counter++                                   ; increment counter
        ; if more than one index found between current and endindex
        if (matches gt 0) then begin
            counter += matches                      ; skip those objects
        endif
    endwhile

    ; store new array elements in a structure and return it
    struct = {array:newobj[0:newcount-1], index:newind[0:newcount-1], $
              count:newcount}

    return, struct
end

;- level 0 --------------------------------------------------------------------

function readpds, filename, SILENT=silent, NOSCALE=noscale, NOVERIFY=noverify, $
			    PATH=path, LABEL=label
    ; error protection
    on_error, 1

    ; check for number of parameters in function call
    if (n_params() lt 1) then begin
        message, "Syntax Error: result = READPDS(filename [,/SILENT,/NOSCALE,PATH=path])"
    endif
    st = keyword_set(silent)
    noscale = keyword_set(noscale)
    nv = KEYWORD_SET(noverify)
    SETENV, 'READPDS_PATH=' + ((N_ELEMENTS(path) NE 0) ? path : '')

    ; save file name
    savefile = filename
    ; obtain PDS label from file name
    label = (st) ? headpds(savefile, /silent) : headpds(savefile)
    ; verify label
    if (~ st) then print, "Now verifying label for: "+filename
    if (~ nv) then test = verify_label(label, filename)
    ;; add end object values and remove comments
    label = preplabel(label)
    ;; extract all viable objects from label
    objects = get_all_objects(label)

    ;; initialize object structure
    result = create_struct('objects', objects.count)
    ;; process all objects
    for i = 0, objects.count - 1 do begin
        name = objects.array[i]        ; store current object name
        index = objects.index[i]       ; store current object index

        ;; process image objects
        if (stregex(name, 'IMAGE$', /boolean)) then begin
            if (~ st) then begin    ; check for silent keyword
                data = (~ noscale) ? imagepds(filename, label, index) : $
                       imagepds(filename, label, index, /noscale)
            endif else begin  ; else include silent keyword
                data = (~ noscale) ? imagepds(filename,label,index,/silent) : $
                       imagepds(filename, label, index, /silent, /noscale)
            endelse
        endif

        ;; process qube objects
        if (stregex(name, 'QUBE$', /boolean)) then begin
            if (~ st) then begin    ; check for silent keyword
                data = (~ noscale) ? qubepds(filename, label, index) : $
                       qubepds(filename, label, index, /noscale)
            endif else begin  ; else include silent keyword
                data = (~ noscale) ? qubepds(filename,label,index,/silent) : $
                       qubepds(filename, label, index, /silent, /noscale)
            endelse
        endif

        ;; process table objects
        expr = '(TABLE$)|(SERIES$)|(PALETTE$)|(SPECTRUM$)'
        if (stregex(name, expr, /boolean)) then begin
            data = (~ st) ? tablepds(filename, label, index) : $
                        tablepds(filename, label, index, /silent)
        endif

        ;; process array and collection objects
        if (stregex(name,'(ARRAY$)|(COLLECTION$)', /boolean)) then begin
            data = (~ st) ? arrcolpds(filename, label, index) : $
                        arrcolpds(filename, label, index, /silent)
        endif

        result = create_struct(result, name, data)
    endfor

    if (~ st) then help, /st, result     ; display structure contents if ! st

    return, result
end
