;;----------------------------------------------------------------------------
;; Name: PREPLABEL
;;
;; Purpose: To prepare the label for adding end object values and removing
;;     comments from the label array
;;
;; Calling Sequence: 
;;     result = preplabel (label)
;;
;; Input:
;;     label - the string array containing the verified PDS label
;; 
;; Output:
;;     result - the string array of PDS label without any comments;
;;              all end object values are added where needed
;;
;; Optional input: none
;;
;; External routines: Pdspar, Clean, Remove, 
;;
;; Modification history:
;;     Written by Puneet Khetarpal, 30 Jun 2005
;;
;;----------------------------------------------------------------------------

;- level 0 ------------------------------------------------------------------

function preplabel, label
    ; error protection
    on_error, 1

    ; check number of parameters in function call
    if (n_params() lt 1) then begin
        message, "Syntax Error: lbl = PREPLABEL(label)"
    endif

    ; extract all object keywords 
    obj_names = pdspar(label, 'object', COUNT=objcnt, INDEX=objindx)
    objarr = strarr(objcnt)
    ;; extract all end object keywords values and indices
    eobj_names = pdspar(label, 'end_object', COUNT=eobjcnt, INDEX=eobjindx)

    lfcr = string([13B, 10B])    ; initialize line feed and cr chars
    
    ; go through the label 
    count = 0               ; initialize counter for object array
    for i = 0, n_elements(label) - 1 do begin
        ; check for comments and remove if present
        if (stregex(label[i], '/\*', /boolean)) then begin
            length = stregex(label[i], '/\*') - 1  ; store length to be saved
            label[i] = strmid(label[i], 0, length) + lfcr ; extract and store
        endif    

        pos = where (objindx eq i, matchcnt) ; match curr index in obj index
        if (matchcnt gt 0) then begin        ; if matched, then add to 
            objarr[count] = clean(obj_names[pos], /space)   ; objarr stack
            count++                          ; increment counter
        endif

        ;; test for end object keyword in label
        test = stregex(label[i], 'END_OBJECT', /boolean)
        if (test && count gt 0) then begin   ; if found and count > 0
            if (eobjcnt gt 0) then begin      ; if eobj keyword values present
                ; check for current index matching
                pos = where(eobjindx eq i, matchcnt)
                if (matchcnt eq 0) then begin   ; if not found then add
                    label[i] = remove(label[i], lfcr)  ; remove lf + cr chars
                    label[i] = strtrim(label[i])     ; remove trailing blanks
                    label[i] += ' = ' + objarr[count - 1] + lfcr
                endif
            endif else begin  ; else add to current label
                label[i] = remove(label[i], lfcr)    ; remove lf and cr chars
                label[i] = strtrim(label[i])         ; remove trailing blanks
                label[i] += ' = ' + objarr[count - 1] + lfcr
            endelse
            count--           ; decrement count
        endif 
    endfor
    
    return, label
end
