;;----------------------------------------------------------------------------
;; Name: PDSPAR
;;
;; Purpose: To obtain the value of a keyword parameter in a PDS header
;;
;; Calling Sequence:
;;    result = pdspar (lbl, pds-keyword, [COUNT=, INDEX=])
;;
;; Input:
;;    lbl - PDS header array, string array, must be a viable PDS label
;;    pds-keyword - string name of the pds-keyword parameter to search for
;;
;; Output:
;;    result - value(s) of the keyword in the PDS label
;;
;; Optional Outputs:
;;    COUNT - optional keyword to return a value equal to the number
;;            of parameters found by PDSPAR, integer scalar
;;    INDEX - optional keyword to return an array of the line numbers
;;            where the values being returned were found in the PDS label.
;;
;; Examples:
;;    Given a PDS header, h, return the values of the axis dimension
;;    values.
;;
;;    IDL> x_axis = pdspar( h, 'lines')  ; extract x axis value
;;    IDL> y_axis = pdspar( h, 'LINE_Samples') ; extract y axis value
;;
;;    To return the OBJECT(s) in the header with count, and index.
;;    IDL> obj = pdspar (h, 'OBJECT', COUNT=cnt, INDEX=indx)
;;
;; External routines: Clean
;;
;; Modification history:
;;    Written by Puneet Khetarpal, 12 May 2003.
;;    12 Feb 08, HJJ:	Rewritten in order to improve execution speed.
;;----------------------------------------------------------------------------

;;- level 0 ------------------------------------------------------------------

function pdspar, lbl, key, COUNT=count, INDEX=index
    ; error protection
    on_error, 1

    ; check for number of parameters
    if (n_params() lt 2) then begin
        message, "Syntax error: result = pdspar(lbl, name [,COUNT=,INDEX=])"
    endif

    ; check whether the label is a string array
    stat = size(lbl)
    if (stat[0] ne 1 || stat[2] ne 7) then begin
        message, "Error: label must be a string array"
    endif
    pmax = N_ELEMENTS(lbl)

    ; convert keyword to upper case and clean it
    key = strtrim(strupcase(key), 2)

    ; get list of matching keywords
    pos = TRANSPOSE(STRPOS(lbl, '='))
    idx = WHERE(pos NE -1, count)
    IF (count GT 0) THEN BEGIN
    	value = STRTRIM(STRMID(lbl[idx], 0, pos[*,idx]), 2)
    	index = WHERE(STRMATCH(value, key), count)
    	IF (count GT 0) THEN BEGIN
    	    index = idx[index]
    	    pos = pos[*,index] + 1
    	    len = TRANSPOSE(STRLEN(lbl[index])-2) - pos
    	    value = STRTRIM(STRMID(lbl[index], pos, len), 2)
    	ENDIF
    ENDIF
    IF (count EQ 0) THEN value = ''

    ; complete string values with no matching end quote in the same line
    idx = WHERE(STRMATCH(value, '"*') AND ~STRMATCH(value, '"*"'), cnt)
    FOR i = 0, cnt-1 DO BEGIN
    	pos = index[idx[i]]
    	val = value[idx[i]]
    	REPEAT BEGIN
    	    ++pos
    	    val += ' ' + STRTRIM(STRMID(lbl[pos], 0, STRLEN(lbl[pos])-2), 2)
    	ENDREP UNTIL (STRMID(val, 0, /REVERSE) EQ '"')
    	value[idx[i]] = val
    ENDFOR

    return, value
end
