;;-----------------------------------------------------------------------------
;; Name: OBJPDS
;;
;; Purpose: To obtain viable data objects from a PDS label.
;;
;; Calling Sequence: 
;;    result = objpds (label, param)
;;
;; Input:
;;    label - string array containing object header information.
;;    param - the object parameter to search for.
;;
;; Output:
;;    result - a structure containing the viable object names, object count,
;;            and the object indicies.
;;
;; Optional input: none.
;;
;; Examples:
;;    To obtain all image objects:
;;       IDL> label = headpds ("TEST.LBL")
;;       IDL> objects = objpds (label, "IMAGE")
;;       IDL> help, /structure, objects
;;            ARRAY         STRING        ARRAY[1]
;;            COUNT           INT                1
;;            INDEX          LONG         ARRAY[1]
;;    To obtain all viable objects:
;;       IDL> label = headpds ("TEST.LBL")
;;       IDL> objects = objpds (label, "ALL")
;;       IDL> help, /structure, objects
;;            ARRAY         STRING        ARRAY[3]
;;            COUNT           INT                3
;;            INDEX          LONG         ARRAY[3]
;; 
;; External routines: Pdspar
;;
;; Modification history:
;;    Written by Puneet Khetarpal, 14 February 2003
;;    For a complete list of modifications, see changelog.txt file.    
;;
;;-----------------------------------------------------------------------------

;- level 1 -------------------------------------------------------------------

;-----------------------------------------------------------------------------
; precondition: param is a scalar string, and object struct has been 
;      compiled properly with all the object
; postcondition: returns a structure containing all the viable objects matching
;      param, with values, count, and index. 

function obtain, param, object
    ; error protection:
    on_error, 2

    ; intialize structure
    struct = {count:0, array:'', index:0} 

    ; process individual objects or all objects:
    if (param ne "ALL") then begin
        ; find position where param has been matched.
        ; in particular, the latter condition is to ensure that only those
        ; object strings are matched with param that end with the value
        ; param, i.e., e.g., param "IMAGE" will match "PRIMARY_IMAGE"
        ; but not "IMAGE_HEADER", and param "TABLE" will match "TABLE" but
        ; not "TABLE_HEADER":
        pos = where(stregex(object.array, param+'$',/boolean) eq 1, srchcount)
    endif else begin
        ; construct expression
        expr = '(COLLECTION$)|(TABLE$)|(SERIES$)|(PALETTE$)|(SPECTRUM$)|'+$
               '(IMAGE$)|(QUBE$)|(WINDOW$)|(ARRAY$)'
        pos = where(stregex(object.array, expr, /boolean) eq 1, srchcount)
    endelse

    ; if above search yields positive definite result, then 
    ; create a structure that contains the matched results
    if (srchcount gt 0) then begin
        struct = {count:srchcount, array:object.array[pos], $
                  index:object.index[pos]}
    endif
        
    return, struct
end

;-----------------------------------------------------------------------------
; precondition: param is a scalar string in upper case, and object contains
;     only viable PDS objects.
; postcondition: the subroutine checks whether param is equal to "ALL", and 
;     if so, then checks whether there exists duplicate names of objects
;     that are not ARRAY, COLLECTION, or WINDOW. If duplicates found, then
;     returns 0, else 1

function check_duplicate, param, object
    ; error protection:
    on_error, 2

    ; if param is not "ALL" then no need to check for duplicates
    if (param ne "ALL") then return, 1

    ; go through each of the object names
    for i = 0, object.count - 2 do begin
        ; if object is not array, collection or window and it 
        ; matches the next one, then issue error
        if (~ stregex(object.array[i], '(ARRAY$)|(COLLECTION$)|(WINDOW$)', $
            /boolean) && object.array[i] eq object.array[i+1]) then begin
            print, "Error: Duplicate object names found in label." + $
                   "Object names must be unique."
            return, 0           ; return 0
        endif
    endfor

    return, 1
end

;- level 0 -------------------------------------------------------------------

;-----------------------------------------------------------------------------
; precondition: label is a viable PDS label, and param is a scalar string;
;     the label has been verified and prepared for processing
; postcondition: returns a structure containing all viable objects that match
;     param.

function objpds, label, param
    ; error protection:
    on_error, 2

    ; obtain all OBJECT keywords from label:
    objarray = pdspar (label, 'OBJECT', COUNT=objcount, INDEX=objindex)
    ; clean the object array and format param:
    for i = 0, objcount - 1 do objarray[i] = clean(objarray[i],/space)
    objarray = strupcase (objarray)
    param = strupcase (param)
    ; create an object structure
    object = {array:objarray, count:objcount, index:objindex}
    ; extract param objects:
    object = obtain (param, object)
    ; check for duplicate names:
    if (~ check_duplicate (param, object)) then begin
        object.count = 0          ; set object count to 0
    endif

    return, object
end
