;;-----------------------------------------------------------------------------
;; Name: HEADPDS
;;
;; Purpose: To read a PDS label into an array variable
;;
;; Calling Sequence: label = headpds (filename [, /SILENT, /FILE ])
;;
;; Input:
;;     filename - scalar string containing the name of the PDS file to read
;;
;; Output:
;;     label - PDS label string array constructed from designated record
;;
;; Optional inputs:
;;     SILENT - suppresses any messages from the procedure
;;     FILE - to be indicated if the file does not contain the label
;;
;; Examples:
;;    To read a PDS file TEST.PDS into a PDS header array, lbl:
;;       IDL> lbl = headpds ('TEST.PDS', /SILENT)
;;    To read a PDS file that may not contain a header:
;;       IDL> lbl = headpds ('TEST2.TXT', /FILE)
;;
;; External routines: Clean, Pdspar, Pointpds
;;
;; Modification history :
;;    Written by Puneet Khetarpal, 24 January 2003
;;    12 Feb 08, HJJ	^STRUCTURE keyword had to be removed from label.
;;
;;    For detailed log of modifications to this routine, please see the
;;    changelog.txt file.
;;
;;-----------------------------------------------------------------------------

function headpds, filename, SILENT=silent, FILE=file
    ; error protection
    on_error, 1
    on_ioerror, signal

    ; check for number of parameters in function call, must be >= 1
    if (n_params() lt 1) then begin
        message, "Syntax Error: lbl = HEADPDS (filename [, /SILENT, /FILE ])"
    endif

    ; check for input of optional keywords
    silent = keyword_set(silent)
    file = keyword_set(file)

    ; check whether the file exists and can be opened
    openr, unit, filename, error=err, /get_lun
    if (err lt 0) then begin
        message, "Error: file " + filename + " could not be opened = "+ $
            "  File either corrupted or invalid file name."
    endif

    ; inform the user of status if not silent
    if (~ silent) then begin
        print, "Now reading header: ", filename
    endif

    ; initialize loop variables
    flag = 0                 ; set to 1 when "END" keyword is encountered
    lbl = strarr(32767)      ; initialize label array to max signed int elem
    count = 0                ; counter to track the elements of lbl array
    line = ""                ; variable to read a single line from file

    ; read the file until "END" or EOF is reached
    while (~ flag && ~ eof(unit)) do begin
        readf, unit, line    ; read one line from file
        ;; since readf removes all linefeed and carriage-return chars
        ;; add lf and cr chars to the line read from file
        lbl[count] = line + string([13B, 10B])
        ;; look for "END" keyword in line
        temp = clean(line, /space)   ; clean the line of all white space
        if (~ file && temp eq "END") then flag++ ; if found, increment flag
        count++                            ; increment label array counter
    endwhile

    ; close file unit and free the unit
    close, unit
    free_lun, unit

    ; set label to actual elements
    lbl = lbl[0:count-1]

    ; obtain STRUCTURE keyword value in label if any
    structure = pdspar(lbl, '\^STRUCTURE', COUNT=structcnt, INDEX=structind)
    if (structcnt gt 0) then begin
        ; obtain pointer attributes for STRUCTURE and read the file
        pointer = pointpds(lbl, filename, 'STRUCTURE')
        datafile = pointer.datafile             ; store datafile name
        fmtlabel = (silent) ? headpds(datafile, /silent, /file) : $
                   headpds(datafile, /file) ; obtain format file data
        ; insert fmtlabel into lbl array
        ; lbl = [lbl[0:structind[0]], fmtlabel, lbl[structind[0] + 1:count-1]]
        ; corrected by HJJ : ^STRUCTURE keyword had to be removed from label !!!
        lbl = [lbl[0:structind[0]-1], fmtlabel, lbl[structind[0]+1:count-1]]
    endif

    return, lbl

    ; error processing for input output
    signal:
        on_ioerror, null
        message, "Error: unable to read file " + filename
end
