;;-----------------------------------------------------------------------------
;; Name: GET_IDL_TYPE
;;
;; Purpose: To obtain the IDL data type value for corresponding PDS data
;;     type objects to be read in the file. 
;;
;; Calling Sequence: 
;;     idltype = get_idl_type(data_type, bytes, inter)
;;
;; Input:
;;     data_type - a scalar string containing a viable PDS data type value
;;     bytes - a long integer containing the number of bytes for current object
;;     inter - the interchange format for the data to be read, either
;;         ASCII or BINARY.
;;
;; Output:
;;     idltype - a long integer ranged from 1 to 15 specifying the IDL
;;         type value for the current data type object.
;;
;; External routines: none.
;;
;; Modification history:
;;     Written by Puneet Khetarpal, 30 June 2005
;;
;;-----------------------------------------------------------------------------

; precondition: data type must be a viable PDS data type value, and
;    bytes must be a viable bytes value; interchange format is either "ASCII"
;    of "BINARY"
; postcondition: the idl type of the element is determined and returned
function get_idl_type, data_type, bytes, inter
    ; convert bytes to bits
    bits = bytes * 8
    type = 0                   ; set to undefined for now
    dt = data_type             ; abbreviate the variable name for use

    ; if interchange format is ascii, then send to ascii label
    if (inter eq "ASCII") then goto, ASCII

    ; start BINARY processing 
    BINARY:

    ; check for character
    if (stregex(dt, 'CHARACTER', /boolean)) then begin
        type = 7
    endif else if (stregex(dt, 'COMPLEX', /boolean)) then begin  ; complex
        type = 6
    endif else if (stregex(dt, 'N/A', /boolean)) then begin      ; spare
        type = 1                            ; set type as byte
    endif else if (bits eq 8) then begin    ; byte
        type = 1
    endif else if (bits eq 16) then begin   ; integer (signed or unsigned)
        type = (stregex(dt, 'UNSIGNED', /boolean)) ? 12 : 2
    endif else if (bits eq 32) then begin   ; long integer or real 
        type = (stregex(dt, 'INTEGER', /boolean)) ? $
            ((stregex(dt, 'UNSIGNED', /boolean)) ? 13 : 3) : 4
    endif else if (bits eq 64) then begin   ; 64-bit int or real
        type = (stregex(dt, 'INTEGER', /boolean)) ? $
            ((stregex(dt, 'UNSIGNED', /boolean)) ? 15 : 14) : 5
    endif

    return, type

    ; start ASCII processing
    ASCII:

    ; if character, time or date
    if (stregex(dt, '(CHARACTER)|(TIME)|(DATE)', /boolean)) then begin
        type = 7
    endif else if (stregex(dt, 'COMPLEX', /boolean)) then begin ; complex
        type = 6
    endif else if (stregex(dt, 'N/A', /boolean)) then begin     ; spare
        type = 1
    endif else if (stregex(dt, 'INTEGER', /boolean)) then begin  ; integer
        type = 3
    endif else if (stregex(dt, '(REAL)|(FLOAT)', /boolean)) then begin ; real
        type = 5
    endif else if (stregex(dt, 'BOOLEAN', /boolean)) then begin ; boolean
        type = 3
    endif else begin
        type = 1
    endelse

    return, type
end
