;;-----------------------------------------------------------------------------
;; Name: APPLY_BITMASK
;;
;; Purpose: To apply bitmask on a Signed or Unsigned Integer array or
;;     scalar as defined in the associated PDS label
;;
;; Calling Sequence: 
;;     result = apply_bitmask (element, bit_mask)
;; 
;; Input:
;;     element - the signed/unsigned integer array/scalar to apply the bitmask
;;     bit_mask - the scalar string extracted from label for element;
;;         must be of format - 2#01..01# 
;; Output:
;;     result - the signed/unsigned integer array/scalar after the
;;         bitmask is applied (if any)
;;
;; Example:
;;     Given an integer x = 56, and bitmask = '2#00000110#'
;;     
;;     IDL> result = apply_bitmask(x, bitmask)
;;     IDL> print, result
;;       6
;;
;; External routines: Clean
;;
;; Modification history:
;;     Written by Puneet Khetarpal, 30 June 2005.
;;
;;-----------------------------------------------------------------------------

;- level 0 -------------------------------------------------------------------

;-----------------------------------------------------------------------------
; precondition: element is an unsigned or signed integer array or
;     scalar, and bit mask is the mask to be applied to the element 
; postcondition: the bitmask for the current element object is
;     applied to the element and returned.

function apply_bitmask, element, bit_mask
    ; initialize variables:
    newvalue = element                         ; save element for processing
    bitmask = 0                                ; decimal value of bitmask

    ; check whether element is integer:
    typerange = [1, 2, 3, 4, 12, 13, 14, 15]   ; declare valid range of values
    stat = size(element, /type)                ; determine the type of element
    pos = where (stat eq typerange, cnt)       ; does element match typerange
    if (cnt eq 0) then return, newvalue        ; if not, then return saved

    ; extract bit mask value in decimal from the input bit mask string    
    ; extract the value from the line: (format 2#01...01#)
    temp = strsplit(bit_mask, "#", /extract)   ; split using # delimeter 
    value = byte(clean(temp[1], /space)) - 48  ; convert to bytarr 0 or 1

    ; determine the added value of the bit elements in decimal:
    for i = n_elements(value) - 1, 0, -1 do begin  ; decrement to 0
        bitmask += value[i] * 2^i              ; add to bitmask value
    endfor

    ; convert bitmask to appropriate decimal type:
    bitmask = fix(bitmask, type = stat)

    ; determine where element is negative:
    negpos = where (element lt 0, negcnt)

    ; apply bitmask to saved absolute value of element:
    newvalue = abs(element) and bitmask       ; "AND" operator applies bitmask

    ; apply negative where element was negative if any:
    if (negcnt gt 0) then begin
        newvalue[pos] *= -1                   ; make element negative
    endif

    return, newvalue
end
