function l2 = READICA_L2(varargin)
%READICA_L2 Read RPCICA L2 EDITED data for one day.
%   l2 = READICA_L2() Reads data for a day.
%        Skips data with less than 96 e_levels.
%        Asks for rootpath & date
%
%   l2 = READICA_L2(stretch) Reads data for a day.
%        Stretches data with less than 96 e_levels to 96 levels.
%        Asks for rootpath & date
%
%   l2 = READICA_L2(stretch, date) Reads data for date given.
%        If stretch is true, stretches data with less than 96 e_levels to 96 levels.
%        Asks for rootpath
%
%   l2 = READICA_L2(stretch, date, rootpath) Reads data for date given.
%        If stretch is true, stretches data with less than 96 e_levels to 96 levels.
%
%  See also: IMPORT_L2, PLOT_ICA, ICAINITIALIZE
stretch = false;
data_date = '';
rootpath = '';
if nargin > 2
    if ~ischar(varargin{3})
        help(mfilename)
        return
    end
    rootpath = varargin{3};
end
if nargin > 1
    if ~ischar(varargin{2})
        help(mfilename)
        return
    end
    data_date = varargin{2};
end
if nargin > 0
    if ~islogical(varargin{1})
        help(mfilename)
        return
    end
    stretch = varargin{1};
end

%% Find rootpath of archive where directory DATA is located
if isempty(rootpath) || ~exist(fullfile(rootpath, 'DATA'),'dir')
    for jj=1:5
        if exist(fullfile(rootpath, 'DATA'),'dir')
            break
        else
            rootpath = fullfile('..', rootpath);
        end
    end
end

%% Get path to data
icapath=fullfile(rootpath, 'DATA', 'EDITED');
while ~exist(icapath, 'dir')
    fprintf(['This script is designed to run inside an archive.\n' ...
        'The DATA/EDITED directory was not found.\n' ...
        'Please provide the path to the root of the archive i.e.:\n' ...
        '/home/user/psa/RO-C-RPCICA-2-ESC1-RAW-V1.0\n' ...
        '\n']);
    rootpath = input('path to the root of the archive:  ','s');
    icapath=fullfile(rootpath, 'DATA', 'EDITED');
end

fprintf('rootpath: %s\n', rootpath);

if isempty(data_date)
    data_date=input('Date of data to import yyyy-mm-dd:  ','s');
end
dn = datenum(data_date);


cdir=fullfile(icapath, ...
    datestr(dn,'yyyy'), ...
    upper(datestr(dn,'mmm')), ...
    ['D' datestr(dn,'dd')]);
fprintf('cdir: %s\n', cdir);

%% Get files
fcdir=fullfile(cdir, '/RPCICA*.TAB');
filelist=dir(fcdir);

%% initialize matrices
l2.iontime = [];
l2.delta_t = [];
l2.quality = [];
l2.mode = [];
l2.noise_reduction = [];
l2.mtable = [];
l2.pacclevel = [];
l2.azim = [];
l2.elev = [];
l2.masschannel = [];
l2.ionspectra = [];
l2.swversion = [];

%% Find function to call
if (exist('IMPORT_L2') == 2)
    import = str2func('IMPORT_L2');
else
    import = str2func('import_l2');
end

%% Read data
e_level_skip = 0;

if ~isempty(filelist)
    for jj=1:length(filelist)
        if ~isempty(strfind(filelist(jj).name,'_HK.TAB'))
            continue;
        end
        if ~isempty(strfind(filelist(jj).name,'_GEOM.TAB'))
            continue;
        end
        disp(filelist(jj).name);
        if ~isempty(strfind(filelist(jj).name,'_L2.TAB'))
            [table, e_levels] = import(fullfile(cdir, filelist(jj).name));
            lbl = strrep(filelist(jj).name, 'TAB', 'LBL');
            l2_lbl = READ_PDS_LBL(fullfile(cdir, lbl));
            curr_sw = str2double(l2_lbl.ROSETTA_COLON_ICA_SW_VERSION);
        else
            % No support for other files
            continue;
        end
        if e_levels ~= 96
            if stretch
                % possibly stretch 8 & 32 e_levels to 96 e_levels
                table.ionspectra = [table.ionspectra nan(size(table.iontime,1),96-e_levels)];
            else
                e_level_skip = e_level_skip + 1;
                continue;
            end
        end
        
        l2.iontime=[l2.iontime; table.iontime];
        l2.delta_t=[l2.delta_t; table.delta_t];
        l2.quality=[l2.quality; table.quality];
        l2.mode= [l2.mode; table.mode];
        l2.noise_reduction=[l2.noise_reduction; table.noise_reduction];
        l2.mtable=[l2.mtable; table.mtable];
        l2.pacclevel=[l2.pacclevel; table.pacclevel];
        l2.azim=[l2.azim; table.azim];
        l2.elev=[l2.elev; table.elev];
        l2.masschannel=[l2.masschannel; table.masschannel];
        l2.ionspectra=[l2.ionspectra; table.ionspectra];
        l2.swversion = [l2.swversion; repmat(curr_sw, length(table.iontime), 1)];
    end
    save(['ica' data_date '.mat'], 'l2');
    if e_level_skip > 0
        fprintf('Skipped reading %d files due to low number of e_levels\n', e_level_skip);
    end
else
    disp(['Can not find data file: ' fcdir]); 
end
