function l4_cts = READICA_L4_CORR_CTS(varargin)
%READICA_L4_CORR_CTS Read RPCICA L4 COUNTS data for one day.
%   l4_corr = READICA_L4_CORR_CTS() Reads data for a day. 
%        Skips data with less than 96 e_levels.
%        Asks for rootpath & date
%
%   l4_corr = READICA_L4_CORR_CTS(stretch) Reads data for a day. 
%        Stretches data with less than 96 e_levels to 96 levels.
%        Asks for rootpath & date
%
%   l4_corr = READICA_L4_CORR_CTS(stretch, date) Reads data for date given. 
%        If stretch is true, stretches data with less than 96 e_levels to 96 levels.
%        Asks for rootpath
%
%   l4_corr = READICA_L4_CORR_CTS(stretch, date, rootpath) Reads data for date given.
%        If stretch is true, stretches data with less than 96 e_levels to 96 levels.
%
% See also: IMPORT_L4_CORR_CTS, PLOT_ICAL4_CORR_CTS
stretch = false;
data_date = '';
rootpath = '';
if nargin > 2
    if ~ischar(varargin{3})
        help(mfilename)
        return
    end
    rootpath = varargin{3};
end
if nargin > 1
    if ~ischar(varargin{2})
        help(mfilename)
        return
    end
    data_date = varargin{2};
end
if nargin > 0
    if ~islogical(varargin{1})
        help(mfilename)
        return
    end
    stretch = varargin{1};
end

%% Find rootpath of archive where directory DATA is located
if isempty(rootpath) || ~exist(fullfile(rootpath, 'DATA'),'dir')
    for jj=1:5
        if exist(fullfile(rootpath, 'DATA'),'dir')
            break
        else
            rootpath = fullfile('..', rootpath);
        end
    end
end

%% Get path to data
icapath=fullfile(rootpath, 'DATA', 'CORR_CTS');
while ~exist(icapath, 'dir')
    fprintf(['This script is designed to run inside an archive.\n' ...
        'The DATA/CORR directory was not found.\n' ...
        'Please provide the path to the root of the archive i.e.:\n' ...
        '/home/user/psa/RO-C-RPCICA-4-ESC1-CTS-V1.0\n' ...
        '\n']);
    rootpath = input('path to the root of the archive:  ','s');
    icapath=fullfile(rootpath, 'DATA', 'CORR_CTS');
end

fprintf('rootpath: %s\n', rootpath);

if isempty(data_date)
    data_date=input('Date of data to import yyyy-mm-dd:  ','s');
end
dn = datenum(data_date);


cdir=fullfile(icapath, ...
    datestr(dn,'yyyy'), ...
    upper(datestr(dn,'mmm')), ...
    ['D' datestr(dn,'dd')]);
fprintf('cdir: %s\n', cdir);

%% Get files
fcdir=fullfile(cdir, '/RPCICA*.TAB');
filelist=dir(fcdir);

%% initialize matrices
l4_cts.time = [];
l4_cts.delta_t = [];
l4_cts.quality = [];
l4_cts.mode = [];
l4_cts.noise_reduction = [];
l4_cts.mtable = [];
l4_cts.pacclevel = [];
l4_cts.azim = [];
l4_cts.elev = [];
l4_cts.mass = [];
l4_cts.counts = [];
l4_cts.swversion = [];
l4_cts.uncertainty = [];

%% Find function to call
if (exist('IMPORT_L4_CORR_CTS') == 2)
    import4_cts = str2func('IMPORT_L4_CORR_CTS');
else
    import4_cts = str2func('import_l4_corr_cts');
end

if (exist('IMPORT_L4_ZERO_CTS') == 2)
    import4_zcts = str2func('IMPORT_L4_ZERO_CTS');
else
    import4_zcts = str2func('import_l4_zero_CTS');
end

%% Read data
e_level_skip = 0;

if ~isempty(filelist)
    for jj=1:length(filelist)
        disp(filelist(jj).name);
        if ~isempty(strfind(filelist(jj).name,'_L4_CTS.TAB'))
            [table, e_levels] = import4_cts(fullfile(cdir, filelist(jj).name));
            lbl = strrep(filelist(jj).name, 'TAB', 'LBL');
            l4_lbl = READ_PDS_LBL(fullfile(cdir, lbl));
            curr_sw = str2double(l4_lbl.ROSETTA_COLON_ICA_SW_VERSION);
            zero = strrep(filelist(jj).name, '_CTS.TAB', '_ZCTS.TAB');
            [ztable, ze_levels] = import4_zcts(fullfile(cdir, zero));
        else
            % No support for other files
            continue;
        end
        if e_levels ~= 96 
            if stretch
                % possibly stretch 8 & 32 e_levels to 96 e_levels
                table.counts = [table.counts nan(size(table.time,1),96-e_levels)];
                ztable.uncertainty = [ztable.uncertainty nan(size(table.time,1),96-e_levels)];
            else
                e_level_skip = e_level_skip + 1;
                continue;
            end
        end
        
        if any(table.time ~= ztable.time)
            disp('time is not equal between corr & zero')
        end
        if any(table.azim ~= ztable.azim)
            disp('azim is not equal between corr & zero')
        end
        if any(table.elev ~= ztable.elev)
            disp('elev is not equal between corr & zero')
        end
        if any(table.mass ~= ztable.mass)
            disp('mass is not equal between corr & zero')
        end
        
        l4_cts.time=[l4_cts.time; table.time];
        l4_cts.delta_t=[l4_cts.delta_t; table.delta_t];
        l4_cts.quality=[l4_cts.quality; table.quality];
        l4_cts.mode= [l4_cts.mode; table.mode];
        l4_cts.noise_reduction=[l4_cts.noise_reduction; table.noise_reduction];
        l4_cts.mtable=[l4_cts.mtable; table.mtable];
        l4_cts.pacclevel=[l4_cts.pacclevel; table.pacclevel];
        l4_cts.azim=[l4_cts.azim; table.azim];
        l4_cts.elev=[l4_cts.elev; table.elev];
        l4_cts.mass=[l4_cts.mass; table.mass];
        l4_cts.counts=[l4_cts.counts; table.counts];
        l4_cts.swversion = [l4_cts.swversion; repmat(curr_sw, length(table.time), 1)];
        l4_cts.uncertainty = [l4_cts.uncertainty; ztable.uncertainty]; 
    end
    save(['ica' data_date '_corr.mat'], 'l4_cts');
    if e_level_skip > 0
        fprintf('Skipped reading %d files due to low number of e_levels\n', e_level_skip);
    end
else
    disp(['Can not find data file: ' fcdir]); 
end
