function [table, e_levels] = IMPORT_L4_ZERO_CTS(filename, startRow, endRow)
%IMPORT_L4_ZERO_CTS Import numeric data from a text file as column vectors.
%   [table, e_levels] = IMPORT_L4_ZERO_CTS(FILENAME) Reads data from text file FILENAME for the default
%   selection.
%
%   [TABLE, E_LEVELS] = IMPORT_L4_ZERO_CTS(FILENAME, STARTROW, ENDROW) Reads data from rows STARTROW
%   through ENDROW of text file FILENAME.
%
% Example:
%   [TABLE, e_levels] = IMPORT_L4_ZERO_CTS('RPCICA150317T04_000_L4_CORR.TAB');
%   [TABLE, e_levels] = IMPORT_L4_ZERO_CTS('RPCICA150317T04_000_L4_CORR.TAB', 1, 16384);
%

%% Initialize variables.
delimiter = ',';
if nargin<=2
    startRow = 1;
    endRow = inf;
end

%% Format string for each line of text:
%   column1: text (%s) time
%   column2: double (%f) azimuth
%   column3: double (%f) elevation
%   column4: double (%f) mass

%   column5-column12: double (%f) uncertainty 8 levels
% or
%   column5-column36: double (%f) uncertainty 32 levels
% or
%   column5-column100: double (%f) uncertainty 96 levels
%
% For more information, see the TEXTSCAN documentation.
formatSpec = ['%s' repmat('%f', 1, 3) repmat('%f', 1, 96) '%[^\n\r]'];

%% Open the text file.
fileID = fopen(filename,'r');

%% Read a line and determine how many e_levels we got
tmp = fgets(fileID);
dataArray = textscan(tmp, formatSpec, endRow(1)-startRow(1)+1, 'Delimiter', delimiter, 'EmptyValue' ,NaN,'HeaderLines', startRow(1)-1, 'ReturnOnError', false);
if isnan(dataArray{1+3+8+1})
    e_levels = 8;
elseif  isnan(dataArray{1+3+32+1})
    e_levels = 32;
else
    e_levels = 96;
end

%% Rewind & remake formatSpec
frewind(fileID);
formatSpec = ['%s' repmat('%f', 1, 3) repmat('%f', 1, e_levels) '%[^\n\r]'];

%% Read columns of data according to format string.
% This call is based on the structure of the file used to generate this code. If
% an error occurs for a different file, try regenerating the code from the
% Import Tool.
dataArray = textscan(fileID, formatSpec, endRow(1)-startRow(1)+1, 'Delimiter', delimiter, 'EmptyValue' ,NaN,'HeaderLines', startRow(1)-1, 'ReturnOnError', false);
for block=2:length(startRow)
    frewind(fileID);
    dataArrayBlock = textscan(fileID, formatSpec, endRow(block)-startRow(block)+1, 'Delimiter', delimiter, 'EmptyValue' ,NaN,'HeaderLines', startRow(block)-1, 'ReturnOnError', false);
    for col=1:length(dataArray)
        dataArray{col} = [dataArray{col};dataArrayBlock{col}];
    end
end

%% Close the text file.
fclose(fileID);

%% Post processing for unimportable data.
% No unimportable data rules were applied during the import, so no post
% processing code is included. To generate code which works for unimportable
% data, select unimportable cells in a file and regenerate the script.

%% Allocate imported array to column variable names
table.time = datenum(dataArray{1}, 'yyyy-mm-ddTHH:MM:SS.FFF');
table.azim = dataArray{2};
table.elev = dataArray{3};
table.mass = dataArray{4};
table.uncertainty = [dataArray{5:end-1}];
table.uncertainty(table.uncertainty>=9.999e99) = nan;

