function l4_corr = READICA_L4_CORR(varargin)
%READICA_L4_CORR Read RPCICA L4 CORRECTED data for one day.
%   l4_corr = READICA_L4_CORR() Reads data for a day. 
%        Skips data with less than 96 e_levels.
%        Asks for rootpath & date
%
%   l4_corr = READICA_L4_CORR(stretch) Reads data for a day. 
%        Stretches data with less than 96 e_levels to 96 levels.
%        Asks for rootpath & date
%
%   l4_corr = READICA_L4_CORR(stretch, date) Reads data for date given. 
%        If stretch is true, stretches data with less than 96 e_levels to 96 levels.
%        Asks for rootpath
%
%   l4_corr = READICA_L4_CORR(stretch, date, rootpath) Reads data for date given.
%        If stretch is true, stretches data with less than 96 e_levels to 96 levels.
%
% See also: IMPORT_L4_CORR, PLOT_ICAL4_CORR
stretch = false;
data_date = '';
rootpath = '';
if nargin > 2
    if ~ischar(varargin{3})
        help(mfilename)
        return
    end
    rootpath = varargin{3};
end
if nargin > 1
    if ~ischar(varargin{2})
        help(mfilename)
        return
    end
    data_date = varargin{2};
end
if nargin > 0
    if ~islogical(varargin{1})
        help(mfilename)
        return
    end
    stretch = varargin{1};
end

%% Find rootpath of archive where directory DATA is located
if isempty(rootpath) || ~exist(fullfile(rootpath, 'DATA'),'dir')
    for jj=1:5
        if exist(fullfile(rootpath, 'DATA'),'dir')
            break
        else
            rootpath = fullfile('..', rootpath);
        end
    end
end

%% Get path to data
icapath=fullfile(rootpath, 'DATA', 'CORR');
while ~exist(icapath, 'dir')
    fprintf(['This script is designed to run inside an archive.\n' ...
        'The DATA/CORR directory was not found.\n' ...
        'Please provide the path to the root of the archive i.e.:\n' ...
        '/home/user/psa/RO-C-RPCICA-4-ESC1-CORR-V1.0\n' ...
        '\n']);
    rootpath = input('path to the root of the archive:  ','s');
    icapath=fullfile(rootpath, 'DATA', 'CORR');
end

fprintf('rootpath: %s\n', rootpath);

if isempty(data_date)
    data_date=input('Date of data to import yyyy-mm-dd:  ','s');
end
dn = datenum(data_date);


cdir=fullfile(icapath, ...
    datestr(dn,'yyyy'), ...
    upper(datestr(dn,'mmm')), ...
    ['D' datestr(dn,'dd')]);
fprintf('cdir: %s\n', cdir);

%% Get files
fcdir=fullfile(cdir, '/RPCICA*.TAB');
filelist=dir(fcdir);

%% initialize matrices
l4_corr.time = [];
l4_corr.delta_t = [];
l4_corr.quality = [];
l4_corr.mode = [];
l4_corr.noise_reduction = [];
l4_corr.mtable = [];
l4_corr.pacclevel = [];
l4_corr.azim = [];
l4_corr.elev = [];
l4_corr.mass = [];
l4_corr.flux = [];
l4_corr.swversion = [];
l4_corr.uncertainty = [];

%% Find function to call
if (exist('IMPORT_L4_CORR') == 2)
    import4_corr = str2func('IMPORT_L4_CORR');
else
    import4_corr = str2func('import_l4_corr');
end

if (exist('IMPORT_L4_ZERO') == 2)
    import4_zero = str2func('IMPORT_L4_ZERO');
else
    import4_zero = str2func('import_l4_zero');
end

%% Read data
e_level_skip = 0;

if ~isempty(filelist)
    for jj=1:length(filelist)
        disp(filelist(jj).name);
        if ~isempty(strfind(filelist(jj).name,'_L4_CORR.TAB'))
            [table, e_levels] = import4_corr(fullfile(cdir, filelist(jj).name));
            lbl = strrep(filelist(jj).name, 'TAB', 'LBL');
            l4_lbl = READ_PDS_LBL(fullfile(cdir, lbl));
            curr_sw = str2double(l4_lbl.ROSETTA_COLON_ICA_SW_VERSION);
            zero = strrep(filelist(jj).name, '_CORR.TAB', '_ZERO.TAB');
            [ztable, ze_levels] = import4_zero(fullfile(cdir, zero));
        else
            % No support for other files
            continue;
        end
        if e_levels ~= 96 
            if stretch
                % possibly stretch 8 & 32 e_levels to 96 e_levels
                table.flux = [table.flux nan(size(table.time,1),96-e_levels)];
                ztable.uncertainty = [ztable.uncertainty nan(size(table.time,1),96-e_levels)];
            else
                e_level_skip = e_level_skip + 1;
                continue;
            end
        end
        
        if any(table.time ~= ztable.time)
            disp('time is not equal between corr & zero')
        end
        if any(table.azim ~= ztable.azim)
            disp('azim is not equal between corr & zero')
        end
        if any(table.elev ~= ztable.elev)
            disp('elev is not equal between corr & zero')
        end
        if any(table.mass ~= ztable.mass)
            disp('mass is not equal between corr & zero')
        end
        
        l4_corr.time=[l4_corr.time; table.time];
        l4_corr.delta_t=[l4_corr.delta_t; table.delta_t];
        l4_corr.quality=[l4_corr.quality; table.quality];
        l4_corr.mode= [l4_corr.mode; table.mode];
        l4_corr.noise_reduction=[l4_corr.noise_reduction; table.noise_reduction];
        l4_corr.mtable=[l4_corr.mtable; table.mtable];
        l4_corr.pacclevel=[l4_corr.pacclevel; table.pacclevel];
        l4_corr.azim=[l4_corr.azim; table.azim];
        l4_corr.elev=[l4_corr.elev; table.elev];
        l4_corr.mass=[l4_corr.mass; table.mass];
        l4_corr.flux=[l4_corr.flux; table.flux];
        l4_corr.swversion = [l4_corr.swversion; repmat(curr_sw, length(table.time), 1)];
        l4_corr.uncertainty = [l4_corr.uncertainty; ztable.uncertainty]; 
    end
    save(['ica' data_date '_corr.mat'], 'l4_corr');
    if e_level_skip > 0
        fprintf('Skipped reading %d files due to low number of e_levels\n', e_level_skip);
    end
else
    disp(['Can not find data file: ' fcdir]); 
end
