; Copyright (c) 2004, Stubbe F. Hviid.  All rights reserved.
;   Unauthorized reproduction prohibited.
;+
; NAME:
;   PRO p_set_comment, header, pos_or_tag, comment
;
; PURPOSE:
;   Set or change a comment in a PDS header
;
; CALLING SEQUENCE:
;   PRO p_set_comment, header, pos_or_tag, comment
;
; INPUTS:
;   header:     The PDS header structure to be modified
;   pos_or_tag: If integer value the input field means the absolute location (linenumber)
;               in the header where the comment should be written
;               If string value then field refers to a label tag
;               and the comment will be inserted just before the location of the tag
;   comment     string with the text to be written
;
; KEYWORD PARAMETERS:
;   REPLACE_EXISTING:     If set Replace any existing comment found at the location
;                         if not set the comment will be inserted
;   AFTER:                Is set the comment will be inserted after the specified label
;                         (Only valid for string values of pos_or_tag)
;   RELATIVE_SHIFT:       Add to index for final position
; OUTPUTS:
;   header:    Will contain the modified header structure
;
; RETURNS:
;   Nothing
;
; RESTRICTIONS:
;    The function can only be used for single line comments
;    Using it on multiline comments will probably cause curroptions
;
; EXAMPLE:
;   h =p_new_header()
;   p_setvalue, h, "TAG1", 0
;   p_setvalue, h, "TAG2", 1
;   p_set_comment, h, "TAG2", "Hello World"
;   p_set_comment, h, "TAG2", "Hello World New", /REPLACE_EXISTING
;   p_set_comment, h, 0, "Comment at beginning"
;
; MODIFICATION HISTORY:
;   Written by:  Stubbe F. Hviid, 12/05-2004
;-

PRO p_set_comment, header, pos_or_tag, comment, REPLACE_EXISTING=REPLACE_EXISTING, AFTER=AFTER, RELATIVE_SHIFT=RELATIVE_SHIFT
  
  tcomment = '/* ' + comment + ' */'
  
  ; first handle tag input 
  if size(pos_or_tag, /type) eq 7 then begin
    tmp = p_value(header, pos_or_tag, FOUND=FOUND, INDEX=INDEX)
    if FOUND eq 0 then begin
      print, 'ERROR: Cannot locate tag: ' + strtrim(pos_or_tag)
      return
    endif
    pos_or_tag = INDEX
    if keyword_set(AFTER) then pos_or_tag++
    if pos_or_tag lt 0 then pos_or_tag = 0
  endif
  
  index = pos_or_tag
  
  if n_elements(RELATIVE_SHIFT) then index += RELATIVE_SHIFT

  if index ge n_elements(header.tags) OR index lt 0 then begin
    ; append the comment to the end of the label
     p_add_tag, header, '', tcomment
     return
  endif 

  if header.tags[index] eq '' AND keyword_set(REPLACE_EXISTING) then begin
    ; ensure that the existing comment is a single line comment
    if strpos(header.values[index], '/*') ge 0 AND strpos(header.values[index], '*/') ge 0 then begin
      header.values[index] = tcomment;
      return;
    endif
  endif
  
  p_add_tag, header, '', tcomment, INSERT_INDEX=index

END