; Copyright (c) 2004, Stubbe F. Hviid.  All rights reserved.
;   Unauthorized reproduction prohibited.
;+
; NAME:
;   PRO p_pds_rhead, file, h, OK=OK, VERBOSE=VERBOSE
;
; PURPOSE:
;   Read header structure from a PDS3 label
;
; CALLING SEQUENCE:
;    p_pds_rhead, file, header
;
; INPUTS:
;   file:   The file to read from
;
; OUTPUT:
;   header:   the output PDS header
;
; KEYWORD PARAMETERS:
;   OK          will be 1 on success else 0
;   VERBOSE     set to enable all progress print statements (only errors will be shown)
;
; MODIFICATION HISTORY:
;   Written by:  Stubbe F. Hviid, 12/05-2004
;                MK, 31 OCt. 2005, added check for string type of file argument to avoid subtle bug which creates an
;                                  infinite loop if a numerical value is erroneously given as file.
;-


FUNCTION read_str, fc, neof
    if fc.push ne '' then begin
       s = fc.push
       fc.push = ''
       return, s
    endif

    if eof(fc.unit) then begin
       neof = 0
       return, ''
    endif else neof = 1

    s = ''
    readf, fc.unit, s
    return, strtrim(s, 2)
END

PRO push_str, fc, s
    fc.push = s;
END


FUNCTION read_pds_token, fc

    s = read_str(fc, neof)

    ; check for empty
    if s eq '' then return, {type: 'empty', tag: '', value: '', comment: ''}

    ; check for comment
    if strmid(s, 0, 2) eq '/*' then begin

       s = strmid(s, 2)	; strip initial /* off the comment

       pe = strpos(s, '*/') ; test for end of comment marker

	   while pe lt 0 AND NOT eof(fc.unit) do begin
	      if n_elements(out) eq 0 then out = [s] else out = [out, s]
	      s = read_str(fc)
	      pe = strpos(s, '*/') ; test for end of comment marker
	   endwhile
	   if pe ge 0 then s = strmid(s, 0, pe)
	   if n_elements(out) eq 0 then out = [s] else out = [out, s]
       return, {type: 'comment', tag: '', value: out}
    endif

    ; hande normal tokens
    p = strpos(s, '=')
    if p gt 0 then begin
       tag = strtrim(strmid(s, 0, p-1), 2)

       value = strtrim( strmid(s, p+1, strlen(s)), 2)

       ; check for multi line items
       cc = strmid(value, 0, 1)
       if cc eq '"' then begin
         if strpos(value, '"', 1) lt 0 then begin
		   cont = 1
           while cont  AND neof do begin
              s = read_str(fc)
              value = value + '\n' + s

			  if strpos(s, '"') ge 0 then cont = 0

           endwhile
         endif
       endif

       if cc eq '(' then begin
         if strpos(value, ')', 1) lt 0 then begin
          while strpos(s, ')') lt 0  AND neof do begin
              s = read_str(fc)
              value = value + ' ' + s
          endwhile
         endif
       endif

       tag = strtrim(tag, 2)
       value = strtrim(value, 2)

       return, {type: 'tag', tag: tag, value: value}

    endif else begin
       if s eq 'END' then return, {type: 'end', tag: '', value: ''}
    endelse

    return, {type: 'empty', tag: '', value: ''}
END


PRO pds_parse_rhead, fc, header, OK=OK, VERBOSE=VERBOSE, HISTORY=HISTORY

    on_ioerror, handle_error

    ; initiate some variables
    tags = ['']
    values = ['']
    ns = ''
    OK=0

    ; verify that the file is a PDS3 file
    if NOT keyword_set(HISTORY) then begin
      token = read_pds_token(fc)
      if token.tag ne 'PDS_VERSION_ID' OR token.value ne 'PDS3' then begin
        free_lun, fc.unit
        return
      endif
      tags = [tags, 'PDS_VERSION_ID']
      values = [values, 'PDS3']
    endif

    found_end = 0
    while NOT eof(fc.unit) AND NOT found_end do begin
       token = read_pds_token(fc)

       if token.type eq 'end' then begin
         found_end = 1
         OK = 1
       endif else if token.type eq 'comment' then begin
       	 for ti = 0, n_elements(token.value)-1 do begin
         	tags = [tags, '']
         	values = [values, token.value[ti]]
         endfor
       endif else if token.type eq 'tag' then begin
         if token.tag eq 'GROUP' then begin
          tags = [tags, 'G>']
          values = [values, token.value]
          if ns eq '' then ns = token.value else ns = ns + '.' + token.value
         endif else if token.tag eq 'END_GROUP' then begin
          tags = [tags, 'G<']
          values = [values, token.value]
          if ns eq token.value then begin
              ns = ''
          endif else begin
              len = strlen(token.value)
              ns = strmid(ns, 0, strlen(ns) - len - 1)
          endelse
         endif else if token.tag eq 'OBJECT' then begin
          tags = [tags, 'O>']
          values = [values, token.value]
          if ns eq '' then ns = token.value else ns = ns + '.' + token.value
         endif else if token.tag eq 'END_OBJECT' then begin
          tags = [tags, 'O<']
          values = [values, token.value]
          if strmid(ns, 1) eq token.value then begin
              ns = ''
          endif else begin
              len = strlen(token.value)
              ns = strmid(ns, 0, strlen(ns) - len - 1)
          endelse
         endif else begin
          tags = [tags, token.tag]
          values = [values, token.value]
         endelse
       endif
    endwhile

    if n_elements(tags) gt 1 then begin
       tags = tags[1:*]
       values = values[1:*]
    endif

    header = {type: 'TAGHDR', filetype: 'PDS3', tags: tags, values: values}

    return

    handle_error:
    OK = 0

END



PRO p_pds_rhead, file, header, OK=OK, VERBOSE=VERBOSE

	on_ioerror, handle_error

    if keyword_set(VERBOSE) then print, 'Reading header of: ' + file

    openr, unit,file, /GET_LUN

    fc = {unit: unit, push: ''}


    ; read primary header
    pds_parse_rhead, fc, header, OK=OK, VERBOSE=VERBOSE

	if OK eq 0 then return		; not a PDS file

    if OK then header = p_set_struct_tag(header, 'filename', file)

    phist = p_value(header, '^HISTORY')
    if phist ne '' then begin
        point_lun, fc.unit, (long(phist)-1) * long(p_value(header, 'RECORD_BYTES'))
         pds_parse_rhead, fc, history, OK=OK_HIST, VERBOSE=VERBOSE, /HISTORY
         if OK_HIST then header = p_set_struct_tag(header, 'history', history)
    endif

    ; close file
    close, unit
    free_lun, unit

    OK=1
    return;

    handle_error:
    OK=0


END