; Copyright (c) 2010, Stubbe F. Hviid.  All rights reserved.
;   Unauthorized reproduction prohibited.
;+
; NAME:
;	FUNCTION p_newcsv, data, COLUMNS=COLUMNS, DESCRIPTIONS=DESCRIPTIONS, DATATYPES=DATATYPES
;
; PURPOSE:
;	Create new csv data structure from data arrays
;
; CATEGORY:
;	Data conversion
;
; CALLING SEQUENCE:
; 	Result = p_newcsv(data, COLUMNS=COLUMNS, DESCRIPTIONS=DESCRIPTIONS, DATATYPES=DATATYPES)
;
; INPUTS:
;	data:	a 1 or 2 dimensional data array of a scalar data type (string, real or integer)
;			formal must me either array[nrows] or array[ncols, nrows]
;
; KEYWORD PARAMETERS:
;	COLUMNS:		Specify the column names to be used (strarr with ncols elements)
;	DESCRIPTIONS:	Specify the column description/unit fields (strarr with ncols elements)
;   DATATYPES:		Specify the IDL data types to associate with the column
;
; OUTPUTS:
;	Result will be a data structure with the form:
;          {type: 'csvdata', filename: file, columns: names, descriptions: descriptions, datatypes: datatypes, data: data}
;
;
; RESTRICTIONS:
;	The input file must be in ascii format with the row fields separated by either ' ', ';', ',' or TAB
;
; EXAMPLE:
;	data = p_newcsv(indgen(5, 20), COLUMNS=['a', 'b', 'c', 'd', 'e'])	; create a csv strcuture with 5 columns and 20 rows with the column names a,b,c,d,e
;
; MODIFICATION HISTORY:
; 	Written by:	Stubbe F. Hviid   Nov 2010
;-
FUNCTION p_newcsv, data, COLUMNS=COLUMNS, DESCRIPTIONS=DESCRIPTIONS, DATATYPES=DATATYPES

	s = size(data, /N_DIMENSIONS)

	if s eq 1 then begin
		ncols = 1
		nrows = n_elements(data)
	endif else if s eq 2 then begin
		s = size(data, /dim)
		ncols = s[0]
		nrows = s[1]
	endif else begin
		print, 'ERROR: input data must be one or two demensional'
		return, -1
	endelse

	if n_elements(COLUMNS) eq 0 then begin
		COLUMNS = strarr(ncols)
		for i=0, ncols-1 do COLUMNS[i] = 'COL' + strtrim(i, 2)
	endif

	if n_elements(DESCRIPTIONS) eq 0 then DESCRIPTIONS = strarr(ncols)
	if n_elements(DATATYPES) eq 0 then begin
		DATATYPES = bytarr(ncols)
		use_auto_datatypes = 1
	endif else use_auto_datatypes = 0

	type = size(data, /type)
	if type eq 7 AND use_auto_datatypes eq 1 then begin
		for i=0, ncols-1 do begin
			if ch eq '"' OR ch eq "'" then begin
				DATATYPES[j] = 7
			endif else begin
				if strpos(tt, '.') ge 0 then DATATYPES[j] = 5 else DATATYPES[j] = 3
			endelse
		endfor
	endif else begin
		DATATYPES[*] = type
	endelse


	return, {type: 'csvdata', filename: '', columns: COLUMNS, descriptions: DESCRIPTIONS, datatypes: byte(DATATYPES), data: strtrim(string(data), 2)}


END