; Copyright (c) 2004, Stubbe F. Hviid.  All rights reserved.
;   Unauthorized reproduction prohibited.
;+
; NAME:
;   PRO p_fits_rhead, file, header, OK=OK, VERBOSE=VERBOSE
;
; PURPOSE:
;   Read header structure from a FITS header
;
; CALLING SEQUENCE:
;    p_fits_rhead, file, header, status
;
; INPUTS:
;   file:   The file to read from
;
; OUTPUT:
;   header:   the output PDS header
;
; KEYWORD PARAMETERS:
;   OK          will be 1 on success else 0
;   VERBOSE     set to enable all progress print statements (only errors will be shown)
;
; MODIFICATION HISTORY:
;   Written by:  Stubbe F. Hviid, 12/05-2004
;                MK, 31 OCt. 2005, added check for string type of file argument to avoid subtle bug which creates an
;                                  infinite loop if a numerical value is erroneously given as file.
;-

FUNCTION lookup_fits_entry, tags, values, tag
	s = where(tags eq tag)
	if s[0] ge 0 then return, values[s[0]]

	return, ''
END


PRO p_fits_rhead, file, header, OK=OK, VERBOSE=VERBOSE

	on_ioerror, handle_error

	openr, unit, file, /GET_LUN

	tsthdr = bytarr(6)
	readu, unit,tsthdr

	if string(tsthdr) ne "SIMPLE" then begin
		free_lun, unit
		OK=0
		return
	endif

	; reopen file
	free_lun, unit
	openr, unit, file, /GET_LUN
	entry_count = 0


	tags = ['']
	values = ['']
	comments = ['']

	done = 0
	while NOT eof(unit) AND done eq 0 do begin
		entry = bytarr(80)
		readu, unit, entry
		entry = strtrim(entry)
		entry_count++
		if entry eq 'END' then begin
			done = 1
		endif else begin
			ntag = ''
			nvalue = ''
			ncmt = ''

			; read tag
			p = strpos(entry, '=')
			if p ge 0 then ntag = strtrim(strmid(entry, 0, p-1))
			entry = strmid(entry, p+1)

			; check for comment
			p = strpos(entry, '/')
			p1 = strpos(entry, "'")
			if p gt 0 AND p1 gt 0 then begin
				p1 = strpos(entry, "'", p1+1)
				if p1 gt p then begin
					p = strpos(entry, '/', p1)
				endif
			endif

			if p ge 0 then begin
				ncmt = strtrim(strmid(entry, p+1))
				entry = strmid(entry, 0, p-1)
			endif

			v = strtrim(entry)
			if strmid(v, 0, 7) eq 'COMMENT' then begin
				ntags = ''
				v = '/*' + strmid(entry, 8) + '*/
			endif
			nvalue = strtrim(v, 2)


			tags = [tags, ntag]
			values = [values, nvalue]
			comments = [comments, ncmt]
		endelse
	endwhile

	; close file
	free_lun, unit

	; calc some information
	lblsize = (long(entry_count) * 80L) / 2880L

	if (float(lblsize) * 2880.0) lt (float(entry_count) * 80.0) then begin
		lblsize++
	endif


	; generate basic header
	tags = ['PDS_VERSION_ID', 'RECORD_TYPE', 'RECORD_BYTES', 'LABEL_RECORDS', '^IMAGE', tags[1:*]]
	values = ['PDS3', 'FIXED_LENGTH', '2880', strtrim(lblsize,2), strtrim(lblsize + 1, 2), values[1:*]]
	comments = ['','','', '', '', comments[1:*]]

	tags = [tags, 'O>'] & values = [values, 'IMAGE']

	; generate IMAGE object
	tags = [tags, 'INTERCHANGE_FORMAT'] & values = [values, 'BINARY']

	line_samples = 1
	lines = 1
	bands = 1

	naxis = long(lookup_fits_entry(tags, values, 'NAXIS'))

	if naxis ge 1 then line_samples = long(lookup_fits_entry(tags, values, 'NAXIS1'))
	if naxis ge 2 then lines        = long(lookup_fits_entry(tags, values, 'NAXIS2'))
	if naxis ge 3 then bands        = long(lookup_fits_entry(tags, values, 'NAXIS3'))

	tags = [tags, 'LINES'] & values = [values, strtrim(lines, 2)]
	tags = [tags, 'LINE_SAMPLES'] & values = [values, strtrim(line_samples, 2)]
	tags = [tags, 'BANDS'] & values = [values, strtrim(bands, 2)]

	fmt = long(lookup_fits_entry(tags, values, 'BITPIX'))

    if fmt eq 8 then begin
    	sample_type = 'MSB_UNSIGNED_INTEGER'
    	sample_bits = 8
	endif else if fmt eq 16 then begin
    	sample_type = 'MSB_INTEGER'
    	sample_bits = 16
	endif else if fmt eq 32 then begin
    	sample_type = 'MSB_INTEGER'
    	sample_bits = 32
	endif else if fmt eq -32 then begin
    	sample_type = 'IEEE_REAL'
    	sample_bits = 32
	endif else if fmt eq -64 then begin
    	sample_type = 'IEEE_REAL'
    	sample_bits = 64
    endif

	tags = [tags, 'SAMPLE_TYPE'] & values = [values, strtrim(sample_type, 2)]
	tags = [tags, 'SAMPLE_BITS'] & values = [values, strtrim(sample_bits, 2)]

	bzero = long(lookup_fits_entry(tags, values, 'BZERO'))
	if bzero ne 0 then begin
		tags = [tags, 'OFFSET']
		values = [values, strtrim(bzero, 2)]
	endif



	tags = [tags, 'O<'] & values = [values, 'IMAGE']

	; build header
	header = {type: 'TAGHDR', filetype: 'FITS', tags: tags, values: values, comments: comments}

	OK = 1
	return



    handle_error:
    OK=0


END