
; Simple program that takes the start ephemeris time as an input and
; returns the Mission Phase abbreviation, Phase name, target code,
; target name and target type for use in the PDS label files. Many of
; the returned values for a given phase are specified by RSOC.
;
; v. 1 2006-10-18  A.J. Steffl
; v. 2 2007-01-22  AJS: update to include observations through end of
;  2006 
; v. 3 2007-03-05 AJS: Update to include observations through end of
; MSB and Jupiter observations
; v. 4 2008-06-11 AJS: Update to include ESB2 and PC8
; v. 5 2008-07-16 AJS: Use dates from issue 2 of mission calendar
; v. 6 2008-09-02 AJS: Update to include Steins fly-by
; v. 7 2009-05-14 AJS: Added OBSERVATION_TYPE keyword and CODMAC level 4
; v. 8 2009-06-09 AJS: Modified OBSERVATION_TYPE keyword to match archive conventions
; v. 9 2010-04-24 AJS: Added PC12 info

PRO ralice_target, levelnum, start_et, phaseabbrev, phasename, obstype, target_id, target_name, $
                   target_type, dataset_target_name, spice_target_name, target_desc

;initialize target description variable
target_desc = ''

;Mission phases
CSPICE_UTC2ET, '2004-03-02T0', leop_beg_et  ; Launch
CSPICE_UTC2ET, '2004-03-05T0', cvp1_beg_et  ; Commissioning part 1
CSPICE_UTC2ET, '2004-06-07T0', cr1_beg_et   ; Cruise 1
CSPICE_UTC2ET, '2004-09-06T0', cvp2_beg_et  ; Commissioning part 2
CSPICE_UTC2ET, '2004-10-17T0', ear1_beg_et  ; Earth Swing-by 1
CSPICE_UTC2ET, '2005-04-05T0', cr2_beg_et   ; Cruise 2
CSPICE_UTC2ET, '2006-07-29T0', mar_beg_et   ; Mars Swing-by
CSPICE_UTC2ET, '2007-05-29T0', cr3_beg_et   ; Cruise 3 
CSPICE_UTC2ET, '2007-09-13T0', ear2_beg_et  ; Earth Swing-by 2
CSPICE_UTC2ET, '2008-01-28T0', cr4_1_beg_et   ; Cruise 4 part 1
CSPICE_UTC2ET, '2008-08-04T0', steins_beg_et  ; Steins flyby 
CSPICE_UTC2ET, '2008-10-06T0', cr4_2_beg_et   ; Cruise 4 part 2
CSPICE_UTC2ET, '2009-09-14T0', ear3_beg_et  ; Earth Swing-by 3
CSPICE_UTC2ET, '2009-12-14T0', cr5_beg_et  ; Cruise 5
CSPICE_UTC2ET, '2010-06-07T0', ast2_beg_et  ; Lutetia Fly-by
CSPICE_UTC2ET, '2010-09-11T0', rvm1_beg_et  ; Rendevous manoeuvre 1
CSPICE_UTC2ET, '2011-07-14T0', cr6_beg_et  ; Earth Swing-by 3

CASE 1 OF
; Launch mission phase (no data)
   start_et LT cvp1_beg_et : BEGIN
      phaseabbrev = 'LEOP'
      phasename = '"LAUNCH"'
      obstype = '"COMMISSIONING"'
      target_name = 'CHECKOUT'
      target_id = 'X'
      dataset_target_name = 'CHECK'
      target_type = '"N/A"'
      spice_target_name = '"N/A"'
   END

; CVP1: Commissioning 1 mission phase
   start_et LE cr1_beg_et : BEGIN
      phaseabbrev = 'CVP1'
      phasename = '"COMMISSIONING 1"'
      obstype = '"COMMISSIONING"'
      target_id = 'C/CAL/X'
      dataset_target_name = '2002T7/CAL/CHECK'

      CSPICE_UTC2ET, '2004-04-22T21:46:52', cvp1_cal1_beg
      CSPICE_UTC2ET, '2004-04-30T0', lineart7_beg
      CSPICE_UTC2ET, '2004-05-28T0', hd203245_beg
      CSPICE_UTC2ET, '2004-05-28T22', alpha_peg_beg
      CSPICE_UTC2ET, '2004-05-29T01:30', alpha_peg_end
      CASE 1 OF 
         start_et GE cvp1_cal1_beg AND start_et LT lineart7_beg : BEGIN
            target_name = 'CALIBRATION'
            target_type = 'CALIBRATION'
            target_desc = 'Sky background'
            spice_target_name = '"N/A"'
          END
         start_et GE lineart7_beg AND start_et LT hd203245_beg : BEGIN
            target_name = 'C/LINEAR (2002 T7)'
            target_type = 'COMET'
            target_desc = 'C/2002 T7 LINEAR'
            spice_target_name = '1000418' ; SPICE doesn't seem to recognize this...
         END
         start_et GE hd203245_beg AND start_et LT alpha_peg_beg : BEGIN
            target_name = 'CALIBRATION'
            target_desc = 'HD 203245'
            target_type = 'CALIBRATION'
            spice_target_name = '"N/A"'
         END
         start_et GE alpha_peg_beg AND start_et LT alpha_peg_end : BEGIN
            target_name = 'CALIBRATION'
            target_desc = 'Alpha Peg'
            target_type = 'CALIBRATION'
            spice_target_name = '"N/A"'
         END
         ELSE : BEGIN
            target_name = 'CHECKOUT'
            target_type = '"N/A"'
            spice_target_name = '"N/A"'
            target_desc = 'Instrument Checkout'
         END
      ENDCASE
   END

; CR1: Cruise 1 mission phase
   start_et LE cvp2_beg_et : BEGIN
      phaseabbrev = 'CR1'
      phasename = '"CRUISE 1"'
      obstype = '"COMMISSIONING"'
      target_name = 'CHECKOUT'
      target_type = 'N/A'
      target_id = 'X'
      dataset_target_name = 'CHECK'
      spice_target_name = '"N/A"'
      target_desc = 'Instrument Checkout'
   END

; CVP2: Commissioning 2 Mission Phase
   start_et LE ear1_beg_et : BEGIN
      phaseabbrev = 'CVP2'
      phasename = '"COMMISSIONING 2"'
      obstype = '"COMMISSIONING"'
      target_id = 'CAL/X'
      dataset_target_name = 'CAL/CHECK'

      CSPICE_UTC2ET,'2004-09-27T04', gamma_grus_beg
      CSPICE_UTC2ET,'2004-09-27T08', gamma_grus_end
      CSPICE_UTC2ET,'2004-10-04T05', alpha_grus_beg
      CSPICE_UTC2ET,'2004-10-04T10', alpha_grus_end
      CASE 1 OF
         start_et GE gamma_grus_beg AND start_et LT gamma_grus_end : BEGIN
            target_name = 'CALIBRATION'
            target_desc = 'Gamma Gru'
            target_type = 'CALIBRATION'
            spice_target_name = '"N/A"'
         END
         start_et GE alpha_grus_beg AND start_et LT alpha_grus_end : BEGIN
            target_name = 'CALIBRATION'
            target_desc = 'Alpha Gru'
            target_type = 'CALIBRATION'
            spice_target_name = '"N/A"'
         END
         ELSE : begin
            target_name = 'CHECKOUT'
            target_type = '"N/A"'
            spice_target_name = '"N/A"'
            target_desc = 'Instrument Checkout'
         END
      ENDCASE
   END

; ESB1: Earth Swing By 1 mission phase
   start_et LE cr2_beg_et : BEGIN
      phaseabbrev = 'EAR1'
      phasename = '"EARTH SWING-BY 1"'
      obstype = '"EARTH SWINGBY 1"'
      target_name = 'CALIBRATION'
      target_type = 'CALIBRATION'
      CASE levelnum OF 
         '2': BEGIN
            dataset_target_name = 'CAL/CHECK'
            target_id = 'CAL/X'
         END
         '3':BEGIN
            dataset_target_name = 'CAL'
            target_id = 'CAL'
         END
         '4':BEGIN
            dataset_target_name = 'CAL'
            target_id = 'CAL'
         END
      ENDCASE

;     spice_target_name = 'MOON'
      CSPICE_UTC2ET,'2005-03-08T04:45:00', moon_ff1_beg
      CSPICE_UTC2ET,'2005-03-08T06:45:00', moon_ff1_end
      CSPICE_UTC2ET,'2005-03-08T06:47:00', sky1_beg
      CSPICE_UTC2ET,'2005-03-08T07:15:00', sky1_end
      CSPICE_UTC2ET,'2005-03-08T07:15:00', moon_ff2_beg
      CSPICE_UTC2ET,'2005-03-08T09:15:00', moon_ff2_end
      CSPICE_UTC2ET,'2005-03-08T09:17:00', sky2_beg
      CSPICE_UTC2ET,'2005-03-08T09:40:00', sky2_end
      CSPICE_UTC2ET,'2005-03-08T09:45:00', moon_ff3_beg
      CSPICE_UTC2ET,'2005-03-08T11:45:00', moon_ff3_end
      CSPICE_UTC2ET,'2005-03-09T04:45:00', moonscatter_beg
      CSPICE_UTC2ET,'2005-03-09T09:15:00', moonscatter_end
      CSPICE_UTC2ET,'2005-03-11T08:00:00', moon_flux1_beg
      CSPICE_UTC2ET,'2005-03-11T12:31:00', moon_flux1_end
      CSPICE_UTC2ET,'2005-03-26T00:00:00', moon_flux2_beg
      CSPICE_UTC2ET,'2005-03-26T06:45:00', moon_flux2_end
      CASE 1 OF
         (start_et GE moon_ff1_beg AND start_et LT moon_ff1_end) OR $
           (start_et GE moon_ff2_beg AND start_et LT moon_ff2_end) OR $
           (start_et GE moon_ff3_beg AND start_et LT moon_ff3_end) OR $
           (start_et GE moon_flux1_beg AND start_et LT moon_flux1_end) OR $
           (start_et GE moon_flux2_beg AND start_et LT moon_flux2_end) : BEGIN
            target_name = 'MOON'
            target_type = 'SATELLITE'
            target_desc = "Earth's moon"
            spice_target_name = 'MOON'
         END
         start_et GE sky1_beg AND start_et LT sky1_end : BEGIN
            target_desc = 'Sky background'
            spice_target_name = '"N/A"'
         END
         start_et GE sky2_beg AND start_et LT sky2_end : BEGIN
            target_desc = 'Sky background'
            spice_target_name = '"N/A"'
         END
         start_et GE moonscatter_beg AND start_et LT moonscatter_end : BEGIN
            target_desc = 'Lunar scattered light'
            spice_target_name = 'MOON'
            target_name = 'MOON'
            target_type = 'SATELLITE'
         END
         start_et GE moon_flux2_end : BEGIN
            target_desc = 'Instrument Checkout'
            obstype = '"PASSIVE CHECKOUT 0"'
            spice_target_name = '"N/A"'
         END
         ELSE : begin
            target_desc = 'Moon'
            spice_target_name = 'MOON'
            target_type = 'SATELLITE'
         END 
      ENDCASE 
   END

; CR2: Cruise 2 mission phase (inc. Deep Impact)
   start_et LE mar_beg_et : BEGIN
      phaseabbrev = 'CR2'
      phasename = '"CRUISE 2"'
      obstype = '"DEEP IMPACT"'
      target_id = 'C/CAL/X'
      dataset_target_name = '9P/CAL/CHECK'

      CSPICE_UTC2ET,'2005-06-27T00', rho_leo_beg
      CSPICE_UTC2ET,'2005-06-28T23', tempel1_beg
      CSPICE_UTC2ET,'2005-07-14T05', tempel1_end
      CSPICE_UTC2ET,'2005-10-01T00', pc1_beg
      CSPICE_UTC2ET,'2005-10-01T11', pc1_end

      CASE 1 OF
         start_et GE rho_leo_beg AND start_et LT tempel1_beg : BEGIN
            target_name = 'CALIBRATION'
            target_desc = 'Rho Leo'
            target_type = 'CALIBRATION'
            spice_target_name = '"N/A"'
         END
         start_et GE tempel1_beg AND start_et LT tempel1_end : BEGIN
            target_name = '9P/TEMPEL 1 (1867 G1)'
            target_type = 'COMET'
            spice_target_name = 'TEMPEL 1'
            target_desc = 'Comet 9P/Tempel 1 (1867 G1)'
         END
         start_et GE pc1_beg AND start_et LT pc1_end : BEGIN
            target_name = 'CHECKOUT'
            target_type = '"N/A"'
            obstype = '"PASSIVE CHECKOUT 1"'
            spice_target_name = '"N/A"'
            target_desc = 'Instrument Checkout'
         END
         ELSE : BEGIN
            target_name = 'CHECKOUT'
            target_type = '"N/A"'
            obstype = '"PASSIVE CHECKOUT 2"'
            spice_target_name = '"N/A"'
            target_desc = 'Instrument Checkout'
         END
      ENDCASE
   END

; MARS: Mars Swing By mission Phase
   start_et LE cr3_beg_et : BEGIN
      phaseabbrev = 'MARS'
      phasename = '"MARS SWING-BY"'
      obstype = '"MARS"'
      dataset_target_name = 'CAL/JUPITER/MARS'
      target_id = 'CAL/J/M'

      CSPICE_UTC2ET, '2006-08-26T00:00:00.00', pc3_beg
      CSPICE_UTC2ET, '2006-08-27T00:00:00.00', pc3_end
      CSPICE_UTC2ET, '2006-08-26T00:00:00.00', pc3_beg
      CSPICE_UTC2ET, '2006-08-27T00:00:00.00', pc3_end
      CSPICE_UTC2ET, '2006-12-03T01:15:00.00', mars1_beg
      CSPICE_UTC2ET, '2006-12-03T05:30:00.00', mars1_end
      CSPICE_UTC2ET, '2006-12-03T06:00:00.00', gammagru1_beg
      CSPICE_UTC2ET, '2006-12-03T12:29:00.00', gammagru1_end
      CSPICE_UTC2ET, '2006-12-03T13:15:00.00', sky1_beg
      CSPICE_UTC2ET, '2006-12-04T16:20:30.00', sky1_end
      CSPICE_UTC2ET, '2006-12-10T03:00:00.00', saturn_beg
      CSPICE_UTC2ET, '2006-12-10T06:20:30.00', saturn_end
      CSPICE_UTC2ET, '2006-12-10T17:00:00.00', vega_beg
      CSPICE_UTC2ET, '2006-12-10T19:30:00.00', vega_end
      CSPICE_UTC2ET, '2006-12-16T09:30:00.00', lac10_beg
      CSPICE_UTC2ET, '2006-12-16T12:05:30.00', lac10_end
      CSPICE_UTC2ET, '2006-12-17T01:45:00.00', zetacas_beg
      CSPICE_UTC2ET, '2006-12-17T06:14:00.00', zetacas_end
      CSPICE_UTC2ET, '2007-02-23T12:00:00.00', alphalyr_beg
      CSPICE_UTC2ET, '2007-02-23T14:05:30.00', alphalyr_end
      CSPICE_UTC2ET, '2007-02-24T17:59:12.00', mars_swingby_beg
      CSPICE_UTC2ET, '2007-02-26T08:53:31.00', mars_swingby_end
      CSPICE_UTC2ET, '2007-02-28T03:45:00.00', jupiter_beg
      CSPICE_UTC2ET, '2007-05-09T17:35:30.00', jupiter_end

      CASE 1 OF
         start_et GE pc3_beg AND start_et LT pc3_end : BEGIN
            target_name = 'CALIBRATION'
            target_type = 'CALIBRATION'
            obstype = '"PASSIVE CHECKOUT 3"'
            target_desc = 'PASSIVE CHECKOUT'
            spice_target_name = '"N/A"'
         END
         start_et GE pc3_end AND start_et LT mars1_beg : BEGIN
            target_name = 'CALIBRATION'
            target_type = 'CALIBRATION'
            target_desc = 'HV and Discriminator calibration'
            spice_target_name = '"N/A"'
            obstype = '"ACTIVE CHECKOUT 4"'
         END
         start_et GE mars1_beg AND start_et LT mars1_end : BEGIN
            target_name = 'MARS'
            target_type = 'PLANET'
            target_desc = 'Mars'
            spice_target_name = 'MARS'
            obstype = '"ACTIVE CHECKOUT 4"'
         END
         start_et GE gammagru1_beg AND start_et LT gammagru1_end : BEGIN
            target_name = 'CALIBRATION'
            target_type = 'CALIBRATION'
            target_desc = 'Gamma Gru'
            spice_target_name = '"N/A"'
            obstype = '"ACTIVE CHECKOUT 4"'
         END
         start_et GE sky1_beg AND start_et LT sky1_end : BEGIN
            target_name = 'CALIBRATION'
            target_type = 'CALIBRATION'
            target_desc = 'Sky background'
            spice_target_name = '"N/A"'
            obstype = '"ACTIVE CHECKOUT 4"'
         END
         start_et GE saturn_beg AND start_et LT saturn_end : BEGIN
            target_name = 'SATURN'
            target_type = 'PLANET'
            target_desc = 'Saturn'
            spice_target_name = 'SATURN'
            obstype = '"ACTIVE CHECKOUT 4"'
         END
         start_et GE vega_beg AND start_et LT vega_end : BEGIN
            target_name = 'CALIBRATION'
            target_type = 'CALIBRATION'
            target_desc = 'Alpha Lyr (Vega)'
            spice_target_name = '"N/A"'
            obstype = '"ACTIVE CHECKOUT 4"'
         END
         start_et GE lac10_beg AND start_et LT lac10_end : BEGIN
            target_name = 'CALIBRATION'
            target_type = 'CALIBRATION'
            target_desc = '10 Lac'
            spice_target_name = '"N/A"'
            obstype = '"ACTIVE CHECKOUT 4"'
         END
         start_et GE zetacas_beg AND start_et LT zetacas_end : BEGIN
            target_name = 'CALIBRATION'
            target_type = 'CALIBRATION'
            target_desc = 'Zeta Cas'
            spice_target_name = '"N/A"'
            obstype = '"ACTIVE CHECKOUT 4"'
         END
         start_et GE alphalyr_beg AND start_et LT alphalyr_end : BEGIN
            target_name = 'CALIBRATION'
            target_type = 'CALIBRATION'
            target_desc = 'Alpha Lyr (Vega)'
            spice_target_name = '"N/A"'
            obstype = '"ACTIVE CHECKOUT 4"'
         END
         start_et GE mars_swingby_beg AND start_et LT mars_swingby_end : BEGIN
            target_name = 'MARS'
            target_type = 'PLANET'
            target_desc = 'Mars'
            spice_target_name = 'MARS'
            obstype = '"MARS"'
         END
         start_et GE jupiter_beg AND start_et LT jupiter_end : BEGIN
            target_name = 'JUPITER'
            target_type = 'PLANET'
            target_desc = 'Jovian aurora and Io plasma torus'
            spice_target_name = 'JUPITER'
            obstype = '"JUPITER"'
         END
         start_et GE jupiter_end : BEGIN
            target_name = 'CALIBRATION'
            target_type = 'CALIBRATION'
            target_desc = 'Instrument Checkout'
            spice_target_name = '"N/A"'
            obstype = '"PASSIVE CHECKOUT 5"'
         END
         ELSE : BEGIN
            target_name = 'CALIBRATION'
            target_type = 'CALIBRATION'
            target_desc = 'See the observing log.'
            spice_target_name = '"N/A"'
            obstype = '"CALIBRATION"'
         END
      ENDCASE
   END

; ESB2: Earth Swing by 2 mission phase (includes PC6)
   start_et LE cr4_1_beg_et : BEGIN
      phaseabbrev = 'EAR2'
      phasename = '"EARTH SWING-BY 2"'
      obstype = '"ACTIVE CHECKOUT 6"'
      target_id = 'CAL/E'
      target_name = 'CALIBRATION'
      target_type = 'CALIBRATION'
      dataset_target_name = 'CAL/EARTH'
      spice_target_name = '"N/A"'
      target_desc = 'Alice calibration observation'

; PC6
      CSPICE_UTC2ET, '2007-09-13T05:00:00.00', vega_beg
      CSPICE_UTC2ET, '2007-09-16T18:00:00.00', vega_end
      CSPICE_UTC2ET, '2007-09-16T21:00:00.00', scat_light_beg
      CSPICE_UTC2ET, '2007-09-17T08:50:00.00', scat_light_end
      CSPICE_UTC2ET, '2007-09-23T10:00:00.00', wd_beg
      CSPICE_UTC2ET, '2007-09-23T15:30:00.00', wd_end

; ESB2
      CSPICE_UTC2ET, '2007-11-11T21:00:00.00', al03_beg
      CSPICE_UTC2ET, '2007-11-11T22:59:00.00', al03_end
      CSPICE_UTC2ET, '2007-11-13T18:45:23.00', al06a_beg
      CSPICE_UTC2ET, '2007-11-13T19:32:22.00', al06a_end
      CSPICE_UTC2ET, '2007-11-13T20:01:23.00', al20a_beg
      CSPICE_UTC2ET, '2007-11-13T20:37:23.00', al20a_end
      CSPICE_UTC2ET, '2007-11-13T20:37:23.00', al06b_beg
      CSPICE_UTC2ET, '2007-11-13T21:02:23.00', al06b_end
      CSPICE_UTC2ET, '2007-11-13T21:05:23.00', al10d_beg
      CSPICE_UTC2ET, '2007-11-13T21:54:03.00', al10d_end
      CSPICE_UTC2ET, '2007-11-13T22:02:23.00', al07_beg
      CSPICE_UTC2ET, '2007-11-14T00:02:23.00', al07_end
      CSPICE_UTC2ET, '2007-11-14T00:17:23.00', al10e_beg
      CSPICE_UTC2ET, '2007-11-14T00:47:23.00', al10e_end
      CSPICE_UTC2ET, '2007-11-14T02:02:23.00', al10f_beg
      CSPICE_UTC2ET, '2007-11-14T02:32:23.00', al10f_end
      CSPICE_UTC2ET, '2007-11-14T03:47:23.00', al08_beg
      CSPICE_UTC2ET, '2007-11-14T06:17:23.00', al08_end
      CSPICE_UTC2ET, '2007-11-14T06:32:23.00', al09_beg
      CSPICE_UTC2ET, '2007-11-14T08:32:23.00', al09_end
      CSPICE_UTC2ET, '2007-11-14T23:02:00.00', al20b_beg
      CSPICE_UTC2ET, '2007-11-15T00:40:00.00', al20b_end
      CSPICE_UTC2ET, '2007-11-15T01:05:00.00', al11_beg
      CSPICE_UTC2ET, '2007-11-15T01:59:10.00', al11_end
      CSPICE_UTC2ET, '2007-11-15T02:30:00.00', al12_beg
      CSPICE_UTC2ET, '2007-11-15T02:59:10.00', al12_end
      CSPICE_UTC2ET, '2007-11-15T05:00:00.00', al05_beg
      CSPICE_UTC2ET, '2007-11-16T07:27:40.00', al05_end
      CSPICE_UTC2ET, '2007-11-16T09:00:00.00', al20c_beg
      CSPICE_UTC2ET, '2007-11-16T10:30:00.00', al20c_end
      CSPICE_UTC2ET, '2007-11-16T10:45:00.00', al20e_beg
      CSPICE_UTC2ET, '2007-11-16T11:14:10.00', al20e_end
      CSPICE_UTC2ET, '2007-11-16T11:30:00.00', al20d_beg
      CSPICE_UTC2ET, '2007-11-16T13:00:00.00', al20d_end

      CASE 1 OF
; PC6
         start_et GE vega_beg AND start_et LT vega_end : $
            target_desc = 'Alpha Lyr (Vega)'

         start_et GE scat_light_beg AND start_et LT scat_light_end : $
            target_desc = 'Solar scattered light test'

         start_et GE wd_beg AND start_et LT wd_end : $
           target_desc = 'WD 0501-289'


; Start of ESB2
         start_et GE al03_beg AND start_et LT al03_end : BEGIN
            target_desc = 'Alpha Lyr (Vega)'
            obstype = '"ESB2"'
         END

; Earth Observations
         (start_et GE al06a_beg AND start_et LT al06a_end) OR $
           (start_et GE al20a_beg AND start_et LT al20a_end) OR $
           (start_et GE al06b_beg AND start_et LT al06b_end) OR $
           (start_et GE al12_beg AND start_et LT al12_end) OR $
           (start_et GE al20c_beg AND start_et LT al20c_end) OR $
           (start_et GE al20e_beg AND start_et LT al20e_end) : BEGIN
            target_name = 'EARTH'
            target_type = 'PLANET'
            target_desc = 'Earth'
            spice_target_name = 'EARTH'
            obstype = '"EARTH SWINGBY 2"'
         END

; Moon Observations
         (start_et GE al07_beg AND start_et LT al07_end) OR $
           (start_et GE al08_beg AND start_et LT al08_end) OR $
           (start_et GE al09_beg AND start_et LT al09_end) OR $
           (start_et GE al20b_beg AND start_et LT al20b_end) OR $
           (start_et GE al20d_beg AND start_et LT al20d_end) : BEGIN
            target_name = 'MOON'
            target_type = 'SATELLITE'
            target_desc = "Earth's Moon"
            spice_target_name = 'MOON'
            obstype = '"EARTH SWINGBY 2"'
         END

         start_et GE al20d_end : obstype = '"PASSIVE CHECKOUT 7"'
         ELSE:
      ENDCASE
   END 

; CR4A: Cruise 4-1 mission phase (includes PC8)
   start_et LE steins_beg_et : BEGIN
      phaseabbrev = 'CR4A'
      phasename = '"CRUISE 4-1"'
      target_id = 'CAL'
      target_name = 'CALIBRATION'
      target_type = 'CALIBRATION'
      dataset_target_name = 'CAL'
      spice_target_name = '"N/A"'
      target_desc = 'Alice calibration observation'
      obstype = '"ACTIVE CHECKOUT 8"'

      CSPICE_UTC2ET,'2008-07-12T18:09', vega_beg
      CSPICE_UTC2ET,'2008-07-12T21:20', vega_end
      CSPICE_UTC2ET,'2008-07-12T22:20', alp_gru_beg
      CSPICE_UTC2ET,'2008-07-13T01:30', alp_gru_end
      CSPICE_UTC2ET,'2008-07-13T03:00', hd93521_beg
      CSPICE_UTC2ET,'2008-07-12T15:10', hd93521_end

      CASE 1 OF
         start_et GE vega_beg AND start_et LT vega_end : $
           target_desc = 'Alpha Lyr (Vega)'
         start_et GE alp_gru_beg AND start_et LT alp_gru_end : $
           target_desc = 'Alpha Gru'
         start_et GE hd93521_beg AND start_et LT hd93521_end : $
           target_desc = 'HD 93521'
         ELSE:
      ENDCASE
   END

; AST1: Steins Fly-By
   start_et LE cr4_2_beg_et : BEGIN
      phaseabbrev = 'AST1'
      phasename = '"STEINS FLY-BY"'
      target_id = 'A/CAL'
      target_name = 'CALIBRATION'
      target_type = 'CALIBRATION'
      dataset_target_name = 'STEINS/CAL'
      spice_target_name = '"N/A"'
      target_desc = 'Alice calibration observation'
      obstype = '"STEINS FLYBY"'

      CSPICE_UTC2ET,'2008-09-02T19:45', gam_gru_beg
      CSPICE_UTC2ET,'2008-09-02T23:40', gam_gru_end
      CSPICE_UTC2ET,'2008-09-03T16:00', vega_beg
      CSPICE_UTC2ET,'2008-09-03T20:45', vega_end
      CSPICE_UTC2ET,'2008-09-04T09:30', steins_beg
      CSPICE_UTC2ET,'2008-09-05T23:50', steins_end
      CSPICE_UTC2ET,'2008-09-06T16:30', bkg_beg
      CSPICE_UTC2ET,'2008-09-07T06:44', bkg_end

      CASE 1 OF
         start_et GE vega_beg AND start_et LT vega_end : $
           target_desc = 'Alpha Lyr (Vega)'
         start_et GE gam_gru_beg AND start_et LT gam_gru_end : $
           target_desc = 'Gamma Gru'
         start_et GE steins_beg AND start_et LT steins_end : BEGIN
            target_desc = '(2867) Steins'
            target_name = '(2867) STEINS'
            target_type = 'ASTEROID'
            spice_target_name = 'STEINS'
         END
         start_et GE bkg_beg AND start_et LT bkg_end : $
           target_desc = 'Background for (2867) Steins observations'
         ELSE:
      ENDCASE
   END

; CR4B: PC9 (maybe 10?)
   start_et LE ear3_beg_et : BEGIN
      phaseabbrev = 'CR4B'
      phasename = '"CRUISE 4-2"'
      target_id = 'CAL'
      target_name = 'CALIBRATION'
      target_type = 'CALIBRATION'
      dataset_target_name = 'CAL'
      spice_target_name = '"N/A"'
      target_desc = 'Alice calibration observation'
      obstype = '"PASSIVE CHECKOUT 9"'
   END

; EAR3: PC10 & Earth Fly-By #3
   start_et LE cr5_beg_et : BEGIN
      phaseabbrev = 'EAR3'
      phasename = '"EARTH SWING-BY 3"'
      obstype = '"ACTIVE CHECKOUT 10"'
      target_id = 'CAL/E'
      target_name = 'CALIBRATION'
      target_type = 'CALIBRATION'
      dataset_target_name = 'CAL/EARTH'
      spice_target_name = '"N/A"'
      target_desc = 'Alice calibration observation'

; PC10
      CSPICE_UTC2ET, '2009-09-19T06:00', alf_gru_beg1
      CSPICE_UTC2ET, '2009-09-20T14:50', alf_gru_end1
      CSPICE_UTC2ET, '2009-09-26T00:50', vega_beg
      CSPICE_UTC2ET, '2009-09-26T03:07', vega_end
      CSPICE_UTC2ET, '2009-09-26T05:00', alf_gru_beg2
      CSPICE_UTC2ET, '2009-09-26T12:00', alf_gru_end2
      CSPICE_UTC2ET, '2009-09-26T12:25', zet_cas_beg
      CSPICE_UTC2ET, '2009-09-26T13:40', zet_cas_end
      CSPICE_UTC2ET, '2009-09-26T14:00', hd57682_beg1
      CSPICE_UTC2ET, '2009-09-26T18:40', hd57682_end1
      CSPICE_UTC2ET, '2009-09-26T19:15', bd28_beg1
      CSPICE_UTC2ET, '2009-09-27T07:40', bd28_end1
      CSPICE_UTC2ET, '2009-09-27T07:50', g191_beg
      CSPICE_UTC2ET, '2009-09-27T20:10', g191_end
      CSPICE_UTC2ET, '2009-09-27T20:20', eta_uma_beg
      CSPICE_UTC2ET, '2009-09-27T21:00', eta_uma_end
      CSPICE_UTC2ET, '2009-09-27T22:45', eps_ori_beg
      CSPICE_UTC2ET, '2009-09-27T23:35', eps_ori_end
      CSPICE_UTC2ET, '2009-09-29T06:15', hd57682_beg2
      CSPICE_UTC2ET, '2009-09-29T07:30', hd57682_end2
      CSPICE_UTC2ET, '2009-09-29T07:50', bd28_beg2
      CSPICE_UTC2ET, '2009-09-29T08:20', bd28_end2
      CSPICE_UTC2ET, '2009-09-29T08:30', lac10_beg
      CSPICE_UTC2ET, '2009-09-29T08:40', lac10_end

; ESB3
      CSPICE_UTC2ET, '2009-11-08T03:00', al11a_beg
      CSPICE_UTC2ET, '2009-11-08T07:15', al11a_end
      CSPICE_UTC2ET, '2009-11-10T18:00', alf_gru_beg3
      CSPICE_UTC2ET, '2009-11-10T19:30', alf_gru_end3
      CSPICE_UTC2ET, '2009-11-10T20:00', vega_beg2
      CSPICE_UTC2ET, '2009-11-10T03:07', vega_end2
      CSPICE_UTC2ET, '2009-11-08T03:00', al11a_beg
      CSPICE_UTC2ET, '2009-11-08T07:15', al11a_end
      CSPICE_UTC2ET, '2009-11-11T22:30', al12_beg
      CSPICE_UTC2ET, '2009-11-13T11:10', al12_end
      CSPICE_UTC2ET, '2009-11-13T15:20', al41_beg
      CSPICE_UTC2ET, '2009-11-13T11:10', al41_end
      CSPICE_UTC2ET, '2009-11-13T15:55', al14_beg
      CSPICE_UTC2ET, '2009-11-13T16:24', al14_end
      CSPICE_UTC2ET, '2009-11-13T16:35', al40_beg
      CSPICE_UTC2ET, '2009-11-13T17:05', al40_end
      CSPICE_UTC2ET, '2009-11-13T17:20', al07_beg
      CSPICE_UTC2ET, '2009-11-18T10:00', al07_end

      CASE 1 OF
; PC10
         (start_et GE alf_gru_beg1 AND start_et LT alf_gru_end1) OR $
         (start_et GE alf_gru_beg2 AND start_et LT alf_gru_end2) : $
           target_desc = 'alpha Gru'
         (start_et GE hd57682_beg1 AND start_et LT hd57682_end1) OR $
         (start_et GE hd57682_beg2 AND start_et LT hd57682_end2) : $
           target_desc = 'HD 57682'
         (start_et GE bd28_beg1 AND start_et LT bd28_end1) OR $
         (start_et GE bd28_beg2 AND start_et LT bd28_end2) : $
           target_desc = 'BD +28 4211'
         start_et GE g191_beg AND start_et LT g191_end : $
           target_desc = 'G191-B2B'
         start_et GE vega_beg AND start_et LT vega_end : $
           target_desc = 'alpha Lyr (Vega)'
         start_et GE zet_cas_beg AND start_et LT zet_cas_end : $
           target_desc = 'zeta Cas'
         start_et GE eta_uma_beg AND start_et LT eta_uma_end : $
           target_desc = 'eta UMa'
         start_et GE eps_ori_beg AND start_et LT eps_ori_end : $
           target_desc = 'eps Ori'
         start_et GE lac10_beg AND start_et LT lac10_end : $
           target_desc = '10 Lac'

; ESB 3
         start_et GE alf_gru_beg3 AND start_et LT alf_gru_end3 : BEGIN
           target_desc = 'alpha Gru'
           obstype = '"EARTH SWINGBY 3"'
        END
         start_et GE vega_beg2 AND start_et LT vega_end2 : BEGIN
           target_desc = 'alpha Lyr (Vega)'
           obstype = '"EARTH SWINGBY 3"'
        END
        (start_et GE al11a_beg AND start_et LT al11a_end) OR $
        (start_et GE al14_beg AND start_et LT al14_end)  OR $
        (start_et GE al07_beg AND start_et LT al07_end) : BEGIN
           target_name = 'MOON'
           target_type = 'SATELLITE'
           target_desc = "Earth's moon"
           spice_target_name = 'MOON'
           obstype = '"EARTH SWINGBY 3"'
        END
        start_et GE al12_beg AND start_et LT al12_end : BEGIN
            target_name = 'EARTH'
            target_type = 'PLANET'
            target_desc = 'Earth'
            spice_target_name = 'EARTH'
            obstype = '"EARTH SWINGBY 3"'
         END
        (start_et GE al41_beg AND start_et LT al41_end) OR $
        (start_et GE al40_beg AND start_et LT al40_end)  : BEGIN
           target_desc = 'rho Lup'
           obstype = '"EARTH SWINGBY 3"'
        END
         ELSE:
      ENDCASE
   END

   start_et LE ast2_beg_et : BEGIN ; PC12

      phaseabbrev = 'CR5'
      phasename = '"CRUISE 5"'
      obstype = '"ACTIVE CHECKOUT 12"'
      target_id = 'CAL'
      target_name = 'CALIBRATION'
      target_type = 'CALIBRATION'
      dataset_target_name = 'CAL'
      spice_target_name = '"N/A"'
      target_desc = 'Alice calibration observation'


      CSPICE_UTC2ET, '2010 Apr 22 08:00', AL01_beg ; passive checkout; 2 hr)
      CSPICE_UTC2ET, '2010 Apr 22 10:00', AL02A_beg ; initial decontamination; 25 hr, 22hr heater + 3hr cooldown)
      CSPICE_UTC2ET, '2010 Apr 23 17:40', AL03C_beg ; Cal Star, Narrow Center [HD 43153]; 4 hr)
      CSPICE_UTC2ET, '2010 Apr 23 22:40', AL03B_beg ; Cal Star, Narrow Center [zet Cas]; 1 hr)
      CSPICE_UTC2ET, '2010 Apr 24 00:40', AL11_beg  ; solar stray light cal; 14 hr 05 min)
      CSPICE_UTC2ET, '2010 Apr 28 00:00', AL10A_beg ; upset frequency test [EEPROM_0]; 24 hr)
      CSPICE_UTC2ET, '2010 Apr 30 20:00', AL03A_beg ; Cal Star, Narrow Center [Vega]; 1 hr)
      CSPICE_UTC2ET, '2010 Apr 30 21:15', AL12_beg  ; boresight cal, cross-slit scan [Vega]; 1 hr)
      CSPICE_UTC2ET, '2010 Apr 30 22:40', AL09_beg ; alignment with OSIRIS [Vega]; 4 hr 6 min)
      CSPICE_UTC2ET, '2010 May 02 11:30', AL10B_beg  ; upset frequency test [EEPROM_1]; 24 hr)
      CSPICE_UTC2ET, '2010 May 05 00:00', AL10C_beg  ; upset frequency test [EEPROM_2]; 24 hr)
      CSPICE_UTC2ET, '2010 May 06 00:00', AL10D_beg  ; upset frequency test [EEPROM_3]; 24 hr)
      CSPICE_UTC2ET, '2010 May 08 21:00', AL03D_beg  ; Cal Star, Narrow Center [G191-B2B]; 12hr)
      CSPICE_UTC2ET, '2010 May 13 22:20', ALIT01_beg ; interference test with ROSINA; 1 hr 35 min)
      CSPICE_UTC2ET, '2010 May 14 00:00', AL02B_beg  ; final decontamination; 25 hr, 22hr heater + 3hr cooldown)

      CASE 1 OF
         start_et GE AL03C_beg AND start_et LT AL03B_beg : target_desc = 'HD 43153'
         start_et GE AL03B_beg AND start_et LT AL11_beg  : target_desc = 'zeta Cas'
         start_et GE AL11_beg  AND start_et LT AL10A_beg : target_desc = 'solar stray light'
         start_et GE AL03A_beg AND start_et LT AL10B_beg : target_desc = 'alpha Lyr (Vega)'
         start_et GE AL03D_beg AND start_et LT ALIT01_beg: target_desc = 'G191-B2B'
       ELSE:
      ENDCASE
   END

; AST2: Lutetia flyby
   start_et LE rvm1_beg_et : BEGIN ; Lutetia

      phaseabbrev = 'AST2'
      phasename = '"LUTETIA FLY-BY"'
      obstype = ''
      target_id = 'A/CAL'
      target_name = '"(21) LUTETIA"'
      target_type = 'ASTEROID'
      dataset_target_name = 'CAL/LUTETIA'
      spice_target_name = 'LUTETIA'
      target_desc = 'Asteroid (21) Lutetia'


      CSPICE_UTC2ET, '2010-Jul-06 14:00:00', AL07_beg1 ;(dark calibration               ; 1 hr)
      CSPICE_UTC2ET, '2010-Jul-06 15:00:00', zet_cas_beg ; 1 hr each boresight)
      CSPICE_UTC2ET, '2010-Jul-08 14:30:00', AL21A_beg; (exosphere search, +26 deg roll      ; 8 hr 20 min)
      CSPICE_UTC2ET, '2010-Jul-09 03:45:00', AL21B_beg; (exosphere search, nominal roll      ; 5 hr 15 min)
      CSPICE_UTC2ET, '2010-Jul-09 09:15:00', AL21C_beg ; (exosphere search, -26 deg roll      ; 9 hr 40 min)
      CSPICE_UTC2ET, '2010-Jul-10 06:45:00', AL21D_beg ; (exosphere search, nominal roll      ; 4 hr 40 min)
      CSPICE_UTC2ET, '2010-Jul-10 11:00:00', AL07_beg2 ; (dark calibration                     ; 1 hr)
      CSPICE_UTC2ET, '2010-Jul-10 12:16:03', AL31_beg ; (closest approach obs ; 3 hr 49 min)
      CSPICE_UTC2ET, '2010-Jul-10 18:30:00', AL09_beg ; (boresight alignment, raster scan [lam Aql] ; 1 hr 45 min)
      CSPICE_UTC2ET, '2010-Jul-11 04:45:00', AL32_beg ; (closest approach bkg                       ; 6 hr)
      CSPICE_UTC2ET, '2010-Jul-11 10:45:00', AL07_beg3 ; (dark calibration                           ; 1 hr)
      CSPICE_UTC2ET, '2010-Jul-11 12:45:00', AL22_beg ; (exosphere bkg 7.5 hr, 2.5 hr at each roll angle: 0,+/-26 deg)
      CSPICE_UTC2ET, '2010-Jul-11 22:45:00', vega_beg ; (stellar cal [Vega]  1 hr each boresight)
      CASE 1 OF
         (start_et GE AL07_beg1 AND start_et LT zet_cas_beg) OR $
           (start_et GE AL07_beg2 AND start_et LT AL31_beg) OR $
           (start_et GE AL07_beg3 AND start_et LT AL22_beg): BEGIN
            target_desc = 'Dark'
            target_name = 'CALIBRATION'
            target_type = 'CALIBRATION'
            spice_target_name = '"N/A"'
         END
         start_et GE zet_cas_beg AND start_et LT AL21A_beg : BEGIN
            target_desc = 'zeta Cas'
            target_name = 'CALIBRATION'
            target_type = 'CALIBRATION'
            spice_target_name = '"N/A"'
         END
         start_et GE AL32_beg AND start_et LT AL07_beg3 : BEGIN
            target_desc = 'Sky background for closest approach observations (AL31)'
            target_name = 'CALIBRATION'
            target_type = 'CALIBRATION'
            spice_target_name = '"N/A"'
         END
         start_et GE AL22_beg AND start_et LT vega_beg : BEGIN
            target_desc = 'Sky background for exosphere search (AL21)'
            target_name = 'CALIBRATION'
            target_type = 'CALIBRATION'
            spice_target_name = '"N/A"'
         END
         start_et GE AL09_beg AND start_et LT AL32_beg : BEGIN
            target_desc = 'Alice boresight alignment raster scan'
            target_name = 'CALIBRATION'
            target_type = 'CALIBRATION'
            spice_target_name = '"N/A"'
         END
         start_et GE vega_beg : BEGIN
            target_desc = 'Vega (alf Lyr)'
            target_name = 'CALIBRATION'
            target_type = 'CALIBRATION'
            spice_target_name = '"N/A"'
         END

       ELSE:
      ENDCASE

   END




   ELSE:BEGIN
      print, '% RALICE_TARGET: Input time not within known phases of ' + $
             'the Rosetta Mission'
      print, '% RALICE_TARGET: Most likley, this program is out of date.'
      phaseabbrev = 'UNK'
      phasename = 'UNK'
      target_id = 'UNK'
      target_name = 'UNK'
      target_type = 'UNK'
      dataset_target_name = 'X'
      spice_target_name = '"N/A"'
      target_desc = 'Unknown'
   END

ENDCASE

END

;            target_name = '(2867) STEINS'
;            target_type = 'ASTEROID'
;            spice_target_name = ''
;            target_desc = 'Asteroid (2867) Steins'

;["67P/CHURYUMOV-GERASIMENKO (1969 R1)","COMET","67P","C"]
;["(2867) STEINS", "ASTEROID", "STEINS", "A"]
;["(21) LUTETIA", "ASTEROID", "LUTETIA", "A"]
;["EARTH", "PLANET", "EARTH", "E"]
;["MARS", "PLANET", "MARS", "M"]
;["CALIBRATION", "CALIBRATION", "CAL", "CAL"]
;["CHECKOUT", "N/A", "CHECK", "X"]
;["INTERPLANETARY DUST", "DUST", "DUST", "D"]
;["INTERSTELLAR DUST", "DUST", "DUST", "D"]
;["METEOROID STREAM", "DUST", "DUST", "D"]
;["SOLAR WIND", "SOLAR SYSTEM", "SW", "SS"]
;["9P/TEMPEL 1 (1867 G1)", "COMET", "9P", "C"]
;["C/LINEAR (2002 T7)", "COMET", "2002T7","C"]
