function l5_moment = READICA_L5(varargin)
%READICA_L5 Read RPCICA L5 MOMENT data for one day.
%   l5 = READICA_L5() Reads data for a day. 
%        Asks for rootpath & date
%
%   l5 = READICA_L5(date) Reads data for date given. 
%        Asks for rootpath
%
%   l5 = READICA_L5(date, rootpath) Reads data for date given.
%
% See also: IMPORT_L5, PLOT_ICAL5
data_date = '';
rootpath = '';
if nargin > 1
    if ~ischar(varargin{2})
        help(mfilename)
        return
    end
    rootpath = varargin{2};
end
if nargin > 0
    if ~ischar(varargin{1})
        help(mfilename)
        return
    end
    data_date = varargin{1};
end

%% Find rootpath of archive where directory DATA is located
if isempty(rootpath) || ~exist(fullfile(rootpath, 'DATA'),'dir')
    for jj=1:5
        if exist(fullfile(rootpath, 'DATA'),'dir')
            break
        else
            rootpath = fullfile('..', rootpath);
        end
    end
end

%% Get path to data
icapath=fullfile(rootpath, 'DATA', 'MOMENT');
while ~exist(icapath, 'dir')
    fprintf(['This script is designed to run inside an archive.\n' ...
        'The DATA/MOMENT directory was not found.\n' ...
        'Please provide the path to the root of the archive i.e.:\n' ...
        '/home/user/psa/RO-C-RPCICA-5-ESC1-MOMENT-V1.0\n' ...
        '\n']);
    rootpath = input('path to the root of the archive:  ','s');
    icapath=fullfile(rootpath, 'DATA', 'MOMENT');
end

fprintf('rootpath: %s\n', rootpath);

if isempty(data_date)
    data_date=input('Date of data to import yyyy-mm-dd:  ','s');
end
dn = datenum(data_date);


cdir=fullfile(icapath, ...
    datestr(dn,'yyyy'), ...
    upper(datestr(dn,'mmm')), ...
    ['D' datestr(dn,'dd')]);
fprintf('cdir: %s\n', cdir);

%% Get files
fcdir=fullfile(cdir, '/RPCICA*.TAB');
filelist=dir(fcdir);

%% initialize matrices
l5_moment.time = [];
l5_moment.mass = [];
l5_moment.charge = [];
l5_moment.density = [];
l5_moment.v = [];
l5_moment.duration = [];
l5_moment.quality = [];

l5_moment.swversion = [];
%% Find function to call
if (exist('IMPORT_L5') == 2)
    import = str2func('IMPORT_L5');
else
    import = str2func('import_l5');
end

%% Read data
if ~isempty(filelist)
    for jj=1:length(filelist)
        disp(filelist(jj).name);
        if ~isempty(strfind(filelist(jj).name,'_L5_MOM.TAB'))
            table = import(fullfile(cdir, filelist(jj).name));
            lbl = strrep(filelist(jj).name, 'TAB', 'LBL');
            l5_lbl = READ_PDS_LBL(fullfile(cdir, lbl));
            curr_sw = str2double(l5_lbl.ROSETTA_COLON_ICA_SW_VERSION);
        else
            % No support for other files
            continue;
        end
        
        l5_moment.time=[l5_moment.time; table.time];
        l5_moment.mass=[l5_moment.mass; table.mass];
        l5_moment.charge= [l5_moment.charge; table.charge];
        l5_moment.density=[l5_moment.density; table.density];
        l5_moment.v=[l5_moment.v; table.v];
        l5_moment.duration=[l5_moment.duration; table.duration];
        l5_moment.quality=[l5_moment.quality; table.quality];
        l5_moment.swversion = [l5_moment.swversion; repmat(curr_sw, length(table.time), 1)];
    end
    save(['ica' data_date '_moment.mat'], 'l5_moment');
else
    disp(['Can not find data file: ' fcdir]); 
end
