function l3 = READICA_L3(varargin)
%READICA_L3 Read RPCICA L3 CALIBRATED data for one day.
%   l3 = READICA_L3() Reads data for a day.
%        Skips data with less than 96 e_levels.
%        Asks for rootpath & date
%
%   l3 = READICA_L3(stretch) Reads data for a day.
%        Stretches data with less than 96 e_levels to 96 levels.
%        Asks for rootpath & date
%
%   l3 = READICA_L3(stretch, date) Reads data for date given.
%        If stretch is true, stretches data with less than 96 e_levels to 96 levels.
%        Asks for rootpath
%
%   l3 = READICA_L3(stretch, date, rootpath) Reads data for date given.
%        If stretch is true, stretches data with less than 96 e_levels to 96 levels.
%
stretch = false;
data_date = '';
rootpath = '';
if nargin > 2
    if ~ischar(varargin{3})
        help(mfilename)
        return
    end
    rootpath = varargin{3};
end
if nargin > 1
    if ~ischar(varargin{2})
        help(mfilename)
        return
    end
    data_date = varargin{2};
end
if nargin > 0
    if ~islogical(varargin{1})
        help(mfilename)
        return
    end
    stretch = varargin{1};
end

%% Find rootpath of archive where directory DATA is located
if isempty(rootpath) || ~exist(fullfile(rootpath, 'DATA'),'dir')
    for jj=1:5
        if exist(fullfile(rootpath, 'DATA'),'dir')
            break
        else
            rootpath = fullfile('..', rootpath);
        end
    end
end

%% Get path to data
icapath=fullfile(rootpath, 'DATA', 'CALIBRATED');
while ~exist(icapath, 'dir')
    fprintf(['This script is designed to run inside an archive.\n' ...
        'The DATA/CALIBRATED directory was not found.\n' ...
        'Please provide the path to the root of the archive i.e.:\n' ...
        '/home/user/psa/RO-C-RPCICA-3-ESC1-CALIB-V1.0\n' ...
        '\n']);
    rootpath = input('path to the root of the archive:  ','s');
    icapath=fullfile(rootpath, 'DATA', 'CALIBRATED');
end

fprintf('rootpath: %s\n', rootpath);

if isempty(data_date)
    data_date=input('Date of data to import yyyy-mm-dd:  ','s');
end
dn = datenum(data_date);


cdir=fullfile(icapath, ...
    datestr(dn,'yyyy'), ...
    upper(datestr(dn,'mmm')), ...
    ['D' datestr(dn,'dd')]);
fprintf('cdir: %s\n', cdir);

%% Get files
fcdir=fullfile(cdir, '/RPCICA*.TAB');
filelist=dir(fcdir);

%% initialize matrices
l3.iontime = [];
l3.delta_t = [];
l3.quality = [];
l3.mode = [];
l3.noise_reduction = [];
l3.mtable = [];
l3.pacclevel = [];
l3.azim = [];
l3.elev = [];
l3.masschannel = [];
l3.flux = [];
l3.swversion = [];

%% Find function to call
if (exist('IMPORT_L3') == 2)
    import = str2func('IMPORT_L3');
else
    import = str2func('import_l3');
end

%% Read data
e_level_skip = 0;

if ~isempty(filelist)
    for jj=1:length(filelist)
        disp(filelist(jj).name);
        if ~isempty(strfind(filelist(jj).name,'_L3.TAB'))
            [table, e_levels] = import(fullfile(cdir, filelist(jj).name));
            lbl = strrep(filelist(jj).name, 'TAB', 'LBL');
            l3_lbl = READ_PDS_LBL(fullfile(cdir, lbl));
            curr_sw = str2double(l3_lbl.ROSETTA_COLON_ICA_SW_VERSION);
        else
            % No support for other files
            continue;
        end
        if e_levels ~= 96
            if stretch
                % possibly stretch 8 & 32 e_levels to 96 e_levels
                table.flux = [table.flux nan(size(table.iontime,1),96-e_levels)];
            else
                e_level_skip = e_level_skip + 1;
                continue;
            end
        end
        
        l3.iontime=[l3.iontime; table.iontime];
        l3.delta_t=[l3.delta_t; table.delta_t];
        l3.quality=[l3.quality; table.quality];
        l3.mode=[l3.mode; table.mode];
        l3.noise_reduction=[l3.noise_reduction; table.noise_reduction];
        l3.mtable=[l3.mtable; table.mtable];
        l3.pacclevel=[l3.pacclevel; table.pacclevel];
        l3.azim=[l3.azim; table.azim];
        l3.elev=[l3.elev; table.elev];
        l3.masschannel=[l3.masschannel; table.masschannel];
        l3.flux=[l3.flux; table.flux];
        l3.swversion = [l3.swversion; repmat(curr_sw, length(table.iontime), 1)];
    end
    save(['ica' data_date '_l3.mat'], 'l3');
    if e_level_skip > 0
        fprintf('Skipped reading %d files due to low number of e_levels\n', e_level_skip);
    end
else
    disp(['Can not find data file: ' fcdir]); 
end
