function [table, e_levels] = import_l3(filename, startRow, endRow)
%IMPORT_L3 Import numeric data from a text file as column vectors.
%   [table, e_levels] = IMPORT_L3(FILENAME) Reads data from text file FILENAME for the default
%   selection.
%
%   [TABLE, E_LEVELS] = IMPORT_L3(FILENAME, STARTROW, ENDROW) Reads data from rows STARTROW
%   through ENDROW of text file FILENAME.
%
% Example:
%   [TABLE, e_levels] = import_l3('RPCICA150317T04_000_L3.TAB');
%   [TABLE, e_levels] = import_l3('RPCICA150317T04_000_L3.TAB', 1, 16384);
%

%% Initialize variables.
delimiter = ',';
if nargin<=2
    startRow = 1;
    endRow = inf;
end

%% Format string for each line of text:
%   column1: text (%s) iontime
%   column2: double (%f) delta_t
%   column3: double (%s) quality flag
%   column4: double (%f) mode
%   column5: double (%f) noise_reduction
%   column6: double (%f) mtable
%   column7: double (%f) pacc
%   column8: double (%f) azimuth
%   column9: double (%f) elev
%   column10: double (%f) mass

%   column11-column18: double (%f) calibrated flux 8 levels
% or
%   column11-column42: double (%f) calibrated flux 32 levels
% or
%   column11-column106: double (%f) calibrated flux 96 levels
%
% For more information, see the TEXTSCAN documentation.
formatSpec = ['%s%f%q' repmat('%f', 1, 7) repmat('%f', 1, 96) '%[^\n\r]'];

%% Open the text file.
fileID = fopen(filename,'r');

%% Read a line and determine how many e_levels we got
tmp = fgets(fileID);
dataArray = textscan(tmp, formatSpec, endRow(1)-startRow(1)+1, 'Delimiter', delimiter, 'EmptyValue' ,NaN,'HeaderLines', startRow(1)-1, 'ReturnOnError', false);
if isnan(dataArray{19})
    e_levels = 8;
elseif  isnan(dataArray{43})
    e_levels = 32;
else
    e_levels = 96;
end

%% Rewind & remake formatSpec
frewind(fileID);
formatSpec = ['%s%f%q' repmat('%f', 1, 7) repmat('%f', 1, e_levels) '%[^\n\r]'];

%% Read columns of data according to format string.
% This call is based on the structure of the file used to generate this code. If
% an error occurs for a different file, try regenerating the code from the
% Import Tool.
dataArray = textscan(fileID, formatSpec, endRow(1)-startRow(1)+1, 'Delimiter', delimiter, 'EmptyValue' ,NaN,'HeaderLines', startRow(1)-1, 'ReturnOnError', false);
for block=2:length(startRow)
    frewind(fileID);
    dataArrayBlock = textscan(fileID, formatSpec, endRow(block)-startRow(block)+1, 'Delimiter', delimiter, 'EmptyValue' ,NaN,'HeaderLines', startRow(block)-1, 'ReturnOnError', false);
    for col=1:length(dataArray)
        dataArray{col} = [dataArray{col};dataArrayBlock{col}];
    end
end

%% Close the text file.
fclose(fileID);

%% Post processing for unimportable data.
% No unimportable data rules were applied during the import, so no post
% processing code is included. To generate code which works for unimportable
% data, select unimportable cells in a file and regenerate the script.

%% Allocate imported array to column variable names
table.iontime = datenum(dataArray{1}, 'yyyy-mm-ddTHH:MM:SS.FFF');
table.delta_t = dataArray{2};
table.quality = dataArray{3};
table.mode =  dataArray{4};
table.noise_reduction = dataArray{5};
table.mtable = dataArray{6};
table.pacclevel = dataArray{7};
table.azim = dataArray{8};
table.elev = dataArray{9};
table.masschannel = dataArray{10};
table.flux = [dataArray{11:end-1}];

