//
//
//
#include "DFMS_utility.hh"


//
// ----------------- set the Terminal for gnuplot -----------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This function issues a uname -a and returns the gnuplot terminal name based
// on the OS returned by the uname -a command
//
// inputs:
//   None
//
// returns:
//   gnuplot term name
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
string setTerminal() {

	const int MAX_BUFFER = 2048;
 	string cmd="uname -a";
    char buffer[MAX_BUFFER];
    FILE *stream = popen(cmd.c_str(), "r");
    if (stream){
    	while (!feof(stream)) {
    		if (fgets(buffer, MAX_BUFFER, stream) != NULL) {
    			string reply = string(buffer);
    			if (reply.find_first_of("Linux") == 0) {
    				pclose(stream);
    				return "X11";
    			} else if (reply.find_first_of("Darwin") == 0) {
    				pclose(stream);
    				//return "aqua";
    				return "X11";
    			} else if (reply.find_first_of("SOLARIS") == 0) {
    				pclose(stream);
    				return "X11";
    			} else {
    				pclose(stream);
    				return "dumb";
    			}
    		}
    		pclose(stream);
    	}
    }

	return "dumb";

}


//
// ------------------------ Check if a file exists -----------------------------
// 
// =============================================================================
// Routine Description
// =============================================================================
// This function checks is an input file exists
//
// inputs:
//   string input filename
//
// returns:
//   true if file exists
//   false if it doesn't
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
bool util_fileExists(string file) {

	string sFunctionName="utility->util_fileExists";
	ifstream ifile;

	ifile.open(file.c_str());
	if (ifile) {
		ifile.close();
		return true;
	} else {
		sErrorMessage="File: "+file+" does not exist";
		writeToLog(sErrorMessage, sFunctionName, WARN);
		return false;
	}
}

//
//  ------------------ Convert an integer to string ----------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This function converts an integer to a string
//
// inputs:
//   integer
//
// returns:
//   integer as string equivalent
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
string util_intToString(int number) {
	stringstream ss;		//create a stringstream
	ss << number;		//add number to the stream
	return ss.str();		//return a string with the contents of the stream
}

//
//  ------------------ Convert an String to int ----------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This function converts an integer to a string
//
// inputs:
//   string
//
// returns:
//   string as integer
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
int util_stringToInt(string str) {

	int val;
	istringstream ss(str);
	ss >> val;
	return val;
}

//
//  ------------------ Convert an String to double ----------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This function converts an string to a double
//
// inputs:
//    str - string to convert
//
// returns:
//   string as double
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
double util_stringToDouble(string str) {

	double val;
	istringstream ss(str);
	ss >> val;
	return val;
}

//
//  ------------------ Convert an double to string ----------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This function converts a double to a string
//
// inputs:
//   double
//
// returns:
//   double as string equivalent
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
string util_doubleToString(double number) {
	stringstream ss;		//create a stringstream
	ss << number;		//add number to the stream
	return ss.str();		//return a string with the contents of the stream
}

//
//  -------------------- Convert an string to bool -----------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This function converts a string to a boolean
//
// inputs:
//   s - string to convert
//
// returns:
//   bool
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
bool util_stringToBool(string s) {
	if (s.compare("true") == 0) return true;
	if (s.compare("false") == 0) return false;
	return false;
}

//
// ---------------------------- Tokenizer --------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This function takes a string of characters and uses white spaces
// to break the string into a vector of string tokens
//
// inputs:
//   String
//
// output:
//   vector of strings (tokens)
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
void util_Tokenize(const string& str, vector<string>& tokens)
{
    string buf; // Have a buffer string
    stringstream ss(str); // Insert the string into a stream

    while (ss >> buf)
        tokens.push_back(buf);

}

//
//-------------------------- util_splitString ----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This function takes a string of characters and a delimiter and returns the
// string into a vector of string tokens
//
// inputs:
//   string s     - input string
//   char delim   - delimiter to break on
//   vector elems - vector elements returned
//
// output:
//   None
// =============================================================================
// History:  Written by Mike Rinaldi May 2013
// =============================================================================
//
void util_splitString(const string &s, char delim, vector<string> &elems) {

    stringstream ss(s);
    string item;
    if (s.size() != 0) {
    	while (getline(ss, item, delim)) {
    		elems.push_back(item);
    	}
    }
}

//
// ------------------- Check if string is a number -----------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This returns true if the string is a numeric
//
// inputs:
//   String
//
// returns:
//   true if string is numeric
//   false if string is not numeric
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
bool util_isNumber(const string& s) {
	return (util_stringToInt(s));
}

//
// -------------------- Check if string is a float -----------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This returns true if the string is a float numeric
//
// inputs:
//   String
//
// returns:
//   true if string is float numeric
//   false if string is not numeric
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
bool util_isFloat(const string& s) {
    istringstream iss(s);
    float dummy=0.0;
    iss >> noskipws >> dummy;

    bool ret = iss && iss.eof();

    iss.str(std::string());  // clear stream release memory
    iss.clear();   // clear eof bit...

    return ret;     // Result converted to bool
}

//
// ----------------- Check if string is an integer -----------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This returns true if the string is an integer numeric
//
// inputs:
//   String
//
// returns:
//   true if string is integer numeric
//   false if string is not numeric
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
bool util_isInteger(const string& s)
{
   if(s.empty() || ((!isdigit(s[0])) && (s[0] != '-') && (s[0] != '+'))) {
       return false ;
   }
   
   char * p ;
   strtol(s.c_str(), &p, 10) ;

   return (*p == 0) ;
}

//
// ------------------- Trim white space from string  ---------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This routine removes leading and ending white spaces
//
// inputs:
//   String
//
// returns:
//   String with white spaces removed
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
string util_trim(const string& str)
{

    string whitespace = " \t";
    int strBegin = str.find_first_not_of(whitespace);
    if (strBegin == string::npos) return ""; // no content

    int strEnd = str.find_last_not_of(whitespace);
    int strRange = strEnd - strBegin + 1;

    return str.substr(strBegin, strRange);
}

//
// ------------------- Trim character from string  ---------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This routine removes leading and ending 'character'
//
// inputs:
//   String
//
// returns:
//   String with character removed
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
string util_trimChar(const string& str, const string ch)
{

    int strBegin = str.find_first_not_of(ch);
    if (strBegin == string::npos) return ""; // no content

    int strEnd = str.find_last_not_of(ch);
    int strRange = strEnd - strBegin + 1;

    return str.substr(strBegin, strRange);
}

//
// ----------------- Trim EOL characters from string  --------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This routine removes EOL 'character(s)'
//
// inputs:
//   String
//
// returns:
//   String with characters removed
// =============================================================================
// History:  Written by Mike Rinaldi OCtober 2015
// =============================================================================
//
string util_trimEOL(string str) {
	str.erase(str.find_last_not_of("\n\r\t")+1);
	return str;
}

//
// -----------------  Compares two L2 file structures --------------------------
//
//==============================================================================
// Routine Description
//==============================================================================
// This routine compares two records of an L2_PDS_file structure and 
// returns a 1 or 0 to indicate the order in which they should be sorted.  
// All dfms data files should be sorted in chronological order, with all time ordered
// GCU spectra files coming before all time ordered non-GCU spectra files.
//
// inputs:
//   file1 - a single record of L2_PDS_file
//   file2 - a single record of L2_PDS_file
//
// returns:
//   true if 'file1' should be placed after 'file2' and 
//   false if 'file2' should be placed after 'file1' in the sorted order
// =============================================================================
// History:  Algorithmn adapted from similar C code in RTOF L2 to L3 software
// This algorithm uses the C++ STL algorithm stable_sort.
// Modified:  Converted to C++ by Mike Rinaldi March 2013
// =============================================================================
//
bool util_compare (const L2FileInfo &file1, const L2FileInfo &file2)
{

	string sFunctionName = "utility->util_compare";

	if (file1.isGCU && !file2.isGCU) return true;
	if (!file1.isGCU && file2.isGCU) return false;

	if (file1.isGCU == file2.isGCU) return ((file1.dateTime).compare(file2.dateTime) < 0);

	return true;
}

//
// -----------------  Compares two L2 file structures --------------------------
//
//==============================================================================
// Routine Description
//==============================================================================
// This routine compares two records of an L2_PDS_file structure and
// returns a 1 or 0 to indicate the order in which they should be sorted.
// The files will be sorted in chronological order,
//
// inputs:
//   file1 - a single record of L2_PDS_file
//   file2 - a single record of L2_PDS_file
//
// returns:
//   true if 'file1' should be placed after 'file2' and
//   false if 'file2' should be placed after 'file1' in the sorted order
// =============================================================================
// History:  Algorithmn adapted from similar C code in RTOF L2 to L3 software
// This algorithm uses the C++ STL algorithm stable_sort.
// Modified:  Converted to C++ by Mike Rinaldi March 2013
// =============================================================================
//
bool util_compare_date (const L2FileInfo &file1, const L2FileInfo &file2)
{

	string sFunctionName = "utility->util_compare_date";

	return ((file1.dateTime).compare(file2.dateTime) < 0);

}

//
// ---------------- Find if string is alphanumerics ----------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This function checks if the input string is an alphanumeric quantity
//
// inputs:
//   String
//
// returns:
//   true if alphanumeric
//   false if not
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
bool util_isalnum(const string& str) {
	
	bool isa = false;
	for (int i=0; i<str.length(); i++) {
		isa = isalnum(str[i]);
		if (isa) return isa;
	}
	return isa;
}

//
//  ----------------------- util_toUpperCase -----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Convert a string to upper case
//
// inputs:
//   s - a C++ string
//
// returns:
//   the upper case string
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
void util_toUpperCase (string &s) {

	for(unsigned int l = 0; l < s.length(); l++)  {
	    s[l] = toupper(s[l]);
	}
}

//
//  -------------------------- util_linIntrp -----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// simple linear interpolation routine
//
// inputs:
//   x0,y0 - beginning interval points
//   y0,y1 - ending interval points
//   x     - independent variable where y is desired
//
// returns:
//   y     - interpolated point
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
double util_linIntrp(double x0, double x1, double y0, double y1, double x) {

	double y = y0+(y1-y0)*(x-x0)/(x1-x0);

	return y;
}

//
//  ----------------------------- getDateFromFileName ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Returns the date portion of a typical L2 file name
//
// inputs:
//   file - L2 File name
//
// returns:
//   fileDate     - L2 File name date portion
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
string getDateFromFileName(string fileName) {

	size_t dpos = fileName.find_first_of("_");
	string fileDate = fileName.substr(dpos+1,15);

	return fileDate;

}

//
//  ----------------------------- massScale ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// simply compute the mass scale given pix0,m0,x
//
// inputs:
//   zoom - zoom value
//   pix0 - pixel0 calib value
//   m0   - target mass
//   numPix - numbers of pixels in massScale
//
// returns:
//   y     - interpolated point
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
void massScale(double *m, double zoom, double pix0, double m0, int numPix, bool p) {

	for (int i=0; i<numPix; ++i) {
		m[i] = pixToMass(zoom, pix0, (double)i, m0);
		if (p) cout << "m[" << i << "] = " << m[i] << endl;
	}
}

//
// ---------------------------- WriteLine --------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This function writes the PDS header as a LeftHandSide = a RightHandSode
// The sides are input.  The LHS is set to be left justified and 39 characters
// long.  The right hand side is also 39 characters long.  The total length of
// any line is 80 characters.  This includes the CR.  RHS's longer than 40 
// characters are formed so that any remaining length is limited to 79 characters
// + CR.
//
// inputs:
//   fstream (output stream object)
//   String LHS -  String on Left side of = sign
//   String RHS -  String on Right side of = sign
//   int addQt - no currently used
//
// returns:
//   true if alphanumeric
//   false if not
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
void writeLine(fstream &os, string sLHS, string sRHS, int addQt)
{

	char cLHS[40];
	char cRHS[80];
	char cOut[80];
	string sOut;
	int eolPos = 0;
	string Remng;

     // If sRHS length > 40 then we have a continuation line
	if (sRHS.length() > 40) {
		sprintf(cLHS,"%-33s=     ",sLHS.c_str());
		sOut = string(cLHS);
		int len = sRHS.length();
		eolPos = sRHS.find_first_of("\n");
		sprintf(cRHS,"%-39s",(sRHS.substr(0,eolPos)).c_str());
		sOut += string(cRHS);
		len -= (sRHS.substr(0,eolPos)).length(); 	
		Remng = sRHS.substr(eolPos+1,string::npos);	
		while (len > 40) {
			eolPos = (Remng.substr(0,string::npos)).find_first_of("\n");
			sprintf(cRHS,"%79s",(Remng.substr(0,eolPos)).c_str());
			sOut += string(cRHS);
			len -= (Remng.substr(0,eolPos)).length();
			Remng = Remng.substr(eolPos+1,string::npos);
		}
	} else {
		sprintf(cOut,"%-33s=     %-39s",sLHS.c_str(),sRHS.c_str());
		sOut = string(cOut);
	}
	os << sOut << dfmsEOL;

}

//
// -------------------------- getProcessTime -----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This function returns the system time formated to Y-M-D:HR:MIN:SEC:
//
// inputs:
//   type  -  Type of format to return
//              1 - YYYY-MM-DDThh:mm:ss 
//              2 - YYYYMMDD_hhmmss 
//              3 - MM/DD/YYYhh:mm:ss 
//
// returns:
//   String with formated system time
// =============================================================================
// History:  Modified from RTOF C version by Mike Rinaldi
// =============================================================================
//
string getProcessTime(int type) {

     char sProcessTime[24];

     string ret;

     // current date/time based on current system time 
     time_t now = time(0);
     tm *ltm = localtime(&now);
     int nowSec = clock();

     switch (type) {
		case 1:
			strftime(sProcessTime, sizeof(sProcessTime), "%Y-%m-%dT%H:%M:%S", ltm);
			break;
          case 2:	// Date/Time for high time res cases
        	strftime(sProcessTime, sizeof(sProcessTime), "%Y%m%d_%H%M%S", ltm);
			break;
          case 3:
        	strftime(sProcessTime, sizeof(sProcessTime), "%Y-%m-%d %H:%M:%S", ltm);
			break;
          case 4:
            sprintf(sProcessTime,"%d", nowSec);
			break;
          case 5:    // Date/Time for low time res cases
        	strftime(sProcessTime, sizeof(sProcessTime), "%Y%m%d_%H%M", ltm);
			break;
		default:
            strftime(sProcessTime, sizeof(sProcessTime), "%Y%m%d_%H%M%S", ltm);
			break;
	}

    ret = string(sProcessTime);

    return ret;

}

//
// -------------------------- getProcessDate -----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This function returns the system date formated to Y-M-D
//
// inputs:
//
// returns:
//   String with formated system date
// =============================================================================
// History:  Derived from RTOF processTime C version, by Mike Rinaldi
// =============================================================================
//
string getProcessDate() {

	char sProcessTime[24];

	// current date/time based on current system time
	time_t now = time(0);
	tm *ltm = localtime(&now);
	int nowSec = clock();

	strftime(sProcessTime, sizeof(sProcessTime), "%Y%m%d", ltm);
	
	return string(sProcessTime);

}

//
// ------------------------- makeUpData ----------------------------------------
//
// Use this to make test data for filling L3 file (probably outdated...)
void makeUpData(str2D &HKdata, str2D &CalData, dbl2D &L3Data, 
			 	 	 	 	 string L2File, string L3File) {

	L3Info->qualID = 1;
	L3Info->cal->pix0_A = 44.56789;
	L3Info->cal->pix0_B = 44.78321;
	L3Info->L2baseName = L2File.substr(0,27);
	L3Info->cal->calID4 = L3File.substr(0,26);
	L3Info->cal->calID5 = "M0150_D_20100702_081924074";
	L3Info->cal->calID6 = "IND_PIXEL_G_TABLE_20130101";

	// First put some data into HK data array
	int k=0;
	for (int i=0; i<31; ++i) {
		for (int j=0; j<4; ++j) {
			HKdata[i][j] = util_intToString(k++);
		}
	}

	// Next put some data into the Cal data array
	k=0;
	for (int i=0; i<2; ++i) {
		for (int j=0; j<7; ++j) {
			CalData[i][j] = util_intToString(k++);
		}
	}

	// Next put some data into the L3 data array
	k=0;
	for (int i=0; i<NUMPIX; ++i) {
		for (int j=0; j<5; ++j) {
			L3Data[i][j] = (double)k++;
		}
	}	


}

//
// --------------------------- timeDiffInSec -----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method calculates returns the time in seconds between to date strings
// of the form  yyyymmdd_hhmmsssss
//
// inputs:
//   date1  -  The first date string
//   date2  -  The second date string
//
// returns:
//   value in seconds
// =============================================================================
// History: Written by Mike Rinaldi, March 2014
// =============================================================================
//
double timeDiffInSec(string date1, string date2) {

	struct tm t1 = {0};
	struct tm t2 = {0};
	double seconds;

	t1.tm_year = util_stringToInt(date1.substr(0,4))-1900;
	t1.tm_mon  = util_stringToInt(date1.substr(4,2))-1;
	t1.tm_mday = util_stringToInt(date1.substr(6,2));
	t1.tm_hour = util_stringToInt(date1.substr(9,2));
	t1.tm_min  = util_stringToInt(date1.substr(11,2));
	t1.tm_sec  = util_stringToInt(date1.substr(13,2));

	/*printf("yr = %d\n",t1.tm_year);
	printf("mon = %d\n",t1.tm_mon);
	printf("mday = %d\n",t1.tm_mday);
	printf("hour = %d\n",t1.tm_hour);
	printf("min = %d\n",t1.tm_min);
	printf("sec = %d\n",t1.tm_sec);*/

	t2.tm_year = util_stringToInt(date2.substr(0,4))-1900;
	t2.tm_mon  = util_stringToInt(date2.substr(4,2))-1;
	t2.tm_mday = util_stringToInt(date2.substr(6,2));
	t2.tm_hour = util_stringToInt(date2.substr(9,2));
	t2.tm_min  = util_stringToInt(date2.substr(11,2));
	t2.tm_sec  = util_stringToInt(date2.substr(13,2));

	/*printf("\n");
	printf("yr = %d\n",t2.tm_year);
	printf("mon = %d\n",t2.tm_mon);
	printf("mday = %d\n",t2.tm_mday);
	printf("hour = %d\n",t2.tm_hour);
	printf("min = %d\n",t2.tm_min);
	printf("sec = %d\n",t2.tm_sec);*/

	seconds = abs(difftime(mktime(&t1),mktime(&t2)));

	return seconds;

}

//
// ------------------------------ calToJD --------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// compute the Julian date given the yr,mon,day,hr,min,sec
//
// inputs:
//   int yr   -  year
//   int mon  -  month
//   int day  -  day
//   int hr   -  hour
//   int min  -  minute
//   double sec - seconds
//
// returns:
//   JD;
// =============================================================================
// History: Written by Mike Rinaldi, March 2014
// =============================================================================
//
double calToJD(int yr, int mon, int day, int hr, int min, double sec) {

	//
	//---COMPUTES THE JULIAN DATE (JD) GIVEN A GREGORIAN CALENDAR
	//   DATE (YEAR,MONTH,DAY).
	//

	    int JD = day-32075 +
	    		      1461*(yr+4800+(mon-14)/12)/4 +
	    		       367*(mon-2-(mon-14)/12*12)/12 -
	    		         3*((yr+4900+(mon-14)/12)/100)/4;

	    double dJD = (double)(JD) + hr*3600.0 + min*60.0 + sec;

	    return dJD;
}

//
// ------------------------------ rosToSec --------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Compute the seconds since 00:00 hours, Jan 1, 1970 UTC given the ROSINA time
// in yyyymmddThh:mm:ss.sss format
//
// inputs:
//   rosTime - ROSINA time string
//   itype   - 0  -  (in yyyy-mm-ddThh:mm:ss.sss format)
//             1  -  (in yyyymmdd_hhmmsssss format)
//			   2  -  (in yyyymmdd format)
//
// returns:
//   seconds since 00:00 hours, Jan 1, 1970 UTC;
// =============================================================================
// History: Written by Mike Rinaldi, July 2014
// =============================================================================
//
double rosToSec(string rosTime, int iType) {

	struct tm rosTm = {0};
	struct tm t1970 = {0};
	double seconds;
	int yr,mon,day,hr,min,sec;

	//cout << "rosTime = " << rosTime << endl;

	// The ROSINA L2 start time in calendar form
	if (iType == 0) {
		yr  = util_stringToInt(rosTime.substr(0,4));
		mon = util_stringToInt(rosTime.substr(5,2));
		day = util_stringToInt(rosTime.substr(8,2));
		hr  = util_stringToInt(rosTime.substr(11,2));
		min = util_stringToInt(rosTime.substr(14,2));
		sec = util_stringToInt(rosTime.substr(17,2));
	} else if (iType == 1) {
		yr  = util_stringToInt(rosTime.substr(0,4));
		mon = util_stringToInt(rosTime.substr(4,2));
		day = util_stringToInt(rosTime.substr(6,2));
		hr  = util_stringToInt(rosTime.substr(9,2));
		min = util_stringToInt(rosTime.substr(11,2));
		sec = util_stringToInt(rosTime.substr(13,2));
	} else if (iType == 2) {
		yr  = util_stringToInt(rosTime.substr(0,4));
		mon = util_stringToInt(rosTime.substr(4,2));
		day = util_stringToInt(rosTime.substr(6,2));
		hr = 0;
		min = 0;
		sec = 0;
	}

	// Build the unix tm struct for the ROSINA time
	rosTm.tm_hour = hr;   rosTm.tm_min = min; rosTm.tm_sec = sec;
	rosTm.tm_year = yr-1900; rosTm.tm_mon = mon-1; rosTm.tm_mday = day;

	// Build the unix tm struct for 1/1/1970 00:00:00 UTC
	t1970.tm_hour = 0;  t1970.tm_min = 0; t1970.tm_sec = 0;
	t1970.tm_year = 70; t1970.tm_mon = 0; t1970.tm_mday = 1;

	seconds = difftime(mktime(&rosTm),mktime(&t1970));

	return seconds;
}

//
// ---------------------------- calcDispersion --------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method calculates the effective dispersion as a function of mass
//
// inputs:
//   DISP  -  Nominal Dispersion
//   m0    -  The commanded mass
//
// returns:
//   eDisp  -  effective dispersion
// =============================================================================
// History: Written by Mike Rinaldi, Oct 2017
// =============================================================================
//
double calcDispersion(double m0) {
    
    double eDisp = DISP;
    
    if (m0 < 70) {
        eDisp = DISP;
    } else {
        eDisp = 382200*pow(m0,-0.34);
    }
    
    return eDisp;
    
}

//
// ---------------------------- pixToMass --------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method calculates m from the relation give as,
//        m = m0 exp((x-x0)*C)
//
// inputs:
//   zoom  -  The zoom value for this mode
//    x    -  The pixel value
//   x0    -  The pix0 parameter
//   m0    -  The Target mass
//
// returns:
//   x0    -  The pix0 parameter
// =============================================================================
// History: Written by Mike Rinaldi, May 2013
// =============================================================================
//
double pixToMass(double zoom, double x0, double x, double m0) {
    
    double eDisp = calcDispersion(m0);
    
    double C = CONV/(eDisp*zoom);
    double m = m0*exp(C*(x-x0));
    
    /*
    cout << "zoom = " << zoom << endl;
    cout << "m0 = " << m0 << endl;
    cout << "x0 = " << x0 << endl;
    cout << "x = " << x << endl;
    cout << "m = " << m << endl;
    */
    
    return m;
    
}

//
// ---------------------------- massToPix0 --------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method calculates x0 from the relation give as,
//        m = m0 exp((x-x0)*C)
//		  m is the Actual MPs Table mass and
//        m0 is the target mass from the L2 HK value ROSINA_DFMS_SCI_MASS
//        x is the peak Center found by fitting
//        C = 25/(D*zoom) ~ 2e-4 (for zoom = 1)
//
// inputs:
//   zoom  -  The zoom value for this mode
//   x     -  The peak Center found by fitting
//   m0    -  Target mass from the L2 HK value ROSINA_DFMS_SCI_MASS
//   m     -  The Actual MPS Table mass
//
// returns:
//   x0    -  The pix0 parameter
// =============================================================================
// History: Written by Mike Rinaldi, May 2013
// =============================================================================
//
double massToPix0(double zoom, double x, double m, double m0) {
    
    double eDisp = calcDispersion(m0);
    
    double C = CONV/(eDisp*zoom);

    double lgTerm = log(m/m0)/C;
    double x0 = x - lgTerm;

    return x0;
    
}

//
// ---------------------------- massToPix --------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method calculates x from the relation give as,
//        m = m0 exp((x-x0)*C)
//		  m is the Actual MPs Table mass and
//        m0 is the target mass from the L2 HK value ROSINA_DFMS_SCI_MASS
//        x0 is the calibration value pix0
//        C = 25/(D*zoom) ~ 2e-4 (for zoom = 1)
//
// inputs:
//   zoom  -  The zoom value for this mode
//   x0    -  the calibration value pix0
//   m0    -  Target mass from the L2 HK value ROSINA_DFMS_SCI_MASS
//   m     -  The Actual MPS Table mass
//
// returns:
//   x    -  The pixel location associated with m,m0, and pix0
// =============================================================================
// History: Written by Mike Rinaldi, March 2014
// =============================================================================
//
double massToPix(double zoom, double x0, double m, double m0) {

    double eDisp = calcDispersion(m0);
    
    double C = CONV/(eDisp*zoom);

    double lgTerm = log(m/m0)/C;
    double x = x0 + lgTerm;

    return x;

}

//
// ----------------------------- writeToLog -----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method writes to the logger stream
// inputs: 
//   uses global fstream toLog
//   
// returns:
//   None
// =============================================================================
// History: Written by Mike Rinaldi, May 2013
// =============================================================================
//
void writeToLog(string sErrMsg, string sFuncName, int nErrType ) {
    
     string sFunctionName="utility->writeToLog";

     string sErrorType;
     string sOut;
     char chout[500];
     
     // use the error type to format for printing handle the error message
	switch (nErrType) {
		case 0:
			sErrorType = "FATAL ERROR  -";
			// format the out put string
			sprintf(chout,"%s  %s in \'%s\': %s", (getProcessTime(3)).c_str(),
					sErrorType.c_str(), sFuncName.c_str(), sErrMsg.c_str());
			// Convert to string type
			sOut = string(chout);
			// Write to standard error
			cerr << sOut << endl;
			// Write to log file
			toLog << sOut << dfmsEOL;
			toLog.close();
			if (L3INFOTOFILE) {
				l3Log << sOut << dfmsEOL;
				l3Log.close();
            }
			exit(EXIT_FAILURE);
			break;
		case 1:
			sErrorType = "ERROR         -";
			break;
		case 2:
			sErrorType = "WARNING       -";
			break;
          case 3:
            sErrorType = "INFORMATIONAL -";
            break;
		default:
			sErrorType = "UNKNOWN       -";
			break;
	}

     sprintf(chout,"%s  %s in \'%s\': %s", (getProcessTime(3)).c_str(), 
            sErrorType.c_str(), sFuncName.c_str(), sErrMsg.c_str());
     // Convert to string type
     sOut = string(chout);              
     // Write to standard error
     if(verbose >= 3) cerr << sOut << dfmsEOL;
     // Write to log file
     toLog << sOut << dfmsEOL;
     
}

//
//  -------------------------- get_env_var -------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Ask system for environment variable value
//
// inputs:
//   key - environment var
//
// returns:
//   envrionrment var value
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
string get_env_var(string const & key ) {
  char * val;
  val = getenv( key.c_str() );
  string retval = "";
  if (val != NULL) {
    retval = val;
  }
  return retval;
}

//
//   -------------------- Print the Quick Look Header ---------------------------------
//
void qlHeader() {
	QuickLookLog << "-----------------------------------------------------------------------------------------------------------------------------------------" << dfmsEOL;
	QuickLookLog << "File#            File                   Mass            PeakLoc               pix0                  ppmDiff              Verif Peak Name" << dfmsEOL;
	QuickLookLog << "                                                      A         B          A         B            A         B" << dfmsEOL;
	QuickLookLog << "-----------------------------------------------------------------------------------------------------------------------------------------" << dfmsEOL;
}

//
//   ---------------- Make a Peak Loc vs ppmDiff histo ------------------------
//
void makePkPPMhisto() {

	long tot0=0L,tot1=0L,tot2=0L,tot3=0L,toth=0L;

	char tmp[200];
	string line;
	vector<pair<double,double> >::iterator it;
	int pkLocHisto[15] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
	int pkLocBin[16] = {0,50,100,125,175,200,225,250,275,300,325,350,375,400,450,500};
	int ppmDiffDistro[4][15];
	int baddataDistro=0;
	for (int i=0; i<4; ++i) {for (int j=0; j<15; ++j) ppmDiffDistro[i][j] = 0;}

	for (it = pkPPMvec.begin(); it != pkPPMvec.end(); it++) {
		double ppmDiff = (*it).second;
		for (int i=0; i<15; ++i) {
			if ((*it).first > pkLocBin[i] && (*it).first <= pkLocBin[i+1]) {
				pkLocHisto[i]++;
				if ((*it).second <= PPMDIFFMIN) ppmDiffDistro[0][i]++;
				if ((*it).second > PPMDIFFMIN && (*it).second <= 2*PPMDIFFMIN) ppmDiffDistro[1][i]++;
				if ((*it).second > 2*PPMDIFFMIN && (*it).second <= 4*PPMDIFFMIN) ppmDiffDistro[2][i]++;
				if ((*it).second > 4*PPMDIFFMIN ) ppmDiffDistro[3][i]++;
				break;
			}
		}
		if (ppmDiff == BADDATA) baddataDistro++;
	}
	cout << endl;
	cout << "-----------------------------------------------------------------------------" << endl;
	cout << "   Peak Pixel                         PPM  Deviation  Distribution" << endl;
	cout << "    Location      -----------------------------------------------------------" << endl;
	cout << "     Range         Totals  |   <500       500-1000     1000-2000      >2000"  << endl;
	cout << "-----------------------------------------------------------------------------" << endl;

	if (QLINFOFILE) {
		QuickLookLog << dfmsEOL;
		QuickLookLog << "-----------------------------------------------------------------------------" << dfmsEOL;
		QuickLookLog << "   Peak Pixel                         PPM  Deviation  Distribution" << dfmsEOL;
		QuickLookLog << "    Location      -----------------------------------------------------------" << dfmsEOL;
		QuickLookLog << "     Range         Totals  |   <500       500-1000     1000-2000      >2000" << dfmsEOL;
		QuickLookLog << "-----------------------------------------------------------------------------" << dfmsEOL;
	}

	if (L3INFOTOFILE) {
			l3Log << dfmsEOL;
			l3Log << "-----------------------------------------------------------------------------" << dfmsEOL;
			l3Log << "   Peak Pixel                         PPM  Deviation  Distribution" << dfmsEOL;
			l3Log << "    Location      -----------------------------------------------------------" << dfmsEOL;
			l3Log << "     Range         Totals  |   <500       500-1000     1000-2000      >2000" << dfmsEOL;
			l3Log << "-----------------------------------------------------------------------------" << dfmsEOL;
	}

	for (int i=0; i<15; i++) {
		sprintf(tmp," [%3.3d -> %3.3d]  =   %5.5d   |   %5.5d    -   %5.5d    -   %5.5d    -   %5.5d",
				pkLocBin[i],pkLocBin[i+1],pkLocHisto[i],ppmDiffDistro[0][i],ppmDiffDistro[1][i],
				ppmDiffDistro[2][i],ppmDiffDistro[3][i]);
		line = string(tmp);
		cout << line << endl;
		if (QLINFOFILE) {
			QuickLookLog << line << dfmsEOL;
		}
		if (L3INFOTOFILE) {
			l3Log << line << dfmsEOL;
		}
		toth += pkLocHisto[i];
		tot0 += ppmDiffDistro[0][i];
		tot1 += ppmDiffDistro[1][i];
		tot2 += ppmDiffDistro[2][i];
		tot3 += ppmDiffDistro[3][i];
	}
	cout << "-----------------------------------------------------------------------------" << endl;
	if (QLINFOFILE) {
		QuickLookLog << "-----------------------------------------------------------------------------" << dfmsEOL;
	}
	if (L3INFOTOFILE) {
		l3Log << "-----------------------------------------------------------------------------" << dfmsEOL;
	}
	sprintf(tmp,"    Totals     =   %5.5d   |   %5.5d    -   %5.5d    -   %5.5d    -   %5.5d",
						toth,tot0,tot1,tot2,tot3);
	line = string(tmp);
	cout << line << endl;
	if (QLINFOFILE) {
		QuickLookLog << line << dfmsEOL;
	}
	if (L3INFOTOFILE) {
		l3Log << line << dfmsEOL;
	}
	sprintf(tmp,"    # of PPMDiffs = N/A    =   %5.5d",baddataDistro);
	line = string(tmp);
	cout << line << endl;
	if (QLINFOFILE) {
		QuickLookLog << line << dfmsEOL;
	}
	if (L3INFOTOFILE) {
		l3Log << line << dfmsEOL;
	}
	cout << "-----------------------------------------------------------------------------" << endl;
	if (QLINFOFILE) {
		QuickLookLog << "-----------------------------------------------------------------------------" << dfmsEOL;
	}
	if (L3INFOTOFILE) {
		l3Log << "-----------------------------------------------------------------------------" << dfmsEOL;
	}
	// -
	cout << "                  [Note: Row  A/B Treated Independently]" << endl;
	cout << "        Therefore the numbers above are 2x the number of L2 files processed" << endl;
	cout << "-----------------------------------------------------------------------------" << endl;
	if (QLINFOFILE) {
		QuickLookLog << "                  [Note: Row  A/B Treated Independently]" << dfmsEOL;
		QuickLookLog << "        Therefore the numbers above are 2x the number of L2 files processed" << dfmsEOL;
		QuickLookLog << "-----------------------------------------------------------------------------" << dfmsEOL;
	}
	if (L3INFOTOFILE) {
		l3Log << "                  [Note: Row  A/B Treated Independently]" << dfmsEOL;
		l3Log << "        Therefore the numbers above are 2x the number of L2 files processed" << dfmsEOL;
		l3Log << "-----------------------------------------------------------------------------" << dfmsEOL;
	}
}

//
//  ---------------- List the modes used in a seesion --------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// List the modes used in a seesion
//
// inputs:
//   none
//
// returns:
//   none
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
void listModes() {

	char tmp[100];
	string line;
	map<int,int>::iterator it;

	cout << endl;
	cout << "---------------------------" << endl;
	cout << "    Modes Distribution" << endl;
	cout << "---------------------------" << endl;
	cout << " Mode        Num Processed" << endl;
	cout << "---------------------------" << endl;
	if (QLINFOFILE) {
		QuickLookLog << "--------------------------" << dfmsEOL;
		QuickLookLog << "    Modes Distribution" << dfmsEOL;
		QuickLookLog << "---------------------------" << dfmsEOL;
		QuickLookLog << " Mode        Num Processed" << dfmsEOL;
		QuickLookLog << "---------------------------" << dfmsEOL;
	}

	if (L3INFOTOFILE) {
		l3Log << "--------------------------" << dfmsEOL;
		l3Log << "    Modes Distribution" << dfmsEOL;
		l3Log << "---------------------------" << dfmsEOL;
		l3Log << " Mode        Num Processed" << dfmsEOL;
		l3Log << "---------------------------" << dfmsEOL;
	}

	for (it=modesPresent.begin(); it!=modesPresent.end(); it++) {
		sprintf(tmp," %4.4d            %5.5d",it->first, it->second);
		line.assign(tmp);
		cout << line << endl;
		if (QLINFOFILE) QuickLookLog << line << dfmsEOL;
		if (L3INFOTOFILE) l3Log << line << dfmsEOL;
	}
	cout << "---------------------------" << endl;
	if (QLINFOFILE) QuickLookLog << "---------------------------" << dfmsEOL;
	if (L3INFOTOFILE) l3Log << "---------------------------" << dfmsEOL;

}

//
//  ---------------- List comm mass fit succ/failure stats ---------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// List comm mass fit succ/failure stats
//
// inputs:
//   none
//
// returns:
//   none
// =============================================================================
// History:  Written by Mike Rinaldi, 10/2/2017
// =============================================================================
//
void listCommMassStats() {
    
    char tmp[100];
    string line;
    map<double,int>::iterator its;
    map<double,int>::iterator itf;
    
    cout << endl;
    cout << "--------------------------------------------" << endl;
    cout << " Commanded     Successfully" << endl;
    cout << "    Mass        Processed" << endl;
    cout << "--------------------------------------------" << endl;
    if (QLINFOFILE) {
        QuickLookLog << "--------------------------------------------" << dfmsEOL;
        QuickLookLog << " Commanded     Successfully" << dfmsEOL;
        QuickLookLog << "    Mass        Processed" << dfmsEOL;
        QuickLookLog << "--------------------------------------------" << dfmsEOL;
    }
    if (L3INFOTOFILE) {
        l3Log << "--------------------------------------------" << dfmsEOL;
        l3Log << " Commanded     Successfully" << dfmsEOL;
        l3Log << "    Mass        Processed" << dfmsEOL;
        l3Log << "--------------------------------------------" << dfmsEOL;
    }
    for (its=thisRunCommMassFitSuccess.begin(); its!=thisRunCommMassFitSuccess.end(); its++) {
        sprintf(tmp," %6.2f            %5.5d",its->first, its->second);
        line.assign(tmp);
        cout << line << endl;
        if (QLINFOFILE) QuickLookLog << line << dfmsEOL;
        if (L3INFOTOFILE) l3Log << line << dfmsEOL;
    }
    
    cout << endl;
    cout << "--------------------------------------------" << endl;
    cout << " Commanded    UnSuccessfully" << endl;
    cout << "    Mass        Processed" << endl;
    cout << "--------------------------------------------" << endl;
    if (QLINFOFILE) {
        QuickLookLog << "--------------------------------------------" << dfmsEOL;
        QuickLookLog << " Commanded    UnSuccessfully" << dfmsEOL;
        QuickLookLog << "    Mass        Processed" << dfmsEOL;
        QuickLookLog << "--------------------------------------------" << dfmsEOL;
    }
    if (L3INFOTOFILE) {
        l3Log << "--------------------------------------------" << dfmsEOL;
        l3Log << " Commanded    UnSuccessfully" << dfmsEOL;
        l3Log << "    Mass        Processed" << dfmsEOL;
        l3Log << "--------------------------------------------" << dfmsEOL;
    }
    for (itf=thisRunCommMassFitFailed.begin(); itf!=thisRunCommMassFitFailed.end(); itf++) {
        sprintf(tmp," %6.2f            %5.5d",itf->first, itf->second);
        line.assign(tmp);
        cout << line << endl;
        if (QLINFOFILE) QuickLookLog << line << dfmsEOL;
        if (L3INFOTOFILE) l3Log << line << dfmsEOL;
    }
    cout << "--------------------------------------------------------" << endl;
    if (QLINFOFILE) QuickLookLog << "--------------------------------------------------------" << dfmsEOL;
    if (L3INFOTOFILE) l3Log << "--------------------------------------------------------" << dfmsEOL;
    
}

//
//  ------------------------ findNumFilesInDir ---------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Return the number of files of type 'type'
//
// inputs:
//   path - path name of directory
//   type - a search string to narrow search to specific type
//
// returns:
//   Number of file found
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
long findNumFilesInDir(string path, string type) {

	string sFunctionName="utility->findNumFilesInDir";

	DIR *dir;
  	struct dirent *entry;
  	string fileName;
  	long num=0L;

  	string ctype = type;
  	std::transform(type.begin(), type.end(), type.begin(), ::toupper);

  	int clen = ctype.length();

	if ((dir = opendir(path.c_str())) == NULL) {
		sErrorMessage="Default L2 Source directory - "+path+" does not exist";
		writeToLog(sErrorMessage, sFunctionName, FATALERROR);
		return 0;
  	} else {
  		while ((entry = readdir(dir)) != NULL) {
  			fileName = string(entry->d_name);
  			string tFile = fileName;
  			std::transform(tFile.begin(), tFile.end(), tFile.begin(), ::toupper);
  			//cout << fileName << endl;
  			string firstChar = tFile.substr(0,1);
			// Skip all fileNames shorter than 3 characters.. (like ./, ../)
			if (tFile.size() > 3 && firstChar.compare(".") != 0) {
				string suffix = tFile.substr(tFile.find_last_of(".") + 1);
				int fstype = tFile.find_first_of(ctype);
				// Only use non-empty .TAB files
				if((suffix == "tab" || suffix == "TAB") && (fstype > 0)) {
					num++;
				}
			}
  		}
  		closedir(dir);
  	}

	return num;
}

//
//  ------------------------------ isEqual -------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Returns true if both arguments are the same
//
// inputs:
//   a - number 1
//   b - number 2
//
// returns:
//   True if same false if diff.
// =============================================================================
// History:  Written by Mike Rinaldi July 2014
// =============================================================================
//
bool isEqual (double a, double b) {
  return (a==b);
}

//
//  ----------------------------- numUnique ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Returns the number of unique values in a vector
//
// inputs:
//   v - input vector
//
// returns:
//   # of nuique values
// =============================================================================
// History:  Written by Mike Rinaldi July 2014
// =============================================================================
//
int numUnique(const vector<double> &v) {

	set<double>::iterator iter;
	set<double> uniqv;
	uniqv.insert(v[0]);
	for (int i=0; i<v.size(); i++) {
		for (int j=i+1; j<v.size(); j++) {
			if (!isEqual(v[i],v[j])) {
				uniqv.insert(v[j]);
				//cout << "v[" << i << "]=" << v[i] << endl;
				//for(iter=uniqv.begin(); iter!=uniqv.end();++iter)  cout<<(*iter)<<endl;
			}
		}
	}

	return uniqv.size();

}

void printLastLine (string key, vector<string> &v) {

	vector<string>::iterator it;
	cout << key <<": ";
	for (it=v.begin() ; it < v.end(); it++ ) {
		cout << *it << endl;
	}
}

//
//  ----------------------------- printxFVector ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Prints contents of the X0FitInfo vector
//
// inputs:
//   vector<x0FitInfo>  - input vector
//
// returns:
//   None
// =============================================================================
// History:  Created by Mike Rinaldi.  October 2016
// =============================================================================

void printxFVector (string mem, vector<x0FitInfo> &xF) {

	if (mem.compare("m0") == 0) {
		for (int i = 0; i<xF.size(); i++) {
			cout << "\n------- Member: " << i+1 << " of " << xF.size() << " -------" << endl;
			xF[i].printM0();
		}
	} else if (mem.compare("m0") == 0) {
		for (int i = 0; i<xF.size(); i++) {
			cout << "\n------- Member: " << i+1 << " of " << xF.size() << " -------" << endl;
			xF[i].printM0();
		}
	} else {
		for (int i = 0; i<xF.size(); i++) {
			cout << "\n------- Member: " << i+1 << " of " << xF.size() << " -------" << endl;
			xF[i].print();
		}
	}
}

//
//  ----------------------------- util_sortInt ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Simple bubble sort routine
//
// inputs:
//   num  - Number of array elements
//   array - pointer to an array
//
// returns:
//   sorted array
// =============================================================================
// History:  Created by Mike Rinaldi.  Feb 12, 2018
// =============================================================================
//
void util_sortInt(int num, int array[]) {
    
    double temp=0;
    
    for (int i = 0; i < num; i++) {
        for (int j = 0; j < (num - i - 1); j++) {
            if (array[j] > array[j + 1]) {
                temp = array[j];
                array[j] = array[j + 1];
                array[j + 1] = temp;
            }
        }
    }
}

//
//  ------------------------------ util_stats ----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Calculate simple stats for input array
//
// inputs:
//   array - pointer to an array
//   num   - number in array
//   st    - empty stats structure
//
// returns:
//   st    - full stats structure
// =============================================================================
// History:  Created by Mike Rinaldi.  Feb 14, 2018
// =============================================================================
//
void util_stats(double array[], int num, statsInfo &st) {
    
    double sum=0.0;
    int *temp = new int[num];
    double min,max,avg,med,sig = {0.0};
    max = -1.0e20;
    min = 1.0e20;
    
    // Calculate Min/Max
    for (int i = 0; i < num; i++) {
        sum += array[i];
        if (array[i] > max) {
            max = array[i];
        }
        if (array[i] < min) {
            min = array[i];
        }
    }
    st.min = (double)min;
    st.max = (double)max;
    
    // Calculate Average
    avg = (double)(sum/num);
    st.avg = avg;

    // Calculate Standard Deviation
    sum = 0.0;
    for (int i = 0; i < num; i++) {
        sum += pow((array[i] - avg),2);
    }
    st.sigma = (double)(sqrt(sum/(num-1)));
    
    // Calculate Median
    for (int i=0; i<num; i++) temp[i] = (int)array[i];
    util_sortInt(num, temp);
    st.median = (double)(temp[num/2]);
    
    delete[] temp; temp=0;
}


//
//  ----------------------------- replace ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Replace one substring with another
//
// inputs:
//   str  - input string
//   sub1 - substring 1 will be replaced with
//   sub2 - ...
//
// returns:
//   true if sub1 found
// =============================================================================
// History:  Found on Stackoverflow.com.  April 2015
// =============================================================================
//
bool replace(string& str, const string& sub1, const string& sub2) {
    size_t start_pos = str.find(sub1);
    if(start_pos == string::npos) {
        return false; //
    } else {
    	str.replace(start_pos, sub1.length(), sub2);
    	return true;
    }
}

//
//  --------------------------- dateException ----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Check if input Julian time is within date exclusion period.
//
// inputs:
//   time  - input Julian time in sec
//
// returns:
//   true if in exclusion zone false otherwise
// =============================================================================
// History:  Created by Mike Rinaldi.  September 2015
// =============================================================================
//
bool dateException(double time) {

	for (int i=0; i<dateExclusion.size(); i++ ){
		if (time >= dateExclusion[i].dStart && time <= dateExclusion[i].dEnd) {
			return true;
		}
	}

	return false;
}

//
//  --------------------------- checkOStype ----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Check if which OS is being used then report and set EOL strings
//
// returns:
//   None
// =============================================================================
// History:  Created by Mike Rinaldi.  October 2015
// =============================================================================
//
void checkOStype() {

	// Check which OS type we running on and set EOL characters/string and dump
	// to screen and various logs
    if (OSTYPE == "MacOSX" || OSTYPE == "Linux" || OSTYPE == "Unix") {
    	sprintf(EOL,"\n");
    	dfmsEOL.assign(EOL);
    	cout <<"****************************************************************" << dfmsEOL;
    	cout << "    Detected \"" << OSTYPE << "\" as the Operating System in use" << dfmsEOL;
    	cout <<"****************************************************************" << dfmsEOL;

    	toLog <<"****************************************************************" << dfmsEOL;
    	toLog << "    Detected \"" << OSTYPE << "\" as the Operating System in use" << dfmsEOL;
    	toLog <<"****************************************************************" << dfmsEOL;

    	if (L3INFOTOFILE) {
    		l3Log <<"****************************************************************" << dfmsEOL;
    		l3Log << "    Detected \"" << OSTYPE << "\" as the Operating System in use" << dfmsEOL;
    		l3Log <<"****************************************************************" << dfmsEOL;
    	}
    } else if (OSTYPE == "Windows" || OSTYPE == "Cygwin") {
    	sprintf(EOL,"\r\n");
    	dfmsEOL.assign(EOL);
    	cout <<"****************************************************************" << dfmsEOL;
    	cout << "    Detected \"" << OSTYPE << "\" as the Operating System in use" << dfmsEOL;
    	cout <<"****************************************************************" << dfmsEOL;

    	toLog <<"****************************************************************" << dfmsEOL;
    	toLog << "    Detected \"" << OSTYPE << "\" as the Operating System in use" << dfmsEOL;
    	toLog <<"****************************************************************" << dfmsEOL;

    	if (L3INFOTOFILE) {
    		l3Log <<"****************************************************************" << dfmsEOL;
    		l3Log << "    Detected \"" << OSTYPE << "\" as the Operating System in use" << dfmsEOL;
    		l3Log <<"****************************************************************" << dfmsEOL;
    	}
    } else {
    	sprintf(EOL,"\n");
    	dfmsEOL.assign(EOL);
    	cout <<"******************************************" << dfmsEOL;
    	cout << "    Operating System Type Unknown" << dfmsEOL;
    	cout <<"******************************************" << dfmsEOL;
    	toLog <<"******************************************" << dfmsEOL;
    	toLog << "    Operating System Type Unknown" << dfmsEOL;
    	toLog <<"******************************************" << dfmsEOL;
    	if (L3INFOTOFILE) {
    		l3Log <<"******************************************" << dfmsEOL;
    		l3Log << "    Operating System Type Unknown" << dfmsEOL;
    		l3Log <<"******************************************" << dfmsEOL;
    	}
    }

}

//
// ----------------------- Create a peakExclusion function to add values ----------------------
//
//
// =============================================================================
// History:  Created by Mike Rinaldi.  September 2018
// =============================================================================
//
//void addPair(pair<int,int> &p, string s1, string s2) {
 
//    int d1,d2;
    
//    d1 = util_stringToInt(s1);
//    d2 = util_stringToInt(s2);
    
//    p = make_pair(d1,d2);
    
//}

//
// ----------------------- Add pair to Vector of pairs ----------------------
//
//
// =============================================================================
// History:  Created by Mike Rinaldi.  September 2018
// =============================================================================
//
//void addToVec(vector<pair<int,int>> &v, pair<int,int> p) {
 //   v.push_back(p);
//}

