

#include "DFMS_stats.hh"

using namespace std;

//
// ------------------------ Find Standard Deviation  ---------------------------
//
//==============================================================================
// Routine Description
//==============================================================================
// This routine takes an input set of values and computes its 
// standard deviation.
//
// inputs:
//   nElements - number of elements in the anValue array
//   anValue - the array of values to compute a standard deviation
//
// returns:
//   the standard deviation of the values in anValue
//==============================================================================
// History:  Most methods of this class were constructed from C++ modifications
// of the RTOF BAM_curve_fit.c code written by Brian Magee. 
// Modifications: C++ version written by Mike Rinaldi, March 2013
//==============================================================================
//
double standardDev (int nElements, double *anValue) {

    string sFunctionName = "standardDev";

	double nMean;
	double nStDev;
	double nPopElements;
	double nPopFactor;
	nMean = 0;
	nStDev = 0;
	nPopElements = nElements;
	nPopFactor = 1.0 / (nPopElements - 1.0);

	// determine the mean
	for (int i=0; i<nElements; i++) nMean = nMean + anValue[i];
	nMean = nMean / nPopElements;

	// determine the stdev
	for (int i=0; i<nElements; i++) {
		nStDev = nStDev + pow(anValue[i] - nMean,2);
	}
	nStDev = sqrt(nPopFactor * nStDev);

	return nStDev;
}


//
// ------------------------ Find Chi Square of fit -----------------------------
//
//==============================================================================
// Routine Description
//==============================================================================
// This routine calculates the reduced chi-squared statistic of 
// fitted data based on the observed reference values, their weights, 
// and the number of degrees of freedom.
//
// inputs:
//   nRecords - the number of records in the data
//   anWeight - the list of weights of the data points
//   anY - the Y-coordinate data values
//   anYfit - the Y-coordinate data values of the fitted function
//   nDegree - the number of degrees of freedom
//
// returns:
//   nChiSquare - the calculated reduced chi-squared statistic
//==============================================================================
// History:  All methods of this class were constructed from C++ modifications
// of the RTOF BAM_curve_fit.c codes written by Brian Magee. 
// Modifications: C++ version written by Mike Rinaldi, March 2013
//==============================================================================
//
double chiSquare (int nRecords, double* anWeight, double* anY, 
                  double* anYfit, int nDegree) {

    string sFunctionName = "chiSquare";

	double nChiSquare;
	nChiSquare = 0;

	for (int i=0; i<nRecords; i++) {
		nChiSquare += anWeight[i] * pow((anY[i] - anYfit[i]),2.0);
	}
	nChiSquare /= nDegree;

	return nChiSquare;
}

//
// ----------------------- Find index of Max value -----------------------------
//
//==============================================================================
// Routine Description
//==============================================================================
// This routine takes as input an array of values and returns the index of the 
// maximum
//
// inputs:
//   anValues - the array of values
//   nRecords - the number of records in the anValues array
//   nMaxThreshold - the upper limit threshold on the input data below which 
//                   the maximum value will be reported
//
// returns:
//   nMaxIndex - the index of the maximum value (below the upper limit threshold)
//==============================================================================
// History:  All methods of this class were constructed from C++ modifications
// of the RTOF BAM_curve_fit.c codes written by Brian Magee. 
// Modifications: C++ version written by Mike Rinaldi, March 2013
//==============================================================================
//
int maxIndex (int nRecords, double *anValues, double nMaxThreshold) {
    
    string sFunctionName = "maxIndex";

	double nMaxValue = -1.0 * pow(10.0,99);
	int nMaxIndex = 0;

	for (int i=0; i < nRecords; i++) {
		if (anValues[i] > nMaxValue && anValues[i] < nMaxThreshold) {
			nMaxIndex = i;
			nMaxValue = anValues[i];
		}
	}

	return nMaxIndex;
}

//
// ----------------------- Find index of Min value -----------------------------
//
//==============================================================================
// Routine Description
//==============================================================================
// This routine takes as input an array of values and returns the index of the 
// minimum
//
// inputs:
//   anValues - the array of values
//   nRecords - the number of records in the anValues array
//   nMinThreshold - the lower limit threshold on the input data above which the
//                   minimum value will be reported
//
// returns:
//   nMinIndex - the index of the minimum value (above the lower limit threshold)
//==============================================================================
// History:  All methods of this class were constructed from C++ modifications
// of the RTOF BAM_curve_fit.c codes written by Brian Magee. 
// Modifications: C++ version written by Mike Rinaldi, March 2013
//==============================================================================
//
int minIndex (int nRecords, double *anValues, double nMinThreshold) {
    
	string sFunctionName = "minIndex";

	double nMinValue = pow(10.0,99);;
	int nMinIndex = 0;

	for (int i=0; i < nRecords; i++) {
		if (anValues[i] < nMinValue && anValues[i] > nMinThreshold) {
			nMinIndex = i;
			nMinValue = anValues[i];
		}
	}

	return nMinIndex;
}

//
// ------------------- propagate uncertainty of products -----------------------
//
//==============================================================================
// Routine Description
//==============================================================================
// This routine takes an input set of values and associated uncertainties 
// and computes the product (allowing for both multiplication and division 
// on a per-element basis - e.g. "2 * 6 / 4") along with the propagated 
// uncertainty in that product.  The routine returns the product Value or 
// Uncertainty depending on the requested return value.
//
// inputs:
//   nReturnType - informs the return of either Value or Uncertainty
//                 0 for value, 1 for uncertainty
//   nElements - number of elements in the anValue and anUncertainty arrays
//   anValue - the array of values to compute a product
//   anUncertainty - the associated uncertainties of each value
//   asOperator - the operator ('*' or '/' for multiplication or division) 
//                for each element (set the first to be '*' for the first 
//                element to be multiplied by 1, or '/' to be set to the 
//                first element to be 1 divided by the first element)
//
// returns:
//   nFunctionValue (product) or nFunctionUncertainty (product uncertainty) 
//   depending on the provided nReturnType value
//==============================================================================
// History:  All methods of this class were constructed from C++ modifications
// of the RTOF BAM_curve_fit.c codes written by Brian Magee. 
// Modifications: C++ version written by Mike Rinaldi, March 2013
//==============================================================================
//
double *propagateErrorProd (int nReturnType, int nElements, double *anValue,
                          double *anUncertainty, char *asOperator) {

    string sFunctionName = "propagateErroProd";

    double *nReturnValue = new double;
	double nFunctionValue;
	double nFunctionValueUncertainty;

	nFunctionValue = 1;
	nFunctionValueUncertainty = 0;

	char sOperator[2];
	for (int i=0; i < nElements; i++) {
		sOperator[0] = asOperator[i];
		if (sOperator[0] == '*') nFunctionValue = nFunctionValue * anValue[i];
		else nFunctionValue = nFunctionValue / anValue[i];
		nFunctionValueUncertainty = nFunctionValueUncertainty + 
                                      pow((anUncertainty[i] / anValue[i]), 2.0);
	}

	nFunctionValueUncertainty = nFunctionValue * sqrt(nFunctionValueUncertainty);

	if (nReturnType) nReturnValue[0] = nFunctionValueUncertainty;
	else nReturnValue[0] = nFunctionValue;

	return nReturnValue;

}

//
// ------------------- propagate uncertainty of sum -----------------------
//
//==============================================================================
// Routine Description
//==============================================================================
// This routine takes an input set of values and associated uncertainties 
// and computes the sum (allowing for both addition and subtraction on a 
// per-element basis - e.g. "2 + 6 - 4") along with the propagated uncertainty
// in that sum.  The routine returns the sum Value or Uncertainty 
// depending on the requested return value.
//
// inputs:
//   nReturnType - informs the return of either Value or Uncertainty
//                 0 for value, 1 for uncertainty
//   nElements - number of elements in the anValue and anUncertainty arrays
//   anValue - the array of values to compute a sum
//   anUncertainty - the associated uncertainties of each value
//   asOperator - the operator ('+' or '-' for addition or subtraction) 
//                for each element (set the first to be '+' for the first 
//                element to be positive, or '-' to set the first element 
//                to be negative)
//
// returns:
//   nFunctionValue (sum) or nFunctionUncertainty (sum uncertainty) 
//   depending on the provided nReturnType value
//==============================================================================
// History:  All methods of this class were constructed from C++ modifications
// of the RTOF BAM_curve_fit.c codes written by Brian Magee. 
// Modifications: C++ version written by Mike Rinaldi, March 2013
//==============================================================================
//
double *propagateErrorSum (int nReturnType, int nElements, double *anValue,
                          double *anUncertainty, char *asOperator) {

	string sFunctionName = "propagateErrorSum";

	double *nReturnValue = new double;
	double nFunctionValue;
	double nFunctionValueUncertainty;

	nFunctionValue = 0;
	nFunctionValueUncertainty = 0;

	char sOperator[2];
	for (int i=0; i < nElements; i++) {
		sOperator[0] = asOperator[i];
		if (sOperator[0] == '+') nFunctionValue = nFunctionValue + anValue[i];
		else nFunctionValue = nFunctionValue - anValue[i];
		nFunctionValueUncertainty = nFunctionValueUncertainty + 
                                      pow(anUncertainty[i],2);
	}

	nFunctionValueUncertainty = sqrt(nFunctionValueUncertainty);

	if (nReturnType) nReturnValue[0] = nFunctionValueUncertainty;
	else nReturnValue[0] = nFunctionValue;

	return nReturnValue;
}

//
// ------------------- propagate uncertainty of exponent -----------------------
//
//==============================================================================
// Routine Description
//==============================================================================
// This routine takes an input set of values and associated uncertainties 
// and computes their value and propagated
// uncertainty when each is raised to an exponent (need not be static for 
// all values).   The routine returns the esulting Values or Uncertainties 
// depending on the requested return value.
//
// inputs:
//   nReturnType - informs the return of either Value or Uncertainty
//                 0 for value, 1 for uncertainty
//   nElements - number of elements in the anValue and anUncertainty arrays
//   anValue - the array of values to compute exponentiation
//   anUncertainty - the associated uncertainties of each value
//   anExponent - the values of the exponents for each element of anValue
//
// returns:
//   anFunctionValue or anFunctionUncertainty depending on the provided 
//   nReturnType value
//==============================================================================
// History:  All methods of this class were constructed from C++ modifications
// of the RTOF BAM_curve_fit.c codes written by Brian Magee. 
// Modifications: C++ version written by Mike Rinaldi, March 2013
//==============================================================================
//
double *propagateErrorExp (int nReturnType, int nElements, double *anValue,
                          double *anUncertainty, double *anExponent) {
    
	string sFunctionName = "propagateErrorExp";

	double *anReturnValue = new double[nElements];
	double anFunctionValue[nElements];
	double anFunctionValueUncertainty[nElements];

	for (int i=0; i < nElements; i++) {
		anFunctionValue[i] = pow(anValue[i], anExponent[i]);
		anFunctionValueUncertainty[i] = anFunctionValue[i] * 
                            fabs(anExponent[i]) * anUncertainty[i] / anValue[i];
	}

	if (nReturnType) for (int i=0; i < nElements; i++) {
         anReturnValue[i] = anFunctionValueUncertainty[i];
	} else {
         for (int i=0; i < nElements; i=i+1) anReturnValue[i] = anFunctionValue[i];
     }

	return anReturnValue;
}

//
// --------------------- propagate uncertainty of log --------------------------
//
//==============================================================================
// Routine Description
//==============================================================================
// This routine takes an input set of values and associated uncertainties 
// and computes their natural logarithm and ropagated uncertainty.   The 
// routine returns the resulting Values or Uncertainties depending on the 
// requested return value.
//
// inputs:
//   nReturnType - informs the return of either Value or Uncertainty
//                 0 for value, 1 for uncertainty
//   nElements - number of elements in the anValue and anUncertainty arrays
//   anValue - the array of values to compute natural logarithms
//   anUncertainty - the associated uncertainties of each value
//
// returns:
//   anFunctionValue or anFunctionUncertainty depending on the provided 
//   nReturnType value
//==============================================================================
// History:  All methods of this class were constructed from C++ modifications
// of the RTOF BAM_curve_fit.c codes written by Brian Magee. 
// Modifications: C++ version written by Mike Rinaldi, March 2013
//==============================================================================
//
double *propagateErrorLog (int nReturnType, int nElements, double *anValue,
                            double *anUncertainty)
{

	string sFunctionName = "propagateErrorLog";

	double *anReturnValue = new double[nElements];
	double anFunctionValue[nElements];
	double anFunctionValueUncertainty[nElements];

	for (int i=0; i < nElements; i++) {
     	anFunctionValue[i] = log(anValue[i]);
		anFunctionValueUncertainty[i] = anUncertainty[i] / anValue[i];
	}

	if (nReturnType) {
		for (int i=0; i < nElements; i++) {
			anReturnValue[i] = anFunctionValueUncertainty[i];
		}
	} else {
		for (int i=0; i < nElements; i=i+1) {
			anReturnValue[i] = anFunctionValue[i];
		}
	}
	return anReturnValue;
}

//
// ----------------- propagate uncertainty of args of exp ----------------------
//
//==============================================================================
// Routine Description
//==============================================================================
// This routine takes an input set of values and associated uncertainties 
// and computes their value and propagated uncertainty when used as 
// exponents for the mathematical constant "e".   The routine returns the 
// resulting Values or Uncertainties depending on the requested return value.
//
// inputs:
//   nReturnType - informs the return of either Value or Uncertainty
//                 0 for value, 1 for uncertainty
//   nElements - number of elements in the anValue and anUncertainty arrays
//   anValue - the array of values to compute exponentiation
//   anUncertainty - the associated uncertainties of each value
//
// returns:
//   anFunctionValue or anFunctionUncertainty depending on the provided 
//   nReturnType value
//==============================================================================
// History:  All methods of this class were constructed from C++ modifications
// of the RTOF BAM_curve_fit.c codes written by Brian Magee. 
// Modifications: C++ version written by Mike Rinaldi, March 2013
//==============================================================================
//
double *propagateErrorExpArgs (int nReturnType, int nElements, double *anValue,
                               double *anUncertainty) {

	string sFunctionName = "propagateErrorExpArgs";

	double *anReturnValue =new double[nElements];
	double anFunctionValue[nElements];
	double anFunctionValueUncertainty[nElements];

	for (int i=0; i < nElements; i++) {
		anFunctionValue[i] = exp(anValue[i]);
		anFunctionValueUncertainty[i] = anFunctionValue[i] * anUncertainty[i];
	}

	if (nReturnType) {
         for (int i=0; i < nElements; i=i+1) {
             anReturnValue[i] = anFunctionValueUncertainty[i];
         }
	} else {
         for (int i=0; i < nElements; i=i+1) anReturnValue[i] = anFunctionValue[i];
     }

	return anReturnValue;
}

//
// ------------------ propagate uncertainty when x consts ----------------------
//
//==============================================================================
// Routine Description
//==============================================================================
// This routine takes an input set of values and associated uncertainties and 
// computes their value and propagated uncertainty when multiplied by a constant
// (need not be the same for all values).   The routine returns the resulting
//  Values or Uncertainties depending on the requested return value.
//
// inputs:
//   nReturnType - informs the return of either Value or Uncertainty
//                 0 for value, 1 for uncertainty
//   nElements - number of elements in the anValue and anUncertainty arrays
//   anValue - the array of values to compute products
//   anUncertainty - the associated uncertainties of each value
//   anConstant - the array of constants to use for multiplication of each 
//               element of anValue
//
// returns:
//   anFunctionValue or anFunctionUncertainty depending on the provided 
//   nReturnType value
//==============================================================================
// History:  All methods of this class were constructed from C++ modifications
// of the RTOF BAM_curve_fit.c codes written by Brian Magee. 
// Modifications: C++ version written by Mike Rinaldi, March 2013
//==============================================================================
//
double *propagateErrorProdConst (int nReturnType, int nElements, double *anValue,
                                   double *anUncertainty, double *anConstant) {

	string sFunctionName = "propagateErrorprodConst";

	double *anReturnValue = new double [nElements];
	double anFunctionValue[nElements];
	double anFunctionValueUncertainty[nElements];

	for (int i=0; i < nElements; i++) {
		anFunctionValue[i] = anValue[i] * anConstant[i];
		anFunctionValueUncertainty[i] = anUncertainty[i] * anConstant[i];
	}

	if (nReturnType) {
         for (int i=0; i < nElements; i++) {
             anReturnValue[i] = anFunctionValueUncertainty[i];
         }
	} else {
         for (int i=0; i < nElements; i=i+1) {
             anReturnValue[i] = anFunctionValue[i];
         }
     }
     
	return anReturnValue;
}
