
#include "DFMS_ProcessMCP_Class.hh"

//
// -------------- Class DFMS_ProcessMCP_Class ------------------------------
//

//
// ------------------------- Constructor ---------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This Class will step through the various processes to convert an L2 to L3. 
// This method is the constructor
// inputs: 
//   p - Mass Peak Search Table path
//   c - path to COPS data
//   l3p - L3 output file path
//   l3 - L3 file name
//   ffp - Path of Fit files
//   gA - gain step value for row A
//   gB - gain step value for row B
//   pg - pixel gain array
//   L2 - L2 file object pointer
//

// returns:
//   None
// =============================================================================
// History: Written by Mike Rinaldi, March 2013
// =============================================================================
//
DFMS_ProcessMCP_Class::DFMS_ProcessMCP_Class (string p, string c, string l3p,
                           string l3, string ffp, double gA, double gB, dbl2D &pg,
                           DFMS_PDS_L2_file_Class *L2)
{
    
	string sFunctionName="DFMS_ProcessMCP_Class::Constructor";
    

	// Initialize various member fields
	mptRows=0;
	PeakF=NULL;
	zoom = 0.0;
	mptIndx = -1;
	COPSobjIn = NULL;
	MPSTobjIn = NULL;

	massType = "";
	resType = "";

	linFitA=NULL;
	linFitB=NULL;

	iXl = 0;		 		// valid starting pixel (left)
	iXr = 0;		 		// valid end pixel (right)

	l3PkMaxIdA = 0;			// Location of the maximum Ion rate for Row A
	l3PkMaxIdB = 0;			// Location of the maximum Ion rate for Row B

	numGoodPks = 0;
	foundVerifPeak = false;  // Set verification peak found flag to false

	noSLFFitFile = false;

	sMPSTpath = p;           // Mass Peak Table path
	sCOPSpath = c;           // COPS Data path
	sL3Path = l3p;           // L3 output path
	L3File = l3;             // L3 file name
    fitFilePath = ffp;       // Path to fit file
	L2Obj = L2;              // L2 file Object
    gainA = gA;              // Gain Step value for Row A
    gainB = gB;              // Gain Step value for Row B

    slfPix0FitFileName = "None";		// Latest GCU calibration fit file
    gcuPix0FitFileName = "None";		// Latest SLF calibration fit file
          
    // Copy the pixel Gain array pg into the local var pixG
    for (int i=0; i<NUMPIX; i++) {
    	// Use vector objects dynamic fill to add values
	 	pixG.push_back(dslice());
	 	for (int j=0; j<5; ++j) {
	 		pixG[i].push_back(0.0);
            pixG[i][j] = pg[i][j];
        }
     }

    // Form the average Pixel Gain
    avgPgA = findAvgPixelGain(1);
    avgPgB = findAvgPixelGain(2);

    // peak Area
    peakAreaA = 0.0;
    peakAreaB = 0.0;

    // Allocate memory for final mass Scale
    fMassScaleA = new double[NUMPIX];
    fMassScaleB = new double[NUMPIX];

    offSetFitA = new double[NUMPIX];
    offSetFitB = new double[NUMPIX];
    for(int i=0; i<NUMPIX; ++i) {
         offSetFitA[i] = 0.0;
         offSetFitB[i] = 0.0;
    }

     // Make local L2 data arrays
     Xin = new double[NUMPIX];
     for(int i=0; i<NUMPIX; ++i) {
          Xin[i] = (double)L2Obj->L2Data[i][0];
     }  
     YAin = new double[NUMPIX];
     yAFit = new double[NUMPIX];
     for(int i=0; i<NUMPIX; ++i) {
         YAin[i] = (double)L2Obj->L2Data[i][1];
         yAFit[i] = 0.0;
     }
     YBin = new double[NUMPIX];
     yBFit = new double[NUMPIX];
     for(int i=0; i<NUMPIX; ++i) {
          YBin[i] = (double)L2Obj->L2Data[i][2];
          yBFit[i] = 0.0;
     }

     // Initialize the 2D L3 data array
     for (int i=0; i<NUMPIX; ++i) {
         L3Data.push_back(dslice());  // Create one row 
         L3Data[i].push_back((double)i); // Place holder for pixel number
         L3Data[i].push_back(0.0);    // Place holder for mass @pix i side A
         L3Data[i].push_back(0.0);    // Place holder for Ion current side A
         L3Data[i].push_back(0.0);    // Place holder for mass @pix i side B
         L3Data[i].push_back(0.0);    // Place holder for Ion current side B             
     }

     // Initialize the 2D L3HKdata array
     for (int i=0; i<L3HKROWS; ++i) {
         L3HKdata.push_back(sslice());  // Create one row 
         L3HKdata[i].push_back(" ");    // Create column 1
         L3HKdata[i].push_back(" ");    // Create column 2
         L3HKdata[i].push_back(" ");    // Create column 3
         L3HKdata[i].push_back(" "); 	// Create column 4
     }
    
    // Variables associted with 3rd order polynomial fit
    xPolyA=NULL;
    xPolyB=NULL;
    yPolyA=NULL;
    yPolyB=NULL;
    
    numGdAPixels=0;
    numGdBPixels=0;
   
}

//
// -------------------------- Destructor ---------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Release dynamic memory
// inputs: 
//   
// returns:
//   None
// =============================================================================
// History: Written by Mike Rinaldi, March 2013
// =============================================================================
//
DFMS_ProcessMCP_Class::~DFMS_ProcessMCP_Class() {
    
    // Release dynamically allocated memory
    delete[] YAin;  YAin=0;
    delete[] YBin;  YBin=0;
    delete[] yAFit; yAFit=0;
    delete[] yBFit; yBFit=0;
    delete[] Xin;   Xin=0;
    delete[] fMassScaleA;  fMassScaleA = 0;
    delete[] fMassScaleB;  fMassScaleB = 0;
    delete[] offSetFitA; offSetFitA=0;
    delete[] offSetFitB; offSetFitB=0;
    delete[] xPolyA; xPolyA=0;
    delete[] xPolyB; xPolyB=0;
    delete[] yPolyA; yPolyA=0;
    delete[] yPolyB; yPolyB=0;
	delete linFitA; linFitA=0;
	delete linFitB; linFitB=0;
    L3Data.clear();
    pixG.clear();
}

//
// ----------------------------- getL2HKdata -----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method assigns the L2 HK data that needs to pass unchanged to L3 HK (row 0-12)
//
// inputs: 
//   L2HKdata - L2 HK data array
//   
// returns:
//   None
// =============================================================================
// History: Written by Mike Rinaldi, March 2013
// =============================================================================
//
void DFMS_ProcessMCP_Class::getL2HKdata(str2D & L2HKdata) {
    
     string sFunctionName="DFMS_ProcessMCP_Class::getL2HKdata";
     
     int numFields = 4;

     // Same as L2 values ROSINA_DFMS_SCI_MODE
     for (int i=0; i<numFields; ++i) L3HKdata[0][i] = L2HKdata[2][i];
     // Same as L2 values ROSINA_DFMS_SCI_MODE
     for (int i=0; i<numFields; ++i) L3HKdata[1][i] = L2HKdata[3][i];
     // Same as L2 values ROSINA_DFMS_SCI_MAG_TEMPERATURE
     for (int i=0; i<numFields; ++i) L3HKdata[2][i] = L2HKdata[43][i];
     // Same as L2 values ROSINA_DFMS_SCI_COVER_POSITION
     for (int i=0; i<numFields; ++i) L3HKdata[3][i] = L2HKdata[44][i];
     // Same as L2 values ROSINA_DFMS_SCI_LEDA_FAR_OFFSET
     for (int i=0; i<numFields; ++i) L3HKdata[4][i] = L2HKdata[48][i];
     // Same as L2 values ROSINA_DFMS_SCI_GAIN_OFF_SPECTRUM
     for (int i=0; i<numFields; ++i) L3HKdata[5][i] = L2HKdata[52][i];
     // Same as L2 values ROSINA_DFMS_IONSRC_TEMPRATURE
     for (int i=0; i<numFields; ++i) L3HKdata[6][i] = L2HKdata[128][i];
     // Same as L2 values ROSINA_DFMS_MAGNET_TEMPERATURE
     for (int i=0; i<numFields; ++i) L3HKdata[7][i] = L2HKdata[170][i];
     // Same as L2 values ROSINA_DFMS_RDP_TEMPERATURE
     for (int i=0; i<numFields; ++i) L3HKdata[8][i] = L2HKdata[173][i];
     // Same as L2 values ROSINA_DFMS_LEDA_TEMPERATURE
     for (int i=0; i<numFields; ++i) L3HKdata[9][i] = L2HKdata[174][i];
     // Same as L2 values ROSINA_DFMS_ASPC_TEMPERATURE
     for (int i=0; i<numFields; ++i) L3HKdata[10][i] = L2HKdata[228][i];
     // Same as L2 values ROSINA_DFMS_FDPA_TEMPERATURE
     for (int i=0; i<numFields; ++i) L3HKdata[11][i] = L2HKdata[244][i];

}

//
// ---------------------------- formL3HKdata -----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method constructs the remaining L3 housekeeping data array (row 13-end)
//
// inputs: None
//   
// returns:
//   none
// =============================================================================
// History: Written by Mike Rinaldi, March 2013
// =============================================================================
//
void DFMS_ProcessMCP_Class::formL3HKdata() {
    
     string sFunctionName="DFMS_ProcessMCP_Class::formL3HKdata";
     
     
     // Define beginning of L3 HK additions
     int HKs = 12;

     // Set COPS NG ----------------------------------------
     L3HKdata[HKs][0] = "ROSINA_COPS_PRESSURE_NG         ";
     L3HKdata[HKs][1] = "     ";
     if (L3Info->copsPressNG != BADDATA) {

    	 L3HKdata[HKs][2] = util_doubleToString(L3Info->copsPressNG);
    	 util_toUpperCase(L3HKdata[HKs][2]);
     } else {
    	 L3HKdata[HKs][2] = "N/A";
     }
     L3HKdata[HKs++][3] = "mbar ";
     
     // Set COPS RGv ----------------------------------------
     L3HKdata[HKs][0] = "ROSINA_COPS_PRESSURE_RG         ";
     L3HKdata[HKs][1] = "     ";
     if (L3Info->copsPressRG != BADDATA) {
    	 L3HKdata[HKs][2] = util_doubleToString(L3Info->copsPressRG);
    	 util_toUpperCase(L3HKdata[HKs][2]);
     } else {
    	 L3HKdata[HKs][2] = "N/A";
     }
     L3HKdata[HKs++][3] = "mbar ";
     
     // Set Cal Signal A ----------------------------------------
     L3HKdata[HKs][0] = "ROSINA_DFMS_SCI_SIGNAL_CAL_VAL_A";
     L3HKdata[HKs][1] = "     ";
     if (L3Info->cal->calValueA != BADDATA) {
         L3HKdata[HKs][2] = util_doubleToString(L3Info->cal->calValueA);
     } else {
    	 L3HKdata[HKs][2] = "N/A";
     }
     L3HKdata[HKs++][3] = "     ";

     // Set Cal signal Dev A ----------------------------------------
     L3HKdata[HKs][0] = "ROSINA_DFMS_SCI_SIGNAL_CAL_DEV_A";
     L3HKdata[HKs][1] = "     ";
     if (L3Info->cal->calValueStdevA != BADDATA) {
         L3HKdata[HKs][2] = util_doubleToString(L3Info->cal->calValueStdevA);
     } else {
    	 L3HKdata[HKs][2] = "N/A";
     }
     L3HKdata[HKs++][3] = "     ";
     
     // Set Cal Signal B ----------------------------------------
     L3HKdata[HKs][0] = "ROSINA_DFMS_SCI_SIGNAL_CAL_VAL_B";
     L3HKdata[HKs][1] = "     ";
     if (L3Info->cal->calValueB != BADDATA) {
         L3HKdata[HKs][2] = util_doubleToString(L3Info->cal->calValueB);
     } else {
    	 L3HKdata[HKs][2] = "N/A";
     }
     L3HKdata[HKs++][3] = "     ";

     // Set Cal signal Dev B ----------------------------------------
     L3HKdata[HKs][0] = "ROSINA_DFMS_SCI_SIGNAL_CAL_DEV_B";
     L3HKdata[HKs][1] = "     ";
     if (L3Info->cal->calValueStdevB != BADDATA) {
         L3HKdata[HKs][2] = util_doubleToString(L3Info->cal->calValueStdevB);
     } else {
    	 L3HKdata[HKs][2] = "N/A";
     }
     L3HKdata[HKs++][3] = "     ";

     // Set Offset A ----------------------------------------
     L3HKdata[HKs][0] = "ROSINA_DFMS_SCI_OFF_LEVEL_A     ";
     L3HKdata[HKs][1] = "     ";
     if (L3Info->offset->offsetLevelA != BADDATA) {
    	 L3HKdata[HKs][2] = util_doubleToString(L3Info->offset->offsetLevelA);
     } else {
    	 L3HKdata[HKs][2] = "N/A";
     }
     L3HKdata[HKs++][3] = "     ";
     
     // Set Offset Dev A ----------------------------------------
     L3HKdata[HKs][0] = "ROSINA_DFMS_SCI_OFF_STDEV_A     ";
     L3HKdata[HKs][1] = "     ";
     if (L3Info->offset->offsetLevelAStdDev != BADDATA) {
    	 L3HKdata[HKs][2] = util_doubleToString(L3Info->offset->offsetLevelAStdDev);
     } else {
    	 L3HKdata[HKs][2] = "N/A";
     }
     L3HKdata[HKs++][3] = "     ";
     
     // Offset Polynomial fit Mode specific File name  ---------------
     L3HKdata[HKs][0] = "ROSINA_DFMS_SCI_OFF_COEFF_FILE  ";
     L3HKdata[HKs][1] = "     ";
     L3HKdata[HKs][2] = pC[L2Info.mode].fileName;
     L3HKdata[HKs++][3] = "     ";

     addPolyOffCoeffInfo(1,&HKs);
     
     // Set Offset B ----------------------------------------
     L3HKdata[HKs][0] = "ROSINA_DFMS_SCI_OFF_LEVEL_B     ";
     L3HKdata[HKs][1] = "     ";
     if (L3Info->offset->offsetLevelB != BADDATA) {
    	 L3HKdata[HKs][2] = util_doubleToString(L3Info->offset->offsetLevelB);
     } else {
    	 L3HKdata[HKs][2] = "N/A";
     }
     L3HKdata[HKs++][3] = "     ";
     
     // Set Offset Dev B ----------------------------------------
     L3HKdata[HKs][0] = "ROSINA_DFMS_SCI_OFF_STDEV_B     ";
     L3HKdata[HKs][1] = "     ";
     if (L3Info->offset->offsetLevelBStdDev != BADDATA) {
    	 L3HKdata[HKs][2] = util_doubleToString(L3Info->offset->offsetLevelBStdDev);
     } else {
    	 L3HKdata[HKs][2] = "N/A";
     }
     L3HKdata[HKs++][3] = "     ";
     
     addPolyOffCoeffInfo(2,&HKs);

     // Set GCU pix0 A ----------------------------------------
     L3HKdata[HKs][0] = "ROSINA_DFMS_SCI_GCU_PIXEL0_A    ";
     if (L3Info->cal->GCUPix0_A_used) {
          L3HKdata[HKs][1] = "ON   ";
     } else {
          L3HKdata[HKs][1] = "OFF  ";
     }
     if (L3Info->cal->GCUPix0_A != BADDATA) {
    	 L3HKdata[HKs][2] = util_doubleToString(L3Info->cal->GCUPix0_A);
    	 setPrecision(L2Info.hiRes,L3HKdata[HKs][2]);
     } else {
    	 L3HKdata[HKs][2] = "N/A";
     }
     L3HKdata[HKs++][3] = "     ";
     
     // Set GCU pix0 Unc A ----------------------------------------
     L3HKdata[HKs][0] = "ROSINA_DFMS_SCI_GCU_PIXEL0_UNC_A";
     if (L3Info->cal->GCUPix0_UncA_used) {
          L3HKdata[HKs][1] = "ON   ";
     } else {
          L3HKdata[HKs][1] = "OFF  ";
     }
     if (L3Info->cal->GCUPix0_UncA != BADDATA) {
    	 L3HKdata[HKs][2] = util_doubleToString(L3Info->cal->GCUPix0_UncA);
    	 setPrecision(L2Info.hiRes,L3HKdata[HKs][2]);
     } else {
    	 L3HKdata[HKs][2] = "N/A";
     }
     L3HKdata[HKs++][3] = "     ";
     
     // Set GCU pix0 B ----------------------------------------
     L3HKdata[HKs][0] = "ROSINA_DFMS_SCI_GCU_PIXEL0_B    ";
     if (L3Info->cal->GCUPix0_B_used) {
          L3HKdata[HKs][1] = "ON   ";
     } else {
          L3HKdata[HKs][1] = "OFF  ";
     }
     if (L3Info->cal->GCUPix0_B != BADDATA) {
    	 L3HKdata[HKs][2] = util_doubleToString(L3Info->cal->GCUPix0_B);
    	 setPrecision(L2Info.hiRes,L3HKdata[HKs][2]);
     } else {
    	 L3HKdata[HKs][2] = "N/A";
     }
     L3HKdata[HKs++][3] = "     ";
     
     // Set GCU pix0 Unc B ----------------------------------------
     L3HKdata[HKs][0] = "ROSINA_DFMS_SCI_GCU_PIXEL0_UNC_B";
     if (L3Info->cal->GCUPix0_UncB_used) {
          L3HKdata[HKs][1] = "ON  ";
     } else {
          L3HKdata[HKs][1] = "OFF  ";
     }
     if (L3Info->cal->GCUPix0_UncB != BADDATA) {
    	 L3HKdata[HKs][2] = util_doubleToString(L3Info->cal->GCUPix0_UncB);
    	 setPrecision(L2Info.hiRes,L3HKdata[HKs][2]);
     } else {
    	 L3HKdata[HKs][2] = "N/A";
     }
     L3HKdata[HKs++][3] = "     ";
     
     // Set Self pix0 A ----------------------------------------
     L3HKdata[HKs][0] = "ROSINA_DFMS_SCI_SELF_PIXEL0_A   ";
     if (L3Info->cal->selfPix0_A_used) {
          L3HKdata[HKs][1] = "ON   ";
     } else {
          L3HKdata[HKs][1] = "OFF  ";
     }
     if (L3Info->cal->selfPix0_A != BADDATA) {
    	 L3HKdata[HKs][2] = util_doubleToString(L3Info->cal->selfPix0_A);
    	 setPrecision(L2Info.hiRes,L3HKdata[HKs][2]);
     } else {
    	 L3HKdata[HKs][2] = "N/A";
     }
     L3HKdata[HKs++][3] = "     ";
     
     // Set Self pix0 Unc A  ----------------------------------------
     L3HKdata[HKs][0] = "ROSINA_DFMS_SCI_SELF_PIXEL0_UNCA";
     if (L3Info->cal->selfPix0_UncA_used) {
          L3HKdata[HKs][1] = "ON   ";
     } else {
          L3HKdata[HKs][1] = "OFF  ";
     }
     if (L3Info->cal->selfPix0_UncA != BADDATA) {
    	 L3HKdata[HKs][2] = util_doubleToString(L3Info->cal->selfPix0_UncA);
    	 setPrecision(L2Info.hiRes,L3HKdata[HKs][2]);
     } else {
    	 L3HKdata[HKs][2] = "N/A";
     }
     L3HKdata[HKs++][3] = "     ";
     
     // Set Self pix0 B ----------------------------------------
     L3HKdata[HKs][0] = "ROSINA_DFMS_SCI_SELF_PIXEL0_B   ";
     if (L3Info->cal->selfPix0_B_used) {
          L3HKdata[HKs][1] = "ON   ";
     } else {
          L3HKdata[HKs][1] = "OFF  ";
     }
     if (L3Info->cal->selfPix0_B != BADDATA) {
    	 L3HKdata[HKs][2] = util_doubleToString(L3Info->cal->selfPix0_B);
    	 setPrecision(L2Info.hiRes,L3HKdata[HKs][2]);
     } else {
    	 L3HKdata[HKs][2] = "N/A";
     }
     L3HKdata[HKs++][3] = "     ";

     // Set Self pix0 Unc B ----------------------------------------
     L3HKdata[HKs][0] = "ROSINA_DFMS_SCI_SELF_PIXEL0_UNCB";
     if (L3Info->cal->selfPix0_UncB_used) {
          L3HKdata[HKs][1] = "ON   ";
     } else {
          L3HKdata[HKs][1] = "OFF  ";
     }
     if (L3Info->cal->selfPix0_UncB != BADDATA) {
    	 L3HKdata[HKs][2] = util_doubleToString(L3Info->cal->selfPix0_UncB);
    	 setPrecision(L2Info.hiRes,L3HKdata[HKs][2]);
     } else {
    	 L3HKdata[HKs][2] = "N/A";
     }
     L3HKdata[HKs++][3] = "     ";
     
     // Set PPM dev A ----------------------------------------
     L3HKdata[HKs][0] = "ROSINA_DFMS_SCI_AVG_PPM_DEV_A   ";
     L3HKdata[HKs][1] = "     ";
     if (L3Info->cal->ppmDiffA != BADDATA) {
    	 L3HKdata[HKs][2] = util_doubleToString(L3Info->cal->ppmDiffA);
    	 setPrecision(L2Info.hiRes,L3HKdata[HKs][2]);
     } else {
    	 L3HKdata[HKs][2] = "N/A";
     }
     L3HKdata[HKs++][3] = "ppm  ";
     
     // Set PPM dev B ----------------------------------------
     L3HKdata[HKs][0] = "ROSINA_DFMS_SCI_AVG_PPM_DEV_B   ";
     L3HKdata[HKs][1] = "     ";
     if (L3Info->cal->ppmDiffB != BADDATA) {
    	 L3HKdata[HKs][2] = util_doubleToString(L3Info->cal->ppmDiffB);
    	 setPrecision(L2Info.hiRes,L3HKdata[HKs][2]);
     } else {
    	 L3HKdata[HKs][2] = "N/A";
     }
     L3HKdata[HKs][3] = "ppm  ";

}

//
// ---------------------------- setPrecision -----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method sets the precision for low/high res double precision.
// Low res is set to xxx.y  (1 decimal place)
// High res is set to xxx   (0 decimal places)
// Input
//   res - high = true,  low = false
//   var -  variable to set the precision
//
// returns:
//   None
// =============================================================================
// History: Written by Mike Rinaldi, September 2014
// =============================================================================
//
void DFMS_ProcessMCP_Class::setPrecision(bool hires, string &sHK) {

	string s;
	char tmp[200];
	double dtmp=0.0;

	// check resolution
	if (hires) {
		dtmp = util_stringToDouble(sHK);
		sprintf(tmp,"%.0f",dtmp);   // Low res use yyy.0 resolution
		s.assign(tmp);
		//cout << "s = " << s << endl;
		sHK = s;
	} else {
		dtmp = util_stringToDouble(sHK);
		dtmp = dtmp+0.05;
		sprintf(tmp,"%.1f",dtmp);   // Low res use yyy.x resolution
		s.assign(tmp);
		//cout << "s = " << s << endl;
		sHK = s;
	}

}

//
// ------------------------- addPolyOffCoeffInfo -------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method adds the polynomial coefficients info
//   iRow - Row A/B
//   HKs - Row number in the Housekeeping object
//
// returns:
//   None
// =============================================================================
// History: Written by Mike Rinaldi, August 2014
// =============================================================================
//
void DFMS_ProcessMCP_Class::addPolyOffCoeffInfo(int iRow, int *HKs) {

	double c1,c2,c3;

	int rowNum = *HKs;

	if (iRow == 1) {

		c1 = pC[L2Info.mode].c[0][1];
		c2 = pC[L2Info.mode].c[0][2];   // Row A values
		c3 = pC[L2Info.mode].c[0][3];
		//
		L3HKdata[rowNum][0] = "ROSINA_DFMS_SCI_OFF_COEFF_C1_A  ";
		L3HKdata[rowNum][1] = "     ";
		if (c1 != BADDATA) {
			L3HKdata[rowNum][2] = util_doubleToString(c1);
			util_toUpperCase(L3HKdata[rowNum][2]);
		} else {
			L3HKdata[rowNum][2] = "N/A";
		}
		L3HKdata[rowNum++][3] = "     ";
		//
		L3HKdata[rowNum][0] = "ROSINA_DFMS_SCI_OFF_COEFF_C2_A  ";
		L3HKdata[rowNum][1] = "     ";
		if (c2 != BADDATA) {
			L3HKdata[rowNum][2] = util_doubleToString(c2);
			util_toUpperCase(L3HKdata[rowNum][2]);
		} else {
			L3HKdata[rowNum][2] = "N/A";
		}
		L3HKdata[rowNum++][3] = "     ";
		//
		L3HKdata[rowNum][0] = "ROSINA_DFMS_SCI_OFF_COEFF_C3_A  ";
		L3HKdata[rowNum][1] = "     ";
		if (c3 != BADDATA) {
			L3HKdata[rowNum][2] = util_doubleToString(c3);
			util_toUpperCase(L3HKdata[rowNum][2]);
		} else {
			L3HKdata[rowNum][2] = "N/A";
		}
		L3HKdata[rowNum++][3] = "     ";

	} else {

		c1 = pC[L2Info.mode].c[1][1];
		c2 = pC[L2Info.mode].c[1][2];   // Row B values
		c3 = pC[L2Info.mode].c[1][3];
		//
		L3HKdata[rowNum][0] = "ROSINA_DFMS_SCI_OFF_COEFF_C1_B  ";
		L3HKdata[rowNum][1] = "     ";
		if (c1 != BADDATA) {
			L3HKdata[rowNum][2] = util_doubleToString(c1);
		  	util_toUpperCase(L3HKdata[rowNum][2]);
		} else {
			L3HKdata[rowNum][2] = "N/A";
		}
		L3HKdata[rowNum++][3] = "     ";
		//
		L3HKdata[rowNum][0] = "ROSINA_DFMS_SCI_OFF_COEFF_C2_B  ";
		L3HKdata[rowNum][1] = "     ";
		if (c2 != BADDATA) {
			L3HKdata[rowNum][2] = util_doubleToString(c2);
		  	util_toUpperCase(L3HKdata[rowNum][2]);
		} else {
			L3HKdata[rowNum][2] = "N/A";
		}
		L3HKdata[rowNum++][3] = "     ";
		//
		L3HKdata[rowNum][0] = "ROSINA_DFMS_SCI_OFF_COEFF_C3_B  ";
		L3HKdata[rowNum][1] = "     ";
		if (c3 != BADDATA) {
			L3HKdata[rowNum][2] = util_doubleToString(c3);
		  	util_toUpperCase(L3HKdata[rowNum][2]);
		} else {
			L3HKdata[rowNum][2] = "N/A";
		}
		L3HKdata[rowNum++][3] = "     ";
	}

	*HKs = rowNum;
}

//
// ------------------------------ processL2 ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method does the processing of L2 to L3 via calls to various methods
// inputs: 
//   L2I - L2 File Info Structure
//   modeInfo - DFMS mode Info Structure
//   
// returns:
//   1 if success, 0 otherwise
// =============================================================================
// History: Written by Mike Rinaldi, March 2013
// =============================================================================
//
int DFMS_ProcessMCP_Class::processL2(L2FileInfo &L2I, DFMSModeInfo &modeInfo) {

	string sFunctionName="DFMS_ProcessMCP_Class::processL2";

	int nSuccess=0;
    verifPeakInfo *peak = new verifPeakInfo();
	DFMS_X0FitFile_Class *x0Fit, *x0FitS;
	DFMS_PeakFit_Class *gPeakFit;
	DFMS_FindAllPeaks_Class *fP;
	double pkH = 0.0;
	double pkW = 0.0;
	bool abT = false;
    
    bool rowAFitSuccess = true;
    bool rowBFitSuccess = true;
    
    string x0FileNameA = "";
    string x0FileNameB = "";

	// Initialize verifPeakInfo peak;
	//peak.init();

	// Initialize calInfo
	//L3Info->cal->init();

	// Assume this L2 peak is not a verification peak
	L3Info->usedAsVerifA = false;
	L3Info->usedAsVerifB = false;
     
	// Set the local copy of L2Info using struct overloaded = operator
	L2Info = L2I;

	// Find the massType
	double m0 = L2Info.commandedMass;
	if (L2Info.lowRes) {
		if (L2Info.isGCU) {
			if (m0 < GCULOWCUTOFF) {
				massType = "L";
			} else {
				massType = "H";
			}
		} else {
			if (m0 < SLFLOWCUTOFF) {    // In the event that GCULOWCUTOFF != SLFLOWCUTOFF
				massType = "L";
			}  else {
				massType = "H";
			}
		}
	} else {
		if (L2Info.isGCU) {
			if (m0 < GCULOWCUTOFF) {
				massType = "L";
			} else if (m0 >= GCULOWCUTOFF && m0 < GCUHICUTOFF) {
				massType = "M";
			} else {
				massType = "H";
			}
		} else {
			if (m0 < SLFLOWCUTOFF) {    // In the event that GCULOWCUTOFF != SLFLOWCUTOFF
				massType = "L";
			} else if (m0 >= SLFLOWCUTOFF && m0 < SLFHICUTOFF) {
				massType = "M";
			} else {
				massType = "H";
			}
		}
	}

    // Find the resType  (resolution)
    if (L2Info.lowRes) {
    	resType = "L";
    } else {
    	resType = "H";
    }

	// STEP 12.12.1 - Find all available Pix0 fit files
	nSuccess = DFMS_X0FitFile_Class::getFitFilesInDir(sx0FitsDataPath, fitInfo);
	if (!nSuccess) {
		sErrorMessage = "Unable to get X0 Fit File dir info at: ";
		sErrorMessage += sx0FitsDataPath;
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		cout << "Unable to get X0Fit File directory info at: " << sx0FitsDataPath << endl;
		exit(EXIT_FAILURE);
	} else {
		sInfoMessage = "Successfully found: "+util_intToString(fitInfo.size());
		sInfoMessage += " X0 fit files";
		writeToLog(sInfoMessage, sFunctionName, INFO);
		if (verbose >= 3) cout << sInfoMessage << endl;
	}

	// Get L3 file and use as calibration id
	string file = L3Info->L3FileName;
	util_toUpperCase(file);

	// Get zoom from mode table if mode is not 0 or 9999
    if (L2Info.mode != 0) {
        L2Info.zoom = modeInfo.zoom;
    }

    // For GCU/SLF with HMHR set zoom = 4.0
    if (L2Info.hiRes && (L2Info.commandedMass >= GCUHICUTOFF) ) {
    	L2Info.zoom = 4.0;
    }

	// STEP 12.12.2 - Get MPST data from file
	cout << "Best MPS Table = " << sBestMPST << endl;
	nSuccess = getMPSTdata(sBestMPST, peak);
	if (!nSuccess) {
		sErrorMessage = "Unable to access proper MPS table or Mass for file: ";
		sErrorMessage += sBestMPST+" - Skipping file processing";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		delete MPSTobjIn; MPSTobjIn=0;
		return 0;
	} else {
		if (mptIndx != -1) {
			L3Info->cal->calID4 = "None";
			L3Info->cal->calID5 = "None";
			L3Info->cal->calID6 = MPSTobjIn->getfileName();  //sBestMPST.substr(0,sBestMPST.length());
		}
		//cout << "mptIndx = " << mptIndx << endl;
		if (L2Info.isGCU) {
			// Assume peak is located in between MPS table defined limits
			if (iXl <= 0) iXl = PKLOWPT;
			if (iXr <= 0) iXr = NUMPIX-PKLOWPT;
		} else {
			// Use entire range
			iXl = PKLOWPT;      // Always look near 255 for target peak
			iXr = NUMPIX-PKLOWPT;
		}
	}

	// STEP 12.12.3 - Perform Mass Scale calibration for each of side A and B
	int fitParams = FITPARAMS;
	int usePk = 0;
	for (int iRow=1; iRow<3; ++iRow) {

		pkMap.clear();

		// Initialize the peakInfo structure
		pkFI.init();

		// STEP 12.12.3.2 - Calculate the Offset
		calcOffset(iRow);

		// STEP 12.12.3.3 - Define peak threshold based on meanOffset
		threshold = meanOffset+meanOffsetStdDev*NUMSDEV;
		if (verbose == 10) {
			cout << "The meanOffset = " << meanOffset << endl;
			cout << "The threshold = " << threshold << endl;
			//
		}

		// STEP 12.12.3.4 - Define object to look for all peaks
		if (iRow == 1) {
			fP = new DFMS_FindAllPeaks_Class(YAin, iXl, iXr, meanOffset, meanOffsetStdDev);
		} else {
			fP = new DFMS_FindAllPeaks_Class(YBin, iXl, iXr, meanOffset, meanOffsetStdDev);
		}

		// STEP 12.12.3.5 - Get all peaks above Threshold
		numGoodPks = fP->findPeaksAboveThreshold(threshold, meanOffset);
		if (iRow == 1) numGoodPksA = numGoodPks;
		if (iRow == 2) numGoodPksB = numGoodPks;
		if (numGoodPks <= 0) {
			sErrorMessage = "No useful Peaks Found for Row: "+sRow[iRow-1];
			writeToLog(sErrorMessage, sFunctionName, WARN);
			pkFI.maxId = fP->maxId;
			if (iRow == 1) sideAexceptionType["peakFinderError"]++;
			if (iRow == 2) sideBexceptionType["peakFinderError"]++;
			delete fP; fP=0;
		} else {
			altPkInfo pkm;
			for (int i=0; i<numGoodPks; ++i) {
				pkFI.peakLoc[i] = fP->peakLoc[i];
				pkFI.peakHght[i] = fP->peakVal[i];
				//cout << "m0 = " << m0 << "  -  pkFI.peakLoc[" << i << "]= " << pkFI.peakLoc[i];
				//cout << " height = " << pkFI.peakHght[i] << endl;
				pkFI.peakWidth[i] = fP->peakWd[i];
				pkFI.pkInBounds[i] = fP->inBounds[i];
				pkFI.pkAboveThresh[i] = fP->pkAboveThresh[i];
			}
			pkFI.maxId = fP->maxId;
			pkFI.maxPkLoc = fP->maxPkLoc;
			pkFI.maxPkVal = fP->maxPkVal;

			if (!L2Info.isGCU) {
				// Build alternate peak info structure
				for (int i=0; i<numGoodPks; ++i) {
					pkm.peakLoc = (double)pkFI.peakLoc[i];
					pkm.pkHght = pkFI.peakHght[i];
					pkm.abT = pkFI.pkAboveThresh[i];
					pkm.pkWd = pkFI.peakWidth[i];
					pkm.fracHght = pkFI.peakHght[i]/pkFI.maxPkVal;
					pkMap.insert(pair<double,altPkInfo>(pkFI.peakHght[i],pkm));
				}
				// find alternate peak if SLF
				if (L2Info.isSLF || L2Info.commandedMass == 36 || L2Info.commandedMass == 28) {
					usePk = findAltPeak(&pkH, &pkW, &abT);
					// Unable to find alternate therefore use tallest peak
					if (pkH == 0 && pkW == 0 && abT == false) {
						pkH   = pkFI.maxPkVal;
						pkW   = pkFI.peakWidth[pkFI.maxId];
						abT   = pkFI.pkAboveThresh[pkFI.maxId];
					}
				} else {
					usePk = pkFI.maxPkLoc;
					pkH   = pkFI.maxPkVal;
					pkW   = pkFI.peakWidth[pkFI.maxId];
					abT   = pkFI.pkAboveThresh[pkFI.maxId];
				}
			} else {
				usePk = pkFI.maxPkLoc;
				pkH   = pkFI.maxPkVal;
				pkW   = pkFI.peakWidth[pkFI.maxId];
				abT   = pkFI.pkAboveThresh[pkFI.maxId];
			}

			// Release fP
			delete fP; fP=0;
		}

		// STEP 12.12.3.6 - print the prePeaks values to screen
		if (verbose == 10) pkFI.printPeaks(numGoodPks);
		//cout << "numGoodPks = "  << numGoodPks << endl;
		//pkFI.printPeaks(numGoodPks);

		// STEP 12.12.3.7.1 - Apply Row B Modulation Correction
		bool doIt = true;
		if (doIt && iRow == 2) {
			rowBModCorr(iRow);
		}

		// STEP 12.12.3.7 - Apply the Offset Correction
		nSuccess = applyOffsetSubtraction(iRow);
		if (!nSuccess) {
			sErrorMessage = "ApplyOffsetSubtraction failed";
			writeToLog(sErrorMessage, sFunctionName, ERROR);
            // Release fP
            delete fP; fP=0;
            return 0;
		}

		// STEP 12.12.3.8 - Apply Overall and Pixel Gain correction to offset subtracted data
		applyGainCorrection(iRow);

		// STEP 12.12.3.9 - Convert corrected Data to Ion Rate
		findIonRate(iRow);

		if (iRow == 1) {
			meanOffsetA = meanOffset;
			gPeakFit = new DFMS_PeakFit_Class (iRow, YAin, fitParams, MPSTdata, meanOffset,
       		 	 	 	 	 	 	 	 	 	 threshold, mptRows, iXl, iXr);
		} else {
			meanOffsetB = meanOffset;
			gPeakFit = new DFMS_PeakFit_Class (iRow, YBin, fitParams, MPSTdata, meanOffset,
 	 	 	 	 	 	 	 	 	 	 	 	 threshold, mptRows, iXl, iXr);
		}

		// STEP 12.12.3.10 - Find peak characteristics
		// Initialize the fit coefficients by histo-gramming Ion Rate data
		nSuccess = gPeakFit->initPeakFit(iRow, usePk, pkH, pkW,abT);
		if (!nSuccess) {
			sErrorMessage = "Unable to find initial peak values";
			writeToLog(sErrorMessage, sFunctionName, ERROR);
			delete MPSTobjIn; MPSTobjIn=0;
			delete gPeakFit;  gPeakFit=0;
			return 0;
		} else if (nSuccess == -1) {
			gPeakFit->fitSucceeded = false;
		}

		// STEP 12.12.3.11 - Perform a non-linear curve fit if Data warrants it
		if (gPeakFit->fitSucceeded) {

			nSuccess = gPeakFit->processCurveFit(iRow, 0);
			if (!nSuccess) {
				sErrorMessage = "processCurveFit returned badly: ";
				writeToLog(sErrorMessage, sFunctionName, ERROR);
				if (QLINFOFILE) {
					QuickLookLog << " - processCurvFit returned badly" << dfmsEOL;
					qlCount++;
					if (qlCount % QLPPAGE == 0) qlHeader();
				}
				gPeakFit->fitSucceeded = false;
                if (iRow == 1) rowAFitSuccess = false;   // Used to dump pix0Del below
                if (iRow == 2) rowBFitSuccess = false;   // Used to dump pix0Del below
			} else {

				// STEP 12.12.3.12 - Use Fisher Matrix to calculate coefficient errors.
				nSuccess = gPeakFit->calcParamErrors(iRow);
				if (!nSuccess) {
					sErrorMessage = "calcParamErrors returned badly: ";
					writeToLog(sErrorMessage, sFunctionName, ERROR);
					gPeakFit->fitSucceeded = false;
				}

			}

			// STEP 12.12.3.13 - Save the fits for later usage
			if (!gPeakFit->fitSucceeded) {

				// STEP 12.12.3.14 - Set the base level L3Info params
				// for bad or no fit data
				assignDefaultFitParams(iRow,gPeakFit);

			} else {

				// set Fit results to local variable
				if (iRow == 1) {
					for (int i=0; i<NUMPIX; ++i) {
						yAFit[i] = gPeakFit->yFitA[i];
					}
				} else if (iRow == 2) {
					for (int i=0; i<NUMPIX; ++i) {
						yBFit[i] = gPeakFit->yFitB[i];
					}
				}

				// STEP 12.12.3.15 - Assign L3Info will values of peakHeight, peakCenter,
				// peakCenterUnc, peakWidth, and offset,...
				nSuccess = assignFitParams(gPeakFit, iRow);

			}

		} else {
			// STEP 12.12.3.13(repeat) - Set the base level L3Info params
			// for bad or no fit data
			assignDefaultFitParams(iRow,gPeakFit);
        }

		// STEP 12.12.3.16 - Read GCU X0 fit file object to handle pix0 fit parameters
	   	x0Fit = new DFMS_X0FitFile_Class(sx0FitsDataPath);
	   	nSuccess = x0Fit->readPix0FitFile(iRow, "GCU", resType,  massType, L2Info.dateTime, fitInfo);
	   	if (!nSuccess) {
	   		sErrorMessage="The pix0 File was not read properly";
	   		writeToLog(sErrorMessage, sFunctionName, ERROR);
	   		delete x0Fit; x0Fit=0;
	   		return 0;
	   	} else {
	   		aGCUA = x0Fit->aA;
	   		bGCUA = x0Fit->bA;
	   		aGCUB = x0Fit->aB;
	   		bGCUB = x0Fit->bB;
            // REMOVE LATER
            x0FileNameA = x0Fit->getFileName();
	   	}

		// STEP 12.12.3.17 - Construct the best mass Scale. Calculate Calibration Value pix0.
		// For     GCU modes -    Use calibGCU()
		// For non-GCU modes - A. Use calibSLF() if verif. peaks found.
		//                     B. Use calibNonGCUnonSLF() if no verif peaks
		if (L2Info.isGCU) {

			L3Info->usedAsVerifA = L3Info->usedAsVerifB = true;

			nSuccess = calibGCU(gPeakFit, x0Fit, modeInfo.sMode, iRow);

			if (!nSuccess) {
				sErrorMessage = "calibGCU returned badly: ";
				writeToLog(sErrorMessage, sFunctionName, ERROR);
				delete gPeakFit;  gPeakFit=0;
				delete MPSTobjIn; MPSTobjIn=0;
				delete x0Fit; x0Fit=0;
				return 0;
			}

		} else {

			// STEP 12.12.3.18 - Open the best SLF file and get the offset
		   	x0FitS = new DFMS_X0FitFile_Class(sx0FitsDataPath);
			noSLFFitFile = false;
		   	nSuccess = x0FitS->readPix0FitFile(iRow, "SLF", resType,  massType, L2Info.dateTime, fitInfo);
		   	if (!nSuccess) {
		   		sErrorMessage="The SLF pix0 File was not read properly";
		   		writeToLog(sErrorMessage, sFunctionName, ERROR);
		   		noSLFFitFile = true;
		   	} else {
		   		aSLFA = x0FitS->aA;		// Just need the offset
		   		bSLFA = x0FitS->bA;
		   		aSLFB = x0FitS->aB;		// Just need the offset
		   		bSLFB = x0FitS->bB;
                // REMOVE LATER
                x0FileNameA = x0FitS->getFileName();
		   	}

            // TODO: -  [July 2017 Calculate new pixelshift for m0 69-70 and above]

			// STEP 12.12.3.19 - In nonGCU mode check iF we have an SLF mass.  If so
			// use it to calculate pix0.
		   	// Before proceeding make sure that we have an SLF mass
		    if (!L2Info.isSLF && (slfSet.find(m0) == slfSet.end())) {
		    	nSuccess = 0;
		    } else {
		    	nSuccess = calibSLF(gPeakFit, x0Fit, x0FitS, modeInfo.sMode, iRow);
		    }

			// STEP 12.12.3.20 - If no verification peak is found use Self Calibration
			if (nSuccess == 0) {
				sInfoMessage="No Verification Peak Found - Use Self Calibration";
				writeToLog(sInfoMessage, sFunctionName, INFO);

				nSuccess = calibNonGCUnonSLF(gPeakFit, x0Fit, x0FitS, modeInfo.sMode, iRow);
				if (!nSuccess) {
					sErrorMessage = "calcNonGCUCalibVals returned badly: ";
					writeToLog(sErrorMessage, sFunctionName, ERROR);
					delete gPeakFit;  gPeakFit=0;
					delete MPSTobjIn; MPSTobjIn=0;
					delete x0Fit; x0Fit=0;
					delete x0FitS; x0FitS=0;
					return 0;
				}

			} else if (nSuccess == 1) {

				sInfoMessage = "Verification Peak found and used for Calibration ";
				writeToLog(sInfoMessage, sFunctionName, INFO);

			} else if (nSuccess == -1) {

				sErrorMessage="No Verification Peak Found - Bad Data";
         		writeToLog(sInfoMessage, sFunctionName, INFO);

			}
			delete x0FitS; x0FitS=0;
		}
        
		// STEP 12.12.3.21 - Optionally dump Raw data and fit to file for GNUPlot plotting method from PeakFit
		// TODO - Possibly remove gnuplot functionality
		//if ((iPltFile == 2 || iPltFile == 4) && iRow == 2 && (L3Info->fitType.compare("CurveFit") == 0)) {
		//	nSuccess = quickPlotDataAndFit(iRow, sPlotFilePath, L2Obj->getFileName(),"Raw", L3Info->peak->pkIDA);
		//	if (!nSuccess) {
		//		sErrorMessage = "gnuplot executable not found";
		//		writeToLog(sErrorMessage, sFunctionName, WARN);
		//	}
		//}
        
		// STEP 12.12.3.22 - Release Memory
		delete gPeakFit;  gPeakFit=0;
		delete x0Fit; x0Fit=0;
	}
    
    // Compare pix0 for row A and B. output delta if required
    if (PIX0DELTOFILE) {
        if (rowAFitSuccess && rowBFitSuccess) {
            pix0D << "L2 Filename = " << L2Info.fileName << endl;
            pix0D << "m0 = " << L2Info.commandedMass <<  " - pix0A = " << L3Info->cal->pix0_A << endl;
            pix0D << "m0 = " << L2Info.commandedMass <<  " - pix0B = " << L3Info->cal->pix0_B << endl;
            pix0D << "x0FileName = " << x0FileNameA << endl;
            if (L3Info->cal->pix0_A != 0 && L3Info->cal->pix0_B !=0) {
                double pix0Delta = abs(L3Info->cal->pix0_A - L3Info->cal->pix0_B);
                if (pix0Delta <= PIX0ROWDIFF) {
                    pix0D << " Delta for pix0 = " << pix0Delta << endl;
                } else {
                    pix0D << " Delta for pix0 = " << pix0Delta << " *" << endl;
                }
                if (pix0Delta <= 3) {
                    pix0Spread["<=3"]++;
                } else if (pix0Delta > 3 && pix0Delta <= 6) {
                    pix0Spread["3 < pix0 <= 6"]++;
                } else if (pix0Delta > 6 && pix0Delta <= 10) {
                    pix0Spread["6 < pix0 <= 10"]++;
                } else {
                    pix0Spread["pix0 > 10"]++;
                }
            } else {
                pix0D << "delta not defined" << endl;
            }
            pix0D << endl;
        }
    }

     
	// STEP 12.12.4 - Create the mass scale portion of the L3 Data data array.
	createL3MassScale();

	// STEP 12.12.5 - Add COPS NG/RG data to L3Info
	addCOPSinfo(sBestNGCOPS, sBestRGCOPS, sBestBGCOPS);

	// STEP 12.12.6 - If LR Special Mass and Not GCU write results to time series file
	if (!L2Info.isGCU && TIMESERIESOUT) {
		nSuccess = writeTimeSeriesFile(L2Obj->getFileDate(),modeInfo);
	}
     
	// STEP 12.12.7 - Optionally printout peak info.
	if (verbose >= 0) L3Info->printDataToScreen();
	if (L3INFOTOFILE) L3Info->printDataToFile(dfmsEOL);
     
	// STEP 12.12.8 - Get needed L2 HK data. Copy these to L3HK data
	getL2HKdata(L2Obj->HKdata);
     
	// STEP 12.12.9 - form the L3HKdata
	// Note that the first 12 HK items are already in place from the call
	// to getL2HKdata above
	formL3HKdata();
     
	// STEP 12.12.10 - form the CalData array
	int totalCalRows = formCalData();
     
	// STEP 12.12.11 -  If L2 is GCU and updateMPSTfile is set then create new MPST file.
	// This should not be done by the code.  Only by hand.  MAR 30-09-2014
	// for now leave as is (as if its never true) which allows for future changes
	if (0) {            //(L2Info.isGCU && updateMPSTfile) {
         
		// Create a new MPST file based on this calibration
		string sMode = "M0"+util_intToString(L2Info.mode);
		string MPSTFileName = sMode+"_MPS_TABLE_"+L2Info.dateTime+".TAB";
		nSuccess = createNewMPSTfile(MPSTFileName);
		if (!nSuccess) {
			sErrorMessage = "Unable to create a new MPST file. ";
			writeToLog(sErrorMessage, sFunctionName, ERROR);
		}
	}

	// STEP 12.12.12 - Define the number of L3 HK data rows
	int HKrows = L3HKROWS;

	// STEP 12.12.13 - Now create the L3 file object
	DFMS_PDS_L3_file_Class *L3Obj = new DFMS_PDS_L3_file_Class (sL3Path, L3File,
										HKrows, totalCalRows, L2Obj->PDSheader,
										L3HKdata, calData, L3Data, L2Info);
	// STEP 12.12.14 - Write the L3 file
	nSuccess = L3Obj->createL3File();
    
	// STEP 12.12.15 - Write L2/L3 Data and fit results to file
	if (DIAGTOFILE) {
		nSuccess = writeL2L3DataFit(1, L2Obj->getFileName(), util_doubleToString(L2Info.commandedMass),
																	util_intToString(L2Info.mode));
		if (!nSuccess) {
			sErrorMessage = "Unable to write L2/L3 Data and Fit to file";
			writeToLog(sErrorMessage, sFunctionName, ERROR);
		}
	}

	// STEP 12.12.16 - Optionally dump Raw data and fit to file for GNUPlot plotting method from PeakFit
	// TODO - Possibly remove gnuplot functionality
	//if (iPltFile == 5 || iPltFile == 6) {
	//	nSuccess = L3Obj->quickPlotL3Data(sPlotFilePath, L2Obj->getFileName(),"Raw", L3Info->peak->pkIDA);
	//	if (!nSuccess) {
	//		sErrorMessage = "gnuplot executable not found";
	//		writeToLog(sErrorMessage, sFunctionName, WARN);
	//	}
	//}

	// STEP 12.12.17 - Release Memory
	delete MPSTobjIn; MPSTobjIn=0;
	delete L3Obj;  L3Obj=0;

	return 1;
}

//
// ----------------------- findAltPeak ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method asks if a second peak is needed and if so, available
//
// input/output:
// 		pH  -  peak Height
//      pW  -  peak Width
//      abT -  above Threshold boolean
// returns:
//   if no alternative is needed it returns the tallest peak location
//   if an alternative peak is required it returns it's peak location
// =============================================================================
// History: Written by Mike Rinaldi, August 2014
// =============================================================================
//
int DFMS_ProcessMCP_Class::findAltPeak(double *pH, double *pW, bool *abT) {

	string sFunctionName="DFMS_ProcessMCP_Class::findAltPeak";

	map<double,altPkInfo>::reverse_iterator itr;
	set<double> altpksLoc;
	map<double,double> altpksH;
	map<double,double> altpksW;
	double pL = 0.0;


	for(itr=pkMap.rbegin(); itr!=pkMap.rend(); itr++) {
		double pkh = itr->first;
		int pkl = itr->second.peakLoc;
		double pkf = itr->second.fracHght;
	}


	//cout << "mass = " << L2Info.commandedMass << endl;

	// Get tallest peak first
	int tallPk = pkMap.rbegin()->second.peakLoc;
	if (tallPk < 200 || tallPk > 310) {
		for(itr=pkMap.rbegin(); itr!=pkMap.rend(); itr++) {
			if (itr != pkMap.rbegin()) {
				if (itr->second.peakLoc >= 200 && itr->second.peakLoc <= 310 && itr->second.fracHght >= 0.50) {
					altpksLoc.insert(itr->second.peakLoc);
					altpksH.insert(pair<double,double>(itr->second.peakLoc,itr->second.pkHght));
					altpksW.insert(pair<double,double>(itr->second.peakLoc,itr->second.pkWd));
				}
			}
		}
		if (altpksLoc.size() == 0) {
			sErrorMessage = "Tallest peak is outside the range 200-310 but no ";
			sErrorMessage += "alternate peaks inside this range exist, therefore use talest peak";
			writeToLog(sErrorMessage, sFunctionName, WARN);
			return tallPk;
		} else {
			pL = *(altpksLoc.rbegin());
			*pH = altpksH[pL];
			*pW = altpksW[pL];
			*abT = true;
		}
	} else {
		//cout << "chosen peak = " << tallPk << endl;
		itr = pkMap.rbegin();
		*pH = itr->second.pkHght;
		*pW = itr->second.pkWd;
		*abT = true;
		return tallPk;
	}



	// Second largest peak fitting the 200-310, frac>0.5 criteria
	int ret = *(altpksLoc.rbegin());

	//cout << "chosen peak = " << ret << endl;
	return ret;

}

//
// ----------------------- createNewMPSTfile ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method creates a new Mass Peak Search Table  
//
// input:
// 		newMPSTfile -  The new MTP file name
// returns:
//   1 if success 0 otherwise
// =============================================================================
// History: Written by Mike Rinaldi, March 2013
// =============================================================================
//
int DFMS_ProcessMCP_Class::createNewMPSTfile(string newMPSTfile) {

	string sFunctionName="DFMS_ProcessMCP_Class::createNewMPSTfile";
     
	DFMS_MPST_Class* MPSTobjOut;
     
	// Update new values
	char tmp[150];

	double newWid = 0.50*(L3Info->peak->peakCenterWidthA + L3Info->peak->peakCenterWidthB);
	sprintf(tmp,"%5.2f",newWid);
	MPSTdata[mptIndx][6] = string(tmp);
     
	// Create Output MPST object
	MPSTobjOut = new DFMS_MPST_Class(sMPSTpath, newMPSTfile, mptRows, MPSTobjIn->PDSheader,
								MPSTobjIn->MPSTobj, MPSTdata);
     
	// Call put method to write the new MPST file
	MPSTobjOut->putMPSTfile();
     
	delete MPSTobjOut; MPSTobjOut=0;
     
	return 1;

}

//
// -------------------- applyOffsetSubtraction -----------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This routine subtracts the offset from the input Data.
//
// inputs:
//   int  iRow
//
// returns:
//   None
// =============================================================================
// History: Written by Mike Rinaldi, March 2013
// =============================================================================
//
int DFMS_ProcessMCP_Class::applyOffsetSubtraction(int iRow)
{

	string sFunctionName = "DFMS_ProcessMCP_Class::applyOffsetSubtraction";
    int nSuccess = 0;
    
	DFMS_PolyFit_Class *pFit;
	double c0=0.0, c1=0.0, c2=0.0, c3=0.0;
	double pOffset=0.0;
	double polyOffsetA[NUMPIX],polyOffsetB[NUMPIX];
	double pYA[NUMPIX],pYB[NUMPIX];
    
    // HISTORICAL METHOD: First calculate the offSet at each pixel using polynomial fit.
    // This fit will only be used for the 1-3rd order coeff.  The 0 order coeff is calculated
    // for each L2 file independently (see meanOffset calc). The 1-3rd order coefficients
    // (c1,c2,c3) are only calculated for the first file of the first new mode in the set of
    // L2 files begin processed.  All other files for a given mode are assumed to share the same
    // coefficients, c1,c2,c3.  THIS METHOD was abandoned on Jan 2018.
    
    // NEWER METHOD as of 1-22-2018.  The 3rd order polynomial fit will now be calculated
    // for every spectrum instead of the previous method of once for the first spectrum of each
    // new mode sequence.  This method uses function DFMS_ProcessMCP_Class::prepOffsetDataLR() (only for LR)
    // above to create the offset pixels that are used by the 3rd order fitting routine.
    
    // FOR HR MODES as of 9/3/2018:  Due to the large number of close peaks the NEW METHOD
    // described above does not work well.  So for HR peaks we will use and alternate method.  Here we
    // read input files (currently: DFMS_20140401-20160127_Peak_Exclusion.dat and
    // DFMS_20160127-20161001_Peak_Exclusion.dat) containing pixel boundaries about significan peaks
    // to remove before proceeding with the 3rd order fit. This method uses
    // functionDFMS_ProcessMCP_Class::prepOffsetDataHR() above to create the offset pixels that are
    // used by the 3rd order fitting routine.

	// Prepara the data for 3rd order polynomial fit of offset.  Use two methods.
    // For HR use input peak Exclusion files to remove peaks before fitting
    // For LR use Dmtry method
    if (L2Info.hiRes) {
        nSuccess = prepOffsetDataHR(iRow);
        if (!nSuccess) {
            sErrorMessage = "HR - Polynomial fit to offset not successful, ";
            sErrorMessage += " Row ="+util_intToString(iRow);
            writeToLog(sErrorMessage, sFunctionName, WARN);
            return 0;
        }
    } else {
        nSuccess = prepOffsetDataLR(iRow);
        if (!nSuccess) {
            sErrorMessage = "LR - Polynomial fit to offset not successful, ";
            sErrorMessage += " Row ="+util_intToString(iRow);
            writeToLog(sErrorMessage, sFunctionName, WARN);
            return 0;
        }
    }
    
	//if (distinctModes[L2Info.mode]) {  // Commented out to perform polyfit for each spectrum
    
		string p1 = (L2Info.fileName).substr(0,18);
		string p2 = (L2Info.fileName).substr(21,6);
		string file = p1+p2;
		(pC[L2Info.mode]).fileName = file;
		if (iRow == 1) {
			pFit = new DFMS_PolyFit_Class(numGdAPixels,4,xPolyA,yPolyA);
			bool Res = pFit->polyFit();
			for (int i=0; i<4; i++) pC[L2Info.mode].c[0][i] = pFit->coeff[i];
			delete pFit;
			//for (int i=0; i<NUMPIX; i++) cout << "YAin[" << i << "] = " << YAin[i] << "  -  yPolyA[" << i << "] = " << yPolyA[i] << endl;
		} else {
			pFit = new DFMS_PolyFit_Class(numGdBPixels,4,xPolyB,yPolyB);
			bool Res = pFit->polyFit();
			for (int i=0; i<4; i++) pC[L2Info.mode].c[1][i] = pFit->coeff[i];
            
			//distinctModes[L2Info.mode] = false;  // Commented out to perform polyfit for each spectrum
			
            delete pFit;
			//for (int i=0; i<NUMPIX; i++) cout << "YBin[" << i << "] = " << YBin[i] << "  -  yPolyB[" << i << "] = " << yPolyB[i] << endl;
			//pC[L2Info.mode].printData();
		}
	//}

	// Define the coefficients by Row
	if (iRow == 1) {
		for (int i=0; i<NUMPIX; i++) pYA[i] = YAin[i];
		c3 = pC[L2Info.mode].c[0][3];
		c2 = pC[L2Info.mode].c[0][2];
		c1 = pC[L2Info.mode].c[0][1];
		c0 = pC[L2Info.mode].c[0][0];
        c0OffsetA = c0;
	} else {
		for (int i=0; i<NUMPIX; i++) pYB[i] = YBin[i];
		c3 = pC[L2Info.mode].c[1][3];
		c2 = pC[L2Info.mode].c[1][2];
		c1 = pC[L2Info.mode].c[1][1];
		c0 = pC[L2Info.mode].c[1][0];
        c0OffsetB = c0;
	}
    
    // Set the c0Offset in case needed later.  This replaces later uses of meanOffset if necessary
    c0Offset = c0;

	if (sInstrumentModel.compare("FM") == 0) {

		if (iRow == 1) {
			for (int i=0; i<NUMPIX; i++) {
				double x = (double)(i+1);
                double offSet = c3*pow(x,3) + c2*pow(x,2) + c1*x + c0;
				YAin[i] -= offSet;
				offSetFitA[i] = offSet;
			}
		} else {
			for (int i=0; i<NUMPIX; i++) {
				double x = (double)(i+1);
                double offSet = c3*pow(x,3) + c2*pow(x,2) + c1*x + c0;
				YBin[i] -= offSet;
				offSetFitB[i] = offSet;
			}
		}

	} else if (sInstrumentModel.compare("FS") == 0) {

		if (iRow == 1) {
			for (int i=0; i<NUMPIX; i++) {
				double x = (double)(i+1);
                double offSet = c3*pow(x,3) + c2*pow(x,2) + c1*x + c0;
				YAin[i] -= offSet;
				offSetFitA[i] = offSet;
			}
		} else {
			for (int i=0; i<NUMPIX; i++) {
				double x = (double)(i+1);
                double offSet = c3*pow(x,3) + c2*pow(x,2) + c1*x + c0;
				YBin[i] -= offSet;
				offSetFitB[i] = offSet;
			}
		}

	} else {
		sErrorMessage = "Instrument Model: "+sInstrumentModel+" is not defined";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}

	return 1;

}

//
// -------------------------- prepOffsetDataLR ---------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This routine removes the beginning and ending parts of the data as well as
// all the major peaks.  Remaining Data is considered offset and put into xA,B,
// yA,B arrays used later for 3rd or fitting.
// As of 9/2/2018 this method is only used for LR mode data
//
// inputs:
//   iRow - Row A/B
//   yA -  Array holding Row A data to pass to poly fit class
//   yB -  Array holding Row B data to pass to poly fit class
// returns:
//   1 for success 0 for failure
// =============================================================================
// History: Written by Mike Rinaldi, August 2014
// Modified: As per Dmtry suggestions: Mar 2018
// =============================================================================
//
int DFMS_ProcessMCP_Class::prepOffsetDataLR(int iRow) {

    string sFunctionName = "DFMS_ProcessMCP_Class::prepOffsetDataLR";
    int nSuccess = 0;
    
	map<double,int> majorPeaks;
	vector<int> myPeaks;
	const int pBound = 20;   // Remove +/- pBound around the peak
    
    int cntsOffset[NUMPIX];

	// Find the largest peaks and put them in a hght/loc map
	// The keys of a map are automatically sorted (lowest to highest).
	// Therefore highest hght/loc is last.
	if (numGoodPks > 1) {
		for (int i=0; i<numGoodPks; i++) {
			//if (iRow == 2) cout << pkFI.peakHght[i] << "," << pkFI.peakLoc[i] << endl;
			majorPeaks[pkFI.peakHght[i]] = pkFI.peakLoc[i];
		}
		for (map<double,int>::reverse_iterator rit = majorPeaks.rbegin(); rit != majorPeaks.rend(); rit++) {
			myPeaks.push_back(rit->second);
			// if (iRow == 2) cout << "key = " << rit->first << " - value = " << rit->second << endl;   // %%%%%%%%%%%%%%%%%%%
		}
		// check if single peak
    } else {   //if (numGoodPks == 1) {
		myPeaks.push_back(pkFI.peakLoc[0]);
    }
    
    // Create holders for beginning and ending of peak removal regions
    int *jStrt = new int[myPeaks.size()];
    int *jStop = new int[myPeaks.size()];

	// Now create an array without peaks and edge (+/-20) pixels
	if (iRow == 1) {
        for (int i=0; i<myPeaks.size(); i++) {
            // Make sure I don't attempt to access memory outside of 20->492
            if (numGoodPks <= 0) {
                jStrt[i] = 0;
                jStop[i] = 0;
            } else {
                if (myPeaks[i]-pBound < 20) jStrt[i] = 20; else jStrt[i] = myPeaks[i]-pBound;
                if (myPeaks[i]+pBound > 492) jStop[i] = 492; else jStop[i] = myPeaks[i]+pBound;
            }
        }
        // Sort peaks bounds arrays into ascending order
        util_sortInt(myPeaks.size(),jStrt);
        util_sortInt(myPeaks.size(),jStop);
        //for (int i=0; i<myPeaks.size(); i++) {
        //    cout << "Peak[" << i << "] = " << jStrt[i] << " - " << jStop[i] << endl;
        //}
        int pLast = myPeaks.size()-1;  // last peak number
        int kinc=0;
        // Initially assume all pixels are good offset candidates (set = 1)
        numGdAPixels = NUMPIX;
        for (int j=0; j<NUMPIX; j++) cntsOffset[j] = 1;
        // Now go through the good pixel candidate array and
        // remove pixels from 0-20, pixels +/- 20 bounding a peak, and pixels 492-512. Setting a
        // cntsOffset array element to zero will remove that pixel from the offSet calculation
        // Discount beginning pixels
        for (int j=0; j<20; j++) {
            cntsOffset[j] = 0;   // pixels < 20
            numGdAPixels--;
        }
        // Discount end pixels
        for (int j=492; j<NUMPIX; j++) {
            cntsOffset[j] = 0;   // pixels > 492
            numGdAPixels--;
        }
        // Take care of middle range with 1 or more peaks
        for (int j=20; j<492; j++) {
            if (j >= jStrt[kinc] && j <= jStop[kinc]) {
                cntsOffset[j] = 0;  // pixels within +/- 20 of peak center
                numGdAPixels--;
            }
            if (j > jStop[kinc]) {
                kinc++;
                if (kinc > pLast) break;
            }
        }
            
        xPolyA = new double[numGdAPixels];
        yPolyA = new double[numGdAPixels];
        //cout << "numGdAPixels = " << numGdAPixels << endl;
        int jinc = 0;
        //for (int j=0; j<NUMPIX; j++) cout << cntsOffset[j];
        //cout << endl;
        for (int j=0; j<NUMPIX; j++) {
            if (cntsOffset[j] == 1) {
                xPolyA[jinc] = (double)j;
                yPolyA[jinc] = YAin[j];
                //cout << xPolyA[jinc] << " - " << yPolyA[jinc] << endl;
                jinc++;
                if (jinc > numGdAPixels-1) break;
            }
        }
        if (jinc != numGdAPixels) {
            sErrorMessage = "A-Houston we have a problem!!";
            writeToLog(sErrorMessage, sFunctionName, WARN);
            cout << "A-Houston we have a problem!!" << endl;
            cout << "jinc = " << jinc << " != numGdAPixels = " << numGdAPixels << endl;
            exit(EXIT_FAILURE);
        }
        statsInfo offInfo;
        util_stats(yPolyA, numGdAPixels, offInfo);
        //cout << "side A" << endl;
        //offInfo.print();
        if (toDebug.is_open()) {
            toDebug << "side A ----------- " << endl;
            offInfo.debug();
            toDebug << endl;
        }

	} else {
        for (int i=0; i<myPeaks.size(); i++) {
            // Make sure I don't attempt to access memory outside of 20->492
            if (numGoodPks <= 0) {
                jStrt[i] = 0;
                jStop[i] = 0;
            } else {
                if (myPeaks[i]-pBound < 20) jStrt[i] = 20; else jStrt[i] = myPeaks[i]-pBound;
                if (myPeaks[i]+pBound > 492) jStop[i] = 492; else jStop[i] = myPeaks[i]+pBound;
            }
        }
        // Sort peaks bounds arrays into ascending order
        util_sortInt(myPeaks.size(),jStrt);
        util_sortInt(myPeaks.size(),jStop);
        //for (int i=0; i<myPeaks.size(); i++) {
        //    cout << "Peak[" << i << "] = " << jStrt[i] << " - " << jStop[i] << endl;
        //}
        int pLast = myPeaks.size()-1;  // last peak number
        int kinc=0;
        // Initially assume all pixels are good offset candidates (set = 1)
        numGdBPixels = NUMPIX;
        for (int j=0; j<NUMPIX; j++) cntsOffset[j] = 1;
        // Now go through the good pixel candidate array and
        // remove pixels from 0-20, pixels +/- 20 bounding a peak, and pixels 492-512. Setting
        // cntsOffset array element to zero will remove that pixel from the offSet calculation
        // Discount beginning pixels
        for (int j=0; j<20; j++) {
            cntsOffset[j] = 0;   // pixels < 20
            numGdBPixels--;
        }
        // Discount end pixels
        for (int j=492; j<NUMPIX; j++) {
            cntsOffset[j] = 0;   // pixels > 492
            numGdBPixels--;
        }
        // Take care of middle range with 1 or more peaks
        for (int j=20; j<492; j++) {
            if (j >= jStrt[kinc] && j <= jStop[kinc]) {
                cntsOffset[j] = 0;  // pixels within +/- 20 of peak center
                numGdBPixels--;
            }
            if (j > jStop[kinc]) {
                kinc++;
                if (kinc > pLast) break;
            }
        }
            
        //for (int j=0; j<NUMPIX; j++) cout << cntsOffset[j];
        //cout << endl;
        xPolyB = new double[numGdBPixels];
        yPolyB = new double[numGdBPixels];
        //cout << "numGdBPixels = " << numGdBPixels << endl;
        int jinc = 0;
        for (int j=0; j<NUMPIX; j++) {
            if (cntsOffset[j] == 1) {
                xPolyB[jinc] = (double)j;
                yPolyB[jinc] = YBin[j];
                //cout << xPolyB[jinc] << " - " << yPolyB[jinc] << endl;
                jinc++;
                if (jinc > numGdBPixels-1) break;
            }
        }
        if (jinc != numGdBPixels) {
            sErrorMessage = "A-Houston we have a problem!!";
            writeToLog(sErrorMessage, sFunctionName, WARN);
            cout << "A-Houston we have a problem!!" << endl;
            cout << "jinc = " << jinc << " != numGdAPixels = " << numGdAPixels << endl;
            exit(EXIT_FAILURE);
        }
        statsInfo offInfo;
        util_stats(yPolyB, numGdBPixels, offInfo);
        //cout << "side B" << endl;
        //offInfo.print();
        if (toDebug.is_open()) {
            toDebug << "side B" << endl;
            offInfo.debug();
            toDebug << endl;
        }
        
	}

	myPeaks.clear();
	majorPeaks.clear();
    delete jStrt;
    delete jStop;

	return 1;

}

//
// ----------------------------- prepOffsetDataHR ---------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This routine is applied only to HR mode data.  It uses the peak boundaries
// defined by the input files (currently: DFMS_20140401-20160127_Peak_Exclusion.dat and
// DFMS_20160127-20161001_Peak_Exclusion.dat) to remove peaks from input data.
// The remaining data is put into arrays xA,B and yA,B for later 3rd order poly
// fitting.
//
// inputs:
//   iRow - Row A/B
//   yA -  Array holding Row A data to pass to poly fit class
//   yB -  Array holding Row B data to pass to poly fit class
// returns:
//   1 for success 0 for failure
// =============================================================================
// History: Written by Mike Rinaldi, September 2018
// =============================================================================
//
int DFMS_ProcessMCP_Class::prepOffsetDataHR(int iRow) {
    
    string sFunctionName = "DFMS_ProcessMCP_Class::prepOffsetDataHR";
    int nSuccess = 0;
    
    vector<pair<int,int> > myPeaks;
    int pksToRmExisting = 0;
    int cntsOffset[NUMPIX];
    
    // Get the Julian Sec since 1970 for time cmparisons
    double peakExclFile1Date = rosToSec(sPeakExclusionFile1.substr(5,8), 2);
    double peakExclFile2Date = rosToSec(sPeakExclusionFile2.substr(5,8), 2);
    
    // Get the current commanded mass
    int m0 = (int)(L2Info.commandedMass +0.5);  // Round up for non integer masses
    
    // Get the associated peak(s) boundaries
    vector<pair<int,int> > peaksToRemove;
    if (L2Info.secSince1970 < peakExclFile2Date) {
        peaksToRemove = peaksToExclude1[m0];
    } else {
        peaksToRemove = peaksToExclude2[m0];
    }
    
    // Count number of peaks
    pair<int,int> p;
    for (int i=0; i<peaksToRemove.size(); i++) {
        p = peaksToRemove[i];
        if (p.first != 0 && p.second != 0) {
            pksToRmExisting++;
            myPeaks.push_back(p);
        }
    }
    
    // Create holders for beginning and ending of peak removal regions
    int *jStrt = new int[myPeaks.size()];
    int *jStop = new int[myPeaks.size()];
    
    // Now create an array without peaks and edge (+/-20) pixels
    if (iRow == 1) {
        for (int i=0; i<myPeaks.size(); i++) {
            // Make sure I don't attempt to access memory outside of 20->492
            if (pksToRmExisting <= 0) {
                jStrt[i] = 0;
                jStop[i] = 0;
            } else {
                if (myPeaks[i].first < 20) jStrt[i] = 20; else jStrt[i] = myPeaks[i].first;
                if (myPeaks[i].second > 492) jStop[i] = 492; else jStop[i] = myPeaks[i].second;
            }
        }
        int pLast = myPeaks.size()-1;  // last peak number
        int kinc=0;
        
        // Initially assume all pixels are good offset candidates (set = 1)
        numGdAPixels = NUMPIX;
        for (int j=0; j<NUMPIX; j++) cntsOffset[j] = 1;
        
        // Now go through the good pixel candidate array and
        // remove pixels from 0-20, pixels +/- 20 bounding a peak, and pixels 492-512. Setting a
        // cntsOffset array element to zero will remove that pixel from the offSet calculation
        
        // Discount beginning pixels
        for (int j=0; j<20; j++) {
            cntsOffset[j] = 0;   // pixels < 20
            numGdAPixels--;
        }
        // Discount end pixels
        for (int j=492; j<NUMPIX; j++) {
            cntsOffset[j] = 0;   // pixels > 492
            numGdAPixels--;
        }
        // Take care of middle range with 1 or more peaks
        for (int j=20; j<492; j++) {
            if (j >= jStrt[kinc] && j <= jStop[kinc]) {
                cntsOffset[j] = 0;  // pixels within +/- 20 of peak center
                numGdAPixels--;
            }
            if (j > jStop[kinc]) {
                kinc++;
                if (kinc > pLast) break;
            }
        }
        
        xPolyA = new double[numGdAPixels];
        yPolyA = new double[numGdAPixels];
        //cout << "numGdAPixels = " << numGdAPixels << endl;
        int jinc = 0;
        //for (int j=0; j<NUMPIX; j++) cout << cntsOffset[j];
        //cout << endl;
        for (int j=0; j<NUMPIX; j++) {
            if (cntsOffset[j] == 1) {
                xPolyA[jinc] = (double)j;
                yPolyA[jinc] = YAin[j];
                //cout << xPolyA[jinc] << " - " << yPolyA[jinc] << endl;
                jinc++;
                if (jinc > numGdAPixels-1) break;
            }
        }
        if (jinc != numGdAPixels) {
            sErrorMessage = "A-Houston we have a problem!!";
            writeToLog(sErrorMessage, sFunctionName, WARN);
            cout << "A-Houston we have a problem!!" << endl;
            cout << "jinc = " << jinc << " != numGdAPixels = " << numGdAPixels << endl;
            exit(EXIT_FAILURE);
        }
        statsInfo offInfo;
        util_stats(yPolyA, numGdAPixels, offInfo);
        //cout << "side A" << endl;
        //offInfo.print();
        if (toDebug.is_open()) {
            toDebug << "side A ----------- " << endl;
            offInfo.debug();
            toDebug << endl;
        }
        
    } else {
        for (int i=0; i<myPeaks.size(); i++) {
            // Make sure I don't attempt to access memory outside of 20->492
            if (pksToRmExisting <= 0) {
                jStrt[i] = 0;
                jStop[i] = 0;
            } else {
                if (myPeaks[i].first < 20) jStrt[i] = 20; else jStrt[i] = myPeaks[i].first;
                if (myPeaks[i].second > 492) jStop[i] = 492; else jStop[i] = myPeaks[i].second;
            }
        }
        int pLast = myPeaks.size()-1;  // last peak number
        int kinc=0;
        
        // Initially assume all pixels are good offset candidates (set = 1)
        numGdBPixels = NUMPIX;
        for (int j=0; j<NUMPIX; j++) cntsOffset[j] = 1;
        
        // Now go through the good pixel candidate array and
        // remove pixels from 0-20, pixels +/- 20 bounding a peak, and pixels 492-512. Setting
        // cntsOffset array element to zero will remove that pixel from the offSet calculation
        // Discount beginning pixels
        for (int j=0; j<20; j++) {
            cntsOffset[j] = 0;   // pixels < 20
            numGdBPixels--;
        }
        // Discount end pixels
        for (int j=492; j<NUMPIX; j++) {
            cntsOffset[j] = 0;   // pixels > 492
            numGdBPixels--;
        }
        // Take care of middle range with 1 or more peaks
        for (int j=20; j<492; j++) {
            if (j >= jStrt[kinc] && j <= jStop[kinc]) {
                cntsOffset[j] = 0;  // pixels within +/- 20 of peak center
                numGdBPixels--;
            }
            if (j > jStop[kinc]) {
                kinc++;
                if (kinc > pLast) break;
            }
        }
        
        //for (int j=0; j<NUMPIX; j++) cout << cntsOffset[j];
        //cout << endl;
        xPolyB = new double[numGdBPixels];
        yPolyB = new double[numGdBPixels];
        //cout << "numGdBPixels = " << numGdBPixels << endl;
        int jinc = 0;
        for (int j=0; j<NUMPIX; j++) {
            if (cntsOffset[j] == 1) {
                xPolyB[jinc] = (double)j;
                yPolyB[jinc] = YBin[j];
                //cout << xPolyB[jinc] << " - " << yPolyB[jinc] << endl;
                jinc++;
                if (jinc > numGdBPixels-1) break;
            }
        }
        if (jinc != numGdBPixels) {
            sErrorMessage = "A-Houston we have a problem!!";
            writeToLog(sErrorMessage, sFunctionName, WARN);
            cout << "A-Houston we have a problem!!" << endl;
            cout << "jinc = " << jinc << " != numGdAPixels = " << numGdAPixels << endl;
            exit(EXIT_FAILURE);
        }
        statsInfo offInfo;
        util_stats(yPolyB, numGdBPixels, offInfo);
        //cout << "side B" << endl;
        //offInfo.print();
        if (toDebug.is_open()) {
            toDebug << "side B" << endl;
            offInfo.debug();
            toDebug << endl;
        }
        
    }
    
    myPeaks.clear();
    delete jStrt;
    delete jStop;
    
    return 1;
    
}

//
// ---------------------- applyGainCorrection ----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This routine applies the overall Gain correction the input Data.  This includes
// the overall gain and the individual pixel Gain correction.
//
// inputs:
//   None
// returns:
//   None
// =============================================================================
// History: Written by Mike Rinaldi, March 2013
// =============================================================================
//
void DFMS_ProcessMCP_Class::applyGainCorrection(int iRow)
{
    
	string sFunctionName = "DFMS_ProcessMCP_Class::::applyGainCorrection";
    
	// Apply the overall gain and pixel gain correction
	// factor on side A
	if (iRow == 1) {
		for (int i=0; i<NUMPIX; ++i) {
			if (pixG[i][3] > 0) {
				double yy = YAin[i];
				YAin[i] /= (gainA*pixG[i][3]);
			} else {
				YAin[i] = 0.0;
			}
		}
	} else {
		// Apply the overall gain and pixel gain correction
		// factor on side B
		for (int i=0; i<NUMPIX; ++i) {
			if (pixG[i][4] > 0) {
				double yy = YBin[i];
				YBin[i] /= (gainB*pixG[i][4]);
			} else {
				YBin[i] = 0.0;
			}
		}
	}

}

//
// ----------------------------- formCalData ---------------------------------
//
//
//  ---------------- Assign the Calibration Data used --------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method assigns the appropriate Cal Data to the CalData array for
// inclusion into the new L3 File
//
// inputs:
//   none
//
// returns:
//   None
// =============================================================================
// History:  Written by Mike Rinaldi, August 2014
// =============================================================================
//
int DFMS_ProcessMCP_Class::formCalData() {
    
	string sFunctionName = "DFMS_ProcessMCP_Class::formCalData";
            
	//cout << "L3 file: " << L3Info->L3FileName << endl;

	// set CAL MASS TABLE values for side A
	for (int i=0; i<L3Info->numCalRows; ++i) {

		calData.push_back(sslice());   // Allocate an empty row

		// Now allocate and assign a value to the individual columns of the new row
		calData[i].push_back(L3Info->vPeaks[i].peakNameA);  // Peak name
		calData[i].push_back(util_intToString(L3Info->vPeaks[i].caltype));   // cal type
		calData[i].push_back(util_intToString(L3Info->vPeaks[i].foundA));   // spectrum peak found
		if (L3Info->vPeaks[i].peakCntrA == 0) {
			calData[i].push_back(util_doubleToString(L3Info->vPeaks[i].peakCntrA));  // Peak Center
		} else {
			string sp = util_doubleToString(L3Info->vPeaks[i].peakCntrA);
			setPrecision(L2Info.hiRes, sp);
			calData[i].push_back(sp);
		}
		if (L3Info->vPeaks[i].peakWidthA == 0) {
			calData[i].push_back(util_doubleToString(L3Info->vPeaks[i].peakWidthA));  // Peak Width
		} else {
			string sp = util_doubleToString(L3Info->vPeaks[i].peakWidthA);
			setPrecision(L2Info.hiRes, sp);
			calData[i].push_back(sp);
		}
		if (L3Info->vPeaks[i].peakHeightA == 0) {
			calData[i].push_back(util_doubleToString(L3Info->vPeaks[i].peakHeightA));  // Peak Height
		} else {
			string sp = util_doubleToString(L3Info->vPeaks[i].peakHeightA);
			setPrecision(L2Info.hiRes, sp);
			calData[i].push_back(sp);
		}
		if (L3Info->vPeaks[i].ppmDiffA == 0) {
			calData[i].push_back(util_doubleToString(L3Info->vPeaks[i].ppmDiffA));  // Peak PPMDiff
		} else {
			string sp = util_doubleToString(L3Info->vPeaks[i].ppmDiffA);
			setPrecision(L2Info.hiRes, sp);
			calData[i].push_back(sp);
		}
	}

	// Next, set CAL MASS TABLE values for side B
	int k = L3Info->numCalRows;
	for (int i=0; i<L3Info->numCalRows; ++i) {

		calData.push_back(sslice());   // Allocate an empty row

		// Now allocate and assign a value to the individual columns of the new row
		calData[k].push_back(L3Info->vPeaks[i].peakNameB);  // Peak name
		calData[k].push_back(util_intToString(L3Info->vPeaks[i].caltype));   // cal type
		calData[k].push_back(util_intToString(L3Info->vPeaks[i].foundB));   // spectrum peak found
		if (L3Info->vPeaks[i].peakCntrB == 0) {
			calData[k].push_back(util_doubleToString(L3Info->vPeaks[i].peakCntrB));  // Peak Center
		} else {
			string sp = util_doubleToString(L3Info->vPeaks[i].peakCntrB);
			setPrecision(L2Info.hiRes, sp);
			calData[k].push_back(sp);
		}
		if (L3Info->vPeaks[i].peakWidthB == 0) {
			calData[k].push_back(util_doubleToString(L3Info->vPeaks[i].peakWidthB));  // Peak Center
		} else {
			string sp = util_doubleToString(L3Info->vPeaks[i].peakWidthB);
			setPrecision(L2Info.hiRes, sp);
			calData[k].push_back(sp);
		}
		if (L3Info->vPeaks[i].peakHeightB == 0) {
			calData[k].push_back(util_doubleToString(L3Info->vPeaks[i].peakHeightB));  // Peak Center
		} else {
			string sp = util_doubleToString(L3Info->vPeaks[i].peakHeightB);
			setPrecision(L2Info.hiRes, sp);
			calData[k].push_back(sp);
		}
		if (L3Info->vPeaks[i].ppmDiffB == 0) {
			calData[k].push_back(util_doubleToString(L3Info->vPeaks[i].ppmDiffB));  // Peak Center
		} else {
			string sp = util_doubleToString(L3Info->vPeaks[i].ppmDiffB);
			setPrecision(L2Info.hiRes, sp);
			calData[k].push_back(sp);
		}
		k++;
	}

	return 2*L3Info->numCalRows;      // Return the 2x the total number of rows.  for A & B DFMS sides

}

//
// ------------------------------ findIonRate ----------------------------------
//
//
//  ------------------ Calculate the Ion Rate given counts ---------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method calculates the ion current for each side on a per pixel basis.
//
// inputs:
//   int iRow  -  Row A or B
//
// returns:
//   None
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
void DFMS_ProcessMCP_Class::findIonRate(int iRow) {

	string sFunctionName = "DFMS_ProcessMCP_Class::findIonRate";

	double ymax = -1.0e99;

	// Integration time
	double intgT = L2Info.intgTime;   // in seconds
	double ys = 1.0;

	// 6.1 - Calculate Dimensional current conversion constant
	currConvConstant = CONVADC*CLEDA/(Q*ys);      		// # of Ions

	// 6.2 - Perform Ion Rate calculation
	// Row A
	if (iRow == 1) {
		for (int i=0; i<NUMPIX; ++i) {

			YAin[i] *= currConvConstant; // Convert corr. cnts. to ion current
			double yCorr = yieldCorr();  // Correct for ion current yield.
			YAin[i] *= yCorr;

			L3Data[i][2] = YAin[i];		// Set the L3 output current value

			// Find the pixel with the maximum ion rate
			if (i >= iXl && i <= iXr) {
				if (YAin[i] > ymax) {
					ymax = YAin[i];
					l3PkMaxIdA = i;    // index of maximum Ion rate
				}
			}
			//cout << "   YAin[" << i << "] = " << YAin[i] << endl;
		}
		gainCorrIonConv = currConvConstant/(gainA*avgPgA);
	} else { 	// Row B
		for (int i=0; i<NUMPIX; ++i) {

			YBin[i] *= currConvConstant; // Convert corr. cnts. to ion current
			double yCorr = yieldCorr();  // Correct for ion current yield.
			YBin[i] *= yCorr;

			L3Data[i][4] = YBin[i];		// Set the L3 output current value

			// Find the pixel with the maximum ion rate
			if (i >= iXl && i <= iXr) {
				if (YBin[i] > ymax) {
					ymax = YBin[i];
					l3PkMaxIdB = i;    // index of maximum Ion rate
				}
			}
			//cout << "   YBin[" << i << "] = " << YBin[i] << endl;
		}
		gainCorrIonConv = currConvConstant/(gainB*avgPgB);
	}

}

//
// ---------------------------- yieldCorr -----------------------------------
//
//
//  -------------- Calculate the current yield correction -----------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method calculates the correction to the current yield.  The formulae used
// were developed by K. Altwegg, et. al.  The exact form implemented were from
// M. Hassig via K. Altwegg 2/15-17/2016.
//
// inputs:
//  None
//
// returns:
//   the yield correction
// =============================================================================
// History: Written by Mike Rinaldi, February 2016
// =============================================================================
//
double DFMS_ProcessMCP_Class::yieldCorr() {

	string sFunctionName = "DFMS_ProcessMCP_Class::yieldCorr";

	double yield = 0.0;
	double m = L2Info.commandedMass;

	if (m < 70) { //-------------------------- Low Mass
		double O_4 = 4.4892e-7*pow(m,4);
		double O_3 = 8.8158e-5*pow(m,3);
		double O_2 = 6.4995e-3*pow(m,2);
		double O_1 = 0.2223*m;
		double O_0 = 3.4922;
		yield=1.0/(O_4 - O_3 + O_2 - O_1 + O_0);
		//cout << "For low mass: yield = " << yield << endl;
	} else {      //-------------------------- High Mass
		if (L2Info.lowRes) {           // Low Res
			double O_1 = -2.400438e-3*m;
			double O_0 = 0.5684252;
			yield=1.0/(O_1 + O_0) + 0.8;
			//cout << "For HM/LR: yield = " << yield << endl;
		} else if (L2Info.hiRes) {    // High res
			double O_1 = -2.400438e-3*m;
			double O_0 = 0.5684252;
			yield=1.0/(O_1 + O_0);
			//cout << "For HM/HR: yield = " << yield << endl;
		} else {                     // Res not defined
			sErrorMessage = "Resolution NOT defined!! ";
			sErrorMessage += "yield correction NOT applied\n";
			writeToLog(sInfoMessage, sFunctionName, INFO);
			yield = 1.0;
		}
	}

	return yield;

}
//
// ---------------------------- createL3MassScale -----------------------------------
//
//
//  ----------------- Create the L3 Data array values --------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method creates the L3 Data mass scale for inclusion into the L3 File
//
// inputs:
//  None
//
// returns:
//   None
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
void DFMS_ProcessMCP_Class::createL3MassScale() {
    
	string sFunctionName = "DFMS_ProcessMCP_Class::createL3MassScale";

	double mA, mB;

	int pkLoc=0;

	peakAreaA = 0.0;
	peakAreaB = 0.0;
	for (int i=0; i<NUMPIX; ++i) {
        
		// side A values
		double ipix = (double)(i+1);
		L3Data[i][0] = ipix;
		if (L2Info.isGCU) {
			mA = pixToMass(L2Info.zoom, L3Info->cal->GCUPix0_A, ipix, L3Info->cal->m0A);
		} else {
			// Use user supplied pix0
			if (useExternalPix0) {
				mA = pixToMass(L2Info.zoom, ePix0A, ipix, L3Info->cal->m0A);

			// Normal nonGCU case.
			} else if (L3Info->cal->pix0_A != 0 || L3Info->cal->pix0_A !=  BADDATA) {
				mA = pixToMass(L2Info.zoom, L3Info->cal->pix0_A, ipix, L3Info->cal->m0A);

			// Normal nonGCU case with Verification peak found, pix0 set in setNonGCUVerifL3Info
			//} else if (L3Info->cal->pix0_A != 0 && L3Info->usedAsVerifA) {
				//mA = pixToMass(L2Info.zoom, L3Info->cal->pix0_A, ipix, L3Info->cal->m0A);

			// Bad fit or Bad data use GCU pix0
			} else {
				mA = pixToMass(L2Info.zoom, L3Info->cal->GCUPix0_A, ipix, L3Info->cal->m0A);
			}
		}
		fMassScaleA[i] = mA;		// Save the final mass scale
		L3Data[i][1] = mA;     		// Assign DFMS row A mass for this pixel
		peakAreaA += yAFit[i];  	// Area of fitted peak - Row A

		// side B values
		if (L2Info.isGCU) {
			mB = pixToMass(L2Info.zoom, L3Info->cal->GCUPix0_B, ipix, L3Info->cal->m0B);
		} else {

			// Use user supplied pix0
			if (useExternalPix0) {
				mB = pixToMass(L2Info.zoom, ePix0B, ipix, L3Info->cal->m0B);

			// Normal nonGCU case
			} else if (L3Info->cal->pix0_B != 0 && L3Info->cal->pix0_B != BADDATA) {
				mB = pixToMass(L2Info.zoom, L3Info->cal->pix0_B, ipix, L3Info->cal->m0B);

			// Normal nonGCU case with Verification peak found, pix0 set in setNonGCUVerifL3Info
			//} else if (L3Info->cal->pix0_B != 0 && L3Info->usedAsVerifB) {
			//	mB = pixToMass(L2Info.zoom, L3Info->cal->pix0_B, ipix, L3Info->cal->m0B);

			// Bad fit or Bad data use GCU pix0
			} else {
				mB = pixToMass(L2Info.zoom, L3Info->cal->GCUPix0_B, ipix, L3Info->cal->m0B);
			}
		}
		//cout << i+1 << "  mB = " << mB << endl;
		fMassScaleB[i] = mB;	 // Save the final mass scale
		L3Data[i][3] = mB;     	 // Assign DFMS row B mass for this pixel
		peakAreaB += yBFit[i];   // Area of fitted peak - Row B
     }

	// 6.2.1 - Get the ion rate at the peak for Row A
	pkLoc = (int)(L3Info->peak->peakCenterA);
	if (pkLoc != BADDATA) {
		L3Info->pkHeightIonRateA = L3Info->peak->peakHeightA;
		L3Info->actualPkHghtA = L3Data[l3PkMaxIdA][2];
        L3Info->peakAreaA = peakAreaA;
	} else {
		L3Info->pkHeightIonRateA = BADDATA;   // Peak center not defined
        L3Info->peakAreaA = BADDATA;
        L3Info->actualPkHghtA = BADDATA;
	}
	// 6.2.2 - Calculate the Signal Cal value for L3 HK - Row A
	L3Info->cal->calValueA = currConvConstant;
	// 6.2.3 - This value varies with FM/FS and is set in the
	//         DFMS_FM_Constants.dat file
	L3Info->cal->calValueStdevA = STDCALDEV;

	pkLoc = (int)(L3Info->peak->peakCenterB);
	if (pkLoc != BADDATA) {
		L3Info->pkHeightIonRateB = L3Info->peak->peakHeightB;
		L3Info->actualPkHghtB = L3Data[l3PkMaxIdB][4];
        L3Info->peakAreaB = peakAreaB;
	} else {
		L3Info->pkHeightIonRateB = BADDATA;   // Peak center not defined
        L3Info->peakAreaB = BADDATA;
        L3Info->actualPkHghtB = BADDATA;
	}
	L3Info->cal->calValueB = currConvConstant;
	// This value varies with FM/FS and is set in the DFMS_FM_Constants.dat file
	L3Info->cal->calValueStdevB = STDCALDEV;

}

//
// ------------------------------ spMassInRange ----------------------------------
//
//
// =============================================================================
// Routine Description
// =============================================================================
// Checks if the mass scale for this file may contain one of the mass peaks in
// the speciall masses array
//
// inputs:
//
// returns:
//   index to spMasses array of special mass found
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
vector<double> DFMS_ProcessMCP_Class::spMassInRange() {

	// Create the mass scale vector from the current mass scale.
	// Note that the args to the vMassScale constructor below are
	// the pointer to the first element in fMassScale and the pointer
	// to the last element, fMassScaleA+NUMPIX
	vector<double> vMassScale(fMassScaleA,fMassScaleA+NUMPIX);
	vector<double> m;
	double mMin = *min_element(vMassScale.begin(),vMassScale.end());
	double mMax = *max_element(vMassScale.begin(),vMassScale.end());
	//cout << "MassScale Range = " << mMin << "  --> " << mMax <<endl;
	for (int j=0; j<NUMSPECIALMASSES; j++) {
		if (spMasses[j] > mMin && spMasses[j] <mMax) {
			m.push_back(spMasses[j]);
		}
	}

	return m;

}

//
// ------------------------------ addCOPSinfo ----------------------------------
//
//
//  ------------------- put COPS Ng/RG Data into L3Info ------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method Gets the COPS Ng and RG data and puts the relevant info into the
// L3Info structure
//
// inputs:
//  sBestNGCOPS - the COPS NG file closest in time to the L2 file
//  sBestRGCOPS - the COPS RG file closest in time to the L2 file
//
// returns:
//   None
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
void DFMS_ProcessMCP_Class::addCOPSinfo(string sBestNGCOPS, string sBestRGCOPS,
																string sBestBGCOPS) {
    
	string sFunctionName = "DFMS_ProcessMCP_Class::addCOPSinfo";

	int nSuccess;

	// Attempt to get COPS BG data

	if (sBestBGCOPS.compare("N/A") != 0) {

		// Now do the BG COPS Data
		COPSobjIn = new DFMS_COPS_Class(sCOPSpath, sBestBGCOPS, "bg");

		// Get the COPS BG data file
		nSuccess = COPSobjIn->getCOPSdata();
		if (!nSuccess) {
			sErrorMessage = "Unable to read the COPS BG file: ";
			sErrorMessage += COPSobjIn->getFileName()+"\n";
			writeToLog(sErrorMessage, sFunctionName, ERROR);
			L3Info->copsPressNG = BADDATA;
			L3Info->copsPressRG = BADDATA;
		} else {
			// put relevant info into L3Info structure
			L3Info->copsPressNG = atof((COPSobjIn->COPSHKdata[31][2]).c_str());
			L3Info->copsPressRG = atof((COPSobjIn->COPSHKdata[32][2]).c_str());
		}
		if (verbose >= 2) COPSobjIn->printCOPSdata("All");

		delete COPSobjIn;

	} else {  // No COPS NG or RG Data

		if (sBestNGCOPS.compare("N/A") == 0) {
			L3Info->copsPressNG = BADDATA;
		}
		if (sBestRGCOPS.compare("N/A") == 0) {
			L3Info->copsPressRG = BADDATA;
		}
	}

	// Attempt to get COPS NG data

	if (sBestNGCOPS.compare("N/A") != 0) {
     
		// Open COPS data using DFMS_COPS_Class...
		COPSobjIn = new DFMS_COPS_Class(sCOPSpath, sBestNGCOPS, "ng");

		// Get the COPS NG data file
		nSuccess = COPSobjIn->getCOPSdata();
		if (!nSuccess) {
			sErrorMessage = "Unable to read the COPS NG file: ";
			sErrorMessage += COPSobjIn->getFileName()+"\n";
			writeToLog(sErrorMessage, sFunctionName, ERROR);
			L3Info->copsPressNG = BADDATA;
		} else {
			// put relevant info into L3Info structure
			L3Info->copsPressNG = atof((COPSobjIn->COPSHKdata[31][2]).c_str());
		}
		if (verbose >= 2) COPSobjIn->printCOPSdata("All");
     
		delete COPSobjIn;

	} else {  // No COPS NG Data
		if (sBestBGCOPS.compare("N/A") == 0) {
			L3Info->copsPressNG = BADDATA;
		}
	}

	// Attempt to get COPS RG data

	if (sBestRGCOPS.compare("N/A") != 0) {

		// Now do the RG COPS Data
		COPSobjIn = new DFMS_COPS_Class(sCOPSpath, sBestRGCOPS, "rg");
     
		// Get the COPS RG data file
		nSuccess = COPSobjIn->getCOPSdata();
		if (!nSuccess) {
			sErrorMessage = "Unable to read the COPS RG file: ";
			sErrorMessage += COPSobjIn->getFileName()+"\n";
			writeToLog(sErrorMessage, sFunctionName, ERROR);
			L3Info->copsPressRG = BADDATA;
		} else {
			// put relevant info into L3Info structure
			L3Info->copsPressRG = atof((COPSobjIn->COPSHKdata[32][2]).c_str());
		}
		if (verbose >= 2) COPSobjIn->printCOPSdata("All");

		delete COPSobjIn;

	} else {  // No COPS RG Data

		if (sBestBGCOPS.compare("N/A") == 0) {
			L3Info->copsPressRG = BADDATA;
		}
	}

}

//
// ------------------------------- getMPSTdata ----------------------------------
//
//
//  --------------------- get the relevant MPST data ----------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method retrieves the Mass Peak Table data
//
// inputs:
//   	sBestMPST - the MPST file closest in time to the L2 file
//      peak - verification peak info
//
// returns:
//   None
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
int DFMS_ProcessMCP_Class::getMPSTdata(string sBestMPST, verifPeakInfo *peak) {
    
	string sFunctionName = "DFMS_ProcessMCP_Class::getMPSTdata";
     
	int nSuccess;

    // 1.1 - Create MPST object using sBestMPST file
	MPSTobjIn = new DFMS_MPST_Class(sMPSTpath, sBestMPST);

	// 1.2 - Get the MPST data file
	nSuccess = MPSTobjIn->getMPSTdata();
	if (!nSuccess) {
		sErrorMessage = "Unable to read the MPST file: ";
		sErrorMessage += MPSTobjIn->getfileName();
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}
	if (verbose >= 2) MPSTobjIn->printMPSTobj("All");

	// 1.3 - Create a local (to this class) copy of the MPSTdata, and verifcation peaks info
	mptRows = MPSTobjIn->getRows();
	// 1.4 - Build local MPST data array
	for (int i=0; i<mptRows; i++) {
		// Use vector objects dynamic fill to create local values
		MPSTdata.push_back(sslice());
		for (int j=0; j<7; ++j) {
			MPSTdata[i].push_back(MPSTobjIn->MPSTdata[i][j]);
        }
	}
     

	// 1.5 - Find peak in MPS Table
	double tMass = L2Info.commandedMass;
	if (L2Info.isGCU) {
		nSuccess = findMPSTpeak(tMass, peak, GCUMASSDEL);
	} else {
		nSuccess = findMPSTpeak(tMass, peak, NONGCUMASSDEL);
	}
	mptIndx = peak->index;
	// 1.6 - Set peak search area
	if (L2Info.isGCU) {
		if (mptIndx == -1) {
			skipReas["gcuMassNotInMPST"]++;
			sErrorMessage = "Unable to find a suitable Mass in MPS Table ";
			sErrorMessage += "for Target mass = "+util_doubleToString(tMass);
			writeToLog(sErrorMessage, sFunctionName, WARN);
			if (QLINFOFILE) {
				QuickLookLog << " - Skipped - " << sErrorMessage << dfmsEOL;
				qlCount++;
				if (qlCount % QLPPAGE == 0) qlHeader();
			}
			cout << " - Skipped - " << sErrorMessage << endl;
			return 0;
		} else {
			iXl = util_stringToInt(MPSTdata[mptIndx][4]);
			iXr = util_stringToInt(MPSTdata[mptIndx][5]);
		}
	// nonGCU file
	} else {
		if (mptIndx != -1) {
			iXl = util_stringToInt(MPSTdata[mptIndx][4]);
			iXr = util_stringToInt(MPSTdata[mptIndx][5]);
			//numVerificPeaksFound++;
		}
	}

	return 1;

}

//
// --------------------------- findMPSTpeak -----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Searches the MPS table for a specific peak corresponding to mass inMass
//
// input:
//      inMass - mass to look for
//      massDel - Mass increment to use for peak ID
// output:
//   p - a struct of verifPeakInfo type
//
// returns:
//   1 for success, 0 for failure
// =============================================================================
// History: Written by Mike Rinaldi, August 2013
// =============================================================================
//
int DFMS_ProcessMCP_Class::findMPSTpeak(double inMass, verifPeakInfo *p,
													double massDel) {

	string sFunctionName = "DFMS_ProcessMCP_Class::findMPSTpeak";

	double mpsTableMass;

	mptIndx = -1;    // Assume no verification peak available

	// Look for target peak data index if in GCU mode
	for (int i=0; i<mptRows; i++) {
		mpsTableMass = atof(MPSTdata[i][1].c_str());
		if (mpsTableMass > inMass-massDel && mpsTableMass < inMass+massDel ) {
			mptIndx = i;
			p->index = mptIndx;
			p->mass = mpsTableMass;
			p->peakNameA = MPSTdata[mptIndx][0];
			break;
		}
	}

	// Set peak struct values for case where no verification peak is present
	if (mptIndx < 0) {
		p->index = mptIndx;
		p->mass = 0.0;
		p->peakNameA = "None Found";
		return 0;
	}

	return 1;
}

//
// ---------------------------- calcPPMDiff ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Calculation of parts per million difference between mass found and target mass
// It also keeps stract
//
// inputs: 
//   m1 - mass 1
//   m2 - mass 2
//   
// returns:
//   ppmdiff - absolute value in ppm
// =============================================================================
// History: Written by Mike Rinaldi, May 2013
// =============================================================================
//
double DFMS_ProcessMCP_Class::calcPPMDiff(double m1, double m2) {
    
	double ppm = abs((m1 - m2)/m2*1.0e6);

	return ppm;

}

//
// ------------------------------ calibGCU -------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Calculates/Assigns the various calibration values pix0, ppmDiff,... for GCU
// mode files;
//
// inputs: 
//   gPF - pointer to Mass Calib object
//   pix0Fit - pointer to fit paramater object
//   sMode - file mode
//   iRow  -  Side A=1, SideB=2
//
// returns:
//   1 for normal and 0 for error
// =============================================================================
// History: Written by Mike Rinaldi, May 2013
// =============================================================================
//
int DFMS_ProcessMCP_Class::calibGCU(DFMS_PeakFit_Class *gPF,
						DFMS_X0FitFile_Class *x0Fit, string sMode, int iRow) {

     string sFunctionName = "DFMS_ProcessMCP_Class::calibGCU";

     int nSuccess=0;

     double pix0=0.0, sPix0=0.0;
     double gcuPix0 = 0.0;
     double aA=0.0, bA=0.0;
     double aB=0.0, bB=0.0;
     double R2coeffA=0.0, R2coeffB=0.0;
     double errorA=0.0, errorB=0.0;
     double x=0.0;
     double xUnc=0.0;
     double m = 0.0, m0 = 0.0;

     string gcuPix0FitFileName = x0Fit->getFileName();

     // STEP 12.12.3.17.1 - Assign target mass
     m0 = L2Info.commandedMass;  // Target mass from ROSINA_SCI value in L2 HK data
     L3Info->commandedMass = m0;
     if (m0 <= 0.0 ) {
    	 sErrorMessage="ROSINA TARGET MASS <= 0.0";
    	 writeToLog(sErrorMessage, sFunctionName, ERROR);
    	 return 0;
     }

     //  STEP 12.12.3.17.2 - If GCU then use MPS mass as actual 'm'
     m = atof(MPSTdata[mptIndx][1].c_str());  // Actual mass from MPS table

     // create mass scale array
     double *mScale = new double[NUMPIX];

     //  STEP 12.12.3.17.3 - If the peak curvfit was successful save the L3 data
     if (gPF->fitSucceeded) {

    	 if (iRow == 1) {
    		 gPFsuccessA = true;
    	     L3Info->peak->pkIDA = MPSTdata[mptIndx][0];
    		 x = L3Info->peak->peakCenterA;
    		 xUnc = L3Info->peak->peakCenterAUnc;

    		 // STEP 12.12.3.17.3a - if GCU file calculate pix0 using formula
    		 sPix0 = massToPix0(L2Info.zoom,x,m,m0);  // Self Pixel 0 calibration
    		 pix0 = aGCUA + bGCUA*m0;     // pix0 from average GCU (from m0 vs pix0 fits)
    		 gcuPix0 = pix0;
    		 L3Info->cal->pix0_A = pix0;
    		 L3Info->cal->GCUPix0_A = gcuPix0;
    		 L3Info->cal->selfPix0_A = sPix0;
    		 L3Info->cal->m0A = m0;
    	 	 // For a HRMASSDELTA error if in High Res mode
    	   	 if (L2Info.hiRes) {
    	   		 // Change to fixed value as of 29/09/2014 (Meeting with Kathrin A.)
    	   		L3Info->cal->selfPix0_UncA = HRPIX0UNC;
    	   		L3Info->cal->GCUPix0_UncA = HRPIX0UNC;
    	   	 } else {
    	   		L3Info->cal->GCUPix0_UncA = x0Fit->sErrorA;
    	   		L3Info->cal->selfPix0_UncA = L3Info->cal->GCUPix0_UncA;
    	   	 }

    		 // STEP 12.12.3.17.3b - Create mode mass scale using our currently best pix0
    		 massScale(mScale, L2Info.zoom, L3Info->cal->pix0_A, m0, NUMPIX, false);
    		 L3Info->pkMassA->num = numGoodPks;
   			 L3Info->pkMassA->pkFit = pkFI.maxPkLoc;
   			 L3Info->pkMassA->iXl = iXl;
   			 L3Info->pkMassA->iXr = iXr;
   			 for (int i=0; i<numGoodPks; i++) {
   				 L3Info->pkMassA->peakLoc[i] = pkFI.peakLoc[i];
   				 L3Info->pkMassA->peakVal[i] = YAin[pkFI.peakLoc[i]];
   				 L3Info->pkMassA->pkMass[i] = mScale[pkFI.peakLoc[i]];
   			 }
   			 L3Info->cal->GCUPix0_A_used = true;
   			 L3Info->cal->GCUPix0_UncA_used = true;
   			 L3Info->cal->selfPix0_A_used = false;
   			 L3Info->cal->selfPix0_UncA_used = false;

    		 // STEP 12.12.3.17.3c - Calculate ppm difference and set quality flag
    		 int pkPixel = (int)(x+0.5);  // Define the nearest integer pixel to indicate the peak
    		 // Use the avg GCU mass Scale defined above
    		 double mFit = pixToMass(L2Info.zoom, pix0, pkPixel, m0);
    		 double ppmDiffGCUavg = calcPPMDiff(m, mFit);
			 L3Info->cal->ppmDiffA = ppmDiffGCUavg;
    		 // make the Self mass scale using the self pix0
			 double sFit = pixToMass(L2Info.zoom, sPix0, pkPixel, m0);
			 double ppmDiffSelf = calcPPMDiff(m, sFit);

			 // Define the quality Flag
			 if (ppmDiffGCUavg <= PPMDIFFMIN) {
				 L3Info->qualIDA = 0;
			 } else if (ppmDiffGCUavg > PPMDIFFMIN && ppmDiffSelf < PPMDIFFMIN) {
				 L3Info->qualIDA = 1;
			 } else if (ppmDiffGCUavg >= PPMDIFFMIN || x0Fit->qualID > 1) {
				 L3Info->qualIDA = 2;
			 }

			 // STEP 12.12.3.17.3d - Set the CAL MASS Table info
			 L3Info->vPeaks[mptIndx].peakNameA = L3Info->peak->pkIDA;
			 L3Info->vPeaks[mptIndx].caltype = 0;
			 L3Info->vPeaks[mptIndx].foundA = L3Info->peak->peakFoundA;
			 L3Info->vPeaks[mptIndx].peakCntrA = x;
			 L3Info->vPeaks[mptIndx].peakWidthA = L3Info->peak->peakCenterWidthA;
			 L3Info->vPeaks[mptIndx].peakHeightA = L3Info->peak->peakHeightA;
			 L3Info->vPeaks[mptIndx].ppmDiffA = ppmDiffGCUavg;

	    	 // Set the calID's for this file
	         calID4A = gcuPix0FitFileName;
	         calID5A = "None";
	         calID6A = MPSTobjIn->getfileName();

    	 } else if (iRow == 2) {

    		 gPFsuccessB = true;
    	     L3Info->peak->pkIDB = MPSTdata[mptIndx][0];
    		 x = L3Info->peak->peakCenterB;
    		 xUnc = L3Info->peak->peakCenterBUnc;

    		 // if GCU file calculate pix0 using formula
    		 sPix0 = massToPix0(L2Info.zoom,x,m,m0);  // Self Pixel 0 calibration
    		 pix0 = aGCUB + bGCUB*m0;     // pix0 from average GCU (from m0 vs pix0 fits)
    		 gcuPix0 = pix0;
    		 L3Info->cal->pix0_B = pix0;
    		 L3Info->cal->GCUPix0_B = gcuPix0;
    	 	 L3Info->cal->selfPix0_B = pix0;
    	 	 L3Info->cal->m0B = m0;
    	 	 // Force a HRMASSDELTA error if in High Res mode
    	   	 if (L2Info.hiRes) {
    	   		 // Change to fixed value as of 29/09/2014 (Meeting with Kathrin A.)
    	   		 L3Info->cal->selfPix0_UncB = HRPIX0UNC;
     	   		 L3Info->cal->GCUPix0_UncB = HRPIX0UNC;
    	   	 } else {
     	   		L3Info->cal->GCUPix0_UncB = x0Fit->sErrorB;
     	   		L3Info->cal->selfPix0_UncB = L3Info->cal->GCUPix0_UncB;
     	   	 }

    	 	 // Create mode mass scale using our currently best pix0
    	 	 massScale(mScale, L2Info.zoom, L3Info->cal->pix0_B, m0, NUMPIX, false);
    	 	 int pkPixel = (int)(x+0.5);
    		 double mFit = mScale[pkPixel];
    		 L3Info->pkMassB->num = numGoodPks;
    		 L3Info->pkMassB->pkFit = pkFI.maxPkLoc;
   			 L3Info->pkMassB->iXl = iXl;
   			 L3Info->pkMassB->iXr = iXr;
    		 for (int i=0; i<numGoodPks; i++) {
    			 L3Info->pkMassB->peakLoc[i] = pkFI.peakLoc[i];
   				 L3Info->pkMassB->peakVal[i] = YBin[pkFI.peakLoc[i]];
    			 L3Info->pkMassB->pkMass[i] = mScale[pkFI.peakLoc[i]];
    		 }
    		 L3Info->cal->GCUPix0_B_used = true;
    		 L3Info->cal->GCUPix0_UncB_used = true;
    		 L3Info->cal->selfPix0_B_used = false;
    		 L3Info->cal->selfPix0_UncB_used = false;

    	 	 for (int i=0; i<NUMPIX; ++i) {
    	 		 fMassScaleB[i] = mScale[i];
    	 	 }

    	 	 // Calculate ppm difference and set quality flag
    		 pkPixel = (int)(x+0.5);  // Define the nearest integer pixel to indicate the peak
    		 // Use the avg GCU mass Scale defined above
    		 mFit = pixToMass(L2Info.zoom, pix0, pkPixel, m0);
    		 double ppmDiffGCUavg = calcPPMDiff(m, mFit);
			 L3Info->cal->ppmDiffB = ppmDiffGCUavg;
    		 // make the Self mass scale using the self pix0
			 double sFit = pixToMass(L2Info.zoom, sPix0, pkPixel, m0);
			 double ppmDiffSelf = calcPPMDiff(m, sFit);

			 // Define the quality Flag
			 if (ppmDiffGCUavg <= PPMDIFFMIN) {
				 L3Info->qualIDB = 0;
			 } else if (ppmDiffGCUavg > PPMDIFFMIN && ppmDiffSelf < PPMDIFFMIN) {
				 L3Info->qualIDB = 1;
			 } else if (ppmDiffGCUavg >= PPMDIFFMIN || x0Fit->qualID > 1) {
				 L3Info->qualIDB = 2;
			 }

    	 	 // Now be safe and take the lowest quality has the representative
    	 	 if (L3Info->qualIDA >= L3Info->qualIDB) {
    	 		 L3Info->qualID = L3Info->qualIDA;
    	 	 } else {
    	 		 L3Info->qualID = L3Info->qualIDB;
    	 	 }

    	 	 L3Info->cal->calType=0;
    	 	 L3Info->calRowUsed = mptIndx;    // Index of CAL MASS used

    	 	 // Set the CAL MASS Table info
    	 	 L3Info->vPeaks[mptIndx].peakNameB = L3Info->peak->pkIDB;
    	 	 L3Info->vPeaks[mptIndx].caltype = 0;
    	 	 L3Info->vPeaks[mptIndx].foundB = L3Info->peak->peakFoundB;
    	 	 L3Info->vPeaks[mptIndx].peakCntrB = x;
    	 	 L3Info->vPeaks[mptIndx].peakWidthB = L3Info->peak->peakCenterWidthB;
    	 	 L3Info->vPeaks[mptIndx].peakHeightB = L3Info->peak->peakHeightB;
    	 	 L3Info->vPeaks[mptIndx].ppmDiffB = ppmDiffGCUavg;

		 	 // Keep stats on pix0/ppmDiff/m distribution
		 	 pkPPMpair = make_pair(L3Info->peak->peakCenterA,L3Info->cal->ppmDiffA);
		 	 pkPPMvec.push_back(pkPPMpair);
		 	 pkPPMpair = make_pair(L3Info->peak->peakCenterB,L3Info->cal->ppmDiffB);
		 	 pkPPMvec.push_back(pkPPMpair);

	    	 // Set the calID's for this file
	         calID4B = gcuPix0FitFileName;
	         calID5B = "None";
	         calID6B = MPSTobjIn->getfileName();

    	 	 // Write to the Quick Look log
    	 	 if (QLINFOFILE) {
    	 		 string star=" ";
    	 		 if (L3Info->cal->ppmDiffA > PPMDIFFMIN) star.assign("*/");
    	 		 if (L3Info->cal->ppmDiffB > PPMDIFFMIN) star+=("*");
    		 	 if (infNanErr) star+=("/InfNaN/");
    		 	 infNanErr = false;
			 	 char tmp[400];
			 	 sprintf(tmp,"%7.2f     %7.2f / %7.2f    %7.2f / %7.2f     %8.2f / %8.2f  %3.3s",
			 			 L3Info->commandedMass,
			 			 L3Info->peak->peakCenterA, L3Info->peak->peakCenterB,
			 			 L3Info->cal->pix0_A, L3Info->cal->pix0_B,
			 			 L3Info->cal->ppmDiffA, L3Info->cal->ppmDiffB, star.c_str());
			 	 string line;
			 	 line.assign(tmp);
			 	 QuickLookLog << line << dfmsEOL;
			 	 qlCount++;
			 	 if (qlCount % QLPPAGE == 0) qlHeader();
    	 	 }
    	 }

     // STEP 12.12.3.17.4 - gPF->fitSucceeded = false - Bad peak fit or No peak found
     } else {

         // STEP 12.12.3.17.4a - make all relevant Bada Data assignments
    	 if (iRow == 1) {
    		 gPFsuccessA = false;
         	 // 2.11.4.3 - Determine pix0 from a/b fit params
        	 pix0 = aGCUA + bGCUA*m0;
    		 L3Info->cal->pix0_A = pix0;
    		 L3Info->cal->GCUPix0_A = pix0;
    		 if (L2Info.lowRes) {
    			 L3Info->cal->GCUPix0_UncA = x0Fit->sErrorA;
    		 } else {
     	   		 L3Info->cal->GCUPix0_UncA = HRPIX0UNC;
    		 }
    		 L3Info->cal->selfPix0_A = BADDATA;
    		 L3Info->cal->selfPix0_UncA = BADDATA;
    		 L3Info->cal->m0A = m0;
    		 L3Info->cal->ppmDiffA = BADDATA;
    		 L3Info->qualIDA = 4;
   			 L3Info->cal->GCUPix0_A_used = false;
   			 L3Info->cal->GCUPix0_UncA_used = false;
   			 L3Info->cal->selfPix0_A_used = false;
   			 L3Info->cal->selfPix0_UncA_used = false;

   	    	 // Set the calID's for this file
   	         calID4A = "None";
   	         calID5A = "None";
   	         calID6A = MPSTobjIn->getfileName();

    	 } else {
    		 gPFsuccessB = false;
             // Determine pix0 from a/b fit params
        	 pix0 = aGCUB + bGCUB*m0;
    		 L3Info->cal->pix0_B = pix0;
    		 L3Info->cal->GCUPix0_B = pix0;
    		 if (L2Info.lowRes) {
    			 L3Info->cal->GCUPix0_UncB = x0Fit->sErrorB;
    		 } else {
     	   		 L3Info->cal->GCUPix0_UncB = HRPIX0UNC;
    		 }
    		 L3Info->cal->selfPix0_B = BADDATA;
    		 L3Info->cal->selfPix0_UncB = BADDATA;
    		 L3Info->cal->m0B = m0;
    		 L3Info->cal->ppmDiffB = BADDATA;
			 L3Info->qualIDB = 4;

    	 	 // Now be safe and take the lowest quality has the representative
    	 	 if (L3Info->qualIDA >= L3Info->qualIDB) {
    	 		 L3Info->qualID = L3Info->qualIDA;
    	 	 } else {
    	 		 L3Info->qualID = L3Info->qualIDB;
    	 	 }
   			 L3Info->cal->GCUPix0_B_used = false;
   			 L3Info->cal->GCUPix0_UncB_used = false;
   			 L3Info->cal->selfPix0_B_used = false;
   			 L3Info->cal->selfPix0_UncB_used = false;

    	 	 L3Info->cal->calType = 0;
    	 	 L3Info->calRowUsed = mptIndx;

   	    	 // Set the calID's for this file
   	         calID4B = "None";
   	         calID5B = "None";
   	         calID6B = MPSTobjIn->getfileName();


    	 	 if (QLINFOFILE) {
    	 		 string star=" ";
    	 		 if (infNanErr) star+=("/InfNaN/");
    	 		 infNanErr = false;
			 	 char tmp[400];
			 	 sprintf(tmp,"%7.2f     %7.2f / %7.2f    %7.2f / %7.2f     %8.2f / %8.2f  %3.3s",
			 			 L3Info->commandedMass,
			 			 L3Info->peak->peakCenterA, L3Info->peak->peakCenterB,
			 			 L3Info->cal->pix0_A, L3Info->cal->pix0_B,
			 			 L3Info->cal->ppmDiffA, L3Info->cal->ppmDiffB, star.c_str());
			 	 string line;
			 	 line.assign(tmp);
			 	 QuickLookLog << line << dfmsEOL;
			 	 qlCount++;
			 	 if (qlCount % QLPPAGE == 0) qlHeader();
    	 	 }
    	 }

     }

 	// Set the final CalID values
 	if (gPFsuccessA) {
 		L3Info->cal->calID4 = calID4A;
 		L3Info->cal->calID5 = calID5A;
 		L3Info->cal->calID6 = calID6A;
 	} else if (gPFsuccessB) {
 		L3Info->cal->calID4 = calID4B;
 		L3Info->cal->calID5 = calID5B;
 		L3Info->cal->calID6 = calID6B;
 	} else {
 		L3Info->cal->calID4 = "None";
 		L3Info->cal->calID5 = "None";
 		L3Info->cal->calID6 = "None";
 	}

     // STEP 12.12.3.17.5 - Release Memory
     delete[] mScale; mScale=0;
     return 1;

}

//
// ----------------------------- calibSLF --------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Calculates/Assigns the various calibration values pix0, ppmDiff,...
// for NON-GCU mode files where a verification peak is available
//
// inputs:
//   gPF - pointer to Mass Calib object
//   pix0Fit - pointer to pix0 fit object
//   sMode - file mode
//   iRow  -  Side A=1, SideB=2
//
// returns:
//   1 for normal and 0 for error
// =============================================================================
// History: Written by Mike Rinaldi, Feb 2014
// =============================================================================
//
int DFMS_ProcessMCP_Class::calibSLF(DFMS_PeakFit_Class *gPF, DFMS_X0FitFile_Class *x0Fit,
						   DFMS_X0FitFile_Class *x0FitS, string sMode, int iRow) {

	string sFunctionName = "DFMS_ProcessMCP_Class::calibSLF";

	int nSuccess=0;

	int calType=0;

	double pix0=0.0;
	double sPix0=BADDATA;
	double aGCU=0.0, bGCU=0.0;
	double aSLF=0.0, bSLF=0.0;
	double x=0.0;
	double xUnc=0.0;
	double m = 0.0, m0 = 0.0;
	double ppmDiff=0.0, ppmDiffSelf=0.0, ppmDiffBest=0.0;
	double ppmDiffGCUavg = 0.0;

	int *peakInd = new int[10];
	int *maxVal = new int[10];
	int *minVal = new int[10];
	int numPeaks = 0;
	int *tmp = new int[10];
	int numTmp = 0;
	double mL=0.0, mR=0.0;

	double vpMass=0.0, tMass=0.0;
	string descrip = "";
	int numGCUmasses = 0;

	int retStatus=0;

	verifPeakInfo *peak = new verifPeakInfo();

	ppmInfo ppm;
	ppm.init();

	double gcuPix0 = 0.0;
	double slfPix0 = 0.0;

	// Mass scale array
	double *mScale = new double[NUMPIX];
	double mScaleG[NUMPIX];
	double mScaleS[NUMPIX];

    // STEP 12.12.3.19.1 - Define the peak center location and uncertainty
    if (iRow == 1) {
    	x = L3Info->peak->peakCenterA;
    	xUnc = L3Info->peak->peakCenterAUnc;
    } else if (iRow == 2) {
    	x = L3Info->peak->peakCenterB;
    	xUnc = L3Info->peak->peakCenterBUnc;
    }

    // STEP 12.12.3.19.2 - Define the target mass from ROSINA_DFMS_SCI_MASS value in L2 HK data
    m0 = L2Info.commandedMass;
    L3Info->commandedMass = m0;

    gcuPix0FitFileName = x0Fit->getFileName();		// Latest GCU calibration fit file
    if (!noSLFFitFile) {
    	slfPix0FitFileName = x0FitS->getFileName();		// Latest SLF calibration fit file
    }

    if (iRow == 1) {
    	aGCU = aGCUA;   // Get the fitting parameters for Row A
    	bGCU = bGCUA;
    	aSLF = aSLFA;
       	bSLF = bSLFA;
    } else {
    	aGCU = aGCUB;   // Get the fitting parameters for Row B
    	bGCU = bGCUB;
    	aSLF = aSLFB;
    	bSLF = bSLFB;
    }

    // STEP 12.12.3.19.5 - Determine gcuPix0 from latest GCU and SLF fit file
	gcuPix0 = aGCU + bGCU*m0;
	slfPix0 = aSLF + bSLF*m0;
	double massGatX=0.0, massSatX=0.0;
    if (isRecentGCUavailable) {
    	pix0 = gcuPix0;
    	massScale(mScaleG, L2Info.zoom, pix0, m0, NUMPIX, false);
    	int pkPixel = (int)(x+0.5);
    	massGatX = mScaleG[pkPixel];    // The actual peak based on GCU fits
    } else {
    	if (noSLFFitFile) {
    		// Since there is no SLF file available use GCU anyway.. Even if not recent
        	pix0 = gcuPix0;
        	massScale(mScaleG, L2Info.zoom, pix0, m0, NUMPIX, false);
        	int pkPixel = (int)(x+0.5);
        	massGatX = mScaleG[pkPixel];    // The actual peak based on GCU fits
    	} else {
    		pix0 = slfPix0;
    		massScale(mScaleS, L2Info.zoom, pix0, m0, NUMPIX, false);
    		int pkPixel = (int)(x+0.5);
    		massSatX = mScaleS[pkPixel];    // The actual peak based on SLF fits
    	}
    }

    // STEP 12.12.3.19.6 - Check if CurveFit succeeded
    if (gPF->fitSucceeded) {

        // STEP 12.12.3.19.7 - Calculate mass scale borders using GCU pix0
    	if (isRecentGCUavailable) {
    		mL = pixToMass(L2Info.zoom, gcuPix0, (double)PKLOWPT, m0);
    		mR = pixToMass(L2Info.zoom, gcuPix0, (double)PKHIGHPT, m0);
    	} else {
    		mL = pixToMass(L2Info.zoom, slfPix0, (double)PKLOWPT, m0);
    		mR = pixToMass(L2Info.zoom, slfPix0, (double)PKHIGHPT, m0);
    	}

        // STEP 12.12.3.19.8 - Search for MPST mass in between m1 and m2
        double pix1,pix2;
        for (int i=0; i<mptRows; i++) {
        	m = util_stringToDouble(MPSTdata[i][1]);
        	// Check if   m1 <= mass <= m2
        	if (m >= mL && m <= mR) {
        		// Calculate the m1/m2 corresponding pixels
				if (isRecentGCUavailable) {
					pix1 = massToPix(L2Info.zoom, gcuPix0, m-NONGCUMASSDEL, m0);
					pix2 = massToPix(L2Info.zoom, gcuPix0, m+NONGCUMASSDEL, m0);
				} else {
					pix1 = massToPix(L2Info.zoom, slfPix0, m-NONGCUMASSDEL, m0);
					pix2 = massToPix(L2Info.zoom, slfPix0, m+NONGCUMASSDEL, m0);
				}
        		// Now check if   pix1 <= peakLoc x <= pix2
        		if (pix1 > pix2) {
            		if (x <= pix1 && x >= pix2) {
            			if (noSLFFitFile) {
            				if (iRow == 1) sPix0 = BADDATA;    //aGCUA + bGCUA*m0;
            				if (iRow == 2) sPix0 = BADDATA;    //aGCUB + bGCUB*m0;
            				pix0 = gcuPix0;
            			} else {
            				if (isRecentGCUavailable) {
            					// Pre 1/3/2015
            					if (iRow == 1) sPix0 = aSLFA + bGCUA*m0;
            					if (iRow == 2) sPix0 = aSLFB + bGCUB*m0;
            				} else {
            					// Post 1/3/2015
            					if (iRow == 1) sPix0 = aSLFA + bSLFA*m0;
            					if (iRow == 2) sPix0 = aSLFB + bSLFB*m0;
            				}
                			pix0 = sPix0;
            			}
                    	mptIndx = i;
            			foundVerifPeak = true;
            			//numVerificPeaksUsed++;
            			break;
            		} else {
            			pix0 = gcuPix0;
            			foundVerifPeak = false;
            		}
        		} else {
            		if (x >= pix1 && x <= pix2) {
            			if (noSLFFitFile) {
            				if (iRow == 1) sPix0 = BADDATA;    //aGCUA + bGCUA*m0;
            				if (iRow == 2) sPix0 = BADDATA;    //aGCUB + bGCUB*m0;
            				pix0 = gcuPix0;
            			} else {
            				if (isRecentGCUavailable) {
            					// Pre 1/3/2015
            					if (iRow == 1) sPix0 = aSLFA + bGCUA*m0;
            					if (iRow == 2) sPix0 = aSLFB + bGCUB*m0;
            				} else {
            					// Post 1/3/2015
            					if (iRow == 1) sPix0 = aSLFA + bSLFA*m0;
            					if (iRow == 2) sPix0 = aSLFB + bSLFB*m0;
            				}
                			pix0 = sPix0;
            			}
            			mptIndx = i;
            			foundVerifPeak = true;
            			//numVerificPeaksUsed++;
            			break;
            		} else {
            			pix0 = gcuPix0;
            			foundVerifPeak = false;
            		}
        		}
        	} else {
    			pix0 = gcuPix0;
        		foundVerifPeak = false;
        	}
        }
        
        // STEP 12.12.3.19.9 - If target mass is an SLF mass then use it as a
    	// verification peak. if Not them return 0 and do Self Calib.
        if (foundVerifPeak) {
            
            // Take care of pixel0 shift for masses > 69.
            // Check if m0 > 69 and if CS2 peak at m0 = 76 was found and used for calibration.
            // If so, then adjust pix0 to pix0+36
            string slfPeakUsed = MPSTdata[mptIndx][0];
            slfPeakUsed = slfPeakUsed.substr(1,6);
            slfPeakUsed = util_trim(slfPeakUsed);
            if (m0 > 69 && slfPeakUsed.compare("CS2")==0) {
                pix0 += 36;
                iicnt++;
            }

        	// Create the mass scale for later use
        	massScale(mScale, L2Info.zoom, pix0, m0, NUMPIX, false);

        	// 2.12.6.4 - Calculate ppm difference and quality flag
        	int pkPixel = (int)(x+0.5);  // Define the nearest integer pixel to indicate the peak
        	// Calc ppmDiff for GCU mass Scale defined above
        	double mFit = pixToMass(L2Info.zoom, gcuPix0, pkPixel, m0);
        	ppmDiffGCUavg = calcPPMDiff(m, mFit);
        	// Calc ppmDiff for Self Mass Scale
        	double sFit = pixToMass(L2Info.zoom, sPix0, pkPixel, m0);
        	ppmDiffSelf = calcPPMDiff(m, sFit);
    		calType = 0;
    		peak->caltype = 1;
    		peak->index = mptIndx;
        	if (sPix0 == BADDATA && ppmDiffGCUavg < ppmDiffSelf) {
        		ppmDiffSelf = ppmDiffGCUavg;
        		peak->caltype = 0;
        	}
        	//


    		// STEP 12.12.3.19.10 - Set Row A/B peak parameters
    		if (iRow == 1) {
    			gPFsuccessA = true;
    			// Define the quality Flag
    			if (ppmDiffSelf <= PPMDIFFMIN) {
    				L3Info->qualIDA = 0;
    			} else if (ppmDiffGCUavg > PPMDIFFMIN && ppmDiffSelf < PPMDIFFMIN) {
    				L3Info->qualIDA = 1;
    			} else if (ppmDiffGCUavg >= PPMDIFFMIN || x0Fit->qualID > 1) {
    				L3Info->qualIDA = 2;
    			}
    			double dtmp = (double)((int)(sFit+0.5));
    			string stmp = (speciesDef[dtmp]).name;
    			//cout << "stmp = " << stmp << endl;
        		peak->peakNameA = MPSTdata[mptIndx][0];
    			L3Info->usedAsVerifA = true;
    			peak->foundA = 1;
    			peak->peakCntrA = x;
    			peak->ppmDiffA = ppmDiffSelf;
    			peak->peakWidthA = L3Info->peak->peakCenterWidthA;
    			peak->peakHeightA = L3Info->peak->peakHeightA;
    			L3Info->cal->GCUPix0_A_used = false;
    			L3Info->cal->GCUPix0_UncA_used = false;
    			L3Info->cal->selfPix0_A_used = true;
    			L3Info->cal->selfPix0_UncA_used = true;

    			// Set the peaksFound related values - Row A
    			L3Info->pkMassA->num = numGoodPks;
    			L3Info->pkMassA->pkFit = pkFI.maxPkLoc;
      			L3Info->pkMassA->iXl = iXl;
      			L3Info->pkMassA->iXr = iXr;
    			for (int i=0; i<numGoodPks; i++) {
    				L3Info->pkMassA->peakLoc[i] = pkFI.peakLoc[i];
      				L3Info->pkMassA->peakVal[i] = YAin[pkFI.peakLoc[i]];
    				L3Info->pkMassA->pkMass[i] = mScale[pkFI.peakLoc[i]];
    			}

    			// Set the CAL MASS Table info
    			L3Info->vPeaks[mptIndx].peakNameA = peak->peakNameA;
    			L3Info->vPeaks[mptIndx].caltype = 0;
    			L3Info->vPeaks[mptIndx].foundA = L3Info->peak->peakFoundA;
    			L3Info->vPeaks[mptIndx].peakCntrA = x;
    			L3Info->vPeaks[mptIndx].peakWidthA = L3Info->peak->peakCenterWidthA;
    			L3Info->vPeaks[mptIndx].peakHeightA = L3Info->peak->peakHeightA;
    			L3Info->vPeaks[mptIndx].ppmDiffA = ppmDiffSelf;
    			// If this is a SLF mass then keep track of the ppmDiff.  This will
    			// be used later when we handle ppmDiff for nonGCU/nonSLF files in
    			// calibNonGCUnonSLF();
    			if (L2Info.isSLF && slfSet.find(L2Info.commandedMass) != slfSet.end()) {
					ppm.ppmDiff = ppmDiffSelf;
					ppm.fileName = L2Info.fileName;
					ppm.mode = L2Info.mode;
					ppm.mass = L2Info.commandedMass;
					ppm.timeTag = L2Info.secSince1970;
    				if (L2Info.lowRes) {
    					if (L2Info.commandedMass < SLFLOWCUTOFF) {
    						ppmSLFLmLrA.push_back(ppm);
    					} else {
        					ppmSLFHmLrA.push_back(ppm);
    					}
    				} else if (L2Info.hiRes) {
    					if (L2Info.commandedMass < SLFLOWCUTOFF) {
    						ppmSLFLmHrA.push_back(ppm);
    					} else if (L2Info.commandedMass >= SLFLOWCUTOFF && L2Info.commandedMass < SLFHICUTOFF) {
    						ppmSLFMmHrA.push_back(ppm);
    					} else {
        					ppmSLFHmHrA.push_back(ppm);
    					}
    				}
    			}
                // STEP 12.12.3.19.11 - Set the L3Info values for this mode file
        		if (L2Info.commandedMass == 0) {
        			setSLFL3Info(iRow,pix0,gcuPix0,m0,m,x,xUnc,ppmDiffSelf, peak, calType, x0Fit->sErrorA,
        		       	x0Fit->sErrorB);
            		calID4A = calID5A = calID6A = "None";
        		} else {
        			if (isRecentGCUavailable) {
        				setSLFL3Info(iRow,pix0,gcuPix0,m0,m,x,xUnc,ppmDiffSelf, peak, calType, x0Fit->sErrorA,
        					x0Fit->sErrorB);
                		calID4A = gcuPix0FitFileName;
                		calID5A = slfPix0FitFileName;
                		calID6A = MPSTobjIn->getfileName();
        			} else {
        				setSLFL3Info(iRow,pix0,gcuPix0,m0,m,x,xUnc,ppmDiffSelf, peak, calType, x0Fit->sErrorA,
        					x0Fit->sErrorB);
                		calID4A = "None";
                		calID5A = slfPix0FitFileName;
                		calID6A = MPSTobjIn->getfileName();
        			}
        		}
    		}

    		if (iRow == 2) {
    			gPFsuccessB = true;
    			// Define the quality Flag
    			if (ppmDiffSelf <= PPMDIFFMIN) {
    				L3Info->qualIDB = 0;
    			} else if (ppmDiffGCUavg > PPMDIFFMIN && ppmDiffSelf < PPMDIFFMIN) {
    				L3Info->qualIDB = 1;
    			} else if (ppmDiffGCUavg >= PPMDIFFMIN || x0Fit->qualID > 1) {
    				L3Info->qualIDB = 2;
    			}
    			double dtmp = (double)((int)(sFit+0.5));
    			string stmp = (speciesDef[dtmp]).name;
    			//cout << "stmp = " << stmp << endl;
        		peak->peakNameB = MPSTdata[mptIndx][0];
    			L3Info->usedAsVerifB = true;
    			peak->foundB = 1;
    			peak->peakCntrB = x;
    			peak->ppmDiffB = ppmDiffSelf;
    			peak->peakWidthB = L3Info->peak->peakCenterWidthB;
    			peak->peakHeightB = L3Info->peak->peakHeightB;
    			L3Info->cal->GCUPix0_B_used = false;
    			L3Info->cal->GCUPix0_UncB_used = false;
    			L3Info->cal->selfPix0_B_used = true;
    			L3Info->cal->selfPix0_UncB_used = true;

    			// Set the peaksFound related values - Row B
    			L3Info->pkMassB->num = numGoodPks;
    			L3Info->pkMassB->pkFit = pkFI.maxPkLoc;
      			L3Info->pkMassB->iXl = iXl;
      			L3Info->pkMassB->iXr = iXr;
    			for (int i=0; i<numGoodPks; i++) {
    				L3Info->pkMassB->peakLoc[i] = pkFI.peakLoc[i];
      				L3Info->pkMassB->peakVal[i] = YBin[pkFI.peakLoc[i]];
    				L3Info->pkMassB->pkMass[i] = mScale[pkFI.peakLoc[i]];
    			}
            	for (int i=0; i<NUMPIX; ++i) {
        			 fMassScaleB[i] = mScale[i];
            	}

                gcuPix0FitFileName = x0Fit->getFileName();		// Latest GCU calibration fit file
                slfPix0FitFileName = x0FitS->getFileName();		// Latest SLF calibration fit file
    			L3Info->calRowUsed = peak->index;

    			// Set the CAL MASS Table info
    			L3Info->vPeaks[mptIndx].peakNameB = peak->peakNameB;
    			L3Info->vPeaks[mptIndx].caltype = 0;
    			L3Info->vPeaks[mptIndx].foundB = L3Info->peak->peakFoundB;
    			L3Info->vPeaks[mptIndx].peakCntrB = x;
    			L3Info->vPeaks[mptIndx].peakWidthB = L3Info->peak->peakCenterWidthB;
    			L3Info->vPeaks[mptIndx].peakHeightB = L3Info->peak->peakHeightB;
    			L3Info->vPeaks[mptIndx].ppmDiffB = ppmDiffSelf;
    			// If this is a SLF mass then keep track of the ppmDiff.  This will
    			// be used later when we handle ppmDiff for nonGCU/nonSLF files in
    			// nonGCUSelfCalibration();
    			if (L2Info.isSLF && slfSet.find(L2Info.commandedMass) != slfSet.end()) {
					ppm.ppmDiff = ppmDiffSelf;
					ppm.fileName = L2Info.fileName;
					ppm.mode = L2Info.mode;
					ppm.mass = L2Info.commandedMass;
					ppm.timeTag = L2Info.secSince1970;
    				if (L2Info.lowRes) {
    					if (L2Info.commandedMass < SLFLOWCUTOFF) {
    						ppmSLFLmLrB.push_back(ppm);
    					} else {
        					ppmSLFHmLrB.push_back(ppm);
    					}
    				} else if (L2Info.hiRes) {
    					if (L2Info.commandedMass < SLFLOWCUTOFF) {
    						ppmSLFLmHrB.push_back(ppm);
    					} else if (L2Info.commandedMass >= SLFLOWCUTOFF && L2Info.commandedMass < SLFHICUTOFF) {
    						ppmSLFMmHrB.push_back(ppm);
    				    } else {
        					ppmSLFHmHrB.push_back(ppm);
    					}
    				}
    			}
                // STEP 12.12.3.19.11 - Set the L3Info values for this mode file
        		if (L2Info.commandedMass == 0) {
        			setSLFL3Info(iRow,pix0,gcuPix0,m0,m,x,xUnc,ppmDiffSelf, peak, calType, x0Fit->sErrorA,
        		       	x0Fit->sErrorB);
            		calID4B = calID5B = calID6B = "None";
        		} else {
        			if (isRecentGCUavailable) {
        				setSLFL3Info(iRow,pix0,gcuPix0,m0,m,x,xUnc,ppmDiffSelf, peak, calType, x0Fit->sErrorA,
        					x0Fit->sErrorB);
                		calID4B = gcuPix0FitFileName;
                		calID5B = slfPix0FitFileName;
                		calID6B = MPSTobjIn->getfileName();
        			} else {
        				setSLFL3Info(iRow,pix0,gcuPix0,m0,m,x,xUnc,ppmDiffSelf, peak, calType, x0Fit->sErrorA,
        					x0Fit->sErrorB);
                		calID4B = "None";
                		calID5B = slfPix0FitFileName;
                		calID6B = MPSTobjIn->getfileName();
        			}
        		}
    		}

    		retStatus = 1;

        } else {   // No verification peak found go to Self Calibration using field method
        	       // calibNonGCUnonSLF()

        	if (iRow == 1) L3Info->peak->pkIDA = "Unknown";
        	if (iRow == 2) L3Info->peak->pkIDB = "Unknown";

        	// Clear all verification peaks memory for next iteration
			delete[] mScale; mScale = 0;
			delete[] peakInd; peakInd = 0;
			delete[] minVal; minVal = 0;
			delete[] maxVal; maxVal = 0;
			delete[] tmp; tmp=0;

			retStatus = 0;
		}

    // STEP 12.12.3.19.12 - gPF->fitSucceeded = false - Bad or No peak fit
    } else {

    	massScale(mScale, L2Info.zoom, gcuPix0, m0, NUMPIX, false);

		peak->caltype = 0;
		peak->index = -1;


    	if (iRow == 1) {
    		gPFsuccessA = false;
    		peak->peakNameA = "Unknown";
    		L3Info->peak->pkIDA = "Unknown";
    		L3Info->cal->pix0_A = pix0;
    		L3Info->cal->GCUPix0_A = gcuPix0;
        	L3Info->cal->selfPix0_A = sPix0;       // Use value from latest fit file
		 	if (L2Info.hiRes) {
		 		// Change to fixed value as of 29/09/2014 (Meeting with Kathrin A.)
		 		L3Info->cal->GCUPix0_UncA = HRPIX0UNC;
		 		L3Info->cal->selfPix0_UncA = HRPIX0UNC;
		 	 } else {
		 		// The total error is the assumed 5 pixel error + the error on the GCU fit
		 		L3Info->cal->GCUPix0_UncA = x0Fit->sErrorA;
		 		L3Info->cal->selfPix0_UncA = sqrt((long double)pow(LRPIX0UNC,2.0) + pow(L3Info->cal->GCUPix0_UncA,2.0));
		   	}
		 	// In case no SLF file is found
		 	if (sPix0 == BADDATA) {
		 		L3Info->cal->selfPix0_UncA = BADDATA;
		 	}
    		L3Info->cal->m0A = m0;
    		L3Info->cal->ppmDiffA = BADDATA;
			L3Info->cal->GCUPix0_A_used = false;
			L3Info->cal->GCUPix0_UncA_used = false;
			L3Info->cal->selfPix0_A_used = false;
			L3Info->cal->selfPix0_UncA_used = false;
			L3Info->qualIDA = 4;
			L3Info->pkMassA->num = numGoodPks;
    		L3Info->pkMassA->pkFit = pkFI.maxPkLoc;
  			L3Info->pkMassA->iXl = iXl;
  			L3Info->pkMassA->iXr = iXr;
    		for (int i=0; i<numGoodPks; i++) {
    			L3Info->pkMassA->peakLoc[i] = pkFI.peakLoc[i];
				L3Info->pkMassA->peakVal[i] = YAin[pkFI.peakLoc[i]];
    			L3Info->pkMassA->pkMass[i] = mScale[pkFI.peakLoc[i]];
    		}
    		// Set calID's at minimum
    		if (m0 > 0) {
    			if (isRecentGCUavailable) {
    				calID4A = gcuPix0FitFileName;
    			} else {
    				calID4A = "None";
    			}
				if (sPix0 == BADDATA) {
					calID5A = "None";
				} else {
					calID5A = slfPix0FitFileName;
				}
				calID6A = "None";
    		} else {
        		calID4A = "None";
        		calID5A = "None";
        		calID6A = "None";
    		}

    	} else {
    		gPFsuccessB = false;
    		peak->peakNameB = "Unknown";
    		L3Info->peak->pkIDB = "Unknown";
    		L3Info->cal->pix0_B = pix0;
    		L3Info->cal->GCUPix0_B = gcuPix0;
    		L3Info->cal->selfPix0_B = sPix0;       // Use value from latest fit file
		 	if (L2Info.hiRes) {
		 		// Change to fixed value as of 29/09/2014 (Meeting with Kathrin A.)
		 		L3Info->cal->GCUPix0_UncB = HRPIX0UNC;
		 		L3Info->cal->selfPix0_UncB = HRPIX0UNC;
		 	 } else {
		 		// The total error is the assumed 5 pixel error + the error on the GCU fit
		 		L3Info->cal->GCUPix0_UncB = x0Fit->sErrorB;
		 		L3Info->cal->selfPix0_UncB = sqrt((long double)pow(LRPIX0UNC,2.0) + pow(L3Info->cal->GCUPix0_UncB,2.0));
		   	}
		 	// In case no SLF file is found
		 	if (sPix0 == BADDATA) {
		 		L3Info->cal->selfPix0_UncB = BADDATA;
		 	}
    		L3Info->cal->m0B = m0;
    		L3Info->cal->ppmDiffB = BADDATA;
			L3Info->cal->GCUPix0_B_used = false;
			L3Info->cal->GCUPix0_UncB_used = false;
			L3Info->cal->selfPix0_B_used = false;
			L3Info->cal->selfPix0_UncB_used = false;
			L3Info->qualIDB = 4;
			L3Info->pkMassB->num = numGoodPks;
    		L3Info->pkMassB->pkFit = pkFI.maxPkLoc;
  			L3Info->pkMassB->iXl = iXl;
  			L3Info->pkMassB->iXr = iXr;
    		for (int i=0; i<numGoodPks; i++) {
    			L3Info->pkMassB->peakLoc[i] = pkFI.peakLoc[i];
  				L3Info->pkMassB->peakVal[i] = YBin[pkFI.peakLoc[i]];
    			L3Info->pkMassB->pkMass[i] = mScale[pkFI.peakLoc[i]];
    		}
        	for (int i=0; i<NUMPIX; ++i) {
    			 fMassScaleB[i] = mScale[i];
        	}
    		// Now be safe and take the lowest quality has the representative
    		if (L3Info->qualIDA >= L3Info->qualIDB) {
    			L3Info->qualID = L3Info->qualIDA;
    		} else {
    			L3Info->qualID = L3Info->qualIDB;
    		}
    		L3Info->cal->calType = 0;
    		L3Info->calRowUsed = mptIndx;
    		if (m0 > 0) {
    			if (isRecentGCUavailable) {
    				calID4B = gcuPix0FitFileName;
    			} else {
    				calID4B = "None";
    			}
				if (sPix0 == BADDATA) {
					calID5B = "None";
				} else {
					calID5B = slfPix0FitFileName;
				}
				calID6B = "None";
    		} else {
        		calID4B = "None";
        		calID5B = "None";
        		calID6B = "None";
    		}
    		// Optionally write to the Quick Look log
    		string star=" ";
    		if (L3Info->cal->ppmDiffA > PPMDIFFMIN) star.assign("*/");
    		if (L3Info->cal->ppmDiffB > PPMDIFFMIN) star+=("*");
	 		if (infNanErr) star+=("/InfNaN/");
	 		infNanErr = false;
    		char ctmp[400];
    		sprintf(ctmp,"%7.2f     %7.2f / %7.2f    %7.2f / %7.2f     %8.2f / %8.2f  %3.3s",
    				L3Info->commandedMass,
    				L3Info->peak->peakCenterA, L3Info->peak->peakCenterB,
    				L3Info->cal->pix0_A, L3Info->cal->pix0_B,
    				L3Info->cal->ppmDiffA, L3Info->cal->ppmDiffB,star.c_str());
    		string line;
    		line.assign(ctmp);

			// Keep stats
			pkPPMpair = make_pair(L3Info->peak->peakCenterA,L3Info->cal->ppmDiffA);
			pkPPMvec.push_back(pkPPMpair);
			pkPPMpair = make_pair(L3Info->peak->peakCenterB,L3Info->cal->ppmDiffB);
			pkPPMvec.push_back(pkPPMpair);
    		if (QLINFOFILE) {
    			QuickLookLog << line << dfmsEOL;
    			qlCount++;
    			if (qlCount % QLPPAGE == 0) qlHeader();
    		}
    	}

		retStatus = -1;

   	}

	// Set the final CalID values
	if (gPFsuccessA) {
		L3Info->cal->calID4 = calID4A;
		L3Info->cal->calID5 = calID5A;
		L3Info->cal->calID6 = calID6A;
	} else if (gPFsuccessB) {
		L3Info->cal->calID4 = calID4B;
		L3Info->cal->calID5 = calID5B;
		L3Info->cal->calID6 = calID6B;
	} else {
		L3Info->cal->calID4 = "None";
		L3Info->cal->calID5 = "None";
		L3Info->cal->calID6 = "None";
	}

    // STEP 12.12.3.19.14 - Clear all verification peaks memory for next iteration
	delete[] mScale; mScale = 0;
	delete[] peakInd; peakInd = 0;
	delete[] minVal; minVal = 0;
	delete[] maxVal; maxVal = 0;
	delete[] tmp; tmp=0;

	return retStatus;
}

//
// -------------------------- calibNonGCUnonSLF --------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Calculates/Assigns the various calibration values pix0, ppmDiff,...
// for NON-GCU mode files;
//
// inputs:
//   gPF - pointer to Mass Calib object
//   pix0Fit - pointer to pix0 fit object
//   sMode - file mode
//   iRow  -  Side A=1, SideB=2
//
// returns:
//   1 for normal and 0 for error
// =============================================================================
// History: Written by Mike Rinaldi, May 2013
// =============================================================================
//
int DFMS_ProcessMCP_Class::calibNonGCUnonSLF(DFMS_PeakFit_Class *gPF,
				DFMS_X0FitFile_Class *x0Fit, DFMS_X0FitFile_Class *x0FitS,
				string sMode, int iRow) {

     string sFunctionName = "DFMS_ProcessMCP_Class::calibNonGCUnonSLF";

     int nSuccess=0;

     int calType=0;

     double pix0=0.0;
     double sPix0=BADDATA;
     double gcuPix0=0.0;
     double a=0.0, b=0.0;
     double x=0.0;
     double xUnc=0.0;
     double m = 0.0, m0 = 0.0;
     double ppmDiff=0.0, ppmDiffSelf=0.0, ppmDiffGCUavg=0.0;

     string descrip = "";

     bool gcuUsed = false;

     // Mass scale array
     double *mScale = new double[NUMPIX];

     // STEP 12.12.3.20.1 - Define the peak center location and uncertainty
     if (iRow == 1) {
    	 x = L3Info->peak->peakCenterA;
		 xUnc = L3Info->peak->peakCenterAUnc;
     } else if (iRow == 2) {
    	 x = L3Info->peak->peakCenterB;
		 xUnc = L3Info->peak->peakCenterBUnc;
     }

     // STEP 12.12.3.20.2 - Define the target mass from ROSINA_DFMS_SCI_MASS value in L2 HK data
     m0 = L2Info.commandedMass;
     L3Info->commandedMass = m0;

     gcuPix0FitFileName = x0Fit->getFileName();		// Latest GCU calibration fit file

     if (iRow == 1) {
    	 gcuPix0 = aGCUA + bGCUA*m0;
    	 if (!noSLFFitFile) {
    	     slfPix0FitFileName = x0FitS->getFileName();	// Latest SLF calibration fit file
    	     if (isRecentGCUavailable) {
    	     // Pre 1/3/2015
    	    	 if (iRow == 1) sPix0 = aSLFA + bGCUA*m0;
    	    	 if (iRow == 2) sPix0 = aSLFB + bGCUB*m0;
    	     } else {
    	    	 // Post 1/3/2015
    	    	 if (iRow == 1) sPix0 = aSLFA + bSLFA*m0;
    	    	 if (iRow == 2) sPix0 = aSLFB + bSLFB*m0;
    	     }
    	     L3Info->cal->GCUPix0_A_used = false;
    	     L3Info->cal->GCUPix0_UncA_used = false;
    	     L3Info->cal->selfPix0_A_used = true;
    	     L3Info->cal->selfPix0_UncA_used = true;
    	 } else {
    		 sPix0 = gcuPix0;		// No SLF file therefore use GCU
    		 L3Info->cal->GCUPix0_A_used = true;
    		 L3Info->cal->GCUPix0_UncA_used = true;
    		 L3Info->cal->selfPix0_A_used = false;
    		 L3Info->cal->selfPix0_UncA_used = false;
    	 }
     } else {
    	 gcuPix0 = aGCUB + bGCUB*m0;
    	 if (!noSLFFitFile) {
    	     slfPix0FitFileName = x0FitS->getFileName();	// Latest SLF calibration fit file
    	     if (isRecentGCUavailable) {
    	     // Pre 1/3/2015
    	    	 if (iRow == 1) sPix0 = aSLFA + bGCUA*m0;
    	    	 if (iRow == 2) sPix0 = aSLFB + bGCUB*m0;
    	     } else {
    	    	 // Post 1/3/2015
    	    	 if (iRow == 1) sPix0 = aSLFA + bSLFA*m0;
    	    	 if (iRow == 2) sPix0 = aSLFB + bSLFB*m0;
    	     }
    	     L3Info->cal->GCUPix0_B_used = false;
    	     L3Info->cal->GCUPix0_UncB_used = false;
    	     L3Info->cal->selfPix0_B_used = true;
    	     L3Info->cal->selfPix0_UncB_used = true;
    	 } else {
    		 sPix0 = gcuPix0;		// No SLF file therefore use GCU
    		 L3Info->cal->GCUPix0_B_used = true;
    		 L3Info->cal->GCUPix0_UncB_used = true;
    		 L3Info->cal->selfPix0_B_used = false;
    		 L3Info->cal->selfPix0_UncB_used = false;
    	 }
     }
	 pix0 = sPix0;

     // For unrealistic or out of bounds pix0 Warn and move on to next file.
     if (pix0 < PKLOWPT || pix0 > PKHIGHPT) {
    	 skipReas["pix0Bounds"]++;
    	 sInfoMessage="pix0 from file is < 20 or > 490... not realistic - skip file";
    	 writeToLog(sInfoMessage, sFunctionName, INFO);
    	 cout << "pix0 = " << pix0 << " Not within bounds" << endl;
         delete[] mScale; mScale = 0;
    	 return 0;
     }


	 // STEP 12.12.3.20.3 - Using pix0 from 2.13.5 create the mass scale
	 massScale(mScale, L2Info.zoom, pix0, m0, NUMPIX, false);
	// for (int jj = 200; jj<300; jj++) {
	//	 cout << "mScale[" << jj << "] = " << mScale[jj] << endl;
	//}

     // STEP 12.12.3.20.4 - Check if the CurveFit exited normally
     if (gPF->fitSucceeded) {

    	 // STEP 12.12.3.20.5 - Set the peak masses to L3Info struct
    	 if (iRow == 1) {
    		 gPFsuccessA = true;
    		 L3Info->peak->pkIDA = "Unknown";
    		 L3Info->pkMassA->num = numGoodPks;
    		 L3Info->pkMassA->pkFit = pkFI.maxPkLoc;
   			 L3Info->pkMassA->iXl = iXl;
   			 L3Info->pkMassA->iXr = iXr;
    		 for (int i=0; i<numGoodPks; i++) {
    			 L3Info->pkMassA->peakLoc[i] = pkFI.peakLoc[i];
   				 L3Info->pkMassA->peakVal[i] = YAin[pkFI.peakLoc[i]];
    			 L3Info->pkMassA->pkMass[i] = mScale[pkFI.peakLoc[i]];
    		 }
    		 // STEP 12.12.3.20.8 - Set the L3Info values needed to write L3 file
        	 if (L2Info.commandedMass == 0) {
        	 	setNonGCUNonSLFL3Info(iRow,pix0,gcuPix0,m0,m,x,xUnc,ppmDiffSelf,calType, x0Fit->sErrorA,
        	     		x0Fit->sErrorB);
        	 	calID4A = calID5A = calID6A = "None";
        	 } else {
        		 if (isRecentGCUavailable) {
        			 setNonGCUNonSLFL3Info(iRow,pix0,gcuPix0,m0,m,x,xUnc,ppmDiffSelf,calType, x0Fit->sErrorA,
        					 x0Fit->sErrorB);
        			 calID4A = gcuPix0FitFileName;
        			 calID5A = slfPix0FitFileName;
        			 calID6A = "None";
        		 } else {
        			 setNonGCUNonSLFL3Info(iRow,pix0,gcuPix0,m0,m,x,xUnc,ppmDiffSelf,calType, x0Fit->sErrorA,
        					 x0Fit->sErrorB);
        			 calID4A = "None";
        			 calID5A = slfPix0FitFileName;
        			 calID6A = "None";
        		 }
        	 }
    	 } else {
    		 gPFsuccessB = true;
    		 L3Info->peak->pkIDB = "Unknown";
    		 L3Info->pkMassB->num = numGoodPks;
    		 L3Info->pkMassB->pkFit = pkFI.maxPkLoc;
   			 L3Info->pkMassB->iXl = iXl;
   			 L3Info->pkMassB->iXr = iXr;
    		 for (int i=0; i<numGoodPks; i++) {
    			 L3Info->pkMassB->peakLoc[i] = pkFI.peakLoc[i];
   				 L3Info->pkMassB->peakVal[i] = YBin[pkFI.peakLoc[i]];
    			 L3Info->pkMassB->pkMass[i] = mScale[pkFI.peakLoc[i]];
 			}
    		 // STEP 12.12.3.20.8 - Set the L3Info values needed to write L3 file
        	 if (L2Info.commandedMass == 0) {
        	 	setNonGCUNonSLFL3Info(iRow,pix0,gcuPix0,m0,m,x,xUnc,ppmDiffSelf,calType, x0Fit->sErrorA,
        	     		x0Fit->sErrorB);
        	 	calID4B = calID5B = calID6B = "None";
        	 } else {
        		 if (isRecentGCUavailable) {
        			 setNonGCUNonSLFL3Info(iRow,pix0,gcuPix0,m0,m,x,xUnc,ppmDiffSelf,calType, x0Fit->sErrorA,
        					 x0Fit->sErrorB);
        			 calID4B = gcuPix0FitFileName;
        			 calID5B = slfPix0FitFileName;
        			 calID6B = "None";
        		 } else {
        			 setNonGCUNonSLFL3Info(iRow,pix0,gcuPix0,m0,m,x,xUnc,ppmDiffSelf,calType, x0Fit->sErrorA,
        					 x0Fit->sErrorB);
        			 calID4B = "None";
        			 calID5B = slfPix0FitFileName;
        			 calID6B = "None";
        		 }
        	 }
    	 }

    	 // STEP 12.12.3.20.6 - Using this mass scale find the mass at actual peak
    	 int pkPixel = (int)(x+0.5);  // Round up
    	 m = mScale[pkPixel];

    	 // STEP 12.12.3.20.7 - Calculate the ppm diff between the target mass and peak mass
    	 // Test if the ppm Diff is within acceptable value. Set Quality flags
    	 double mFit = pixToMass(L2Info.zoom, gcuPix0, pkPixel, m0);
    	 ppmDiffGCUavg = calcPPMDiff(m, mFit);

    	 // Use closest SLF calculated ppmdiff
    	 int numPts=0;
    	 ppmDiffSelf = setPPMDiff(iRow, &numPts);

		 // Define the quality Flag
    	 if (iRow == 1) {
    		  if (ppmDiffSelf != BADDATA && ppmDiffSelf <= PPMDIFFMIN) {
    			 L3Info->qualIDA = 0;
    		 } else if (ppmDiffGCUavg > PPMDIFFMIN && ppmDiffSelf < PPMDIFFMIN) {
    			 L3Info->qualIDA = 1;
    		 } else if (ppmDiffGCUavg >= PPMDIFFMIN || x0Fit->qualID > 1) {
    			 L3Info->qualIDA = 2;
    		 } else if (ppmDiffSelf == BADDATA || numPts < 2) {
    			 L3Info->qualIDA = 4;
    		 }
    	 } else {
    		 if (ppmDiffSelf != BADDATA && ppmDiffSelf <= PPMDIFFMIN) {
    			 L3Info->qualIDB = 0;
    		 } else if (ppmDiffGCUavg > PPMDIFFMIN && ppmDiffSelf < PPMDIFFMIN) {
    			 L3Info->qualIDB = 1;
    		 } else if (ppmDiffGCUavg >= PPMDIFFMIN || x0Fit->qualID > 1) {
    			 L3Info->qualIDB = 2;
    		 } else if (ppmDiffSelf == BADDATA || numPts < 2) {
    			 L3Info->qualIDB = 4;
    		 }
    	 }

     // STEP 12.12.3.20.9 - gPF->fitSucceeded = false - Bad or No peak fit
     } else {

		if (iRow == 1) {
    		gPFsuccessA = false;
	 		L3Info->cal->pix0_A = pix0;
	 		L3Info->cal->GCUPix0_A = gcuPix0;       // Use value from latest fit file
	 		if (sPix0 == BADDATA) {
	 			L3Info->cal->selfPix0_A = BADDATA;
	 		} else {
	 			L3Info->cal->selfPix0_A = sPix0;        // Use value from latest fit file
	 		}
		 	if (L2Info.hiRes) {
		 		// Change to fixed value as of 29/09/2014 (Meeting with Kathrin A.)
		 		L3Info->cal->GCUPix0_UncA = HRPIX0UNC;
		 		L3Info->cal->selfPix0_UncA = HRPIX0UNC;
		 	 } else {
		 		// The total error is the assumed 5 pixel error + the error on the GCU fit
		 		L3Info->cal->GCUPix0_UncA = x0Fit->sErrorA;
		 		L3Info->cal->selfPix0_UncA = sqrt((long double)pow(LRPIX0UNC,2.0) + pow(L3Info->cal->GCUPix0_UncA,2.0));
		   	}
		 	if (sPix0 == BADDATA) {
		 		L3Info->cal->selfPix0_UncA = BADDATA;
		 	}
			L3Info->cal->m0A = m0;
			L3Info->cal->ppmDiffA = BADDATA;
			L3Info->qualIDA = 4;
			L3Info->pkMassA->num = numGoodPks;
	    	L3Info->pkMassA->pkFit = pkFI.maxPkLoc;
  			L3Info->pkMassA->iXl = iXl;
  			L3Info->pkMassA->iXr = iXr;
	    	for (int i=0; i<numGoodPks; i++) {
	    		L3Info->pkMassA->peakLoc[i] = pkFI.peakLoc[i];
  				L3Info->pkMassA->peakVal[i] = YAin[pkFI.peakLoc[i]];
	    		L3Info->pkMassA->pkMass[i] = mScale[pkFI.peakLoc[i]];
	    	}
			L3Info->cal->GCUPix0_A_used = false;
			L3Info->cal->GCUPix0_UncA_used = false;
			L3Info->cal->selfPix0_A_used = false;
			L3Info->cal->selfPix0_UncA_used = false;
   			// set calID's
    		if (m0 > 0) {
    			if (isRecentGCUavailable) {
    				calID4A = gcuPix0FitFileName;
    			} else {
    				calID4A = "None";
    			}
				if (sPix0 == BADDATA) {
					calID5A = "None";
				} else {
					calID5A = slfPix0FitFileName;
				}
				calID6A = "None";
    		} else {
    			calID4A = "None";
    			calID5A = "None";
    			calID6A = "None";
    		}

		} else {
    		gPFsuccessB = false;
	 		L3Info->cal->pix0_B = pix0;
	 		L3Info->cal->GCUPix0_B = gcuPix0;       // Use value from latest fit file
	 		if (sPix0 == BADDATA) {
	 			L3Info->cal->selfPix0_B = BADDATA;
	 		} else {
	 			L3Info->cal->selfPix0_B = sPix0;       // Use value from latest fit file
	 		}
		 	if (L2Info.hiRes) {
		 		// Change to fixed value as of 29/09/2014 (Meeting with Kathrin A.)
		 		L3Info->cal->GCUPix0_UncB = HRPIX0UNC;
		 		L3Info->cal->selfPix0_UncB = HRPIX0UNC;
		 	 } else {
		 		// The total error is the assumed 5 pixel error + the error on the GCU fit
		 		L3Info->cal->GCUPix0_UncB = x0Fit->sErrorA;
		 		L3Info->cal->selfPix0_UncB = sqrt((long double)pow(LRPIX0UNC,2.0) + pow(L3Info->cal->GCUPix0_UncA,2.0));
		   	}
		 	if (sPix0 == BADDATA) {
		 		L3Info->cal->selfPix0_UncB = BADDATA;
		 	}
			L3Info->cal->m0B = m0;
   			L3Info->cal->ppmDiffB = BADDATA;
			L3Info->qualIDB = 4;
   			L3Info->pkMassB->num = numGoodPks;
   			L3Info->pkMassB->pkFit = pkFI.maxPkLoc;
  			L3Info->pkMassB->iXl = iXl;
  			L3Info->pkMassB->iXr = iXr;
   			for (int i=0; i<numGoodPks; i++) {
   				L3Info->pkMassB->peakLoc[i] = pkFI.peakLoc[i];
  				L3Info->pkMassB->peakVal[i] = YBin[pkFI.peakLoc[i]];
   				L3Info->pkMassB->pkMass[i] = mScale[pkFI.peakLoc[i]];
   			}
   			L3Info->cal->GCUPix0_B_used = false;
   			L3Info->cal->GCUPix0_UncB_used = false;
   			L3Info->cal->selfPix0_B_used = false;
   			L3Info->cal->selfPix0_UncB_used = false;
   			// Now be safe and take the lowest quality has the representative
   			if (L3Info->qualIDA >= L3Info->qualIDB) {
   				L3Info->qualID = L3Info->qualIDA;
   			} else {
   				L3Info->qualID = L3Info->qualIDB;
   			}
   			L3Info->cal->calType = 0;
   			L3Info->calRowUsed = mptIndx;

   			// Optionally write to the Quick Look log
   			string star=" ";
   			if (L3Info->cal->ppmDiffA > PPMDIFFMIN) star.assign("*/");
   			if (L3Info->cal->ppmDiffB > PPMDIFFMIN) star+=("*");
	 		if (infNanErr) star+=("/InfNaN/");
	 		infNanErr = false;
   			char ctmp[400];
   			sprintf(ctmp,"%7.2f     %7.2f / %7.2f    %7.2f / %7.2f     %8.2f / %8.2f  %3.3s",
   	 			 L3Info->commandedMass,
   	 			 L3Info->peak->peakCenterA, L3Info->peak->peakCenterB,
   	 			 L3Info->cal->pix0_A, L3Info->cal->pix0_B,
   	 			 L3Info->cal->ppmDiffA, L3Info->cal->ppmDiffB,star.c_str());
   			string line;
   			line.assign(ctmp);
   			// Keep stats
   			pkPPMpair = make_pair(L3Info->peak->peakCenterA,L3Info->cal->ppmDiffA);
   			pkPPMvec.push_back(pkPPMpair);
   			pkPPMpair = make_pair(L3Info->peak->peakCenterB,L3Info->cal->ppmDiffB);
   			pkPPMvec.push_back(pkPPMpair);
   			if (QLINFOFILE) {
   				QuickLookLog << line << dfmsEOL;
   				qlCount++;
   				if (qlCount % QLPPAGE == 0) qlHeader();
   			}
   			// set calID's
    		if (m0 > 0) {
    			if (isRecentGCUavailable) {
    				calID4B = gcuPix0FitFileName;
    			} else {
    				calID4B = "None";
    			}
				if (sPix0 == BADDATA) {
					calID5B = "None";
				} else {
					calID5B = slfPix0FitFileName;
				}
				calID6B = "None";
    		} else {
    			calID4B = "None";
    			calID5B = "None";
    			calID6B = "None";
    		}
		}

	}

 	// Set the final CalID values
 	if (gPFsuccessA) {
 		L3Info->cal->calID4 = calID4A;
 		L3Info->cal->calID5 = calID5A;
 		L3Info->cal->calID6 = calID6A;
 	} else if (gPFsuccessB) {
 		L3Info->cal->calID4 = calID4B;
 		L3Info->cal->calID5 = calID5B;
 		L3Info->cal->calID6 = calID6B;
 	} else {
 		L3Info->cal->calID4 = "None";
 		L3Info->cal->calID5 = "None";
 		L3Info->cal->calID6 = "None";
 	}

    // STEP 12.12.3.20.10 - Release memory
	delete[] mScale; mScale = 0;

	return 1;
}

//
// --------------------------- setPPMDiff ----------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method calculates the best ppmDiff based on the values from the SLF
// ppmDiff.  Choose closest in time then mode
//
// inputs:
//   iRow     - side A/B
//
// returns:
//   best ppmDiff
// =============================================================================
// History: Written by Mike Rinaldi, Feb 2014
// =============================================================================
//
double DFMS_ProcessMCP_Class::setPPMDiff(int iRow, int *num) {

	if (iRow == 1) {
		// Row 1 - Low Res - Low Mass
		if (L2Info.lowRes) {
			if (L2Info.commandedMass < SLFLOWCUTOFF) {
				if (ppmSLFLmLrA.size() > 0) {
					*num = ppmSLFLmLrA.size();
					for (int i=0; i<ppmSLFLmLrA.size()-1; i++) {
						if (L2Info.secSince1970 > ppmSLFLmLrA[i].timeTag &&
							L2Info.secSince1970 < ppmSLFLmLrA[i+1].timeTag ) {
							return ppmSLFLmLrA[i].ppmDiff;
						}
						if (i == ppmSLFLmLrA.size()-2) {
							return ppmSLFLmLrA[i+1].ppmDiff;
						}
					}
				} else {
					return BADDATA;
				}
			// Row 1 - Low Res - High Mass
			} else {
				if (ppmSLFHmLrA.size() > 0) {
					*num = ppmSLFHmLrA.size();
					for (int i=0; i<ppmSLFHmLrA.size()-1; i++) {
						if (L2Info.secSince1970 > ppmSLFHmLrA[i].timeTag &&
							L2Info.secSince1970 < ppmSLFHmLrA[i+1].timeTag ) {
							return ppmSLFHmLrA[i].ppmDiff;
						}
						if (i == ppmSLFHmLrA.size()-2) {
							return ppmSLFHmLrA[i+1].ppmDiff;
						}
					}
				} else {
					return BADDATA;
				}
			}
		// Row 1 - High Res - Low Mass
		} else if (L2Info.hiRes) {
			if (L2Info.commandedMass < SLFLOWCUTOFF) {
				if (ppmSLFLmHrA.size() > 0) {
					*num = ppmSLFLmHrA.size();
					for (int i=0; i<ppmSLFLmHrA.size()-1; i++) {
						if (L2Info.secSince1970 > ppmSLFLmHrA[i].timeTag &&
							L2Info.secSince1970 < ppmSLFLmHrA[i+1].timeTag ) {
							return ppmSLFLmHrA[i].ppmDiff;
						}
						if (i == ppmSLFLmHrA.size()-2) {
							return ppmSLFLmHrA[i+1].ppmDiff;
						}
					}
				} else {
					return BADDATA;
				}
			// Row 1 - High Res - Medium Mass
			} else if (L2Info.commandedMass >= SLFLOWCUTOFF && L2Info.commandedMass < SLFHICUTOFF) {
				if (ppmSLFMmHrA.size() > 0) {
					*num = ppmSLFMmHrA.size();
					for (int i=0; i<ppmSLFMmHrA.size()-1; i++) {
						if (L2Info.secSince1970 > ppmSLFMmHrA[i].timeTag &&
							L2Info.secSince1970 < ppmSLFMmHrA[i+1].timeTag ) {
							return ppmSLFMmHrA[i].ppmDiff;
						}
						if (i == ppmSLFMmHrA.size()-2) {
							return ppmSLFMmHrA[i+1].ppmDiff;
						}
					}
				} else {
					return BADDATA;
				}
			// Row 1 - High Res - High Mass
			} else {
				if (ppmSLFHmHrA.size() > 0) {
					*num = ppmSLFHmHrA.size();
					for (int i=0; i<ppmSLFHmHrA.size()-1; i++) {
						if (L2Info.secSince1970 > ppmSLFHmHrA[i].timeTag &&
							L2Info.secSince1970 < ppmSLFHmHrA[i+1].timeTag ) {
							return ppmSLFHmHrA[i].ppmDiff;
						}
						if (i == ppmSLFHmHrA.size()-2) {
							return ppmSLFHmHrA[i+1].ppmDiff;
						}
					}
				} else {
					return BADDATA;
				}
			}
		}

	} else {
		// Row 2 - Low Res - Low Mass
		if (L2Info.lowRes) {
			if (L2Info.commandedMass < SLFLOWCUTOFF) {
				if (ppmSLFLmLrB.size() > 0) {
					*num = ppmSLFLmLrB.size();
					for (int i=0; i<ppmSLFLmLrB.size()-1; i++) {
						if (L2Info.secSince1970 > ppmSLFLmLrB[i].timeTag &&
							L2Info.secSince1970 < ppmSLFLmLrB[i+1].timeTag ) {
							return ppmSLFLmLrB[i].ppmDiff;
						}
						if (i == ppmSLFLmLrB.size()-2) {
							return ppmSLFLmLrB[i+1].ppmDiff;
						}
					}
				} else {
					return BADDATA;
				}
			// Row 2 - Low Res - High Mass
			} else {
				if (ppmSLFHmLrB.size() > 0) {
					*num = ppmSLFHmLrB.size();
					for (int i=0; i<ppmSLFHmLrB.size()-1; i++) {
						if (L2Info.secSince1970 > ppmSLFHmLrB[i].timeTag &&
							L2Info.secSince1970 < ppmSLFHmLrB[i+1].timeTag ) {
							return ppmSLFHmLrB[i].ppmDiff;
						}
						if (i == ppmSLFHmLrB.size()-2) {
							return ppmSLFHmLrB[i+1].ppmDiff;
						}
					}
				} else {
					return BADDATA;
				}
			}
		// Row 2 - High Res - Low Mass
		} else if (L2Info.hiRes) {
			if (L2Info.commandedMass < SLFLOWCUTOFF) {
				if (ppmSLFLmHrB.size() > 0) {
					*num = ppmSLFLmHrB.size();
					for (int i=0; i<ppmSLFLmHrB.size()-1; i++) {
						if (L2Info.secSince1970 > ppmSLFLmHrB[i].timeTag &&
							L2Info.secSince1970 < ppmSLFLmHrB[i+1].timeTag ) {
							return ppmSLFLmHrB[i].ppmDiff;
						}
						if (i == ppmSLFLmHrB.size()-2) {
							return ppmSLFLmHrB[i+1].ppmDiff;
						}
					}
				} else {
					return BADDATA;
				}
			// Row 2 - High Res - Medium Mass
			} else if (L2Info.commandedMass >= SLFLOWCUTOFF && L2Info.commandedMass < SLFHICUTOFF) {
				if (ppmSLFMmHrB.size() > 0) {
					*num = ppmSLFMmHrB.size();
					for (int i=0; i<ppmSLFMmHrB.size()-1; i++) {
						if (L2Info.secSince1970 > ppmSLFMmHrB[i].timeTag &&
							L2Info.secSince1970 < ppmSLFMmHrB[i+1].timeTag ) {
							return ppmSLFMmHrB[i].ppmDiff;
						}
						if (i == ppmSLFMmHrB.size()-2) {
							return ppmSLFMmHrB[i+1].ppmDiff;
						}
					}
				} else {
					return BADDATA;
				}
			// Row 2 - High Res - High Mass
			} else {
				if (ppmSLFHmHrB.size() > 0) {
					*num = ppmSLFHmHrB.size();
					for (int i=0; i<ppmSLFHmHrB.size()-1; i++) {
						if (L2Info.secSince1970 > ppmSLFHmHrB[i].timeTag &&
							L2Info.secSince1970 < ppmSLFHmHrB[i+1].timeTag ) {
							return ppmSLFHmHrB[i].ppmDiff;
						}
						if (i == ppmSLFHmHrB.size()-2) {
							return ppmSLFHmHrB[i+1].ppmDiff;
						}
					}
				} else {
					return BADDATA;
				}
			}
		}
	}

	return BADDATA;

}

//
// --------------------------- setNonGCUVerifL3Info ----------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Set the needed L3 values that will be written to the final L3 file
// This method gathers many of the needed L3 values from the various structures
// and adds them to the final L3 data object L3Info.
//
// inputs:
//   iRow     - side A/B
//   pix0      - calibration param pix0
//   gcuPix0   - if calibration param pix0 came from a GCU fit file
//   m0        - target mass
//   m         - calculated or theoretical mass
//   x         - peak center location
//   xUnc      - peak center uncertainty
//   ppmDiff   - deviation in ppm between m/m0
//   peak      - specific verification peak info
//   calType   - 0 - self calibration  1 - Verification peak calib
//   calID4    - source 1 of calibration
//   calID5    - source 2 of calibration
//   calID6    - source 3 of calibration
//
// returns:
//   None
// =============================================================================
// History: Written by Mike Rinaldi, Feb 2014
// =============================================================================
//

void DFMS_ProcessMCP_Class::setSLFL3Info(int iRow, double pix0,
				    double gcuPix0, double m0, double m, double x, double xUnc,
				    double ppmDiff, verifPeakInfo *peak, int calType,
				    double gcuPix0UncA, double gcuPix0UncB) {

	// set the peak name from peak Verification info

	double pUncTot=0.0;

	if (iRow == 1) {
		L3Info->peak->pkIDA = peak->peakNameA;
		L3Info->cal->m0A = m0;
		L3Info->cal->ppmDiffA = ppmDiff;
 		if (pix0 <= 0.0) {
 			L3Info->cal->pix0_A = gcuPix0;
 		} else {
 			L3Info->cal->pix0_A = pix0;       // Use value from latest fit file
 		}

 		// Set Pix0/Unc values for LR/HR
 		L3Info->cal->GCUPix0_A = gcuPix0;
 		L3Info->cal->selfPix0_A = pix0;
   		if (L2Info.hiRes) {
   			// Change to fixed value as of 29/09/2014 (Meeting with Kathrin A.)
   	 		L3Info->cal->GCUPix0_UncA = HRPIX0UNC;
   			L3Info->cal->selfPix0_UncA = HRPIX0UNC;
   		} else {
   	 		// The total error is the assumed 5 pixel error + the error on the GCU fit
   	 		L3Info->cal->GCUPix0_UncA = gcuPix0UncA;
   	 		L3Info->cal->selfPix0_UncA = sqrt((long double)pow(LRPIX0UNC,2.0) + pow(L3Info->cal->GCUPix0_UncA,2.0));
   		}
 		if (calType == 1 && peak->index >= 0) {
 			L3Info->vPeaks[peak->index].peakNameA = peak->peakNameA;
 			L3Info->vPeaks[peak->index].caltype = calType;
 			L3Info->vPeaks[peak->index].foundA = 1;
 			L3Info->vPeaks[peak->index].peakCntrA = x;
 			L3Info->vPeaks[peak->index].peakWidthA = L3Info->peak->peakCenterWidthA;
 			L3Info->vPeaks[peak->index].peakHeightA = L3Info->peak->peakHeightA;
 			L3Info->vPeaks[peak->index].ppmDiffA = ppmDiff;
 		}
	} else if (iRow == 2) {
        L3Info->peak->pkIDB = peak->peakNameB;
		L3Info->cal->m0B = m0;
		L3Info->cal->ppmDiffB = ppmDiff;
 		if (pix0 <= 0.0) {
 			L3Info->cal->pix0_B = gcuPix0;
 		} else {
 			L3Info->cal->pix0_B = pix0;       // Use value from latest fit file
 		}

 		// Set Pix0/Unc values for LR/HR
 		L3Info->cal->GCUPix0_B = gcuPix0;
 		L3Info->cal->selfPix0_B = pix0;
   		if (L2Info.hiRes) {
   			// Change to fixed value as of 29/09/2014 (Meeting with Kathrin A.)
   	 		L3Info->cal->GCUPix0_UncB = HRPIX0UNC;
   			L3Info->cal->selfPix0_UncB = HRPIX0UNC;
   		} else {
   	 		// The total error is the assumed 5 pixel error + the error on the GCU fit
   	 		L3Info->cal->GCUPix0_UncB = gcuPix0UncB;
   	 		L3Info->cal->selfPix0_UncB = sqrt((long double)pow(LRPIX0UNC,2.0) + pow(L3Info->cal->GCUPix0_UncB,2.0));
   		}

		// Items independent of DFMS Side
		L3Info->cal->calType = calType;
		if (calType == 1 && peak->index >= 0) {
			L3Info->vPeaks[peak->index].peakNameB = peak->peakNameB;
			L3Info->vPeaks[peak->index].caltype = calType;
			L3Info->vPeaks[peak->index].foundB = 1;
			L3Info->vPeaks[peak->index].peakCntrB = x;
			L3Info->vPeaks[peak->index].peakWidthB = L3Info->peak->peakCenterWidthB;
			L3Info->vPeaks[peak->index].peakHeightB = L3Info->peak->peakHeightB;
			L3Info->vPeaks[peak->index].ppmDiffB = ppmDiff;
		}

		// Now be safe and take the lowest quality between side A/b
		// reasonable has the representative
		if (L3Info->qualIDA >= L3Info->qualIDB) {
			L3Info->qualID = L3Info->qualIDA;
		} else {
			L3Info->qualID = L3Info->qualIDB;
		}

		if (calType == 1 && peak->index >= 0) {
			// Write to the Quick Look log
			string star=" ";
			if (L3Info->cal->ppmDiffA > PPMDIFFMIN) star.assign("*/");
			if (L3Info->cal->ppmDiffB > PPMDIFFMIN) star+=("*");
	 		if (infNanErr) star+=("/InfNaN/");
	 		infNanErr = false;
			char tmp[400];
			sprintf(tmp,"%7.2f     %7.2f / %7.2f    %7.2f / %7.2f     %8.2f / %8.2f      %s  %3.3s",
					 L3Info->commandedMass,
					 L3Info->peak->peakCenterA, L3Info->peak->peakCenterB,
					 L3Info->cal->pix0_A, L3Info->cal->pix0_B,
					 L3Info->cal->ppmDiffA, L3Info->cal->ppmDiffB,
					 L3Info->vPeaks[peak->index].peakNameB.c_str(),star.c_str());
			string line;
			line.assign(tmp);
			// Keep stats
			pkPPMpair = make_pair(L3Info->peak->peakCenterA,L3Info->cal->ppmDiffA);
			pkPPMvec.push_back(pkPPMpair);
			pkPPMpair = make_pair(L3Info->peak->peakCenterB,L3Info->cal->ppmDiffB);
			pkPPMvec.push_back(pkPPMpair);
			if (QLINFOFILE) {
				QuickLookLog << line << dfmsEOL;
				qlCount++;
				if (qlCount % QLPPAGE == 0) qlHeader();
			}
		} else {
			// Write to the Quick Look log
			string star=" ";
			if (L3Info->cal->ppmDiffA > PPMDIFFMIN) star.assign("*/");
			if (L3Info->cal->ppmDiffB > PPMDIFFMIN) star+=("*");
	 		if (infNanErr) star+=("/InfNaN/");
	 		infNanErr = false;
			char tmp[400];
			sprintf(tmp,"%7.2f     %7.2f / %7.2f    %7.2f / %7.2f     %8.2f / %8.2f  %3.3s",
					 L3Info->commandedMass,
					 L3Info->peak->peakCenterA, L3Info->peak->peakCenterB,
					 L3Info->cal->pix0_A, L3Info->cal->pix0_B,
					 L3Info->cal->ppmDiffA, L3Info->cal->ppmDiffB,star.c_str());
			string line;
			line.assign(tmp);
			// Keep stats
			pkPPMpair = make_pair(L3Info->peak->peakCenterA,L3Info->cal->ppmDiffA);
			pkPPMvec.push_back(pkPPMpair);
			pkPPMpair = make_pair(L3Info->peak->peakCenterB,L3Info->cal->ppmDiffB);
			pkPPMvec.push_back(pkPPMpair);
			if (QLINFOFILE) {
				QuickLookLog << line << dfmsEOL;
				qlCount++;
				if (qlCount % QLPPAGE == 0) qlHeader();
			}
		}
	}

}

//
// --------------------------- setNonGCUSelfL3Info ----------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Set the needed L3 values that will be written to the final L3 file
// This method gathers many of the needed L3 values from the various structures
// and adds them to the final L3 data object L3Info.
//
// inputs:
//   iRow     - side A/B
//   pix0      - calibration param pix0
//   gcuPix0   - if calibration param pix0 came from a GCU fit file
//   m0        - target mass
//   m         - calculated or theoretical mass
//   x         - peak center location
//   xUnc      - peak center uncertainty
//   ppmDiff   - deviation in ppm between m/m0
//   peak      - specific verification peak info
//   calType   - 0 - self calibration  1 - Verification peak calib
//   calID4    - source 1 of calibration
//   calID5    - source 2 of calibration
//   calID6    - source 3 of calibration
//
// returns:
//   None
// =============================================================================
// History: Written by Mike Rinaldi, Feb 2014
// =============================================================================
//

void DFMS_ProcessMCP_Class::setNonGCUNonSLFL3Info(int iRow, double pix0,
				    double gcuPix0, double m0, double m, double x, double xUnc,
				    double ppmDiff, int calType, double gcuPix0UncA,
				    double gcuPix0UncB) {

	 if (iRow == 1) {
		 L3Info->cal->m0A = m0;
		 L3Info->cal->ppmDiffA = ppmDiff;
		 if (pix0 <= 0.0) {
			 L3Info->cal->pix0_A = gcuPix0;
		 } else {
			 L3Info->cal->pix0_A = pix0;         // Use value from latest fit file
		 }

	 	 // Set pix0/Unc for LR/HR
		 L3Info->cal->GCUPix0_A = gcuPix0;
	 	 L3Info->cal->selfPix0_A = pix0;
	 	 if (L2Info.hiRes) {
	 		 // Change to fixed value as of 29/09/2014 (Meeting with Kathrin A.)
	 		 L3Info->cal->GCUPix0_UncA = HRPIX0UNC;
	 		 L3Info->cal->selfPix0_UncA = HRPIX0UNC;
	 	 } else {
	 		 // The total error is the assumed 5 pixel error + the error on the GCU fit
	 		 L3Info->cal->GCUPix0_UncA = gcuPix0UncA;
	 		 L3Info->cal->selfPix0_UncA = sqrt((long double)pow(LRPIX0UNC,2.0) + pow(L3Info->cal->GCUPix0_UncA,2.0));
	   	}
	 } else if (iRow == 2) {
		 L3Info->cal->m0B = m0;
		 L3Info->cal->ppmDiffB = ppmDiff;
		 if (pix0 <= 0.0) {
			 L3Info->cal->pix0_B = gcuPix0;
		 } else {
			 L3Info->cal->pix0_B = pix0;         // Use value from latest fit file
		 }
	 	 // Set pix0/Unc for LR/HR
	 	 L3Info->cal->GCUPix0_B = gcuPix0;
		 L3Info->cal->selfPix0_B = pix0;
	 	 if (L2Info.hiRes) {
	 		 // Change to fixed value as of 29/09/2014 (Meeting with Kathrin A.)
	 		 L3Info->cal->GCUPix0_UncB = HRPIX0UNC;
	 		 L3Info->cal->selfPix0_UncB = HRPIX0UNC;
	 	 } else {
	 		 // The total error is the assumed 5 pixel error + the error on the GCU fit
	 		 L3Info->cal->GCUPix0_UncB = gcuPix0UncB;
	 		 L3Info->cal->selfPix0_UncB = sqrt((long double)pow(LRPIX0UNC,2.0) + pow(L3Info->cal->GCUPix0_UncB,2.0));
	   	}

		 // Items independent of DFMS Side
		 L3Info->cal->calType = calType;
		 // Now be safe and take the lowest quality between side A/b
		 // reasonable has the representative
		 if (L3Info->qualIDA >= L3Info->qualIDB) {
			 L3Info->qualID = L3Info->qualIDA;
		 } else {
			 L3Info->qualID = L3Info->qualIDB;
		 }

		 if (calType == 1) {
			 // Write to the Quick Look log
			 string star=" ";
			 if (L3Info->cal->ppmDiffA > PPMDIFFMIN) star.assign("*/");
			 if (L3Info->cal->ppmDiffB > PPMDIFFMIN) star+=("*");
		 	 if (infNanErr) star+=("/InfNaN/");
		 	 infNanErr = false;
			 char tmp[400];
			 sprintf(tmp,"%7.2f     %7.2f / %7.2f    %7.2f / %7.2f     %8.2f / %8.2f      %s",
					 L3Info->commandedMass,
					 L3Info->peak->peakCenterA, L3Info->peak->peakCenterB,
					 L3Info->cal->pix0_A, L3Info->cal->pix0_B,
					 L3Info->cal->ppmDiffA, L3Info->cal->ppmDiffB);
			 string line;
			 line.assign(tmp);
			 // Keep stats
			 pkPPMpair = make_pair(L3Info->peak->peakCenterA,L3Info->cal->ppmDiffA);
			 pkPPMvec.push_back(pkPPMpair);
			 pkPPMpair = make_pair(L3Info->peak->peakCenterB,L3Info->cal->ppmDiffB);
			 pkPPMvec.push_back(pkPPMpair);
			 if (QLINFOFILE) {
				 QuickLookLog << line << dfmsEOL;
				 qlCount++;
				 if (qlCount % QLPPAGE == 0) qlHeader();
			 }
		 } else {
			 // Write to the Quick Look log
			 string star=" ";
			 if (L3Info->cal->ppmDiffA > PPMDIFFMIN) star.assign("*/");
			 if (L3Info->cal->ppmDiffB > PPMDIFFMIN) star+=("*");
		 	 if (infNanErr) star+=("/InfNaN/");
		 	 infNanErr = false;
			 char tmp[400];
			 sprintf(tmp,"%7.2f     %7.2f / %7.2f    %7.2f / %7.2f     %8.2f / %8.2f  %3.3s",
					 L3Info->commandedMass,
					 L3Info->peak->peakCenterA, L3Info->peak->peakCenterB,
					 L3Info->cal->pix0_A, L3Info->cal->pix0_B,
					 L3Info->cal->ppmDiffA, L3Info->cal->ppmDiffB,star.c_str());
			 string line;
			 line.assign(tmp);
			 // Keep stats
			 pkPPMpair = make_pair(L3Info->peak->peakCenterA,L3Info->cal->ppmDiffA);
			 pkPPMvec.push_back(pkPPMpair);
			 pkPPMpair = make_pair(L3Info->peak->peakCenterB,L3Info->cal->ppmDiffB);
			 pkPPMvec.push_back(pkPPMpair);
			 if (QLINFOFILE) {
				 QuickLookLog << line << dfmsEOL;
				 qlCount++;
				 if (qlCount % QLPPAGE == 0) qlHeader();
			 }
		 }
	 }

}

//
// -------------------------- peakCenterMass ----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Find the best mass for the peak center based on the peakCenter pixel value.
// Uses simple linear interpolation
//
// inputs:
//   L2Info - Level 2 file info
//   pix0 - pix0 calibration value
//   x - Peak center location
//   m - mass at peak center
//
// returns:
//   mpc - peak center mass
// =============================================================================
// History: Written by Mike Rinaldi, June 2013
// =============================================================================
//
double DFMS_ProcessMCP_Class::peakCenterMass(L2FileInfo &L2Info, double pix0,
													double x, double m) {

	string sFunctionName = "DFMS_ProcessMCP_Class::peakCenterMass";

	double mpc = 0.0;
    // create mass scale array
    double *ms = new double[NUMPIX];

    for (int i=1; i<NUMPIX; ++i) {
		ms[i] = pixToMass(L2Info.zoom, pix0, (double)i, m);
    	if (x > i-1 && x <= i && i > 0) {
    		mpc = ms[i-1] + (ms[i]-ms[i-1])*(x-(i-1));
    		break;
    	}
    }

    delete ms; ms=0;

	return mpc;
}

//
// ----------------------- assignDefaultFitParams ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Set the base level L3Info data object values for the case where the fit is
// not possible or failed
//
// inputs:
//   iRow - DFMS side
//   gPF - pointer to Mass Calib object
//
// returns:
//   None
// =============================================================================
// History: Written by Mike Rinaldi, October 2013
// =============================================================================
//
void DFMS_ProcessMCP_Class::assignDefaultFitParams(int iRow, DFMS_PeakFit_Class *gPF) {

	string sFunctionName = "DFMS_ProcessMCP_Class::assignDefaultFitParams";

	if (iRow == 1) {
		L3Info->peak->pkIDA = "Unknown";
		L3Info->peak->peakFoundA = 0;
		L3Info->peak->peakHeightA = BADDATA;
		L3Info->peak->peakCenterA = BADDATA;
		L3Info->peak->peakCenterAUnc = BADDATA;
		L3Info->peak->peakCenterWidthA = BADDATA;
		//L3Info->offset->offsetLevelA = meanOffset;
        L3Info->offset->offsetLevelA = c0OffsetA;
		L3Info->offset->offsetLevelAStdDev = meanOffsetStdDev;
		L3Info->pkMassA->iXl = iXl;
		L3Info->pkMassA->iXr = iXr;
		L3Info->cal->selfPix0_A = BADDATA;
		L3Info->cal->selfPix0_UncA = BADDATA;
	} else {
		L3Info->peak->pkIDB = "Unknown";
		L3Info->peak->peakFoundB = 0;
		L3Info->peak->peakHeightB = BADDATA;
		L3Info->peak->peakCenterB = BADDATA;
		L3Info->peak->peakCenterBUnc = BADDATA;
		L3Info->peak->peakCenterWidthB = BADDATA;
        //L3Info->offset->offsetLevelB = meanOffset;
        L3Info->offset->offsetLevelB = c0OffsetB;
        L3Info->offset->offsetLevelBStdDev = meanOffsetStdDev;
		L3Info->pkMassB->iXl = iXl;
		L3Info->pkMassB->iXr = iXr;
		L3Info->cal->selfPix0_B = BADDATA;
		L3Info->cal->selfPix0_UncB = BADDATA;
		if (L3Info->peak->peakCenterA != BADDATA && L3Info->peak->peakCenterB == BADDATA) {
			L3Info->fitType = "CurveFit";
		} else if (L3Info->peak->peakCenterA == BADDATA && L3Info->peak->peakCenterB != BADDATA) {
			L3Info->fitType = "CurveFit";
		} else if (numGoodPksA == 0 && numGoodPksB != 0) {
			L3Info->fitType = "No Useful Peaks Row A";
		} else if (numGoodPksA != 0 && numGoodPksB == 0) {
			L3Info->fitType = "No Useful Peaks Row B";
		} else if (numGoodPksA == 0 && numGoodPksB == 0) {
					L3Info->fitType = "No Useful Peaks Row A/B";
		} else {
			L3Info->fitType = "Fit Unsuccessful";
		}
	}


}


//
// ----------------------- assignFitParams ----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Assigns the fit resutls to the L3Info structure.
//
// inputs: 
//   gPF - pointer to the mass Cal fit object
//   iRow - DFMS side A/B
//
// returns:
//   1 if success, 0 if failure
// =============================================================================
// History: Written by Mike Rinaldi, June 2013
// =============================================================================
//
int DFMS_ProcessMCP_Class::assignFitParams(DFMS_PeakFit_Class* gPF, int iRow) {

	string sFunctionName = "DFMS_ProcessMCP_Class::assignFitParams";

    // 2.7.1 - Make relevant assignments if Curve fit is best
    // set useFitRes to TRUE... later remove PeakFinder logic
   	L3Info->fitType = "CurveFit";	// Use CurveFit results
    // Find the errors of fit coeff at the peak
 	//gPF->calcParamErrors(iRow);
   	if (iRow == 1 ) {
   		// if GCU mode then use the MPST value
   		if (L2Info.isGCU) L3Info->peak->pkIDA = util_trimChar(MPSTdata[mptIndx][0],"\"");
   		L3Info->peak->peakFoundA = 1;
   		L3Info->peak->peakHeightA = gPF->coeff[0];   			// This uses the peak fit results
   		L3Info->peak->peakCenterA = gPF->coeff[1];   			// This uses the peak fit results
   		L3Info->actualPkHghtA = gPF->yAin[gPF->yAmaxId];  	// This uses the actual data
   		//L3Info->peak->peakCenterA = p->nPeakCenter;
   		if (L3Info->peak->peakCenterA < PKLOWPT || L3Info->peak->peakCenterA > PKHIGHPT) {
   			if(!CONVERTOUTBPEAK) {
                sideAexceptionType["peakOutOfBounds"]++;
   				sErrorMessage = "Peak Location "+util_intToString(L3Info->peak->peakCenterA);
   				sErrorMessage += " is out of Bounds";
   				writeToLog(sErrorMessage, sFunctionName, ERROR);
   				if (QLINFOFILE) {
   					QuickLookLog << " - Skipped - Peak Location: " <<  L3Info->peak->peakCenterA << " is out of Bounds" << dfmsEOL;
   					qlCount++;
   					if (qlCount % QLPPAGE == 0) qlHeader();
   				}
   				cout << " - Skipped - Peak Location: " <<  L3Info->peak->peakCenterA << " is out of Bounds" << endl;
   				return 0;
   			} else {
   				sInfoMessage = "Alert: Peak is outside of bounds: "+util_doubleToString(L3Info->peak->peakCenterA);
   				writeToLog(sErrorMessage, sFunctionName, WARN);
   				L3Info->Pix0Calc = false;
   			}
   		}
   		L3Info->peak->peakCenterAUnc = gPF->errCoeff[1];
   		L3Info->peak->peakCenterWidthA = gPF->coeff[2];
   		//L3Info->offset->offsetLevelA = meanOffset;
        L3Info->offset->offsetLevelA = c0OffsetA;
   		L3Info->offset->offsetLevelAStdDev = meanOffsetStdDev;
   	} else {
   		// if GCU mode then use the MPST value
   		if (L2Info.isGCU) L3Info->peak->pkIDB = util_trimChar(MPSTdata[mptIndx][0],"\"");
   		L3Info->peak->peakFoundB = 1;
   		L3Info->peak->peakHeightB = gPF->coeff[0];   			// This uses the peak fit results
   		L3Info->peak->peakCenterB = gPF->coeff[1];   			// This uses the peak fit results
   		L3Info->actualPkHghtB = gPF->yBin[gPF->yBmaxId];   // This uses the actual data
   		//L3Info->peak->peakCenterB = p->nPeakCenter;
   		if (L3Info->peak->peakCenterB < PKLOWPT || L3Info->peak->peakCenterB > PKHIGHPT) {
   			if(!CONVERTOUTBPEAK) {
                sideBexceptionType["peakOutOfBounds"]++;
   				sErrorMessage = "Peak Location "+util_intToString(L3Info->peak->peakCenterB);
   				sErrorMessage += " is out of Bounds";
   				writeToLog(sErrorMessage, sFunctionName, ERROR);
   				if (QLINFOFILE) {
   					QuickLookLog << " - Skipped - Peak Location: " <<  L3Info->peak->peakCenterB << " is out of Bounds" << dfmsEOL;
   					qlCount++;
   					if (qlCount % QLPPAGE == 0) qlHeader();
   				}
   				cout << " - Skipped - Peak Location: " <<  L3Info->peak->peakCenterB << " is out of Bounds" << endl;
   				return 0;
   			} else {
   				sInfoMessage = "Alert: Peak is outside of bounds: "+util_doubleToString(L3Info->peak->peakCenterB);
   				writeToLog(sErrorMessage, sFunctionName, WARN);
   				L3Info->Pix0Calc = false;
            }
   		}
   		L3Info->peak->peakCenterBUnc = gPF->errCoeff[1];
   		L3Info->peak->peakCenterWidthB = gPF->coeff[2];
   		//L3Info->offset->offsetLevelB = meanOffset;
        L3Info->offset->offsetLevelB = c0OffsetB;
   		L3Info->offset->offsetLevelBStdDev = meanOffsetStdDev;
   	}

    return 1;

} 

//
// ------------------------- calcPixUnc -------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Calculate the uncertainty in pix0 using the uncertainty in the peak center value
//
// inputs: 
//   x  - peakCenter
//   m  - Verification Mass
//   m0 - target Mass
//   xUnc - Uncertainty in peak Center
//
// returns:
//   pix0 uncertainty
// =============================================================================
// History: Written by Mike Rinaldi, June 2013
// =============================================================================
//
double DFMS_ProcessMCP_Class::calcPixUnc(double x, double m, double m0,double xUnc) {
    
    double pix0l = massToPix0(L2Info.zoom,x-xUnc,m,m0);
    double pix0h = massToPix0(L2Info.zoom,x+xUnc,m,m0);
    return abs(pix0h-pix0l);

}

//
// ------------------------ findAvgPixelGain -----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Calculate the average of the pixel Gain between pixels 20-490
//
// inputs:
//   None
//
// returns:
//   average pixel gain
// =============================================================================
// History: Written by Mike Rinaldi, October 2013
// =============================================================================
//
double DFMS_ProcessMCP_Class::findAvgPixelGain(int iRow) {

	double sum=0.0;

	int high = PKHIGHPT;
	int low = PKLOWPT;
	int span = high - low;
	if (iRow == 1) {
		for (int i=low; i<high; ++i) {
			sum += pixG[i][3];
		}
	} else {
		for (int i=20; i<490; ++i) {
			sum += pixG[i][4];
		}
	}

    return sum/span;
}

//
// ------------------------------ quickPlotL2Data ---------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method writes the L2 data to a gnuplot file for a quick view.
//
// inputs:
//   iRow  - DFMS side
//   path  - Path to output data
//   fileName - base filename from L2 file
//   type - Raw or Data Type
//
// returns:
//   pix0 uncertainty
// =============================================================================
// History: Written by Mike Rinaldi, June 2013
// =============================================================================
//
int DFMS_ProcessMCP_Class::quickPlotL2Data(int iRow, string path, string fileName,
                                         string type) {


	string sFunctionName = "DFMS_ProcessMCP_Class::quickPlotL2Data";

	string sSide;
	string pFout;

	// the file to hold the gnuplot commands
	string file=path+fileName.substr(0,fileName.length()-4)+"_"+type+"_plot.dat";
	fstream os;
	os.open(file.c_str(), fstream::out | fstream::binary);

	// set the terminal type
	string term = setTerminal();

	double yfit=0.0;

	os << "Pixel    Counts    Fit" << dfmsEOL;
	char tmp[200];
	for (int i=0; i<NUMPIX; ++i) {
		sprintf(tmp,"%3d    %f    %f    %7.3f    %f    %f    %7.3f",
				  i,YAin[i],0.0,0.0,YBin[i],0.0,0.0);
		if (iRow == 1) {
			sSide = "A";
		} else if (iRow == 2) {
			sSide = "B";
		}
		os << string(tmp) << dfmsEOL;
	}

	os.close();

	string fileBase = fileName.substr(0,fileName.length()-4)+"_"+type+"_plot.dat";

	// Set up a Gnuplot script and then use system call to plot
	if (iPltFile == 1) {					// plot to terminal window
		pFout = string("# simple gnuplot script \n") +
			    "set term "+term+" size 900,1200 \n\n";
	} else if (iPltFile == 3) {				// plot to postscript file
        string outfile = sPlotFilePath+fileName.substr(0,fileName.length()-4)+"_"+type+"_plot.ps";
		//string outfile = sPlotFilePath+fileName.substr(0,fileName.length()-4)+"_"+type+"_plot.png";
		pFout = string("# simple gnuplot script \n") +
			"set terminal postscript portrait enhanced color font \'Helvetica,10\'\n" +
			//"set terminal pngcairo size 350,262 enhanced font \'Verdana,10\'\n" +
			"set output \'"+outfile+"\'\n";
	}
	pFout += string("# simple gnuplot script \n") +
		"set multiplot layout 2, 1 \n\n" +
		string("file1 = \"")+file+string("\"\n") +
		string("set title \"")+fileBase+string(" - DFMS side A\n")+
		"set autoscale\n\n" +
		"set xrange [0:512]\n" +
		"set xlabel \"Pixel number\"\n" +
		"set ylabel \"Raw Counts\"\n\n" +
		"set style line 1 lt rgb \"blue\" lw 1\n\n" +
		"# the uncommented line below is short for\n" +
		"# plot file1 using 1:2 with lines\n" +
		"plot file1 us 1:2 title \"L2 Data\" ls 1 w l\n\n";

	// Plot Commands for side B
	if (term.compare("aqua") == 0 || term.compare("X11") == 0) {  // no pause command on Mac OS X
		pFout += string("set title \"")+fileBase+string(" - DFMS side B\n") +
				"plot file1 us 1:5 title \"L2 Data\" ls 1 w l\n";
	} else {
		pFout += string("set title \"")+fileBase+string(" - DFMS side B\n") +
				"plot file1 us 1:5 title \"L2 Data\" ls 1 w l\n" +
				"pause 120\n";
	}



	string pfile=path+"plotFitAuto.p";
	os.open(pfile.c_str(), fstream::out | fstream::binary);
	os << pFout << dfmsEOL;
	os.close();

	if (util_fileExists(sGNUplotPath+"gnuplot")) {

		string comm = sGNUplotPath+"gnuplot "+pfile+"&";
		system(comm.c_str());

		// Delay for 1 sec to give gnuplot a chance to finish
		sleep(1000);
		return 1;
	} else {
		return 0;
	}
}

//
// ------------------------------ writeL2L3DataFit ---------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method writes the L2 data and fit and L3 data to a file
// for later plotting using the users prefered plotting package.
//
// inputs:
//   ifrom - whether the call was from Convert_L2_to_L3 or main
//           0 - from main.. implies file was skipped.. just write raw counts
//           1 - from convert.. implies normal fit
//   fileName - base filename from L2 file
//   mass -  target mass
//   mode -  L2 file mode
//
// returns:
//   pix0 uncertainty
// =============================================================================
// History: Written by Mike Rinaldi, February 2014
// =============================================================================
//
int DFMS_ProcessMCP_Class::writeL2L3DataFit(int ifrom, string fileName,
													string mass, string mode) {

	string sFunctionName = "DFMS_ProcessMCP_Class::writeL2L3DataFit";

	// Remove quotes (") from mass string
	string sMass = util_trim(util_trimChar(mass,"\""));

	// The data file name
	string file=sDiagPath+fileName.substr(0,fileName.length()-4)+"_Mass_"+sMass+"_plot.dat";
	//cout << "file = " << file << endl;


	fstream os;
	os.open(file.c_str(), fstream::out | fstream::binary);

	if (ifrom) {
		os << "L2 file: " << L2Obj->getFileName() << " Processed" << dfmsEOL;
		os << "Mode = " << mode << dfmsEOL;
		os << "Mass = " << sMass << dfmsEOL;
		os << "iXl = " << iXl << dfmsEOL;
		os << "iXr = " << iXr << dfmsEOL;
		os << "pix0A = " << L3Info->cal->pix0_A << dfmsEOL;
		os << "pix0B = " << L3Info->cal->pix0_B << dfmsEOL;
		os << "gcuPix0A = " << L3Info->cal->GCUPix0_A << dfmsEOL;
		os << "gcuPix0B = " << L3Info->cal->GCUPix0_B << dfmsEOL;
		os << "meanOffsetA = " << meanOffsetA << dfmsEOL;
		os << "meanOffsetB = " << meanOffsetB << dfmsEOL;
		os << "Peak Name A = " << L3Info->peak->pkIDA << dfmsEOL;
		os << "Peak Name B = " << L3Info->peak->pkIDB << dfmsEOL;
		os << "Peak Center A = " << L3Info->peak->peakCenterA << dfmsEOL;
		os << "Peak Center B = " << L3Info->peak->peakCenterB << dfmsEOL;
		os << "Peak Width A = " << L3Info->peak->peakCenterWidthA << dfmsEOL;
		os << "Peak Width B = " << L3Info->peak->peakCenterWidthB << dfmsEOL;
		os << "peakAreaA = " << L3Info->peakAreaA << dfmsEOL;
		os << "peakAreaB = " << L3Info->peakAreaB << dfmsEOL;
		os << "Pixel CountsA     CorrCntsA        FitA        offSetA      MassA     IonsA     CountsB     CorrCntsB        FitB        offSetB     MassB      IonsB" << dfmsEOL;
		char tmpA[200];
		char tmpB[200];
		for (int i=0; i<NUMPIX; ++i) {
			sprintf(tmpA,"%3d   %7.7d  %12.4f  %12.4f  %12.4f   %7.3f  %10.3e",i+1,L2Obj->L2Data[i][1],YAin[i],yAFit[i],offSetFitA[i],L3Data[i][1],L3Data[i][2]);
			sprintf(tmpB,"   %7.7d  %12.4f  %12.4f  %12.4f   %7.3f  %10.3e",       L2Obj->L2Data[i][2],YBin[i],yBFit[i],offSetFitB[i],L3Data[i][3],L3Data[i][4]);
			os << string(tmpA) << string(tmpB) << dfmsEOL;
		}
	} else {
		os << "L2 file: " << L2Obj->getFileName() << " Skipped" << dfmsEOL;
		os << "Mode = " << mode << dfmsEOL;
		os << "Mass = " << sMass << dfmsEOL;
		os << "iXl = " << PKLOWPT << dfmsEOL;
		os << "iXr = " << PKHIGHPT << dfmsEOL;
		os << "pix0A = " << 0.0 << dfmsEOL;
		os << "pix0B = " << 0.0 << dfmsEOL;
		os << "gcuPix0A = " << 0.0 << dfmsEOL;
		os << "gcuPix0B = " << 0.0 << dfmsEOL;
		os << "meanOffsetA = " << 0.0 << dfmsEOL;
		os << "meanOffsetB = " << 0.0 << dfmsEOL;
		os << "Peak Name A = " << L3Info->peak->pkIDA << dfmsEOL;
		os << "Peak Name B = " << L3Info->peak->pkIDB << dfmsEOL;
		os << "Peak Center A = " << 0.0 << dfmsEOL;
		os << "Peak Center B = " << 0.0 << dfmsEOL;
		os << "Peak Width A = " << 0.0 << dfmsEOL;
		os << "Peak Width B = " << 0.0 << dfmsEOL;
		os << "peakAreaA = " << 0.0 << dfmsEOL;
		os << "peakAreaB = " << 0.0 << dfmsEOL;
		os << "Pixel CountsA     CorrCntsA        FitA     MassA      IonsA     CountsB     CorrCntsB        FitB     MassB      IonsB" << dfmsEOL;
		char tmpA[200];
		char tmpB[200];
		for (int i=0; i<NUMPIX; ++i) {
			sprintf(tmpA,"%3d   %7.7d  %12.4f  %12.4f  %12.4f  %7.3f  %10.3e",i+1,L2Obj->L2Data[i][1],0.0,0.0,0.0,0.0,0.0);
			sprintf(tmpB,"   %7.7d  %12.4f  %12.4f  %12.4f  %7.3f  %10.3e",       L2Obj->L2Data[i][2],0.0,0.0,0.0,0.0,0.0);
			os << string(tmpA) << string(tmpB) << dfmsEOL;
		}
	}
	os.close();
	return 1;
}

//
// ------------------------------ writeTimeSeriesFile ---------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method writes to the time series file.  this file contains L3 peak area
// calculated density and time
//
// inputs:
//
// returns:
//   1 if successful,  0 if failure
// =============================================================================
// History: Written by Mike Rinaldi, March 2014
// =============================================================================
//
int DFMS_ProcessMCP_Class::writeTimeSeriesFile(string date, DFMSModeInfo &modeInfo) {

	string sFunctionName = "DFMS_ProcessMCP_Class::writeTimeSeriesFile";

	double areaA=0, areaB=0;
	double densityA=0, densityB=0;
	string mass="";

	char tmp[300];
	char ftmp[300];
	string line="";
	string name="";
	double lrDel=0.2, hrDel=0.05;

	string L3f = L3Info->L3FileName;
	double tMass = L2Info.commandedMass;
	string sTime = (L2Info.startTime).substr(0,23);
	fstream os;

	// Check if special mass peak exists within the mass scale range
	// using mass scale created in DFMS_ProcessMCP_Class::createL3MassScale()
	vector<double> masses = spMassInRange();

	//int spMidx = L2Info.indexToSpcMass;
	// If we have a special mass then proceed
	if (masses.size() > 0) {
		//cout << "masses.size() = " << masses.size() << endl;
		for (int i=0; i<masses.size(); i++) {
			double sumA=0.00, sumB=0.0;
			//if (spMidx != -1) {
			//mass = util_intToString(spMasses[spMidx]);
			mass = util_doubleToString(masses[i]);
			//cout << "mass = " << mass << endl;
			for (int iRow = 1; iRow < 3; iRow++) {
				cout << endl;
				// Do as a Low Res file
				if (L2Info.lowRes) {
					sprintf(ftmp,"DFMS_LR_m%2.2s_TimeSeries.dat",mass.c_str());
					name.assign(ftmp);
					string file=sTimeSeriesPath+name;
					// Check if file exists. If not create it and write header line
					if (!ifstream(file.c_str())) {
						os.open(file.c_str(), fstream::out | fstream::binary);
						os << "         L3 file Name                  Start Time         CMass   SMass  GS     NG-COPS        RG-COPS       Ions/Spec A    Ions/Spec B    Ions/Spec A+B" << dfmsEOL;
						os.close();
					}
					os.open(file.c_str(), fstream::app | fstream::out | fstream::binary);
					// Now find the peak ion density using mass scale created in
					// DFMS_ProcessMCP_Class::createL3MassScale()
					for (int j=0; j<NUMPIX; j++) {
						if (iRow ==1) {
							//if (fMassScaleA[j] >= spMasses[spMidx]-lrDel && fMassScaleA[j] <= spMasses[spMidx]+lrDel) {
							if (fMassScaleA[j] >= masses[i]-lrDel && fMassScaleA[j] <= masses[i]+lrDel) {
								sumA += L3Data[j][2];
								//cout << "Low Res - m[" << j << "] = " << fMassScaleA[j] << " L3Data[" << j << "][2] = " << L3Data[j][2] << endl;
							}
						} else {
							//if (fMassScaleB[j] >= spMasses[spMidx]-lrDel && fMassScaleB[j] <= spMasses[spMidx]+lrDel) {
							if (fMassScaleB[j] >= masses[i]-lrDel && fMassScaleB[j] <= masses[i]+lrDel) {
								sumB += L3Data[j][4];
							//cout << "Low Res - m[" << j << "] = " << fMassScaleB[j] << " L3Data[" << j << "][4] = " << L3Data[j][4] << endl;
							}
						}
					}
					// Write ion density and other needed vals to file
					if (os.is_open()) {
						if (sumA > 0 && sumB > 0) {
							sprintf(tmp,"%s  %s  %6.2f  %6.2f  %2.2d  %12.6e   %12.6e   %12.6e   %12.6e   %12.6e",
									L3f.c_str(), sTime.c_str(), tMass, masses[i], L2Info.GS, L3Info->copsPressNG, L3Info->copsPressRG, sumA, sumB, sumA+sumB);
							line.assign(tmp);
							os << line << dfmsEOL;
						}
					} else {
						sErrorMessage = "Cannot open time series file: "+file;
						writeToLog(sErrorMessage, sFunctionName, ERROR);
						return 0;
					}
					os.close();
				// Do as High Res file
				} else if (L2Info.hiRes) {
					sprintf(ftmp,"DFMS_HR_m%2.2s_TimeSeries.dat",mass.c_str());
					name.assign(ftmp);
					string file=sTimeSeriesPath+name;
					// Check if file exists. If not create it and write header line
					if (!ifstream(file.c_str())) {
						os.open(file.c_str(), fstream::out |fstream::binary);
						os << "         L3 file Name                  Start Time         CMass   SMass  GS     NG-COPS        RG-COPS       Ions/Spec A    Ions/Spec B    Ions/Spec A+B" << dfmsEOL;
						os.close();
					}
					os.open(file.c_str(), fstream::app | fstream::out | fstream::binary);
					// Now find the peak ion density
					for (int j=0; j<NUMPIX; j++) {
						if (iRow ==1) {
							//cout << "High Res - m[" << j << "] = " << fMassScaleA[j] << " L3Data[" << j << "][2] = " << L3Data[j][2] << endl;
							//if (fMassScaleA[j] >= spMasses[spMidx]-hrDel && fMassScaleA[j] <= spMasses[spMidx]+hrDel) {
							if (fMassScaleA[j] >= masses[i]-hrDel && fMassScaleA[j] <= masses[i]+hrDel) {
								sumA += L3Data[j][2];
								//cout << "High Res - m[" << j << "] = " << fMassScaleA[j] << " L3Data[" << j << "][2] = " << L3Data[j][2];
								//cout << "  -  sumA = " << sumA << endl;
							}
						} else {
							//if (fMassScaleB[j] >= spMasses[spMidx]-hrDel && fMassScaleB[j] <= spMasses[spMidx]+hrDel) {
							if (fMassScaleB[j] >= masses[i]-hrDel && fMassScaleB[j] <= masses[i]+hrDel) {
								sumB += L3Data[j][4];
								//cout << "High Res - m[" << j << "] = " << fMassScaleB[j] << " L3Data[" << j << "][4] = " << L3Data[j][4] << endl;
							}
						}
					}
					// Write ion density and other needed vals to file
					if (os.is_open()) {
						if (sumA > 0 && sumB > 0) {
							sprintf(tmp,"%s  %s  %6.2f  %6.2f  %2.2d  %12.6e   %12.6e   %12.6e   %12.6e   %12.6e",
									L3f.c_str(), sTime.c_str(), tMass, masses[i], L2Info.GS, L3Info->copsPressNG, L3Info->copsPressRG, sumA, sumB, sumA+sumB);
							line.assign(tmp);
							os << line << dfmsEOL;
						}
					} else {
						sErrorMessage = "Cannot open time series file: "+file;
						writeToLog(sErrorMessage, sFunctionName, ERROR);
						return 0;
					}
					os.close();
				}
			}
		}
	}

	return 1;

}

//
// ------------------------------ quickPlotDataAndFit ---------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method writes the L2 data and resulting curve fit data to a gnuplot file
// for a quick view.
//
// inputs:
//   iRow  - DFMS side
//   path  - Path to output data
//   fileName - base filename from L2 file
//   type - Raw or Data Type
//
// returns:
//   pix0 uncertainty
// =============================================================================
// History: Written by Mike Rinaldi, June 2013
// =============================================================================
//
int DFMS_ProcessMCP_Class::quickPlotDataAndFit(int iRow, string path, string fileName,
                             string type, string sstr) {


	string sFunctionName = "DFMS_ProcessMCP_Class::quickPlotDataAndFit";

	string sSide;
	string pFout;

	string term = setTerminal();

	string leg1 = "\"L2 Data\"";
	string leg2 = "\"Fit\"";

	// the file to hold the gnuplot commands
	string file=path+fileName.substr(0,fileName.length()-4)+"_"+type+"_plot.dat";
    cout << "fileName = " << fileName << endl;

	fstream os;
	os.open(file.c_str(), fstream::out | fstream::binary);

	os << "Pixel    Counts    Fit" << dfmsEOL;
	char tmp[300];
	for (int i=0; i<NUMPIX; ++i) {
		sprintf(tmp,"%3d    %f    %f    %7.3f    %f    %f    %7.3f",
				  i,YAin[i],yAFit[i],fMassScaleA[i],YBin[i],yBFit[i],fMassScaleB[i]);
		if (iRow == 1) {
			sSide = "A";
		} else if (iRow == 2) {
			sSide = "B";
		}
		os << string(tmp) << dfmsEOL;
	}

	string sMass = util_trim(util_trimChar(sstr,"\""));
	os.close();

	string fileBase = fileName.substr(0,fileName.length()-4)+"_"+type+"_plot.dat";

	// Set up a Gnuplot command script and then use a system call to execute
	// the gnuplot commands
	if (iPltFile == 2) {					// plot to terminal window
		pFout = string("# simple gnuplot script \n") +
			"set term "+term+" size 900,1200 \n\n";
	} else if (iPltFile == 4) {				// plot to postscript file
		string outfile = sPlotFilePath+fileName.substr(0,fileName.length()-4)+"_"+type+"_plot.ps";
		//string outfile = sPlotFilePath+fileName.substr(0,fileName.length()-4)+"_"+type+"_plot.png";
		pFout = string("# simple gnuplot script \n") +
			"set terminal postscript portrait enhanced color font \'Helvetica,10\'\n" +
			//"set terminal pngcairo size 350,262 enhanced font \'Verdana,10\'\n" +
			"set output \'"+outfile+"\'\n";
	}

	// Plot Commands for side A
	pFout +=  string("file1 = \"")+file+string("\"\n") +
		string("set title \"")+fileBase+string(" - DFMS side A\n") +
		"set multiplot layout 2, 1 \n" +
		"set autoscale\n\n" +
		"set xlabel \"Pixel number\"\n" +
		"set x2label \"Mass Scale (for Mass: "+sMass+" )\"\n" +
		"set xrange [0:512]\n"
		"set x2range ["+util_doubleToString(fMassScaleA[0])+":"+util_doubleToString(fMassScaleB[NUMPIX-1])+"]\n"
		"set xtics nomirror \n" +
		"set x2tics \n" +
		"set grid x2tics \n" +
		"set grid ytics \n" +
		"set ylabel \"Raw Counts\"\n\n" +
		"set style line 2 lt rgb \"red\" lw 1\n" +
		"set style line 1 lt rgb \"blue\" lw 1\n\n" +
		"plot file1 us 1:2 title "+leg1+" ls 1 w l, file1 us 1:3 title "+leg2+" ls 2 w l\n\n\n";

	// Plot Commands for side B
	if (term.compare("aqua") == 0 || term.compare("X11") == 0) {
		pFout += string("set title \"")+fileBase+string(" - DFMS side B\n") +
			"set x2range ["+util_doubleToString(fMassScaleA[0])+":"+util_doubleToString(fMassScaleB[NUMPIX-1])+"]\n" +
			"plot file1 us 1:5 title "+leg1+" ls 1 w l, file1 us 1:6 title "+leg2+" ls 2 w l\n";
	} else {
		pFout += string("set title \"")+fileBase+string(" - DFMS side B\n") +
			"set x2range ["+util_doubleToString(fMassScaleA[0])+":"+util_doubleToString(fMassScaleB[NUMPIX-1])+"]\n" +
			"plot file1 us 1:5 title "+leg1+" ls 1 w l, file1 us 1:6 title "+leg2+" ls 2 w l\n" +
			"pause 120\n";
	}
 
	string pfile=path+"plotFitAuto.p";
	os.open(pfile.c_str(), fstream::out | fstream::binary);
	os << pFout << dfmsEOL;
	os.close();

	if (util_fileExists(sGNUplotPath+"gnuplot")) {
		string comm = sGNUplotPath+"gnuplot "+pfile+"&";

		system(comm.c_str());

		// Delay for 1 sec to give gnuplot a chance to finish
		sleep(1000);
		return 1;
	} else {
		return 0;
    }
}

//
// ------------------------------- calcOffset ----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method will calculate the average offset by histogramming the log of
// the raw counts into 50-100 bins.  It will then search for the largest bin.  Since
// most of the 512 pixels are just offset the method will use these pixels to
// fid the mean offset and std Dev.
//
// inputs:
//	int iRow  -  Flag to indicate Row A or B
//
// returns:
//   1 for success 0 for failure
// =============================================================================
// History: Written by Mike Rinaldi, March 2014
// =============================================================================
//
int DFMS_ProcessMCP_Class::calcOffset(int iRow) {

	string sFunctionName = "DFMS_ProcessMCP_Class::calcOffset";

	// 2.3.1 - Create the histogram object
	DFMS_Histogram_Class *hOff;

	// 2.3.2 - Use full data are for offset calculation
	if (iRow == 1) {
		hOff = new DFMS_Histogram_Class(YAin, NUMPIX, HISTONUMBINS, string("offset"));
	} else {
		hOff = new DFMS_Histogram_Class(YBin, NUMPIX, HISTONUMBINS, string("offset"));
	}
	// 2.3.3 - Make the entire data range histogram
	hOff->createHisto(iRow);

	// 2.3.4 - Find the Mean & standard Deviation of the Offset using the Histogram technique
	double mean,stdDev;
	hOff->meanOffset(&mean,&stdDev);
	meanOffset = mean;
	meanOffsetStdDev = stdDev;

	double pk = hOff->getHmax();

	// 2.3.5 - Release memory
    delete hOff; hOff=0;

    return 1;

}

//
// ------------------------------- rowBModCorr ----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method will attempt to correct the count modulation in the Row B offset.
// the formula"  Offset = Offset +/- N*(-1)^pixelNumber.  Where N is set in the
// DFMS constants input file
//
// inputs:
//	int iRow  -  Flag to indicate Row A(1) or B(2).
//               This will always be B(2) for this method.
//
// returns:
//   None
// =============================================================================
// History: Written by Mike Rinaldi, January 2016
// =============================================================================
//
void DFMS_ProcessMCP_Class::rowBModCorr(int iRow) {

	bool decreasing = true;

	for (int i=1; i<NUMPIX; i++) {
		double modCorr = MODAMP*pow(-1,i+1);
		double Ybefore = YBin[i];
		// Check what the phase is. Y1>Y2 is positive phase
		if (i == 1) {
			if (YBin[i] >= YBin[i+1]) {
				decreasing = true;
			} else {
				decreasing = false;
			}
		}
		// Use either + MODAMP or - depending on posPhase
		if (decreasing) {
			YBin[i] -= modCorr;
		} else {
			YBin[i] += modCorr;
		}
		//cout << i+1 << "," << Ybefore << "," << YBin[i] << endl;
	}

	//exit(0);

}

