
#include "DFMS_PDS_L3_file_Class.hh"

//
// ---------------------------- Constructor ------------------------------------
//
DFMS_PDS_L3_file_Class::DFMS_PDS_L3_file_Class (string p, string f, 
							int H, int C, mapstr &P, str2D &HKd,
							str2D &Cald, dbl2D &L3d, L2FileInfo &L2) {

	path = p;
	fileName = f;
	L3File = path+fileName;
	HKrows = H;
	CalRows = C;  // Total Cal Rows (includes sides A/B)

	// Copy L2 PDSheader map using overloaded map::operator=
	// A simple assignment of a map object does a deep copy
	PDSheader = P;

	// A simple assignment of a struct does a deep copy
	L2Info = L2;
     
	// Initialize the needed local 2D vectors. unfortunately simply copying
	// only does a shallow copy (not element by element). We need a deep copy
	initializeData(HKd, Cald, L3d);
     
}

//
// ----------------------------- Destructor ------------------------------------
//
DFMS_PDS_L3_file_Class::~DFMS_PDS_L3_file_Class () {

	HKobj.clear();
	CalObj.clear();
	CalData.clear();
	L3HKdata.clear();
	L3DataObj.clear();
}

//
// -------------------------- initializeData -----------------------------------
//
void DFMS_PDS_L3_file_Class::initializeData(str2D &HKd, str2D &Cald, dbl2D &L3d) {
    
	string sFunctionName = "DFMS_PDS_L3_file_Class::intializeData";

	int calItems = 7;
	int l3Items = 5;
	int hkItems = 4;
	// Allocate memory and copy data arrays	
	for(int i=0; i<HKrows; ++i) {
		L3HKdata.push_back(sslice());
		for (int j=0; j<hkItems; ++j) {
			L3HKdata[i].push_back(HKd[i][j]);
			//cout << " HKd[" << i << "][" << j << "] = " << HKd[i][j];
		}
		//cout << endl;
	}
	// Allocate memory and copy data arrays	
	for(int i=0; i<CalRows; ++i) {
		CalData.push_back(sslice());
		for (int j=0; j<calItems; ++j) {
			CalData[i].push_back(Cald[i][j]);
		}
	}
     
	// Allocate memory and copy data arrays
	for(int i=0; i<NUMPIX; ++i) {
		L3Data.push_back(dslice());
		for (int j=0; j<l3Items; ++j) {
			L3Data[i].push_back(L3d[i][j]);
		}
	}
}

//
// ----------------------------- formHKobj -------------------------------------
//
int DFMS_PDS_L3_file_Class::formHKobj() {

	// Modify the L2 HK Desc Object (L3 has fewer rows)
	HKobj["OBJECT"]             = "DFMS_HK_TABLE";
	HKobj["NAME"]               = "DFMS_HOUSEKEEPING_TABLE";
	HKobj["INTERCHANGE_FORMAT"] = "ASCII";
	HKobj["ROWS"] = util_intToString(HKrows);
	HKobj["COLUMNS"]            = "5";
	HKobj["ROW_BYTES"]          = "80";
	HKobj["^STRUCTURE"]         = "\"DFMS_L3_HK.FMT\"";
	HKobj["END_OBJECT"]         = "DFMS_HK_TABLE";

	return 1;
}

//
// ----------------------------- formCalObj -------------------------------------
//
int DFMS_PDS_L3_file_Class::formCalObj() {

	CalObj["OBJECT"]             = "DFMS_MASS_CAL_TABLE";
	CalObj["NAME"]               = "DFMS_MASS_CALIBRATION_TABLE";
	CalObj["INTERCHANGE_FORMAT"] = "ASCII";
	CalObj["ROWS"]               = util_intToString(CalRows);
	CalObj["COLUMNS"]            = "8";
	CalObj["ROW_BYTES"]          = "80";
	CalObj["^STRUCTURE"]         = "\"DFMS_L3_CALINFO.FMT\"";
	CalObj["END_OBJECT"]         = "DFMS_MASS_CAL_TABLE";

	return 1;

}

//
// ----------------------------- formL3DataObj -------------------------------------
//
int DFMS_PDS_L3_file_Class::formL3DataObj() {

	L3DataObj["OBJECT"]             = "MCP_DATA_L3_TABLE";
	L3DataObj["NAME"]               = "MCP_DATA_L3_TABLE";
	L3DataObj["INTERCHANGE_FORMAT"] = "ASCII";
	L3DataObj["ROWS"]               = util_intToString(NUMPIX);
	L3DataObj["COLUMNS"]            = "6";
	L3DataObj["ROW_BYTES"]          = "80";
	L3DataObj["^STRUCTURE"]         = "\"DFMS_L3_DATA.FMT\"";
	L3DataObj["END_OBJECT"]         = "MCP_DATA_L3_TABLE";	

	return 1;

}	

//
// ------------------------------ writePDSheader ------------------------------------
//
/*
int DFMS_PDS_L3_file_Class::writePDSheader(fstream &os) {

	string sFunctionName = "DFMS_PDS_L3_file_Class::writePDSheader";
	
	char cline[80];
	string sline;

	// Write the L3 PDS Header
	writeLine(os,"PDS_VERSION_ID",PDSheader["PDS_VERSION_ID"],0);
	writeLine(os,"LABEL_REVISION_NOTE",PDSheader["LABEL_REVISION_NOTE"],1);
	writeLine(os,"SOFTWARE_NAME",PDSheader["SOFTWARE_NAME"],0);	
	writeLine(os,"SOFTWARE_VERSION_ID",PDSheader["SOFTWARE_VERSION_ID"],0);
	writeLine(os,"ROSETTA:ROSINA_PIXEL0_A_MASS",PDSheader["ROSETTA:ROSINA_PIXEL0_A_MASS"],1);
	writeLine(os,"ROSETTA:ROSINA_PIXEL0_B_MASS",PDSheader["ROSETTA:ROSINA_PIXEL0_B_MASS"],1);
	writeLine(os,"SOURCE_FILE_NAME",PDSheader["SOURCE_FILE_NAME"],0);
	writeLine(os,"ROSETTA:ROSINA_CAL_ID1",PDSheader["ROSETTA:ROSINA_CAL_ID1"],0);
	writeLine(os,"ROSETTA:ROSINA_CAL_ID2",PDSheader["ROSETTA:ROSINA_CAL_ID2"],0);
	writeLine(os,"ROSETTA:ROSINA_CAL_ID3",PDSheader["ROSETTA:ROSINA_CAL_ID3"],0);
	writeLine(os,"RECORD_TYPE",PDSheader["RECORD_TYPE"],0);
	writeLine(os,"RECORD_BYTES",PDSheader["RECORD_BYTES"],0);
	writeLine(os,"FILE_RECORDS",PDSheader["FILE_RECORDS"],0);
	writeLine(os,"LABEL_RECORDS",PDSheader["LABEL_RECORDS"],0);
	writeLine(os,"^DFMS_HK_TABLE",PDSheader["^DFMS_HK_TABLE"],0);
	writeLine(os,"^DFMS_MASS_CAL_TABLE",PDSheader["^DFMS_MASS_CAL_TABLE"],0);
	writeLine(os,"^MCP_DATA_L3_TABLE",PDSheader["^MCP_DATA_L3_TABLE"],0);
	writeLine(os,"DATA_SET_ID",PDSheader["DATA_SET_ID"],1);
	writeLine(os,"DATA_SET_NAME",PDSheader["DATA_SET_NAME"],1);
	writeLine(os,"PRODUCT_ID",PDSheader["PRODUCT_ID"],0);
	writeLine(os,"PRODUCT_CREATION_TIME",PDSheader["PRODUCT_CREATION_TIME"],0);
	writeLine(os,"PRODUCT_TYPE",PDSheader["PRODUCT_TYPE"],0);
	writeLine(os,"PROCESSING_LEVEL_ID",PDSheader["PROCESSING_LEVEL_ID"],1);
	writeLine(os,"MISSION_ID",PDSheader["MISSION_ID"],0);
	writeLine(os,"MISSION_NAME",PDSheader["MISSION_NAME"],1);
	writeLine(os,"TARGET_NAME",PDSheader["TARGET_NAME"],1);
	writeLine(os,"TARGET_TYPE",PDSheader["TARGET_TYPE"],1);
	writeLine(os,"MISSION_PHASE_NAME",PDSheader["MISSION_PHASE_NAME"],1);
	writeLine(os,"INSTRUMENT_HOST_NAME",PDSheader["INSTRUMENT_HOST_NAME"],1);
	writeLine(os,"INSTRUMENT_HOST_ID",PDSheader["INSTRUMENT_HOST_ID"],0);
	writeLine(os,"INSTRUMENT_NAME",PDSheader["INSTRUMENT_NAME"],1);
	writeLine(os,"INSTRUMENT_ID",PDSheader["INSTRUMENT_ID"],0);
	writeLine(os,"INSTRUMENT_MODE_ID",PDSheader["INSTRUMENT_MODE_ID"],0);
	writeLine(os,"^INSTRUMENT_MODE_DESC",PDSheader["^INSTRUMENT_MODE_DESC"],1);
	writeLine(os,"INSTRUMENT_TYPE",PDSheader["INSTRUMENT_TYPE"],1);
	writeLine(os,"DETECTOR_ID",PDSheader["DETECTOR_ID"],0);
	writeLine(os,"DETECTOR_DESC",PDSheader["DETECTOR_DESC"],1);
	writeLine(os,"CHANNEL_ID",PDSheader["CHANNEL_ID"],0);
	writeLine(os,"START_TIME",PDSheader["START_TIME"],0);
	writeLine(os,"STOP_TIME",PDSheader["STOP_TIME"],0);
	writeLine(os,"SPACECRAFT_CLOCK_START_COUNT",PDSheader["SPACECRAFT_CLOCK_START_COUNT"],1);
	writeLine(os,"SPACECRAFT_CLOCK_STOP_COUNT",PDSheader["SPACECRAFT_CLOCK_STOP_COUNT"],1);
	writeLine(os,"PRODUCER_ID",PDSheader["PRODUCER_ID"],0);
	writeLine(os,"PRODUCER_FULL_NAME",PDSheader["PRODUCER_FULL_NAME"],1);
	writeLine(os,"PRODUCER_INSTITUTION_NAME",PDSheader["PRODUCER_INSTITUTION_NAME"],1);
	writeLine(os,"DATA_QUALITY_ID",PDSheader["DATA_QUALITY_ID"],1);
	writeLine(os,"DATA_QUALITY_DESC",PDSheader["DATA_QUALITY_DESC"],1);
	writeLine(os,"SC_SUN_POSITION_VECTOR",PDSheader["SC_SUN_POSITION_VECTOR"],1);
	writeLine(os,"SC_TARGET_POSITION_VECTOR",PDSheader["SC_TARGET_POSITION_VECTOR"],1);
	writeLine(os,"COORDINATE_SYSTEM_ID",PDSheader["COORDINATE_SYSTEM_ID"],1);
	writeLine(os,"COORDINATE_SYSTEM_NAME",PDSheader["COORDINATE_SYSTEM_NAME"],1);
	writeLine(os,"SC_TARGET_VELOCITY_VECTOR",PDSheader["SC_TARGET_VELOCITY_VECTOR"],1);
	writeLine(os,"SPACECRAFT_ALTITUDE",PDSheader["SPACECRAFT_ALTITUDE"],1);
	writeLine(os,"SUB_SPACECRAFT_LATITUDE",PDSheader["SUB_SPACECRAFT_LATITUDE"],1);
	writeLine(os,"SUB_SPACECRAFT_LONGITUDE",PDSheader["SUB_SPACECRAFT_LONGITUDE"],1);
	writeLine(os,"DESCRIPTION",PDSheader["DESCRIPTION"],1);
	writeLine(os,"NOTE",PDSheader["NOTE"],1);

	// Add a blank line to seperate PDSheader from first Object descriptor
	sprintf(cline,"%-78s"," ");
	sline = string(cline);
	os << sline << dfmsEOL;

	return 1;

}
*/

//
// ------------------------------ writeOldPDSheader ------------------------------------
//
int DFMS_PDS_L3_file_Class::writeOldPDSheader(fstream &os) {

	string sFunctionName = "DFMS_PDS_L3_file_Class::writeOldPDSheader";

    char tmp[810];
    string stmp;
    string ctmp="None";
    int sEnd;
	int lblRec = 102;
	int L3recs = NUMPIX;
	int lastLineNum = lblRec+HKrows+CalRows+L3recs;

	// Start forming and writing the PDS Header.
	// Old L2 values that are to be directly transfered to L3 header contain a
	// reference to PDSheader["xxxxx"].
	// New L3 values are created from L3 relevant data
	// Note: the variable PDSheader["xxxxx"] contains unmodified L2 data.

    stmp = "PDS_VERSION_ID                   =     ";
		PDSheader["PDS_VERSION_ID"] = util_trimEOL(PDSheader["PDS_VERSION_ID"]);
		sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["PDS_VERSION_ID"].c_str());
		os << string(tmp) << dfmsEOL;

    stmp = "LABEL_REVISION_NOTE              =     \"2015-01-01,Thierry Semon(UoB),";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "                                        Version1.0 release\"";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "SOFTWARE_NAME                    =     ";
       string ttmp = "\""+sCurrentSoftwareName+"\"";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),ttmp.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "SOFTWARE_VERSION_ID              =     ";
       ttmp = "\""+sCurrentSoftwareVersionID+"\"";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),ttmp.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "ROSETTA:ROSINA_PIXEL0_A_MASS     =     ";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),(util_doubleToString(L3Info->cal->m0A)).c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "ROSETTA:ROSINA_PIXEL0_B_MASS     =     ";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),(util_doubleToString(L3Info->cal->m0B)).c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "SOURCE_FILE_NAME                 =     ";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),(L3Info->L2baseName).c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "ROSETTA:ROSINA_CAL_ID4           =     ";
    	if (L3Info->cal->calID4.compare("None") != 0) {
    		sEnd = L3Info->cal->calID4.length()-4;
    		ctmp = L3Info->cal->calID4.substr(0,sEnd);
    	} else {
    		ctmp = L3Info->cal->calID4;
    	}
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),ctmp.c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "ROSETTA:ROSINA_CAL_ID5           =     ";
    	if (L3Info->cal->calID5.compare("None") != 0) {
    		sEnd = L3Info->cal->calID5.length()-4;
    		ctmp = L3Info->cal->calID5.substr(0,sEnd);
    	} else {
    		ctmp = L3Info->cal->calID5;
    	}
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),ctmp.c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "ROSETTA:ROSINA_CAL_ID6           =     ";
    	if (L3Info->cal->calID6.compare("None") != 0) {
    		sEnd = L3Info->cal->calID6.length()-4;
    		ctmp = L3Info->cal->calID6.substr(0,sEnd);
    	} else {
    		ctmp = L3Info->cal->calID6;
    	}
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),ctmp.c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "RECORD_TYPE                      =     ";
    	PDSheader["RECORD_TYPE"] = util_trimEOL(PDSheader["RECORD_TYPE"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["RECORD_TYPE"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "RECORD_BYTES                     =     ";
    	PDSheader["RECORD_BYTES"] = util_trimEOL(PDSheader["RECORD_BYTES"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["RECORD_BYTES"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "FILE_RECORDS                     =     ";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),(util_intToString(lastLineNum)).c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "LABEL_RECORDS                    =     ";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),(util_intToString(lblRec)).c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "^DFMS_HK_TABLE                   =     ";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),(util_intToString(lblRec+1)).c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "^DFMS_MASS_CAL_TABLE             =     ";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),(util_intToString(lblRec+HKrows+1)).c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "^MCP_DATA_L3_TABLE               =     ";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),(util_intToString(lblRec+HKrows+CalRows+1)).c_str());
       os << string(tmp) << dfmsEOL;

    string newDName = PDSheader["DATA_SET_ID"];
    replace(newDName,"-2-","-3-");
    stmp = "DATA_SET_ID                      =     ";
    	newDName = util_trimEOL(newDName);
		sprintf(tmp,"%-39s%-39s",stmp.c_str(),newDName.c_str());
		os << string(tmp) << dfmsEOL;

    newDName = PDSheader["DATA_SET_NAME"];
    replace(newDName,"2","3");
    stmp = "DATA_SET_NAME                    =     ";
    	newDName = util_trimEOL(newDName);
		sprintf(tmp,"%-39s%-39s",stmp.c_str(),newDName.c_str());
		os << string(tmp) << dfmsEOL;

    stmp = "PRODUCT_ID                       =     ";
       string name = fileName.substr(0,26);
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),name.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "PRODUCT_CREATION_TIME            =     ";
       string tTime = getProcessTime(1);
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),tTime.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "PRODUCT_TYPE                     =     RDR";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "PROCESSING_LEVEL_ID              =   \"3\"";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "MISSION_ID                       =     ";
    	PDSheader["MISSION_ID"] = util_trimEOL(PDSheader["MISSION_ID"]);
		sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["MISSION_ID"].c_str());
		os << string(tmp) << dfmsEOL;

    stmp = "MISSION_NAME                     =     ";
    	PDSheader["MISSION_NAME"] = util_trimEOL(PDSheader["MISSION_NAME"]);
		sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["MISSION_NAME"].c_str());
		os << string(tmp) << dfmsEOL;

    stmp = "TARGET_NAME                      =     ";
    	PDSheader["TARGET_NAME"] = util_trimEOL(PDSheader["TARGET_NAME"]);
		sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["TARGET_NAME"].c_str());
		os << string(tmp) << dfmsEOL;

    stmp = "TARGET_TYPE                      =     ";
    	PDSheader["TARGET_TYPE"] = util_trimEOL(PDSheader["TARGET_TYPE"]);
        sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["TARGET_TYPE"].c_str());
		os << string(tmp) << dfmsEOL;

    stmp = "MISSION_PHASE_NAME               =     ";
    	PDSheader["MISSION_PHASE_NAME"] = util_trimEOL(PDSheader["MISSION_PHASE_NAME"]);
		sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["MISSION_PHASE_NAME"].c_str());
		os << string(tmp) << dfmsEOL;

    stmp = "INSTRUMENT_HOST_NAME             =     ";
    	PDSheader["INSTRUMENT_HOST_NAME"] = util_trimEOL(PDSheader["INSTRUMENT_HOST_NAME"]);
		sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["INSTRUMENT_HOST_NAME"].c_str());
		os << string(tmp) << dfmsEOL;

    stmp = "INSTRUMENT_HOST_ID               =     ";
    	PDSheader["INSTRUMENT_HOST_ID"] = util_trimEOL(PDSheader["INSTRUMENT_HOST_ID"]);
		sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["INSTRUMENT_HOST_ID"].c_str());
		os << string(tmp) << dfmsEOL;

    stmp = "INSTRUMENT_NAME                  =     ";
    	PDSheader["INSTRUMENT_NAME"] = util_trimEOL(PDSheader["INSTRUMENT_NAME"]);
		sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["INSTRUMENT_NAME"].c_str());
		os << string(tmp) << dfmsEOL;

    stmp = "INSTRUMENT_ID                    =     ";
    	PDSheader["INSTRUMENT_ID"] = util_trimEOL(PDSheader["INSTRUMENT_ID"]);
		sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["INSTRUMENT_ID"].c_str());
		os << string(tmp) << dfmsEOL;

    stmp = "INSTRUMENT_MODE_ID               =     ";
    	PDSheader["INSTRUMENT_MODE_ID"] = util_trimEOL(PDSheader["INSTRUMENT_MODE_ID"]);
		sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["INSTRUMENT_MODE_ID"].c_str());
		os << string(tmp) << dfmsEOL;

    stmp = "^INSTRUMENT_MODE_DESC            =     ";
    	PDSheader["^INSTRUMENT_MODE_DESC"] = util_trimEOL(PDSheader["^INSTRUMENT_MODE_DESC"]);
		sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["^INSTRUMENT_MODE_DESC"].c_str());
		os << string(tmp) << dfmsEOL;

    stmp = "INSTRUMENT_TYPE                  =     ";
    	PDSheader["INSTRUMENT_TYPE"] = util_trimEOL(PDSheader["INSTRUMENT_TYPE"]);
		sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["INSTRUMENT_TYPE"].c_str());
		os << string(tmp) << dfmsEOL;

    stmp = "DETECTOR_ID                      =     ";
    	PDSheader["DETECTOR_ID"] = util_trimEOL(PDSheader["DETECTOR_ID"]);
		sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["DETECTOR_ID"].c_str());
        os << string(tmp) << dfmsEOL;

    stmp = "DETECTOR_DESC                    =     ";
    	PDSheader["DETECTOR_DESC"] = util_trimEOL(PDSheader["DETECTOR_DESC"]);
		sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["DETECTOR_DESC"].c_str());
		os << string(tmp) << dfmsEOL;

    stmp = "CHANNEL_ID                       =     ";
    	PDSheader["CHANNEL_ID"] = util_trimEOL(PDSheader["CHANNEL_ID"]);
		sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["CHANNEL_ID"].c_str());
		os << string(tmp) << dfmsEOL;

    stmp = "START_TIME                       =     ";
    	PDSheader["START_TIME"] = util_trimEOL(PDSheader["START_TIME"]);
		sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["START_TIME"].c_str());
		os << string(tmp) << dfmsEOL;

    stmp = "STOP_TIME                        =     ";
    	PDSheader["STOP_TIME"] = util_trimEOL(PDSheader["STOP_TIME"]);
		sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["STOP_TIME"].c_str());
		os << string(tmp) << dfmsEOL;

    stmp = "SPACECRAFT_CLOCK_START_COUNT     =     ";
    	PDSheader["SPACECRAFT_CLOCK_START_COUNT"] = util_trimEOL(PDSheader["SPACECRAFT_CLOCK_START_COUNT"]);
		sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["SPACECRAFT_CLOCK_START_COUNT"].c_str());
		os << string(tmp) << dfmsEOL;

    stmp = "SPACECRAFT_CLOCK_STOP_COUNT      =     ";
		PDSheader["SPACECRAFT_CLOCK_STOP_COUNT"] = util_trimEOL(PDSheader["SPACECRAFT_CLOCK_STOP_COUNT"]);
		sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["SPACECRAFT_CLOCK_STOP_COUNT"].c_str());
		os << string(tmp) << dfmsEOL;

    stmp = "PRODUCER_ID                      =     ";
    	PDSheader["PRODUCER_ID"] = util_trimEOL(PDSheader["PRODUCER_ID"]);
		sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["PRODUCER_ID"].c_str());
		os << string(tmp) << dfmsEOL;

    stmp = "PRODUCER_FULL_NAME               =     ";
    	PDSheader["PRODUCER_FULL_NAME"] = util_trimEOL(PDSheader["PRODUCER_FULL_NAME"]);
		sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["PRODUCER_FULL_NAME"].c_str());
		os << string(tmp) << dfmsEOL;

    stmp = "PRODUCER_INSTITUTION_NAME        =     ";
    	PDSheader["PRODUCER_INSTITUTION_NAME"] = util_trimEOL(PDSheader["PRODUCER_INSTITUTION_NAME"]);
		sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["PRODUCER_INSTITUTION_NAME"].c_str());
		os << string(tmp) << dfmsEOL;

    stmp = "DATA_QUALITY_ID                  =     ";
       string qid = "\""+util_intToString(L3Info->qualID)+"\"";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),qid.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "DATA_QUALITY_DESC                =     \"";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "      0 means 'Nominal quality, avg. PPM deviance < 500\'";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "      1 means 'Self-calibrated, GCU avg. PPM deviance >= 500, SELF < 500\'";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "      2 means 'Adopted mass scale avg. PPM deviance >= 500\'";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "      3 means 'Enhanced Noise\'";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "      4 means 'Not enough peaks found for accurate calibration/verification\'";
    	sprintf(tmp,"%-78s",stmp.c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "SC_SUN_POSITION_VECTOR           =     ";
    	PDSheader["SC_SUN_POSITION_VECTOR"] = util_trimEOL(PDSheader["SC_SUN_POSITION_VECTOR"]);
        sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["SC_SUN_POSITION_VECTOR"].c_str());
        os << string(tmp) << dfmsEOL;

    stmp = "SC_TARGET_POSITION_VECTOR        =     ";
    	PDSheader["SC_TARGET_POSITION_VECTOR"] = util_trimEOL(PDSheader["SC_TARGET_POSITION_VECTOR"]);
        sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["SC_TARGET_POSITION_VECTOR"].c_str());
        os << string(tmp) << dfmsEOL;

    stmp = "COORDINATE_SYSTEM_ID             =     ";
        sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["COORDINATE_SYSTEM_ID"].c_str());
        os << string(tmp) << dfmsEOL;

    stmp = "COORDINATE_SYSTEM_NAME           =     ";
        sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["COORDINATE_SYSTEM_NAME"].c_str());
        os << string(tmp) << dfmsEOL;

    stmp = "SC_TARGET_VELOCITY_VECTOR        =     ";
    	PDSheader["SC_TARGET_VELOCITY_VECTOR"] = util_trimEOL(PDSheader["SC_TARGET_VELOCITY_VECTOR"]);
        sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["SC_TARGET_VELOCITY_VECTOR"].c_str());
        os << string(tmp) << dfmsEOL;

    stmp = "SPACECRAFT_ALTITUDE              =     ";
    	PDSheader["SPACECRAFT_ALTITUDE"] = util_trimEOL(PDSheader["SPACECRAFT_ALTITUDE"]);
        sprintf(tmp,"%-40.40s%-38.38s",stmp.c_str(),PDSheader["SPACECRAFT_ALTITUDE"].c_str());
        os << string(tmp) << dfmsEOL;

    stmp = "SUB_SPACECRAFT_LATITUDE          =     ";
    	PDSheader["SUB_SPACECRAFT_LATITUDE"] = util_trimEOL(PDSheader["SUB_SPACECRAFT_LATITUDE"]);
        sprintf(tmp,"%-40.40s%-38.38s",stmp.c_str(),PDSheader["SUB_SPACECRAFT_LATITUDE"].c_str());
        os << string(tmp) << dfmsEOL;

    stmp = "SUB_SPACECRAFT_LONGITUDE         =     ";
    	PDSheader["SUB_SPACECRAFT_LONGITUDE"] = util_trimEOL(PDSheader["SUB_SPACECRAFT_LONGITUDE"]);
        sprintf(tmp,"%-40.40s%-38.38s",stmp.c_str(),PDSheader["SUB_SPACECRAFT_LONGITUDE"].c_str());
        os << string(tmp) << dfmsEOL;

    stmp = "DESCRIPTION                      =     ";
    	PDSheader["DESCRIPTION"] = util_trimEOL(PDSheader["DESCRIPTION"]);
        sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["DESCRIPTION"].c_str());
        os << string(tmp) << dfmsEOL;

    stmp = "NOTE                             =     ";
    	PDSheader["NOTE"] = util_trimEOL(PDSheader["NOTE"]);
        sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["NOTE"].c_str());
        os << string(tmp) << dfmsEOL;

    // Add an 79 character blank line to separate PDSheader from first Object descriptor
	sprintf(tmp,"%-78s"," ");
	os << string(tmp) << dfmsEOL;

	return 1;

}

//
// ------------------------------ writeNewPDSheader ------------------------------------
//
int DFMS_PDS_L3_file_Class::writeNewPDSheader(fstream &os) {

	string sFunctionName = "DFMS_PDS_L3_file_Class::writeNewPDSheader";

    char tmp[810];
    string stmp;
    string ctmp="None";
    int sEnd;
	int lblRec = 0;
	int L3recs = NUMPIX;

	if (PDSheader["SPICE_FILE_NAME"].empty()) {
		PDSheader["SPICE_FILE_NAME"] = "\"N/A\"";
		lblRec = 101;
	} else {
		lblRec = 114;
	}
	int lastLineNum = lblRec+HKrows+CalRows+L3recs;

	// Start forming and writing the PDS Header.
	// Old L2 values that are to be directly transfered to L3 header contain a
	// reference to PDSheader["xxxxx"].
	// New L3 values are created from L3 relevant data
	// Note: the variable PDSheader["xxxxx"] contains unmodified L2 data.

    stmp = "PDS_VERSION_ID                   =     ";
    	PDSheader["PDS_VERSION_ID"] = util_trimEOL(PDSheader["PDS_VERSION_ID"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["PDS_VERSION_ID"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "LABEL_REVISION_NOTE              =     \"2015-01-01,Thierry Semon(UoB),";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "                                        Version1.0 release\"";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "SOFTWARE_NAME                    =     ";
       string ttmp = "\""+sCurrentSoftwareName+"\"";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),ttmp.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "SOFTWARE_VERSION_ID              =     ";
       ttmp = "\""+sCurrentSoftwareVersionID+"\"";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),ttmp.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "ROSETTA:ROSINA_PIXEL0_A_MASS     =     ";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),(util_doubleToString(L3Info->cal->m0A)).c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "ROSETTA:ROSINA_PIXEL0_B_MASS     =     ";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),(util_doubleToString(L3Info->cal->m0B)).c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "SOURCE_FILE_NAME                 =     ";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),(L3Info->L2baseName).c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "ROSETTA:ROSINA_CAL_ID4           =     ";
    	if (L3Info->cal->calID4.compare("None") != 0) {
    		sEnd = L3Info->cal->calID4.length()-4;
    		ctmp = L3Info->cal->calID4.substr(0,sEnd);
    	} else {
    		ctmp = L3Info->cal->calID4;
    	}
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),ctmp.c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "ROSETTA:ROSINA_CAL_ID5           =     ";
    	if (L3Info->cal->calID5.compare("None") != 0) {
    		sEnd = L3Info->cal->calID5.length()-4;
    		ctmp = L3Info->cal->calID5.substr(0,sEnd);
    	} else {
    		ctmp = L3Info->cal->calID5;
    	}
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),ctmp.c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "ROSETTA:ROSINA_CAL_ID6           =     ";
    	if (L3Info->cal->calID6.compare("None") != 0) {
    		sEnd = L3Info->cal->calID6.length()-4;
    		ctmp = L3Info->cal->calID6.substr(0,sEnd);
    	} else {
    		ctmp = L3Info->cal->calID6;
    	}
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),ctmp.c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "RECORD_TYPE                      =     ";
    	PDSheader["RECORD_TYPE"] = util_trimEOL(PDSheader["RECORD_TYPE"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["RECORD_TYPE"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "RECORD_BYTES                     =     ";
    	PDSheader["RECORD_BYTES"] = util_trimEOL(PDSheader["RECORD_BYTES"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["RECORD_BYTES"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "FILE_RECORDS                     =     ";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),(util_intToString(lastLineNum)).c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "LABEL_RECORDS                    =     ";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),(util_intToString(lblRec)).c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "^DFMS_HK_TABLE                   =     ";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),(util_intToString(lblRec+1)).c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "^DFMS_MASS_CAL_TABLE             =     ";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),(util_intToString(lblRec+HKrows+1)).c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "^MCP_DATA_L3_TABLE               =     ";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),(util_intToString(lblRec+HKrows+CalRows+1)).c_str());
       os << string(tmp) << dfmsEOL;

    string newDName = PDSheader["DATA_SET_ID"];
    replace(newDName,"-2-","-3-");
    stmp = "DATA_SET_ID                      =     ";
    	newDName = util_trimEOL(newDName);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),newDName.c_str());
    	os << string(tmp) << dfmsEOL;

    newDName = PDSheader["DATA_SET_NAME"];
    replace(newDName,"2","3");
    stmp = "DATA_SET_NAME                    =     ";
		newDName = util_trimEOL(newDName);
		sprintf(tmp,"%-39s%-39s",stmp.c_str(),newDName.c_str());
		os << string(tmp) << dfmsEOL;

    stmp = "PRODUCT_ID                       =     ";
       string name = fileName.substr(0,26);
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),name.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "PRODUCT_CREATION_TIME            =     ";
       string tTime = getProcessTime(1);
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),tTime.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "PRODUCT_TYPE                     =     RDR";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "PROCESSING_LEVEL_ID              =     \"3\"";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "MISSION_ID                       =     ";
    	PDSheader["MISSION_ID"] = util_trimEOL(PDSheader["MISSION_ID"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["MISSION_ID"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "MISSION_NAME                     =     ";
    	PDSheader["MISSION_NAME"] = util_trimEOL(PDSheader["MISSION_NAME"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["MISSION_NAME"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "TARGET_NAME                      =     ";
    	PDSheader["TARGET_NAME"] = util_trimEOL(PDSheader["TARGET_NAME"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["TARGET_NAME"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "TARGET_TYPE                      =     ";
    	PDSheader["TARGET_TYPE"] = util_trimEOL(PDSheader["TARGET_TYPE"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["TARGET_TYPE"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "MISSION_PHASE_NAME               =     ";
    	PDSheader["MISSION_PHASE_NAME"] = util_trimEOL(PDSheader["MISSION_PHASE_NAME"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["MISSION_PHASE_NAME"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "INSTRUMENT_HOST_NAME             =     ";
    	PDSheader["INSTRUMENT_HOST_NAME"] = util_trimEOL(PDSheader["INSTRUMENT_HOST_NAME"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["INSTRUMENT_HOST_NAME"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "INSTRUMENT_HOST_ID               =     ";
    	PDSheader["INSTRUMENT_HOST_ID"] = util_trimEOL(PDSheader["INSTRUMENT_HOST_ID"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["INSTRUMENT_HOST_ID"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "INSTRUMENT_NAME                  =     ";
    	PDSheader["INSTRUMENT_NAME"] = util_trimEOL(PDSheader["INSTRUMENT_NAME"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["INSTRUMENT_NAME"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "INSTRUMENT_ID                    =     ";
    	PDSheader["INSTRUMENT_ID"] = util_trimEOL(PDSheader["INSTRUMENT_ID"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["INSTRUMENT_ID"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "INSTRUMENT_MODE_ID               =     ";
    	PDSheader["INSTRUMENT_MODE_ID"] = util_trimEOL(PDSheader["INSTRUMENT_MODE_ID"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["INSTRUMENT_MODE_ID"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "^INSTRUMENT_MODE_DESC            =     ";
    	PDSheader["^INSTRUMENT_MODE_DESC"] = util_trimEOL(PDSheader["^INSTRUMENT_MODE_DESC"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["^INSTRUMENT_MODE_DESC"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "INSTRUMENT_TYPE                  =     ";
    	PDSheader["INSTRUMENT_TYPE"] = util_trimEOL(PDSheader["INSTRUMENT_TYPE"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["INSTRUMENT_TYPE"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "DETECTOR_ID                      =     ";
    	PDSheader["DETECTOR_ID"] = util_trimEOL(PDSheader["DETECTOR_ID"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["DETECTOR_ID"].c_str());
        os << string(tmp) << dfmsEOL;

    stmp = "DETECTOR_DESC                    =     ";
    	PDSheader["DETECTOR_DESC"] = util_trimEOL(PDSheader["DETECTOR_DESC"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["DETECTOR_DESC"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "CHANNEL_ID                       =     ";
    	PDSheader["CHANNEL_ID"] = util_trimEOL(PDSheader["CHANNEL_ID"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["CHANNEL_ID"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "START_TIME                       =     ";
    	PDSheader["START_TIME"] = util_trimEOL(PDSheader["START_TIME"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["START_TIME"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "STOP_TIME                        =     ";
    	PDSheader["STOP_TIME"] = util_trimEOL(PDSheader["STOP_TIME"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["STOP_TIME"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "SPACECRAFT_CLOCK_START_COUNT     =     ";
    	PDSheader["SPACECRAFT_CLOCK_START_COUNT"] = util_trimEOL(PDSheader["SPACECRAFT_CLOCK_START_COUNT"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["SPACECRAFT_CLOCK_START_COUNT"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "SPACECRAFT_CLOCK_STOP_COUNT      =     ";
		PDSheader["SPACECRAFT_CLOCK_STOP_COUNT"] = util_trimEOL(PDSheader["SPACECRAFT_CLOCK_STOP_COUNT"]);
		sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["SPACECRAFT_CLOCK_STOP_COUNT"].c_str());
		os << string(tmp) << dfmsEOL;

    stmp = "PRODUCER_ID                      =     ";
    	PDSheader["PRODUCER_ID"] = util_trimEOL(PDSheader["PRODUCER_ID"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["PRODUCER_ID"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "PRODUCER_FULL_NAME               =     ";
    	PDSheader["PRODUCER_FULL_NAME"] = util_trimEOL(PDSheader["PRODUCER_FULL_NAME"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["PRODUCER_FULL_NAME"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "PRODUCER_INSTITUTION_NAME        =     ";
    	PDSheader["PRODUCER_INSTITUTION_NAME"] = util_trimEOL(PDSheader["PRODUCER_INSTITUTION_NAME"]);
    	sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["PRODUCER_INSTITUTION_NAME"].c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "DATA_QUALITY_ID                  =     ";
       string qid = "\""+util_intToString(L3Info->qualID)+"\"";
       sprintf(tmp,"%-39s%-39s",stmp.c_str(),qid.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "DATA_QUALITY_DESC                =     \"";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "      0 means 'Nominal quality, avg. PPM deviance < 500\'";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "      1 means 'Self-calibrated, GCU avg. PPM deviance >= 500, SELF < 500\'";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "      2 means 'Adopted mass scale avg. PPM deviance >= 500\'";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "      3 means 'Enhanced Noise\'";
       sprintf(tmp,"%-78s",stmp.c_str());
       os << string(tmp) << dfmsEOL;

    stmp = "      4 means 'Not enough peaks found for accurate calibration/verification\'\"";
    	sprintf(tmp,"%-78s",stmp.c_str());
    	os << string(tmp) << dfmsEOL;

    stmp = "SC_SUN_POSITION_VECTOR           =     ";
    	PDSheader["SC_SUN_POSITION_VECTOR"] = util_trimEOL(PDSheader["SC_SUN_POSITION_VECTOR"]);
        sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["SC_SUN_POSITION_VECTOR"].c_str());
        os << string(tmp) << dfmsEOL;

    stmp = "SC_TARGET_POSITION_VECTOR        =     ";
    	PDSheader["SC_TARGET_POSITION_VECTOR"] = util_trimEOL(PDSheader["SC_TARGET_POSITION_VECTOR"]);
        sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["SC_TARGET_POSITION_VECTOR"].c_str());
        os << string(tmp) << dfmsEOL;

    stmp = "SC_TARGET_VELOCITY_VECTOR        =     ";
    	PDSheader["SC_TARGET_VELOCITY_VECTOR"] = util_trimEOL(PDSheader["SC_TARGET_VELOCITY_VECTOR"]);
        sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["SC_TARGET_VELOCITY_VECTOR"].c_str());
        os << string(tmp) << dfmsEOL;

    stmp = "SPACECRAFT_ALTITUDE              =      ";
    	PDSheader["SPACECRAFT_ALTITUDE"] = util_trimEOL(PDSheader["SPACECRAFT_ALTITUDE"]);
        sprintf(tmp,"%-40.40s%-38.38s",stmp.c_str(),PDSheader["SPACECRAFT_ALTITUDE"].c_str());
        os << string(tmp) << dfmsEOL;

    stmp = "SUB_SPACECRAFT_LATITUDE          =      ";
    	PDSheader["SUB_SPACECRAFT_LATITUDE"] = util_trimEOL(PDSheader["SUB_SPACECRAFT_LATITUDE"]);
        sprintf(tmp,"%-40.40s%-38.38s",stmp.c_str(),PDSheader["SUB_SPACECRAFT_LATITUDE"].c_str());
        os << string(tmp) << dfmsEOL;

    stmp = "SUB_SPACECRAFT_LONGITUDE         =      ";
    	PDSheader["SUB_SPACECRAFT_LONGITUDE"] = util_trimEOL(PDSheader["SUB_SPACECRAFT_LONGITUDE"]);
        sprintf(tmp,"%-40.40s%-38.38s",stmp.c_str(),PDSheader["SUB_SPACECRAFT_LONGITUDE"].c_str());
        os << string(tmp) << dfmsEOL;

    stmp = "SPICE_FILE_NAME                  =     ";
    	PDSheader["SPICE_FILE_NAME"] = util_trimEOL(PDSheader["SPICE_FILE_NAME"]);
        sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["SPICE_FILE_NAME"].c_str());
        os << string(tmp) << dfmsEOL;

    stmp = "DESCRIPTION                      =     ";
    	PDSheader["DESCRIPTION"] = util_trimEOL(PDSheader["DESCRIPTION"]);
        sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["DESCRIPTION"].c_str());
        os << string(tmp) << dfmsEOL;

    stmp = "NOTE                             =     ";
    	PDSheader["NOTE"] = util_trimEOL(PDSheader["NOTE"]);
        sprintf(tmp,"%-39s%-39s",stmp.c_str(),PDSheader["NOTE"].c_str());
        os << string(tmp) << dfmsEOL;

    // Add an 79 character blank line to separate PDSheader from first Object descriptor
	sprintf(tmp,"%-78s"," ");
	os << string(tmp) << dfmsEOL;

	return 1;

}

//
// ------------------------------ writeHKdata ------------------------------------
//
int DFMS_PDS_L3_file_Class::writeHKdata(fstream &os) {

	string sFunctionName = "DFMS_PDS_L3_file_Class::writeHKdata";

	char line[80];
	string sOut;

	for (int i=0; i<HKrows; i++) {
		sprintf(line,"\"%-32s\",\"%-5s\",\"%-24s\",\"%-5s\"%1s",
		(L3HKdata[i][0]).c_str(),(L3HKdata[i][1]).c_str(),
		(L3HKdata[i][2]).c_str(),(L3HKdata[i][3]).c_str()," ");
		sOut = string(line);
		os << sOut << dfmsEOL;
	}
	
	return 1;	

}


//
// ------------------------------ writeCalData ------------------------------------
//
int DFMS_PDS_L3_file_Class::writeCalData(fstream &os) {

	string sFunctionName = "DFMS_PDS_L3_file_Class::writeCalData";

	char line[80];
	string sOut;
	string blank = " ";
     
	for (int i=0; i<CalRows; i++) {
		sprintf(line,"%17.17s,  %1.1s,  %1.1s,     %6.6s,   %6.6s,  %10.10s,   %10.10s,%3.3s",
		(util_trim(CalData[i][0])).c_str(), (CalData[i][1]).c_str(),
		(CalData[i][2]).c_str(), (CalData[i][3]).c_str(),
		(CalData[i][4]).c_str(), (CalData[i][5]).c_str(),
		(CalData[i][6]).c_str(), blank.c_str());

		sOut = string(line);
		os << sOut << dfmsEOL;
	}

	return 1;	

}


//
// ------------------------------ writeL3data ------------------------------------
//
int DFMS_PDS_L3_file_Class::writeL3data(fstream &os) {

	string sFunctionName = "DFMS_PDS_L3_file_Class::writeL3data";

	char line[80];
	string sOut;

	for (int i=0; i<NUMPIX; i++) {
        if (L3SET_MASS_PRECISION == 2) {   // Output 2 digit mass precision
            sprintf(line,"%3d,%17.2f,%16.1f,%17.2f,%16.1f,%4s",
                    (int)L3Data[i][0],L3Data[i][1],L3Data[i][2],
                    L3Data[i][3],L3Data[i][4]," ");
        } else if (L3SET_MASS_PRECISION == 3) {   // Output 3 digit mass precision
            sprintf(line,"%3d,%17.3f,%16.1f,%17.3f,%16.1f,%4s",
                    (int)L3Data[i][0],L3Data[i][1],L3Data[i][2],
                    L3Data[i][3],L3Data[i][4]," ");
        } else if (L3SET_MASS_PRECISION == 4) {   // Output 4 digit mass precision
            sprintf(line,"%3d,%17.4f,%16.1f,%17.4f,%16.1f,%4s",
                    (int)L3Data[i][0],L3Data[i][1],L3Data[i][2],
                    L3Data[i][3],L3Data[i][4]," ");
        } else if (L3SET_MASS_PRECISION == 5) {   // Output 5 digit mass precision
            sprintf(line,"%3d,%17.5f,%16.1f,%17.5f,%16.1f,%4s",
                    (int)L3Data[i][0],L3Data[i][1],L3Data[i][2],
                    L3Data[i][3],L3Data[i][4]," ");
        } else {
            sErrorMessage = "L3SET_MASS_PRECISION value not allowed \n";
            sErrorMessage += "L3SET_MASS_PRECISION = " + util_intToString(L3SET_MASS_PRECISION);
            sErrorMessage += "\n allowed values are 2,3,4, or 5";
            writeToLog(sErrorMessage, sFunctionName, FATALERROR);
            exit(EXIT_FAILURE);
        }
		sOut = string(line);
        //cout << sOut << endl;
		os << sOut << dfmsEOL;
	}	

	return 1;	
}

//
// -----------------------------------------  createL3file ----------------------------------------
//
int DFMS_PDS_L3_file_Class::createL3File() {

	string sFunctionName = "DFMS_PDS_L3_file_Class::createL3File";
	
	int rows,cols;
	fstream os;
	int nSuccess;
	
	// Open the L3 File stream for output
	openFile(L3File,os,0);

	// Form the HK data descp obj
	nSuccess = formHKobj();
	if (!nSuccess) {
		sErrorMessage = "Unable to form the L3 HK data Object \n";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}

	// Form the Mass Cal data descp obj
	nSuccess = formCalObj();
	if (!nSuccess) {
		sErrorMessage = "Unable to form the L3 Cal data Object \n";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}

	// Form the L3 data descp obj
	nSuccess = formL3DataObj();
	if (!nSuccess) {
		sErrorMessage = "Unable to form the L3 data Object\n";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}

	// write the L3 file PDS header
	// New PDS header applies for dates >= 01/1/2014 00:00:00
	if (isNewPDS) {
		nSuccess = writeNewPDSheader(os);
		if (!nSuccess) {
			sErrorMessage = "Unable to write the New PDS header for file: ";
			sErrorMessage += L3File;
			writeToLog(sErrorMessage, sFunctionName, ERROR);
			return 0;
		} else {
			//cout << "Writing NEW PDS header" << endl;
		}
	} else {
		nSuccess = writeOldPDSheader(os);
		if (!nSuccess) {
			sErrorMessage = "Unable to write the Old PDS header for file: ";
			sErrorMessage += L3File;
			writeToLog(sErrorMessage, sFunctionName, ERROR);
			return 0;
		} else {
			//cout << "Writing OLD PDS header" << endl;
		}
	}
	//if (verbose == 1) printL3File("PDSHeader");

	// Write the L3 file HK object description
	nSuccess = writeObj(os, HKobj, 0);
	if (!nSuccess) {
		sErrorMessage = "Unable to write the HK Object Descp. for file: ";
		sErrorMessage += L3File;
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}
	//if (verbose == 1) printL3File("L3HKobj");
	// Write the L3 file Cal Data object description
	nSuccess = writeObj(os, CalObj, 0);
	if (!nSuccess) {
		sErrorMessage = "Unable to write the Cal Object Descp. for file: ";
		sErrorMessage += L3File;
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}

	// Write the L3 Data object description
	nSuccess = writeObj(os, L3DataObj, 1);
	if (!nSuccess) {
		sErrorMessage = "Unable to write the L3 Object Descp. for file: ";
		sErrorMessage += L3File;
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}
	//if (verbose == 1) printL3File("L3CalObj");

	// Write the L3 file HK Data
	nSuccess = writeHKdata(os);
	if (!nSuccess) {
		sErrorMessage = "Unable to write the HK Data to file: ";
		sErrorMessage += L3File;
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}
	//if (verbose == 1) printL3File("L3HKdata");

	// Write the L3 file Cal Data
	nSuccess = writeCalData(os);
	if (!nSuccess) {
		sErrorMessage = "Unable to write the Cal Data to file: ";
		sErrorMessage += L3File;
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}
	//if (verbose == 1) printL3File("L3CalData");

	// Write the L3 file L3 Data
	nSuccess = writeL3data(os);
	if (!nSuccess) {
		sErrorMessage = "Unable to write the L3 Data to file: ";
		sErrorMessage += L3File;
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}
	//if (verbose == 1) printL3File("L3data");

	// Close the L3 File stream
	closeFile(os);

	return 1;

}

//
// ------------------------------ printL3File ------------------------------------
//
void DFMS_PDS_L3_file_Class::printL3File(string type) {

	// First print the PDS Header
	
	map<string,string>::const_iterator it;

	if (type == "PDSHeader" || type == "All") {
		for (it = PDSheader.begin(); it != PDSheader.end(); ++it) {
			printf("%-32s =     %-s\n",(it->first).c_str(),(it->second).c_str());
		}
		cout << endl;
	}

	// Next print out HK Object description
	if (type == "L3HKobj" || type == "All") {
		for (it = HKobj.begin(); it != HKobj.end(); ++it) {
			printf("%32s =     %-s\n",(it->first).c_str(),(it->second).c_str());
		}
		cout << endl;
	}

	// Next print out Cal Object description
	if (type == "L3Calobj" || type == "All") {
		for (it = CalObj.begin(); it != CalObj.end(); ++it) {
			printf("%32s =     %-s\n",(it->first).c_str(),(it->second).c_str());
		}
		cout << endl;
	}

	// Next print out Data Object description
	if (type == "L3dataObj" || type == "All") {
		for (it = L3DataObj.begin(); it != L3DataObj.end(); ++it) {
			printf("%32s =     %-s\n",(it->first).c_str(),(it->second).c_str());
		}
	     cout << "END" << endl;
	}

	// Next print out the HK data
	if (type == "L3HKdata" || type == "All") {
	 	for (int i=0; i<HKrows; i++) {
			printf("\"%-32s\",\"%-5s\",\"%-15s\",\"%-5s\"\n",
				(L3HKdata[i][0]).c_str(),
				(L3HKdata[i][1]).c_str(),
				(L3HKdata[i][2]).c_str(),
				(L3HKdata[i][3]).c_str());
		}
	}

	// Finally print out the data
	if (type == "CalData" || type == "All") {
		for (int i=0; i<CalRows; i++) {
			printf("%12s,%3s,%3s,%12s,%9s,%12s,%12s,\n",
				(CalData[i][0]).c_str(), (CalData[i][1]).c_str(), 
				(CalData[i][2]).c_str(), (CalData[i][3]).c_str(), 
				(CalData[i][4]).c_str(), (CalData[i][5]).c_str(), 
				(CalData[i][6]).c_str());
		}	
	}

	// Finally print out the data
	if (type == "L3data" || type == "All") {
		for (int i=0; i<NUMPIX; i++) {
			printf("%6f,%17.6f,%16.6f,%17.6f,%16.6f,\n",
				L3Data[i][0], L3Data[i][1], L3Data[i][2], L3Data[i][3], L3Data[i][4]);
		}	
	}

}

//
// --------------------------------------- DumpL3toFile ----------------------------------------
//
int DFMS_PDS_L3_file_Class::DumpL3toFile(string path) {

	string file=path + fileName.substr(0,fileName.length()-4)  + "_plot.dat";
	fstream os;
	os.open(file.c_str(), fstream::out | fstream::binary);
	os << "Pixel    Mass-A   IonRate-A    Mass-B    IoRate-B" << dfmsEOL;
	char tmp[80];
	for (int i=0; i<NUMPIX; ++i) {
		sprintf(tmp,"%5d    %f    %f     %f      %f",
				L3Data[i][0],L3Data[i][1],L3Data[i][2],
				L3Data[i][3],L3Data[i][4]);
		os << string(tmp) << dfmsEOL;
	}

	os.close();

	return 1;
}

//
// ------------------------------ quickPlotL2Data ---------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method writes the L3 data to a gnuplot file for a quick view.
//
// inputs:
//   path  - Path to output data
//   fileName - base filename from L2 file
//   type - Raw or Data Type
//
// returns:
//   1 for success, 0 for failure
// =============================================================================
// History: Written by Mike Rinaldi, Feb 2014
// =============================================================================
//
int DFMS_PDS_L3_file_Class::quickPlotL3Data(string path, string fileName,
													string type, string sstr) {


	string sFunctionName = "DFMS_PDS_L3_file_Class::quickPlotL3Data";

	string pFout;

	// the file to hold the gnuplot commands
	string file=path+fileName.substr(0,fileName.length()-4)+"_"+type+"_L3plot.dat";
	fstream os;
	os.open(file.c_str(), fstream::out | fstream::binary);

	// set the terminal type
	string term = setTerminal();

	// A row max ions
	double aMaxIons = -1.0 * pow(10.0,99);
	for (int i=0; i < NUMPIX; i++) {
		if (L3Data[i][2] > aMaxIons && L3Data[i][2] < pow(10.0,99)) {
			aMaxIons = L3Data[i][2];
		}
	}

	// B row max ions
	double bMaxIons = -1.0 * pow(10.0,99);
	for (int i=0; i < NUMPIX; i++) {
		if (L3Data[i][4] > bMaxIons && L3Data[i][4] < pow(10.0,99)) {
			bMaxIons = L3Data[i][4];
		}
	}

	double MaxIons=aMaxIons;
	if (aMaxIons < bMaxIons) MaxIons = bMaxIons;

	os << "Pixel    Counts    Fit" << dfmsEOL;
	char tmp[100];
	for (int i=0; i<NUMPIX; ++i) {
		sprintf(tmp,"%3d    %f    %f    %7.3f    %f    %f    %7.3f",
				i,L3Data[i][2],0.0,L3Data[i][1],L3Data[i][4],0.0,L3Data[i][3]);
		cout << string(tmp) << endl;
		os << string(tmp) << dfmsEOL;
	}

	string sMass = util_trim(util_trimChar(sstr,"\""));
	os.close();

	string fileBase = fileName.substr(0,fileName.length()-4)+"_"+type+"_L3plot.dat";

	// Set up a Gnuplot script and then use system call to plot
	if (iPltFile == 5) {					// plot L3 to terminal window
		pFout = string("# simple gnuplot script \n") +
				    "set term "+term+" size 900,1200 \n\n";
	} else if (iPltFile == 6) {				// plot L3 to postscript file
		string outfile = path+fileName.substr(0,fileName.length()-4)+"_"+type+"_L3plot.ps";
		//string outfile = sPlotFilePath+fileName.substr(0,fileName.length()-4)+"_"+type+"_plot.png";
		pFout = string("# simple gnuplot script \n") +
				"set terminal postscript portrait enhanced color font \'Helvetica,10\'\n" +
				//"set terminal pngcairo size 350,262 enhanced font \'Verdana,10\'\n" +
				"set output \'"+outfile+"\'\n";
	}
	pFout += string("# simple gnuplot script \n") +
			"set multiplot layout 2, 1 \n\n" +
			string("file1 = \"")+file+string("\"\n") +
			string("set title \"")+fileBase+string(" - DFMS Row A\n")+
			"set autoscale\n\n" +
			"set xlabel \"Pixel number\"\n" +
			"set x2label \"Mass Scale (for Mass: "+sMass+" )\"\n" +
			"set xrange [0:512]\n" +
			"set yrange [0:"+util_doubleToString(MaxIons)+"]\n" +
			"set x2range ["+util_doubleToString(L3Data[0][1])+":"+util_doubleToString(L3Data[NUMPIX-1][1])+"]\n"
			"set xtics nomirror \n" +
			"set x2tics \n" +
			"set grid ytics \n" +
			"set grid x2tics \n" +
			"set ylabel \"Ions/s\"\n\n" +
			"set style line 1 lt rgb \"blue\" lw 1\n\n" +
			"# the uncommented line below is short for\n" +
			"# plot file1 using 1:2 with lines\n" +
			"plot file1 us 1:2 title \"L3 Data\" ls 1 w l\n\n";

	// Plot Commands for side B
	if (term.compare("aqua") == 0) {  // no pause command on Mac OS X
		pFout += string("set title \"")+fileBase+string(" - DFMS Row B\n") +
			"set yrange [0:"+util_doubleToString(MaxIons)+"]\n" +
			"set x2range ["+util_doubleToString(L3Data[0][3])+":"+util_doubleToString(L3Data[511][3])+"]\n" +
			"plot file1 us 1:5 title \"L3 Data\" ls 1 w l\n";
	} else {
		pFout += string("set title \"")+fileBase+string(" - DFMS Row B\n") +
			"set yrange [0:"+util_doubleToString(MaxIons)+"]\n" +
			"set x2range ["+util_doubleToString(L3Data[0][3])+":"+util_doubleToString(L3Data[511][3])+"]\n" +
			"plot file1 us 1:5 title \"L3 Data\" ls 1 w l\n" +
			"pause 120\n";
	}


	string pfile=path+"plotFitAuto.p";
	os.open(pfile.c_str(), fstream::out | fstream::binary);
	os << pFout << dfmsEOL;
	os.close();

	if (util_fileExists(sGNUplotPath+"gnuplot")) {

		string comm = sGNUplotPath+"gnuplot "+pfile+"&";
		system(comm.c_str());

		// Delay for 1 sec to give gnuplot a chance to finish
		sleep(1000);
		return 1;
	} else {
		return 0;
	}
}
