
#include "DFMS_PDS_L2_dir_Class.hh"

//
// --------------------------- Constructor -------------------------------------
//
DFMS_PDS_L2_dir_Class::DFMS_PDS_L2_dir_Class (string p)  
{
	path = p;
	numFiles = 0;
	numDmodes=0;
}

//
// --------------------------- Destructor -----------------------------------
//
DFMS_PDS_L2_dir_Class::~DFMS_PDS_L2_dir_Class () {
    
     // Release dynamically allocated memory
     L2Info.clear();
     gcuLmLrFiles.clear();
     gcuHmLrFiles.clear();
     slfLmLrFiles.clear();
     slfHmLrFiles.clear();
     gcuLmHrFiles.clear();
     gcuMmHrFiles.clear();
     gcuHmHrFiles.clear();
     slfLmHrFiles.clear();
     slfMmHrFiles.clear();
     slfHmHrFiles.clear();
}

//
// ------------------------ getL2FilesInDir ----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Get the full pathnames of all L2 PDS file in the L2 data path
//
// inputs: None
//
// returns:
//   1 if success, 0 if failure
// =============================================================================
// History:  Written by Mike Rinaldi, Mar 2013
// =============================================================================
int DFMS_PDS_L2_dir_Class::getL2FilesInDir(int imode) {

     string sFunctionName="DFMS_PDS_L2_dir_Class::getL2FilesInDir";

  	DIR *dir;
  	struct dirent *entry;
  	L2FileInfo L;
  	allL2.clear();
  	long numGCU=0L;

	if ((dir = opendir(path.c_str())) == NULL) {
		sErrorMessage="Default L2 Source directory - "+path+" does not exist";
		writeToLog(sErrorMessage, sFunctionName, FATALERROR);
		return 0;
  	} else {
  		while ((entry = readdir(dir)) != NULL) {
  			fileName = string(entry->d_name);
  			//cout << fileName << endl;
			bool isL2 = verifyL2();
			// check if file is an MCP L2 PDS type
			if (isL2) {
				int mode = this->findL2Mode();
				// If -m CLA was use then only do intMode otherwise to them all
				if (intMode != 0 && intMode != mode) continue;
				dateTime = this->findL2FileDate();
				int fileSize = this->findL2FileSize();
				//Push back new FileInfo created with default constructor.
				L.fileName = fileName;
				L.dateTime = dateTime;
				L.fileSize = fileSize;
				L.secSince1970 = 0;		// This will be filled by getHKInfo() below;
				L.gainStep = 0.0;   	// This will be filled by getHKInfo() below;
				L.commandedMass = 0.0;  // This will be filled by getHKInfo() below;
				L.intgTime = 0.0;   	// This will be filled by getHKInfo() below;
				if (verbose >= 3) L.printData();
				//L.printData();
				allL2.push_back(L);
			}
  		}
  		closedir(dir);
  	}

	// Check if L2 files have been found. Return error if not
	if (allL2.size() == 0) {
		sErrorMessage="Default L2 Source directory - "+path+" does not contain expected L2 files";
		writeToLog(sErrorMessage, sFunctionName, FATALERROR);
		return 0;
	} else {
		sInfoMessage = "Completed cataloging "+util_intToString(allL2.size());
		sInfoMessage += " L2 files in Directory: "+path;
		writeToLog(sInfoMessage, sFunctionName, INFO);
		// Now run through all the allL2 files and get the relevant PDS and HK info
		// that will be needed later
		for (int i=0; i<allL2.size(); i++) {
			int nSuccess = getHKInfo(i, &numGCU);
			if (!nSuccess) {
				sErrorMessage = "Error in gathering HK Info from PDS File: "+fileName;
				writeToLog(sErrorMessage, sFunctionName, ERROR);
				return 0;
			}
		}
		//allL2[fId].printData();
		// get the number of distinct modes contained in the L2 files
		numDmodes = distinctModes.size();

		// Find the min and max L2 time
		for (int i=0; i<allL2.size(); i++) {
			if (allL2[i].secSince1970 > L2TimeMax) L2TimeMax = allL2[i].secSince1970;
			if (allL2[i].secSince1970 < L2TimeMin) L2TimeMin = allL2[i].secSince1970;
			//cout << "allL2[" << i << "] = " << allL2[i].secSince1970 << endl;
		}

		printf("allL2[0].secSince1970 = %14.1f\n",allL2[0].secSince1970);
		printf("L2TimeMin = %14.1f\n",L2TimeMin);
		printf("L2TimeMax = %14.1f\n",L2TimeMax);


		// Sort All L2 files by GCU mode/date and then remaining nonGCU files by date.
		sortL2Files();

		// Warn if there are an insufficient number of GCU files
		if (!DOCEM) {
			if (numGCU < MIN4GCUFIT) {
				sInfoMessage="Warning: There are less than "+ util_intToString(MIN4GCUFIT);
				sInfoMessage += " GCU files in this batch of L2 files";
				writeToLog(sInfoMessage, sFunctionName, INFO);
				cout << sInfoMessage << endl;
			}
		}
    }

	return 1;

}

//
// --------------------- findL2FileSize ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Check if filename is a coherent L2 PDS filename
//
// inputs: None
//
// returns:
//   true if good filename false if bad
// =============================================================================
// History:  Written by Mike Rinaldi, Mar 2013
// =============================================================================
bool DFMS_PDS_L2_dir_Class::verifyL2() {

	bool fileStatus = true;
	string prefix, modePart, suffix;

	int len = fileName.length();
	if (len != 31) {
		return false;
	}

	suffix = fileName.substr(fileName.find_last_of(".") + 1);
	if ((suffix.compare("tab") == 0 || suffix.compare("TAB") == 0)) {
		fileStatus = true;
	} else {
		fileStatus = false;
	}

	prefix = fileName.substr(0,3);
	modePart = fileName.substr(22,1);
	if (prefix.compare("MC_") != 0 && prefix.compare("CE_") != 0) fileStatus = false;
	if (modePart.compare("M") != 0 ) fileStatus = false;

	return fileStatus;
}

//
// --------------------- findL2FileSize ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Returns the number of bytes in L2 file
//
// inputs: None
//
// returns:
//   bytes in file
// =============================================================================
// History:  Written by Mike Rinaldi, Mar 2013
// =============================================================================
int DFMS_PDS_L2_dir_Class::findL2FileSize()
{
	
	string sFunctionName="DFMS_PDS_L2_dir_Class::findL2FileSize";

	struct stat st;
	stat((path+fileName).c_str(), &st);
	int fileSize = st.st_size;

	return fileSize;

}

//
// --------------------- findResState ---------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Returns the L2 file Resolution Mode (Low or High)
//
// inputs: None
//
// returns:
//   true if Low Res, false if High Res
// =============================================================================
// History:  Written by Mike Rinaldi, Mar 2013
// =============================================================================
bool DFMS_PDS_L2_dir_Class::findResState() {
    
    string sFunctionName="DFMS_PDS_L2_dir_Class::findResState";
    
    // Return true if in Low Res State.  False if in High Res State
    //size_t dpos = fileName.find_last_of(".");
    //string sRes = fileName.substr(dpos-1,1);

	string tfile = fileName;
	std::transform(tfile.begin(), tfile.end(), tfile.begin(), ::toupper);

    size_t dpos = tfile.find_last_of("M");
    string sRes = tfile.substr(dpos+4,1);

    int iRes = util_stringToInt(sRes);
    if (iRes <= 1 || iRes == 5 || iRes == 6) {
        return true;          // Low Res mode
    } else if (iRes == 2 || iRes == 3 || iRes == 7 || iRes == 8) {
        return false;         // High Res mode
    } else {
    	sErrorMessage = "Cannot determine Resolution mode, sRes =  "+sRes+"/n";
    	writeToLog(sErrorMessage, sFunctionName, ERROR);
        return true;        
    }
    
}

//
// ---------------------- findL2Mode ---------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Returns the DFMS file mode
//
// inputs: none
//
// returns:
//   DFMS file mode
// =============================================================================
// History:  Written by Mike Rinaldi, Mar 2013
// =============================================================================
int DFMS_PDS_L2_dir_Class::findL2Mode()
{
	int mode = 0;
	
	string sFunctionName="DFMS_PDS_L2_dir_Class::findL2Mode";

	string tfile = fileName;
	std::transform(tfile.begin(), tfile.end(), tfile.begin(), ::toupper);
	size_t dpos = tfile.find_last_of("M");
	string smode = tfile.substr(dpos+1,4);
	mode = util_stringToInt(smode);

	return mode;

}

//
// -------------------- findL2FileDate --------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Returns the portion of the filename that represents the file date
//
// inputs: none
//
// returns:
//   the file date string
// =============================================================================
// History:  Written by Mike Rinaldi, Mar 2013
// =============================================================================
string DFMS_PDS_L2_dir_Class::findL2FileDate()
{
	
	string sFunctionName="DFMS_PDS_L2_dir_Class::findL2FileDate";

	size_t dpos = fileName.find_first_of("_");
	string fileDate = fileName.substr(dpos+1,18);

	// In case it's needed
	//int yr = util_stringToInt(fileDate.substr(0,4));
	//int mon = util_stringToInt(fileDate.substr(4,2));
	//int day = util_stringToInt(fileDate.substr(6,2));
	//int hr = util_stringToInt(fileDate.substr(9,2));
	//int min = util_stringToInt(fileDate.substr(11,2));
	float sec = (atof((fileDate.substr(13,5)).c_str()))/1000.;
	

	return fileDate;
}

//
// --------------------- sortL2Files ---------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Perform a sort on the set of L2 files
//
// inputs: None
//
// returns:
//   None
// =============================================================================
// History:  Written by Mike Rinaldi, Mar 2013
// =============================================================================
void DFMS_PDS_L2_dir_Class::sortL2Files()
{

	string sFunctionName="DFMS_PDS_L2_dir_Class::sortL2Files";

		string numL2 = util_intToString(allL2.size());

		if (allL2.size() <= 1) {
			sErrorMessage = "Default L2 Source directory - "+path+" does not ";
			sErrorMessage += "contain enough L2 files to sort.  Need at least 2 but have: "+numL2;
			writeToLog(sErrorMessage, sFunctionName, ERROR);
		}

		// use C++ STL sort algorithm function with user supplied comparator
		//stable_sort(allL2.begin(), allL2.end(), util_compare);

		vector<L2FileInfo> SLFL2;
		vector<L2FileInfo> nonSLFL2;
		vector<L2FileInfo> GCUL2;

		// Now build an SLF L2 vector
		for (int i=0; i< allL2.size(); i++) {
			const bool is_in = slfSet.find(allL2[i].commandedMass) != slfSet.end();
			if (!is_in && allL2[i].isGCU) {
				GCUL2.push_back(allL2[i]);
			} else if (is_in && !allL2[i].isGCU) {
				SLFL2.push_back(allL2[i]);
			} else if (is_in && allL2[i].isGCU) {
				GCUL2.push_back(allL2[i]);
			} else {
				nonSLFL2.push_back(allL2[i]);
			}
		}

		// Sort each one separately
		stable_sort(GCUL2.begin(), GCUL2.end(), util_compare_date);
		stable_sort(SLFL2.begin(), SLFL2.end(), util_compare_date);
		stable_sort(nonSLFL2.begin(), nonSLFL2.end(), util_compare_date);

		L2Info.reserve( GCUL2.size() + SLFL2.size() + nonSLFL2.size() ); // preallocate memory
		// Now insert each in the order GU/SLF/nonSLF
		L2Info.insert( L2Info.end(), GCUL2.begin(), GCUL2.end() );
		L2Info.insert( L2Info.end(), SLFL2.begin(), SLFL2.end() );
		L2Info.insert( L2Info.end(), nonSLFL2.begin(), nonSLFL2.end() );

		// Simply dumps the L2 files to process in the specific order in which they will be
		// processed to a time tagged file.  dumpL2Order is set on the command line as (-O)
        if (dumpL2Order) {
        	char l2out[200];
        	string line;
        	int k=0;
        	fstream os;
        	string l2date = L2Info[0].dateTime.substr(0,8);
        	string file = "L2orderOfProcessing_"+l2date+".txt";
        	os.open (file.c_str(), fstream::out | fstream::binary);
        	if (!os.is_open()) {
        	    cout << "Error opening file:" << file << endl;
        	}
        	for(vector<L2FileInfo>::iterator it=L2Info.begin(); it != L2Info.end(); it++) {
            	string type = "nonGCUnonSLF";
        		const bool is_in = slfSet.find(it->commandedMass) != slfSet.end();
        		if (it->isGCU) type = "GCU";
        		if (is_in) type = "SLF";
        		if (it->isGCU && is_in) type = "GCU";
        		if (!it->isGCU && !is_in) string type = "nonGCUnonSLF";
        		if (k % 50 == 0) os << dfmsEOL << "Index               File Name                        Type    comm Mass" << dfmsEOL << dfmsEOL;
        		sprintf(l2out,"%5.5d     %s    %12.12s     %6.2f\n",k++,it->fileName.c_str(),type.c_str(),it->commandedMass);
        		line.assign(l2out);
        		os << line;
        		//printf("  -  rosTime = %14.1f",it->secSince1970);
        	}

        	os.close();

        	cout << "Created file: " << file << " located in DFMS_INSTALL_DIR" << endl;

        	exit(EXIT_SUCCESS);
        }


}

//
// ------------------------------ getHKInfo ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Run quickly through all the L2 file and look for the commanded mass, Gain step,
// start/stop time, accumulation time, and
// number of accumulations.
//
// inputs:
//  fId - The index number of the sorted L2 file
//
// returns:
//   1 if success, 0 if failure
// =============================================================================
// History:  Written by Mike Rinaldi, July 2014
// =============================================================================
int DFMS_PDS_L2_dir_Class::getHKInfo(int fId, long *numGCU) {

	string sFunctionName="DFMS_PDS_L2_dir_Class::getHKInfo";

	char input[100];
	string line;
	fstream is;
	vector<string> tokens;
	bool done = false;
	int numFound=0;
	double accumTime=0.0;
	int scanCnts = 0;
    double switch1 = 0.0;
    double switch2 = 0.0;

	// HouseKeeping items needed
	string mtitle[NUMREQHK] = {"ROSINA_DFMS_SCI_MASS",
			            	    "START_TIME",
			            	    "STOP_TIME",
			            	    "ROSINA_DFMS_SCI_GAIN_OF_SPECTRUM",
			            	    "ROSINA_DFMS_LEDA_CEM_ACCU_TIME",
			            	    "ROSINA_DFMS_ACCU_SCAN_COUNTS",
			            	    "ROSINA_DFMS_FIL_EMISSION",
			            	    "ROSINA_DFMS_SCI_MODE",
                                "ROSINA_DFMS_ENTR_SLIT_SWITCH2",
                                "ROSINA_DFMS_ENTR_SLIT_SWITCH1"};
	bool found[NUMREQHK] = {false,false,false,false,false,false,false,false};
	vector<string> itemsFound;

	// Quickly run through the L2 file and look for the items in the mtitle array
	// above
    string file = path+allL2[fId].fileName;
	is.open(file.c_str(), fstream::in);
	if(!is.is_open()) {
		sErrorMessage = "Error opening for reading PDS File: "+file;
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}
	// Run through the entire L2 file and look for lines with one
	// of each of the items in mtitle above until all NUMREQHK are found
	int linecnt=0;
	while(!done) {
		is.getline(input,100);
		linecnt++;
		if (is.eof() || linecnt >= 840) break;
		line.assign(input);
		if (line.length() > 80) {
			sErrorMessage =  "Error - L2 file: "+file+"\n";
			sErrorMessage += "  line overrun found - line size = " + util_intToString(line.length())+" > 80\n";
			sErrorMessage += "  line = " + line+"\n";
			sErrorMessage += "\n            ***** TERMINATING CODE EXECUTION ****** \n";
			cout << sErrorMessage;
			writeToLog(sErrorMessage, sFunctionName, FATALERROR);
			toLog.close();
			exit(EXIT_FAILURE);
		}
		for (int i=0; i<NUMREQHK; ++i) {
			size_t pos = line.find(mtitle[i]);
			if (pos != line.npos && !found[i]) {
				numFound++;
				found[i] = true;
				itemsFound.push_back(line);
				break;
			}
		}
		if (numFound == NUMREQHK) done=true;
	}

	// Make sure all NUMREQHK items were found
	for (int i=0; i<NUMREQHK; ++i) {
		if (!found[i]) {
			sErrorMessage = "Failed to find: "+ mtitle[i] + "in PDS File: "+fileName;
			writeToLog(sErrorMessage, sFunctionName, ERROR);
			return 0;
		}
	}

	string res;
	string name;
	// Set the various allL2 quantities needed Later
	for (int i=0; i<NUMREQHK; ++i) {
		string line = itemsFound[i];
		//cout << "line = " << line << endl;
		if (line.find(',') != line.npos) {        // HK value
            tokens.clear();
			util_splitString(line, ',', tokens);
			name = util_trim(util_trimChar(tokens[0],"\""));
            //cout << "name = " << name << endl;
			if (name.compare("ROSINA_DFMS_FIL_EMISSION") == 0) {
				res = util_trim(util_trimChar(tokens[1],"\""));
				int mpos = res.find("u");
				allL2[fId].emission = util_stringToDouble(res.substr(0,mpos))*1.0e-6;
				continue;
			} else {
				res = util_trim(util_trimChar(tokens[2],"\""));
			}
			tokens.clear();
		} else if (line.find('=') != line.npos) { // PDSHeader value
			util_splitString(line, '=', tokens);
			name = util_trim(tokens[0]);
			res = util_trim(tokens[1]);
			tokens.clear();
		}
		if (name.compare("ROSINA_DFMS_SCI_MASS") == 0) {
			allL2[fId].commandedMass = util_stringToDouble(res);
			const bool is_in = slfSet.find(allL2[fId].commandedMass) != slfSet.end();
			if (is_in) {
				if (!DOCEM) {
					allL2[fId].isSLF = true;
				} else {
					allL2[fId].isSLF = false;
				}
			} else {
				allL2[fId].isSLF = false;
			}
			//cout << "allL2[fId].commandedMass = " << allL2[fId].commandedMass << endl;
		} else if (name.compare("ROSINA_DFMS_SCI_MODE") == 0) {
			allL2[fId].mode = util_stringToInt(res);
			int mode = allL2[fId].mode;
			allL2[fId].lowRes = modeData[mode].lowRes;
			allL2[fId].hiRes = modeData[mode].hiRes;
			//L.mode = mode;   // mode is now set below in getHKInfo()
			// Use a map container to keep track of unique modes
			distinctModes.insert(pair<int,bool>(mode,true));
			allL2[fId].isGCU = modeData[mode].isGCU;
			if (allL2[fId].isGCU) {
				*numGCU = *numGCU+1;
			} else {
				allL2[fId].isGCU = false;
			}
			allL2[fId].emission = modeData[mode].emission;
			allL2[fId].zoom = modeData[mode].zoom;
		} else if (name.compare("START_TIME") == 0) {
			allL2[fId].startTime = res;
		} else if (name.compare("STOP_TIME") == 0) {
			allL2[fId].stopTime = res;
		} else if (name.compare("ROSINA_DFMS_SCI_GAIN_OF_SPECTRUM") == 0) {
			allL2[fId].gainStep = util_stringToDouble(res);
		} else if (name.compare("ROSINA_DFMS_LEDA_CEM_ACCU_TIME") == 0) {
			accumTime = util_stringToDouble(res);
		} else if (name.compare("ROSINA_DFMS_ACCU_SCAN_COUNTS") == 0) {
			scanCnts = util_stringToInt(res);
        } else if (name.compare("ROSINA_DFMS_ENTR_SLIT_SWITCH1") == 0) {
            switch1 = util_stringToDouble(res);
        } else if (name.compare("ROSINA_DFMS_ENTR_SLIT_SWITCH2") == 0) {
            switch2 = util_stringToDouble(res);
		} else {
			sInfoMessage = "Unknown HK variable in PDS File: "+fileName;
			writeToLog(sErrorMessage, sFunctionName, WARN);

		}
	}
    
    // Set the zoom factor for mode 0 or 9999.
    if ((abs((int) switch2) < 10) && ((abs((int) switch1)< 10))) {
        allL2[fId].zoom = 6.4;
    } else {
        allL2[fId].zoom = 1.0;
    }

	// Form the integration time and Julian date in sec from Start Time
	allL2[fId].intgTime = accumTime/1000.0*scanCnts; // in sec
	allL2[fId].secSince1970 = rosToSec(allL2[fId].startTime, 0);

	return 1;

}



//
// -------------------------- getFileIndex -----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Return the sorted index of of the L2 file in the array
//
// inputs:
//  dfmsFileName - the L2 file name
//
// returns:
//   the L2 array file id
// =============================================================================
// History:  Written by Mike Rinaldi, Mar 2013
// =============================================================================
int DFMS_PDS_L2_dir_Class::getFileIndex(string dfmsFileName) {

	for (int i=0; i<L2Info.size(); ++i) {
		if (dfmsFileName.compare(L2Info[i].fileName) == 0) {
			return i;
		}
	}

	return -1;

}

//
// ------------------ printL2FileList -------------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Prints the name of the L2 file for each member of the L2 file array
//
// inputs: None
//
// returns:
//   None
// =============================================================================
// History:  Written by Mike Rinaldi, Mar 2013
// =============================================================================
void DFMS_PDS_L2_dir_Class::printL2FileList() 
{

	int k=0;

	vector<L2FileInfo>::iterator it;
	cout << "Total number of files: " << L2Info.size() << endl;
	cout << "List of L2 files in path: " << path << endl;
	cout << "-------------------------------------------------------------------" << endl;
	for(it=L2Info.begin() ; it < L2Info.end(); it++ ) { 
		printf("%3.3d : %s is GCU: %d%s", k++, ((*it).fileName).c_str(), (*it).isGCU, EOL);
	}
	cout << "-------------------------------------------------------------------" << endl;
	
}

//
// ------------------ printL2FileList -----------------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Prints the info associated with each L2 file in the L2 array
//
// inputs:
//  k - The index to the L2 file array
//
// returns:
//   None
// =============================================================================
// History:  Written by Mike Rinaldi, Mar 2013
// =============================================================================
void DFMS_PDS_L2_dir_Class::printFileInfo(int k)
{

	string ans;

	cout << "-----------------------------------------------------------" << endl;
	cout << "------------------ L2 file Info ---------------------------" << endl;
	cout << "-----------------------------------------------------------" << endl;
	cout << " Name:      " << L2Info[k].fileName << endl;
	cout << " Date:      " << L2Info[k].dateTime << endl;
	cout << " Size:      " << L2Info[k].fileSize << " bytes" << endl;
	cout << " Mode:      " << L2Info[k].mode << endl;
	cout << " IsGCU:     " << (ans = (L2Info[k].isGCU) ? "Yes" : "No") << endl;
	cout << " startTime: " << L2Info[k].startTime << endl;
	cout << " stopTime:  " << L2Info[k].stopTime << endl;
	cout << endl << endl;

}

//
// ------------------------ processPix0FitData ---------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This routine will look through all the sorted L2 files and create pix0 fit files
// for both GCU and SLF type files if possible.
//
// inputs:
//  sMassPeakTablePath - The path to the MPS table
//  pgtPath - The path to the Pixel Gain table
//  begId - the number of the first L2 file to process (first in sort order)
//  endId - the number of the last L2 file to process (first in sort order)
//  gTObject - The overall gain object pointer
//  modeIDobject - The mode ID object pointer
//  pgObj - the Pixel Gain Interpolation Object
//  PGTFileInfo - the DirInfo structure containing the Pixel Gain info
//  gcuMPSTinfo - The relevant information contained in the GCU MPS table
//  nonGCUMPSTinfo - The relevant information contained in the nonGCU MPS table
//
// output:
//  numPix0Created - the Number of pix0 fit files created in this process
//
// returns:
//   1 if success, 0 if failure
// =============================================================================
// History:  Written by Mike Rinaldi, July 2014
// =============================================================================
int DFMS_PDS_L2_dir_Class::processPix0FitData(int begId, int endId,
			DFMS_GainTable_Interp_Class* gtObject, vector<gtFileInfo> &GTFileInfo,
            DFMS_ModeID_Table_Class* modeIDobject, DFMS_PixelGain_Interp_Class *pgObj,
            vector<pgFileInfo> &PGTFileInfo, vector<L2FileInfo> &gcuMPSTinfo,
            vector<L2FileInfo> &nonGCUMPSTinfo, int *numPix0Created)

{

	string sFunctionName = "DFMS_PDS_L2_dir_Class::processPix0FitData";

	DFMSModeInfo modeInfo;
	int nSuccess=0;
	string type, massType;
	string dateStrt;

	double maxSecsInGCUPix0Fit = NUMDAYSTOFITGCU*24*3600.0;
	double maxSecsInSLFPix0Fit = NUMDAYSTOFITSLF*24*3600.0;

	// Create the pix0 fit files from the new GCU/SLF L2 files.  A minimum of 4 GCU files are
	// need to create a new fit and a min of 3 SLF files.  There is also an upper bound on the amount
	// of time separating the GCU/SLF files. For GCU it's 7 days for SLF 1 day.

	// STEP 11.1- Get GCU files to use and put them into vector gcuFiles
	nSuccess = gatherGCUL2Files(begId, endId, maxSecsInGCUPix0Fit);

	// STEP 11.2 - Get SLF files to use and put them into vector slfFiles
	nSuccess = gatherSLFL2Files(begId, endId, maxSecsInSLFPix0Fit);

	// ------------ GCU Block ----------
	// STEP 11.3 - Do the Low Mass/Low Res GCU files
	if (gcuLmLrFiles.size() >= MIN4GCUFIT) {
		cout << "Trying gcuLmLrFiles files" << endl;
		nSuccess = findPix0ForGCUSLFType(begId, endId, "GCU", "L", "L", gcuLmLrFiles, gtObject,
				 GTFileInfo, modeIDobject, pgObj, PGTFileInfo, gcuMPSTinfo, nonGCUMPSTinfo, numPix0Created);
	}
	// STEP 11.4 - Do the High Mass/Low Res GCU files
	if (gcuHmLrFiles.size() >= MIN4GCUFIT) {
		cout << "Trying gcuHmLrFiles files" << endl;
		nSuccess = findPix0ForGCUSLFType(begId, endId, "GCU", "H", "L", gcuHmLrFiles, gtObject,
				 GTFileInfo, modeIDobject, pgObj, PGTFileInfo, gcuMPSTinfo, nonGCUMPSTinfo, numPix0Created);
	}
	// STEP 11.5 - Do the Low Mass/High Res GCU files
	if (gcuLmHrFiles.size() >= MIN4GCUFIT) {
		cout << "Trying gcuLmHrFiles files" << endl;
		nSuccess = findPix0ForGCUSLFType(begId, endId, "GCU", "L", "H", gcuLmHrFiles, gtObject,
				 GTFileInfo, modeIDobject, pgObj, PGTFileInfo, gcuMPSTinfo, nonGCUMPSTinfo, numPix0Created);
	}
	// STEP 11.6 - Do the Medium Mass/High Res GCU files
	if (gcuMmHrFiles.size() >= MIN4GCUFIT) {
		cout << "Trying gcuMmHrFiles files" << endl;
		nSuccess = findPix0ForGCUSLFType(begId, endId, "GCU", "M", "H", gcuMmHrFiles, gtObject,
				 GTFileInfo, modeIDobject, pgObj, PGTFileInfo, gcuMPSTinfo, nonGCUMPSTinfo, numPix0Created);
	}
	// STEP 11.7 - Do the High Mass/High Res GCU files
	if (gcuHmHrFiles.size() >= MIN4GCUFIT) {
		cout << "Trying gcuHmHrFiles files" << endl;
		nSuccess = findPix0ForGCUSLFType(begId, endId, "GCU", "H", "H", gcuHmHrFiles, gtObject,
				 GTFileInfo, modeIDobject, pgObj, PGTFileInfo, gcuMPSTinfo, nonGCUMPSTinfo, numPix0Created);
	}

	// ------------ SLF Block -----------
	// STEP 11.8 - Do the Low Mass/Low Res SLF files
	if (slfLmLrFiles.size() >= MIN4SLFFIT && !onlyOneDistinctMass(slfLmLrFiles)) {
		cout << "Trying slfLmLrFiles files" << endl;
		nSuccess = findPix0ForGCUSLFType(begId, endId, "SLF", "L", "L", slfLmLrFiles, gtObject,
				 GTFileInfo, modeIDobject, pgObj, PGTFileInfo, gcuMPSTinfo, nonGCUMPSTinfo, numPix0Created);
	}
	// STEP 11.9 - Do the High Mass/Low Res SLF files
	if (slfHmLrFiles.size() >= MIN4SLFFIT && !onlyOneDistinctMass(slfHmLrFiles)) {
		cout << "Trying slfHmLrFiles files" << endl;
		nSuccess = findPix0ForGCUSLFType(begId, endId, "SLF", "H", "L", slfHmLrFiles, gtObject,
				 GTFileInfo, modeIDobject, pgObj, PGTFileInfo, gcuMPSTinfo, nonGCUMPSTinfo, numPix0Created);
	}
	// STEP 11.10 - Do the Low Mass/High Res SLF files
	if (slfLmHrFiles.size() >= MIN4SLFFIT && !onlyOneDistinctMass(slfLmHrFiles)) {
		cout << "Trying slfLmHrFiles files" << endl;
		nSuccess = findPix0ForGCUSLFType(begId, endId, "SLF", "L", "H", slfLmHrFiles, gtObject,
				 GTFileInfo, modeIDobject, pgObj, PGTFileInfo, gcuMPSTinfo, nonGCUMPSTinfo, numPix0Created);
	}
	// STEP 11.11 - Do the Medium Mass/High Res SLF files
	if (slfMmHrFiles.size() >= MIN4SLFFIT && !onlyOneDistinctMass(slfMmHrFiles)) {
		cout << "Trying slfMmHrFiles files" << endl;
		nSuccess = findPix0ForGCUSLFType(begId, endId, "SLF", "M", "H", slfMmHrFiles, gtObject,
				 GTFileInfo, modeIDobject, pgObj, PGTFileInfo, gcuMPSTinfo, nonGCUMPSTinfo, numPix0Created);
	}
	// STEP 11.12 - Do the High Mass/High Res SLF files
	if (slfHmHrFiles.size() >= MIN4SLFFIT && !onlyOneDistinctMass(slfHmHrFiles)) {
		cout << "Trying slfHmHrFiles files" << endl;
		nSuccess = findPix0ForGCUSLFType(begId, endId, "SLF", "H", "H", slfHmHrFiles, gtObject,
				 GTFileInfo, modeIDobject, pgObj, PGTFileInfo, gcuMPSTinfo, nonGCUMPSTinfo, numPix0Created);
	}

	return 1;

}

//
// -------------------------- findPix0ForGCUSLFType ---------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This routine will run through all the file of type and massType/resType and
// if there are enough it will perform a linear fit
//
//  inputs:
//  begId - the number of the first L2 file to process (first in sort order)
//  endId - the number of the last L2 file to process (first in sort order)
//  gTObject - The overall gain object pointer
//  modeIDobject - The mode ID object pointer
//  pgObj - the Pixel Gain Interpolation Object
//  PGTFileInfo - the DirInfo structure containing the Pixel Gain info
//  gcuMPSTinfo - The relevant information contained in the GCU MPS table
//  nonGCUMPSTinfo - The relevant information contained in the nonGCU MPS table
//
//  outputs:
//  numPix0Created - the number of fits created.
//
// returns:
//   1 if success, 0 if failure
// =============================================================================
// History:  Written by Mike Rinaldi, July 2014
// =============================================================================
int DFMS_PDS_L2_dir_Class::findPix0ForGCUSLFType (int begId, int endId, string type, string massType,
				string resType, vector<fitFileInfo> &typeFile, DFMS_GainTable_Interp_Class* gtObject,
				vector<gtFileInfo> &GTFileInfo, DFMS_ModeID_Table_Class* modeIDobject,
                DFMS_PixelGain_Interp_Class *pgObj, vector<pgFileInfo> &PGTFileInfo,
                vector<L2FileInfo> &gcuMPSTinfo, vector<L2FileInfo> &nonGCUMPSTinfo,
                int *numPix0Created)
{

	string sFunctionName = "DFMS_PDS_L2_dir_Class::findPix0ForGCUSLFType";

	string L2F = "";

	DFMSModeInfo modeInfo;
	DFMS_CreateGCUSLFPix0_Class* doOneL2=NULL;
	DFMS_PDS_L2_file_Class* L2Obj=NULL;
	int nSuccess=0;
	set<double> distinctMasses;

	string dateStrt;
	cout << "typeFile[0].dateTime = " << typeFile[0].dateTime << endl;
	dateStrt = (typeFile[0].dateTime).substr(0,15);

	cout << "There are a total of " << typeFile.size() << " file types to process" << endl;

	for (int i=0; i<typeFile.size(); i++) {

		// The index of the L2Info file to use
		int id = typeFile[i].L2Id;

		// STEP 11.x.0 - Ascertain Date and compare to 1/3/2015
	    // On December 27,2014 the GCU stopped working.  In order to continue processing data
	    // the code will use SLF masses to act as surrogates in place of the GCU masses.  The code
	    // will first ascertain whether the file being processed is pre or post 12/27/14 + 1 week.
		// It will use this information to either use the original mode of calibration or the new one
	    // based on SLF fits. The post GCU available date is set at one week past the
	    // last available GCU data or 1/3/2015.
		// We first get the seconds since 1970 for the date January 3,2015.  This represents 1 week
		// after which the GCU stopped working.  After this date we will use the SLF offset
		// and slope.  Before this date we will instead use the SLF offset and GCU slope.
		string sJan032015 = "20150103_000000000";
		double jan032015 = rosToSec(sJan032015, 1);
		double thisFileDate = rosToSec(L2Info[id].dateTime, 1);

		// Check if we are in the time domain before or after the GCU failure
		if (thisFileDate < jan032015) {
			isRecentGCUavailable = true;
		} else {
			isRecentGCUavailable = false;
		}

		// Check for New PDS Header time boundary. New L2 PDS Header is assumed for
		// all time after 01/01/2014 00:00:00
		string sJan012014 = "20140101_000000000";
		double jan012014 = rosToSec(sJan012014, 1);
		if (thisFileDate < jan012014) {
			isNewPDS = false;
		} else {
			isNewPDS = true;
		}

        // Check if file date is in exclusion zone.  Files with dates within this
        // zone will not be processed for various reasons.
        bool exclude = dateException(thisFileDate);
        if (exclude) {
        	cout << "Pix0 Loop - Skipped - File is within date/time exclusion region" << endl;
        	continue;
        }

		// STEP 11.x.1 - Get Mode Info
		modeInfo = modeData[L2Info[id].mode];
		//modeInfo.printData();

		// Get the current L2 file name
		string L2File = L2Info[id].fileName;

		cout << endl;
        cout << "--------------------- Trying (findPix0ForGCUSLFType): " << i << " of " << typeFile.size();
        cout << " - L2 File: " << L2File << " - commandedMass = " << L2Info[id].commandedMass;
        cout << " ---------------------" << endl;

		// STEP 11.x.2 - Get Best Mass Peak Table file - sets name to global variable, sBestMPST
		if (L2Info[id].isGCU) {
			nSuccess = DFMS_MPST_Class::findBestMPSTfile(sMassPeakTablePath,
								L2Info[id].dateTime, L2Info[id].isGCU, gcuMPSTinfo);
		} else {
			nSuccess = DFMS_MPST_Class::findBestMPSTfile(sMassPeakTablePath,
   		                        L2Info[id].dateTime, L2Info[id].isGCU, nonGCUMPSTinfo);
		}
		if (!nSuccess) {
			skipReas["noMPSTfile"]++;
			sErrorMessage = "Pix0 Loop -  Unable to find an appropriate MPST file: ";
			sErrorMessage += sBestMPST+"\n";
			writeToLog(sErrorMessage, sFunctionName, ERROR);
			totFilesSkipped++;
			cout << sFunctionName << " - Pix0 Loop -  - Unable to find an appropriate MPST file" << endl;
			continue;
		} else {
			cout << "Pix0 Loop - Best MPST file found: " << sBestMPST << endl;
			sInfoMessage = "Pix0 Loop - Best MPST file found: "+sBestMPST;
			writeToLog(sInfoMessage, sFunctionName, INFO);
		}

		// STEP 11.x.3 - Get Best Pixel Gain Table data.  If None or Singular GS use closest
		// PG Table in GS and Time.  If multiple GS available then use interpolation between existing
		// PG Tables for the L2 time and Gain Step.
		DFMS_PixelGain_Table_Class *pixGTObject = NULL;
		int GSin = L2Info[id].gainStep;
		vector<pgFileInfo> PGSin;
		int mGS = pgObj->findMulti_GS_PGT(GSin, PGTFileInfo, PGSin);
		if (mGS <= 1) {
			int GSuse = 0;
			string dType = "";
            string subDNme = "";
			nSuccess = DFMS_PixelGain_Table_Class::findBestPGTtime(pgtPath, L2Info[id].dateTime,
   							 	 	 	 	 	 GSin, &GSuse, dType, subDNme, PGTFileInfo);
			if (!nSuccess) {
				sErrorMessage = "Initial Loop - Unable to find Best Pixel Gain Table Time ";
				writeToLog(sErrorMessage, sFunctionName, ERROR);
				cout << sFunctionName << " - Initial Loop - Unable to find Best Pixel Gain Table Time " << endl;
				exit(EXIT_FAILURE);
			}

			// Create pixGT fileName
			char tmp[10];
			sprintf(tmp,"%2.2d",GSuse);
			string sGS = string(tmp);
			string pgfile = sPixelGainTableBaseName+sBestPGTtime+"_"+dType+"_"+sInstrumentModel+"_GS"+sGS+".TAB";
			cout << "Using Pixel Gain File: " << pgfile << endl;
			// Create the Pixel Gain Table file object
			string pgpath = pgtPath+subDNme+"/";
			//cout << "Pix0 Loop - Using Pixel Gain Table Folder: " << pgpath << endl;

			// Declare the pixel Gain Table object then get the best pixel gain table data
			// based on the nearest time and closest Gain Step file
			pixGTObject = new DFMS_PixelGain_Table_Class(pgpath, pgfile);
			//nSuccess = DFMS_get_PixelGain_Table(pgpath, pgfile, pixGTObject);
			nSuccess = pixGTObject->getPGTable(pgpath, pgfile);
			if (!nSuccess) {
				sErrorMessage = "Pix0 Loop - Reading Pixel Gain Table was unsuccessful ";
				writeToLog(sErrorMessage, sFunctionName, ERROR);
				cout << sFunctionName << " - Pix0 Loop - Reading Pixel Gain Table was unsuccessful " << endl;
				exit(EXIT_FAILURE);
			}
		} else {
			// When PGSin contains 2 or more PG Tables for this GS use simple DFMS_PixelGain_Table_Class
        	// constructor.  This will create a simple PG Table object that will simply allow the usage
        	// of the pixGTdata field later in the processing.  Instead of reading pixel Gain data from
        	// file use the results of the interpolation to populate the pixGTdata array.
        	// Fill pixGTObject->pixGTdata with data from interpolation.
			pixGTObject = new DFMS_PixelGain_Table_Class();
			// Build the pixGTObject->pixGTdata array
			nSuccess = pgObj->buildPixGTdata(pixGTObject, GSin, thisFileDate, PGSin);
			if (!nSuccess) {
				// This should never happen but....
				sErrorMessage = "Pix0 Loop - Pixel Gain Interpolation was unsuccessful for unknown reason.";
				writeToLog(sErrorMessage, sFunctionName, ERROR);
				if (L3INFOTOFILE) l3Log << sErrorMessage << dfmsEOL;
				cout << sErrorMessage << endl;
				cout << " - Pix0 Loop - Skipped - Pixel Gain Interpolation was unsuccessful for unknown reason." << endl;
				continue;
			}
		}
        
        // Now using the current L2 file date interpolate to find the correct Gain from the
        // available tables
        nSuccess = gtObject->buildGTdata(thisFileDate, GTFileInfo);
        if (!nSuccess) {
            // This should never happen but....
            sErrorMessage = "Gain Table Interpolation was unsuccessful for unknown reason.";
            writeToLog(sErrorMessage, sFunctionName, ERROR);
            if (L3INFOTOFILE) l3Log << sErrorMessage << dfmsEOL;
            cout << sErrorMessage << endl;
            delete L2Obj; L2Obj=0;
            delete L3Info, L3Info=0;
            totFilesSkipped++;
            cout << " - Skipped - Gain Table Interpolation was unsuccessful for unknown reason." << endl;
            continue;
        }

		// STEP 11.x.4 - Get the overall gain for this L2 file
		double gainA = gtObject->gtFit[GSin-1][2];
		double gainB = gtObject->gtFit[GSin-1][3];

		// STEP 11.x.5 - Get the L2 file
		L2Obj = new DFMS_PDS_L2_file_Class(sL2sourcePath,L2File);
		nSuccess = L2Obj->getL2file(L2Info[id]);
		if (!nSuccess) {
			sErrorMessage = "Pix0 Loop - Unable to open/read/form L2 file: "+L2File;
			sErrorMessage += " - Skipping file processing";
			writeToLog(sErrorMessage, sFunctionName, ERROR);
			delete L2Obj; L2Obj=0;
			cout << sFunctionName << " - Pix0 Loop - Skipped - Unable to open/read/form proper L2 file" << endl;
			continue;
		}

		// STEP 11.x.6 - Create the pix0 Fit file object that does all the work
		doOneL2 = new DFMS_CreateGCUSLFPix0_Class( sMassPeakTablePath, sx0FitsDataPath,
											gainA, gainB, pixGTObject->pixGTdata, L2Obj);

		//typeFile[i].printData(type);

		// STEP 11.x.7 - Process an L2 File and find the pix0/m0 value
		nSuccess = doOneL2->processL2forPix0Fit(L2Info[id], modeInfo, type, distinctMasses);
		if (!nSuccess) {
			sErrorMessage = "Pix0 Loop - Creation of GCU or SLF pix0 file from L2 file: "+L2File+" NOT successful";
			writeToLog(sErrorMessage, sFunctionName, ERROR);
			// Keep track of the L2 files that failed so that I can remove them
			// from the L2Info list.
			badL2Ids.push_back(id);
		} else {
			sInfoMessage = "Pix0 Loop - L2 File: "+L2File+" was successfully used for pix0 fit";
			writeToLog(sInfoMessage, sFunctionName, INFO);
		}

		// copy the gcu fit parameters for later use
		gcuP = doOneL2->gcuP;

		// STEP 11.x.8 - Release dynamically allocated memory of local variables
		delete doOneL2;  doOneL2=0;
		delete L2Obj;  L2Obj=0;
		delete pixGTObject; pixGTObject=0;

	}

	// Dump all the variables collected to do the Linear fit
	if (verbose >= 3) printFitVars(type, massType, resType);
    
	// STEP 11.x.9 - If there are enough points then do the fit and write it to file
	int numInFitA = 0;
	int numInFitB = 0;
	int min4Fit = 0;
	if (type.compare("GCU") == 0) {
		min4Fit = MIN4GCUFIT;
		if (resType.compare("L") == 0) {
			if (massType.compare("L") == 0) {
				numInFitA = gcuLmLrA.size();
				numInFitB = gcuLmLrB.size();
				if (gcuLmLrA.size() > 0) L2F = gcuLmLrA[0].file;
			} else if (massType.compare("H") == 0) {
				numInFitA = gcuHmLrA.size();
				numInFitB = gcuHmLrB.size();
				if (gcuHmLrA.size() > 0) L2F = gcuHmLrA[0].file;
			}
		} else if (resType.compare("H") == 0) {
			if (massType.compare("L") == 0) {
				numInFitA = gcuLmHrA.size();
				numInFitB = gcuLmHrB.size();
				if (gcuLmHrA.size() > 0) L2F = gcuLmHrA[0].file;
			} else if (massType.compare("M") == 0) {
				numInFitA = gcuMmHrA.size();
				numInFitB = gcuMmHrB.size();
				if (gcuMmHrA.size() > 0) L2F = gcuMmHrA[0].file;
			}else if (massType.compare("H") == 0) {
				numInFitA = gcuHmHrA.size();
				numInFitB = gcuHmHrB.size();
				if (gcuHmHrA.size() > 0) L2F = gcuHmHrA[0].file;
			}
		}
	} else if (type.compare("SLF") == 0) {
		min4Fit = MIN4SLFFIT;
		if (resType.compare("L") == 0) {
			if (massType.compare("L") == 0) {
				numInFitA = slfLmLrA.size();
				numInFitB = slfLmLrB.size();
				if (slfLmLrA.size() > 0) L2F = slfLmLrA[0].file;
			} else if (massType.compare("H") == 0) {
				numInFitA = slfHmLrA.size();
				numInFitB = slfHmLrB.size();
				if (slfHmLrA.size() > 0) L2F = slfHmLrA[0].file;
			}
		} else if (resType.compare("H") == 0) {
			if (massType.compare("L") == 0) {
				numInFitA = slfLmHrA.size();
				numInFitB = slfLmHrB.size();
				if (slfLmHrA.size() > 0) L2F = slfLmHrA[0].file;
			} else if (massType.compare("M") == 0) {
				numInFitA = slfMmHrA.size();
				numInFitB = slfMmHrB.size();
				if (slfMmHrA.size() > 0) L2F = slfMmHrA[0].file;
			} else if (massType.compare("H") == 0) {
				numInFitA = slfHmHrA.size();
				numInFitB = slfHmHrB.size();
				if (slfHmHrA.size() > 0) L2F = slfHmHrA[0].file;
			}
		}
	}

	// STEP 11.x.10 - Create the Pix0 fit file if appropriate files values exist
	if (numInFitA >= min4Fit && numInFitB >= min4Fit) {
		// Create the file name for new fit file
		string x0FileName = sx0FitsDataPath+"X0_"+type+"_"+dateStrt+"_" +
				massType+"M"+resType+"R"+".TAB";
		cout << "x0FileName = " << x0FileName << endl;
		// Open the pix0 fit file object and write file
		DFMS_X0FitFile_Class *x0Fit = new DFMS_X0FitFile_Class(sx0FitsDataPath, x0FileName,
										  	  	  	  	  	  	numInFitA, numInFitB);
		nSuccess = x0Fit->doTheLinFit(type, massType, resType, gcuP);
		if (type.compare("SLF") == 0 && !isRecentGCUavailable &&
				resType.compare("L") == 0 && massType.compare("H") == 0) {
			//exit(0);
		}
		// Now write the pix0 fit file
		if (nSuccess) {
			x0Fit->writePix0FitFile(type, massType, resType, L2F);
			numPix0Created++;
		} else {
			sInfoMessage = "Pix0 Loop - Linfit returned offset for either row A or B = 0";
			sInfoMessage += " This maybe due to a single m0 present";
			writeToLog(sInfoMessage, sFunctionName, INFO);
		}
		delete x0Fit;
	} else {
   		sInfoMessage = "Pix0 Loop - Not enough "+ massType + "mass "+resType+"res of type "+type;
   	    sInfoMessage += " files successfully fit to create a pix0 vs m0 fit";
   		writeToLog(sInfoMessage, sFunctionName, INFO);
	}

	return 1;
}

//
// -------------------------- gatherGCUL2Files ---------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This routine will look through all the sorted GCU L2 files and organize
// the files into blocks of at least 4 files and not greater than 7 days in extent.
//
// inputs:
//  maxSecsInGCUPix0Fit - The maximum time to be used for making a pix0 fit
//
// returns:
//   1 if success, 0 if failure
// =============================================================================
// History:  Written by Mike Rinaldi, July 2014
// =============================================================================
int DFMS_PDS_L2_dir_Class::gatherGCUL2Files(int begId, int endId, double maxTime) {

	string sFunctionName = "DFMS_PDS_L2_dir_Class::gatherGCUL2Files";

	fitFileInfo gT;

	gT.fitPeriod = 1;
	double initTime = 0.0;
	for (int i=begId; i<endId; i++) {
		// Look for GCU files but skip those that have HK ROSINA_DFMS_SCI_MASS = 0.0
		if (L2Info[i].isGCU && L2Info[i].commandedMass > 0.0) {
			if (i == begId) initTime = L2Info[i].secSince1970;
			if (L2Info[i].secSince1970-initTime > gT.fitPeriod*maxTime) {
				gT.fitPeriod++;
			}
			gT.L2Id = i;
			gT.fileName = L2Info[i].fileName;
			gT.dateTime = L2Info[i].dateTime;
			gT.timeTag = L2Info[i].secSince1970;
			gT.type = "GCU";
			gT.commandedMass = L2Info[i].commandedMass;
			if (L2Info[i].lowRes) {
				if (L2Info[i].commandedMass < GCULOWCUTOFF) {
					gT.massType = "L";
					gT.resType = "L";
					gcuLmLrFiles.push_back(gT);
				} else {
					gT.massType = "H";
					gT.resType = "L";
					gcuHmLrFiles.push_back(gT);
				}
			} else {
				if (L2Info[i].commandedMass < GCULOWCUTOFF) {
					//cout << "filename = " << L2Info[i].fileName << endl;
					//cout << "mass = " << L2Info[i].commandedMass << endl;
					gT.massType = "L";
					gT.resType = "H";
					gcuLmHrFiles.push_back(gT);
				} else if (L2Info[i].commandedMass >= GCULOWCUTOFF && L2Info[i].commandedMass < GCUHICUTOFF) {
					gT.massType = "M";
					gT.resType = "H";
					gcuMmHrFiles.push_back(gT);
				} else {
					gT.massType = "H";
					gT.resType = "H";
					gT.printData("S");
					gcuHmHrFiles.push_back(gT);
				}
			}
		}
	}

	if (gcuLmLrFiles.size() < MIN4GCUFIT) {
		sInfoMessage = "Not enough Low Mass/Low Res GCU files found to create a New GCU pix0 fit file";
		writeToLog(sInfoMessage, sFunctionName, INFO);
	} else {
		sInfoMessage = "Found "+ util_intToString(gcuLmLrFiles.size());
		sInfoMessage += " Low Mass/Low Res GCU files which maybe used to create a New GCU pix0 fit file";
		cout << sInfoMessage << endl;
		if (L3INFOTOFILE) l3Log << sInfoMessage << dfmsEOL;
		writeToLog(sInfoMessage, sFunctionName, INFO);
	}

	if (gcuHmLrFiles.size() < MIN4GCUFIT) {
		sInfoMessage = "Not enough High Mass/Low Res GCU files found to create a New GCU pix0 fit file";
		writeToLog(sInfoMessage, sFunctionName, INFO);
	} else {
		sInfoMessage = "Found "+ util_intToString(gcuHmLrFiles.size());
		sInfoMessage += " High Mass/Low Res GCU files which maybe used to create a New GCU pix0 fit file";
		cout << sInfoMessage << endl;
		if (L3INFOTOFILE) l3Log << sInfoMessage << dfmsEOL;
		writeToLog(sInfoMessage, sFunctionName, INFO);
	}

	if (gcuLmHrFiles.size() < MIN4GCUFIT) {
		sInfoMessage = "Not enough Low Mass/High Res GCU files found to create a New GCU pix0 fit file";
		writeToLog(sInfoMessage, sFunctionName, INFO);
	} else {
		sInfoMessage = "Found "+ util_intToString(gcuLmHrFiles.size());
		sInfoMessage += " Low Mass/High Res GCU files which maybe used to create a New GCU pix0 fit file";
		cout << sInfoMessage << endl;
		if (L3INFOTOFILE) l3Log << sInfoMessage << dfmsEOL;
		writeToLog(sInfoMessage, sFunctionName, INFO);
	}

	if (gcuMmHrFiles.size() < MIN4GCUFIT) {
		sInfoMessage = "Not enough Medium Mass High Res GCU files found to create a New GCU pix0 fit file";
		writeToLog(sInfoMessage, sFunctionName, INFO);
	} else {
		sInfoMessage = "Found "+ util_intToString(gcuMmHrFiles.size());
		sInfoMessage += " Medium Mass/High Res GCU files which maybe used to create a New GCU pix0 fit file";
		cout << sInfoMessage << endl;
		if (L3INFOTOFILE) l3Log << sInfoMessage << dfmsEOL;
		writeToLog(sInfoMessage, sFunctionName, INFO);
	}

	if (gcuHmHrFiles.size() < MIN4GCUFIT) {
		sInfoMessage = "Not enough High Mass/High Res GCU files found to create a New GCU pix0 fit file";
		writeToLog(sInfoMessage, sFunctionName, INFO);
	} else {
		sInfoMessage = "Found "+ util_intToString(gcuHmHrFiles.size());
		sInfoMessage += " High Mass/High Res GCU files which maybe used to create a New GCU pix0 fit file";
		cout << sInfoMessage << endl;
		if (L3INFOTOFILE) l3Log << sInfoMessage << dfmsEOL;
		writeToLog(sInfoMessage, sFunctionName, INFO);
	}

	return 1;

}

//
// -------------------------- gatherSLFL2Files ---------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This routine will look through all the sorted nonGCU L2 files and organize
// the files into blocks of at least 4 files and not greater than 7 days in extent.
//
// inputs:
//  maxSecsInGCUPix0Fit - The maximum time to be used for making a pix0 fit
//
// returns:
//   1 if success, 0 if failure
// =============================================================================
// History:  Written by Mike Rinaldi, July 2014
// =============================================================================
int DFMS_PDS_L2_dir_Class::gatherSLFL2Files(int begId, int endId, double maxTime) {

	string sFunctionName = "DFMS_PDS_L2_dir_Class::gatherSLFL2Files";

	fitFileInfo sT;
	bool isIn=false;

	sT.fitPeriod = 1;
	bool foundFirst = true;
	double initTime = 0.0;
	for (int i=begId; i<endId; i++) {
		if (!L2Info[i].isGCU) {
			isIn = slfSet.find(L2Info[i].commandedMass) != slfSet.end();
			if (isIn) {
				if (foundFirst) {
					initTime = L2Info[i].secSince1970;
					foundFirst = false;
				}
				sT.L2Id = i;
				sT.fileName = L2Info[i].fileName;
				sT.dateTime = L2Info[i].dateTime;
				sT.timeTag = L2Info[i].secSince1970;
				sT.type = "SLF";
				sT.commandedMass = L2Info[i].commandedMass;
				if (L2Info[i].lowRes) {
					if (L2Info[i].commandedMass < SLFLOWCUTOFF) {
						if (L2Info[i].secSince1970-initTime > sT.fitPeriod*maxTime) {
							sT.fitPeriod++;
						}
						sT.massType = "L";
						sT.resType = "L";
						slfLmLrFiles.push_back(sT);
					} else {
						sT.massType = "H";
						sT.resType = "L";
						slfHmLrFiles.push_back(sT);
					}
				} else if (L2Info[i].hiRes) {
					if (L2Info[i].commandedMass < SLFLOWCUTOFF) {
						sT.massType = "L";
						sT.resType = "H";
						slfLmHrFiles.push_back(sT);
					} else if (L2Info[i].commandedMass >= SLFLOWCUTOFF && L2Info[i].commandedMass < SLFHICUTOFF) {
						sT.massType = "M";
						sT.resType = "H";
						slfMmHrFiles.push_back(sT);
					} else {
						sT.massType = "H";
						sT.resType = "H";
						slfHmHrFiles.push_back(sT);
					}
				}
			}
		}
	}

	// Low Mass / Low Res
	if (slfLmLrFiles.size() < MIN4SLFFIT || onlyOneDistinctMass(slfLmLrFiles)) {
		sInfoMessage = "Not enough Low Mass/Low Res SLF files found OR ";
		sInfoMessage += "only one distinct mass Found. Therefore no New SLF pix0 fit file";
		writeToLog(sInfoMessage, sFunctionName, INFO);
	} else {
		sInfoMessage = "Found "+ util_intToString(slfLmLrFiles.size());
		sInfoMessage += " Low Mass/Low Res SLF files which maybe used to create a New SLF pix0 fit file";
		cout << sInfoMessage << endl;
		if (L3INFOTOFILE) l3Log << sInfoMessage << dfmsEOL;
		writeToLog(sInfoMessage, sFunctionName, INFO);
	}
	// High Mass / Low Res
	if (slfHmLrFiles.size() < MIN4SLFFIT || onlyOneDistinctMass(slfHmLrFiles)) {
		sInfoMessage = "Not enough High Mass/Low Res SLF files found OR ";
		sInfoMessage += "only one distinct mass Found. Therefore no New SLF pix0 fit file";
		writeToLog(sInfoMessage, sFunctionName, INFO);
	} else {
		sInfoMessage = "Found "+ util_intToString(slfHmLrFiles.size());
		sInfoMessage += " High Mass/Low Res SLF files which maybe used to create a New SLF pix0 fit file";
		cout << sInfoMessage << endl;
		if (L3INFOTOFILE) l3Log << sInfoMessage << dfmsEOL;
		writeToLog(sInfoMessage, sFunctionName, INFO);
	}
	// Low Mass / High Res
	if (slfLmHrFiles.size() < MIN4SLFFIT || onlyOneDistinctMass(slfLmHrFiles)) {
		sInfoMessage = "Not enough Low Mass/High Res SLF files found OR ";
		sInfoMessage += "only one distinct mass Found. Therefore no New SLF pix0 fit file";
		writeToLog(sInfoMessage, sFunctionName, INFO);
	} else {
		sInfoMessage = "Found "+ util_intToString(slfLmHrFiles.size());
		sInfoMessage += " Low Mass/High Res SLF files which maybe used to create a New SLF pix0 fit file";
		cout << sInfoMessage << endl;
		if (L3INFOTOFILE) l3Log << sInfoMessage << dfmsEOL;
		writeToLog(sInfoMessage, sFunctionName, INFO);
	}
	// Medium Mass / High Res
	if (slfMmHrFiles.size() < MIN4SLFFIT || onlyOneDistinctMass(slfMmHrFiles)) {
		sInfoMessage = "Not enough Medium Mass/High Res SLF files found OR ";
		sInfoMessage += "only one distinct mass Found. Therefore no New SLF pix0 fit file";
		writeToLog(sInfoMessage, sFunctionName, INFO);
	} else {
		sInfoMessage = "Found "+ util_intToString(slfMmHrFiles.size());
		sInfoMessage += " Medium Mass/High Res SLF files which maybe used to create a New SLF pix0 fit file";
		cout << sInfoMessage << endl;
		if (L3INFOTOFILE) l3Log << sInfoMessage << dfmsEOL;
		writeToLog(sInfoMessage, sFunctionName, INFO);
	}
	// Low Mass / High Res
	if (slfHmHrFiles.size() < MIN4SLFFIT || onlyOneDistinctMass(slfHmHrFiles)) {
		sInfoMessage = "Not enough High Mass/High Res SLF files found OR ";
		sInfoMessage += "only one distinct mass Found. Therefore no New SLF pix0 fit file";
		writeToLog(sInfoMessage, sFunctionName, INFO);
	} else {
		sInfoMessage = "Found "+ util_intToString(slfHmHrFiles.size());
		sInfoMessage += " High Mass/High Res SLF files which maybe used to create a New SLF pix0 fit file";
		cout << sInfoMessage << endl;
		if (L3INFOTOFILE) l3Log << sInfoMessage << dfmsEOL;
		writeToLog(sInfoMessage, sFunctionName, INFO);
	}


	return 1;

}

//
//  -------------------------- onlyOneDistinctMass ---------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Check if there is only one distinct mass
//
// inputs:
//   typeFile - fit file type
//
// returns:
//   true if only one distinct mass, false otherwise
// =============================================================================
// History:  Written by Mike Rinaldi September 2014
// =============================================================================
//
bool DFMS_PDS_L2_dir_Class::onlyOneDistinctMass(vector<fitFileInfo> &typeFile) {

	// Just put all the masses into an STL set type.  This container
	// will automatically sort and only hold unique values.
	set<double> masses;
	for (int i=0; i<typeFile.size(); i++) {
		masses.insert(typeFile[i].commandedMass);
		//cout << "typeFile[i].commandedMass = " << typeFile[i].commandedMass << endl;
	}

	// Check if there are more than one distinct masses
	if (masses.size() <= 1) {
		return true;
	} else {
		return false;
	}

}

//
//  -------------------------- printGCUFitVars ---------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Print the contents of the Global variables gcuLA, gcuHA, gcuLA,...
//
// inputs:
//   type - The file type (GCU or SLF)
//
// returns:
//   None
// =============================================================================
// History:  Written by Mike Rinaldi July 2014
// =============================================================================
//
void DFMS_PDS_L2_dir_Class::printFitVars(string type, string massType, string resType) {

	char tmp[100];
	string line;
	if (type.compare("GCU") == 0) {
		if (resType.compare("L") == 0) {
			if (massType.compare("L") == 0) {
				cout << endl;
				cout << "------------------------------------------------------" << endl;
				cout << "   GCU Lm/Lr values used to construct New Fit File" << endl;
				cout << "------------------------------------------------------" << endl;
				cout << "   #     m0     m     pix0     pix0Unc             fileName" << endl;
				cout << "------------------------------------------------------" << endl;
				for(int i=0; i<gcuLmLrA.size(); i++) {
					sprintf(tmp,"  %-3.3d  %-6.2f  %-6.2f  %-6.2f  %-6.2f  %s\n",
							i,gcuLmLrA[i].m0, gcuLmLrA[i].m, gcuLmLrA[i].pix0, gcuLmLrA[i].pix0Unc,
							gcuLmLrA[i].file.c_str());
					line.assign(tmp);
					cout << line;
				}
			} else if (massType.compare("H") == 0) {
				cout << endl;
				cout << "------------------------------------------------------" << endl;
				cout << "   GCU Hm/Lr values used to construct New Fit File" << endl;
				cout << "------------------------------------------------------" << endl;
				cout << "   #     m0     m     pix0     pix0Unc             fileName" << endl;
				cout << "------------------------------------------------------" << endl;
				for(int i=0; i<gcuHmLrA.size(); i++) {
					sprintf(tmp,"  %-3.3d  %-6.2f  %-6.2f  %-6.2f  %-6.2f  %s\n",
							i,gcuHmLrA[i].m0, gcuHmLrA[i].m, gcuHmLrA[i].pix0, gcuHmLrA[i].pix0Unc,
							gcuHmLrA[i].file.c_str());
					line.assign(tmp);
					cout << line;
				}
			}
		} else if (resType.compare("H") == 0) {
			if (massType.compare("L") == 0) {
				cout << endl;
				cout << "------------------------------------------------------" << endl;
				cout << "   GCU Lm/Hr values used to construct New Fit File" << endl;
				cout << "------------------------------------------------------" << endl;
				cout << "   #     m0     m     pix0     pix0Unc             fileName" << endl;
				cout << "------------------------------------------------------" << endl;
				for(int i=0; i<gcuLmHrA.size(); i++) {
					sprintf(tmp,"  %-3.3d  %-6.2f  %-6.2f  %-6.2f  %-6.2f  %s\n",
							i,gcuLmHrA[i].m0, gcuLmHrA[i].m, gcuLmHrA[i].pix0, gcuLmHrA[i].pix0Unc,
							gcuLmHrA[i].file.c_str());
					line.assign(tmp);
					cout << line;
				}
			} else if (massType.compare("M") == 0) {
				cout << endl;
				cout << "------------------------------------------------------" << endl;
				cout << "   GCU Mm/Hr values used to construct New Fit File" << endl;
				cout << "------------------------------------------------------" << endl;
				cout << "   #     m0     m     pix0     pix0Unc             fileName" << endl;
				cout << "------------------------------------------------------" << endl;
				for(int i=0; i<gcuMmHrA.size(); i++) {
					sprintf(tmp,"  %-3.3d  %-6.2f  %-6.2f  %-6.2f  %-6.2f  %s\n",
							i,gcuMmHrA[i].m0, gcuMmHrA[i].m, gcuMmHrA[i].pix0, gcuMmHrA[i].pix0Unc,
							gcuMmHrA[i].file.c_str());
					line.assign(tmp);
					cout << line;
				}
			} else if (massType.compare("H") == 0) {
				cout << endl;
				cout << "------------------------------------------------------" << endl;
				cout << "   GCU Hm/Hr values used to construct New Fit File" << endl;
				cout << "------------------------------------------------------" << endl;
				cout << "   #     m0     m     pix0     pix0Unc             fileName" << endl;
				cout << "------------------------------------------------------" << endl;
				for(int i=0; i<gcuHmHrA.size(); i++) {
					sprintf(tmp,"  %-3.3d  %-6.2f  %-6.2f  %-6.2f  %-6.2f  %s\n",
							i,gcuHmHrA[i].m0, gcuHmHrA[i].m, gcuHmHrA[i].pix0, gcuHmHrA[i].pix0Unc,
							gcuHmHrA[i].file.c_str());
					line.assign(tmp);
					cout << line;
				}
			}
		}
	} else if (type.compare("SLF") == 0) {
		if (resType.compare("L") == 0) {
			if (massType.compare("L") == 0) {
				cout << endl;
				cout << "------------------------------------------------------" << endl;
				cout << "  SLF Lm/Lr values used to construct New Fit File" << endl;
				cout << "------------------------------------------------------" << endl;
				cout << "   #     m0     m     pix0     pix0Unc             fileName" << endl;
				cout << "------------------------------------------------------" << endl;
				for(int i=0; i<slfLmLrA.size(); i++) {
					sprintf(tmp,"  %-3.3d  %-6.2f  %-6.2f  %-6.2f  %-6.2f  %s\n",
							i,slfLmLrA[i].m0, slfLmLrA[i].m, slfLmLrA[i].pix0, slfLmLrA[i].pix0Unc,
							slfLmLrA[i].file.c_str());
					line.assign(tmp);
					cout << line;
				}
			} else if (massType.compare("H") == 0) {
				cout << endl;
				cout << "------------------------------------------------------" << endl;
				cout << " SLF Hm/Lr values used to construct New Fit File" << endl;
				cout << "------------------------------------------------------" << endl;
				cout << "   #     m0     m     pix0     pix0Unc             fileName" << endl;
				cout << "------------------------------------------------------" << endl;
				for(int i=0; i<slfHmLrA.size(); i++) {
					sprintf(tmp,"  %-3.3d  %-6.2f  %-6.2f  %-6.2f  %-6.2f  %s\n",
							i,slfHmLrA[i].m0, slfHmLrA[i].m, slfHmLrA[i].pix0, slfHmLrA[i].pix0Unc,
							slfHmLrA[i].file.c_str());
					line.assign(tmp);
					cout << line;
				}
			}
		} else if (resType.compare("H") == 0) {
			if (massType.compare("L") == 0) {
				cout << endl;
				cout << "------------------------------------------------------" << endl;
				cout << "  SLF Lm/Hr values used to construct New Fit File" << endl;
				cout << "------------------------------------------------------" << endl;
				cout << "   #     m0     m     pix0     pix0Unc             fileName" << endl;
				cout << "------------------------------------------------------" << endl;
				for(int i=0; i<slfLmHrA.size(); i++) {
					sprintf(tmp,"  %-3.3d  %-6.2f  %-6.2f  %-6.2f  %-6.2f  %s\n",
							i,slfLmHrA[i].m0, slfLmHrA[i].m, slfLmHrA[i].pix0, slfLmHrA[i].pix0Unc,
							slfLmHrA[i].file.c_str());
					line.assign(tmp);
					cout << line;
				}
			} else if (massType.compare("M") == 0) {
				cout << endl;
				cout << "------------------------------------------------------" << endl;
				cout << " SLF Mm/Hr values used to construct New Fit File" << endl;
				cout << "------------------------------------------------------" << endl;
				cout << "   #     m0     m     pix0     pix0Unc             fileName" << endl;
				cout << "------------------------------------------------------" << endl;
				for(int i=0; i<slfMmHrA.size(); i++) {
					sprintf(tmp,"  %-3.3d  %-6.2f  %-6.2f  %-6.2f  %-6.2f  %s\n",
							i,slfMmHrA[i].m0, slfMmHrA[i].m, slfMmHrA[i].pix0, slfMmHrA[i].pix0Unc,
							slfMmHrA[i].file.c_str());
					line.assign(tmp);
					cout << line;
				}
			} else if (massType.compare("H") == 0) {
				cout << endl;
				cout << "------------------------------------------------------" << endl;
				cout << " SLF Hm/Hr values used to construct New Fit File" << endl;
				cout << "------------------------------------------------------" << endl;
				cout << "   #     m0     m     pix0     pix0Unc             fileName" << endl;
				cout << "------------------------------------------------------" << endl;
				for(int i=0; i<slfHmHrA.size(); i++) {
					sprintf(tmp,"  %-3.3d  %-6.2f  %-6.2f  %-6.2f  %-6.2f  %s\n",
							i,slfHmHrA[i].m0, slfHmHrA[i].m, slfHmHrA[i].pix0, slfHmHrA[i].pix0Unc,
							slfHmHrA[i].file.c_str());
					line.assign(tmp);
					cout << line;
				}
			}
		}
	}
}

