/* 
 * File:   DFMS_Histogram_Class.cc
 * Author: marinaldi@@
 * 
 * Created on March 29, 2013, 4:14 PM
 */

#include "DFMS_Histogram_Class.hh"

DFMS_Histogram_Class::DFMS_Histogram_Class(double* inArr, int n, int b, string t) {
    
     string sFunctionName = "DFMS_Histogram_Class::Constructor";
    
 	 hMin=0;
 	 hMax=0;
 	 iMin=0;
 	 iMax=0;
 	 binSize=0;
 	 type = t;

     numBins = b;
     numPix = n;
     array = new double[numPix];

     if (type.compare("offset") == 0) {
    	 for (int i=0; i<numPix; ++i) {
    		 if (inArr[i] <= 0.0) {
    			 array[i] = 0.0;
    		 } else {
    			 array[i] = log10(inArr[i]);
    		 }
        	 //cout << "offset - array[" << i << "] = " << array[i] << endl;
    	 }
     } else {
    	 for (int i=0; i< numPix; ++i) {
			 array[i] = inArr[i];
    	 }
    	 //cout << "peak - array[" << i << "] = " << array[i] << endl;
     }
    
     histo = new int[numBins];
     sumInHisto = new double[numBins];
     for (int i=0; i<numBins; ++i) {
         histo[i] = 0;
         sumInHisto[i] = 0;
     }

 	// Create [numBins x NumPix] array
 	countsInBin = new double*[numBins];
 	for(int i=0; i<numBins; i++){
 		countsInBin[i] = new double[NUMPIX];
 		for (int j=0; j<NUMPIX; j++) countsInBin[i][j] = 0.0;
 	}
    
}

//
// -------------------------------- Destructor ---------------------------------
//
DFMS_Histogram_Class::~DFMS_Histogram_Class() {

    // Release dynamically allocated memory
    delete[] histo;
    delete[] sumInHisto;
    delete[] array;
    for (int i = 0; i < numBins; i++) delete[] countsInBin[i];
    delete[] countsInBin;  countsInBin = 0;

}

//
// -------------------------- create Histogram ---------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method will create the histogram and an array containing the sum of
// all the values in the bins of the histogram.
//
// outputs:
//	string type - type of histogram, offset or peak
//  int iROw - DFMS side A/B
//
// returns:
//   none
// =============================================================================
// History: Written by Mike Rinaldi, March 2013
// =============================================================================
//
void DFMS_Histogram_Class::createHisto(int iRow) {
    
    string sFunctionName = "DFMS_Histogram_Class::createHisto";

    // 2.3.3.1 - Find the min and max arrays indices of the data
    iMin = minIndex (numPix, array, -1.0e99);
    iMax = maxIndex (numPix, array, 1.0e99);

    
    // 2.3.3.2 - Define the values of the min and max
    if (type.compare("offset") == 0) {
    	hMin = pow(10.0,array[iMin]);
    	hMax = pow(10.0,array[iMax]);
    } else {
    	hMin = array[iMin];
    	hMax = array[iMax];
    }
    
    if (verbose == 100) {
    	cout << type << " - " << iRow << " - hMin = " << hMin <<  endl;
    	cout << type << " - " << iRow << " - hMax = " << hMax << endl;
    }

    // 2.3.3.3 - Define the binsize of the histogram
    binSize = (hMax-hMin)/numBins;
    
    double counts=0;

    // 2.3.3.4 - Build the histogram
    for (int i=0; i<numPix; ++i) {
    	for (int j=0; j<numBins; ++j) {
    		double binMin = binSize*(j);
    		double binMax = binSize*(j+1);
    		if (type.compare("offset") == 0) {
    			counts = pow(10.0,array[i]);
    		} else {
    			counts = array[i];
    		}
        	if (counts >= binMin && counts < binMax) {

        		countsInBin[j][i] = counts;
    			histo[j]++;
    			//if (type.compare("offset") == 0) cout << " i = " << i << " counts = " << counts << " - j = " << j << endl;
    			sumInHisto[j] += counts;   //pow(10.0,array[i]);
    			if (type.compare("offset") == 0 && verbose == 100) {
    				cout << type << " - " << iRow << " - array[" << i << "] = " << array[i];
    				cout << "  histo[" << j << "] = " << histo[j];
    				cout << "  sumInHisto[" << j << "] = " << sumInHisto[j] << endl;
    			}
    			break;
            }
        }
    }

    if (type.compare("offset") == 0 && verbose == 100) {
    	for(int i=0; i<numBins; i++) {
    		cout << "  histo[" << i << "] = " << histo[i] << endl;
    	}
    	int iiP = 0;
    	int hPk = histoPeak(&iiP);
    	cout << "histoPeak[ = " << iiP << "] = " << hPk << endl;
    }

}

//
// ---------- Find the Histogram_Class peak value and index --------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method will calculate location of the highest peak
//
// outputs:
//	int iPeak - the location of the highest peak
//
// returns:
//   The value of the peak
// =============================================================================
// History: Written by Mike Rinaldi, March 2014
// =============================================================================
//
double DFMS_Histogram_Class::histoPeak(int *iPeak){
    
     string sFunctionName = "DFMS_Histogram_Class::histoPeak";

     double peakVal = -1.0e99;
     for (int j=0; j<numBins; ++j) { 
    	 if (histo[j] >= peakVal) {
    		 peakVal = histo[j];
    		 *iPeak = j+1;
         }
     }
     return peakVal;
}

//
// ------------------------------ meanOffset ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method will calculate the average offset by histogramming the log of
// the raw counts into 50 bins.  It will then search for the largest bin.  Since
// most of the 512 pixels are just offset the method will use these pixels to
// fid the mean offset and std Dev.
//
// outputs:
//	double mean - the calculated mean offset
//  double stdev - the error on the offset
//
// returns:
//   none
// =============================================================================
// History: Written by Mike Rinaldi, March 2014
// =============================================================================
//
void DFMS_Histogram_Class::meanOffset(double* mean, double* stdev) {
        
    double binMin=0.0;
    double binMax=0.0;
    
    // 2.3.6.1 - Find the histogram bin with the largest value.  This will nominally be
    // the background as other than the peak(s) most bins are within +/- binsize
    // of the background.
    int maxH = histo[0];
    int idx=0;

    for (int i=1; i<numBins; ++i) {
    	if(histo[i] > maxH) {
    		idx = i;
            maxH = histo[i];
            binMin = binSize*i;      // Need this in order to bracket data below
            binMax = binSize*(i+1);  // Need this in order to bracket data below
        }
    }
    
    //cout << "maxH = " << maxH << endl;
    //cout << "binMin = " << binMin << endl;
    //cout << "binMax = " << binMax << endl;
    //cout << "binSize = " << binSize << endl;

    // 2.3.6.2 - Number of values in largest histo bin.  This is assumed to be
    // sampling the Offset region of the data since most pixels are just offset
    // values
    int numVals = histo[idx];

    // 2.3.6.3 - sumInHisto contains the sum of all values counted for the bin idx
    // Use the sum of the largest bin and find the mean
    *mean = sumInHisto[idx]/numVals;
    
    // create background histo bin array
    double *offset = new double[numVals];
    
    // 2.3.6.4 - Find the std dev of the background by sampling the background bin
    int k = 0;
    for (int i=0; i< NUMPIX; i++) {
    	if (countsInBin[idx][i] > 0.0) {
    		offset[k] = countsInBin[idx][i];
        	//cout << "offset[" << k << "] = " << offset[k] << endl;
        	k++;
    	}
    }
    *stdev = standardDev(numVals, offset);

    /*
    int k=0;
    for(int i=0; i<numPix; ++i) {
    	//if (array[i] >= binMin && array[i] < binMax) {
    	double counts = pow(10.0,array[i]);
    	if (counts >= binMin && counts < binMax) {
    		offset[k] = counts;
    		if (type.compare("offset") == 00  && verbose == 100) cout << "offset[" << k << "] = " <<  offset[k] << endl;
    		k++;
    		//offset[k++] = array[i];
    	}
    }
    */
    
    // 2.3.6.5 - Find the std Dev of the background
    
    // 2.3.6.6 - Release memory
    delete[] offset;
    
}
// ---------------------------- print histo info -------------------------------
//
void DFMS_Histogram_Class::printHistoInfo() {
    
     string sFunctionName = "DFMS_Histogram_Class::printHistoInfo";
        
     int iPeak;
     double peak = histoPeak(&iPeak);
      
     cout << "-----------------------------------------------------" << endl;
     cout << "Input Array values:" << endl;
     cout << "-----------------------------------------------------" << endl;
     cout << "Main Peak Index = " << getImax() << endl;
     //cout << "Main Peak Value = " << pow(10.0,array[getImax()]) << endl;
     cout << "Main Peak Value = " << array[getImax()] << endl;
     cout << endl;
     cout << "Histogram Bin size = " << getBinSize() << endl;
     //cout << "Background = binSize*peakLoc = " << pow(10.0,getBinSize()*iPeak) << endl;
     cout << "Background = binSize*peakLoc = " << getBinSize()*iPeak << endl;
     cout << "Histogram Peak Value = " << peak << endl;
     cout << "Histogram Peak Value index= " << iPeak << endl;
     for (int i=0; i<numBins; ++i){
    	 cout << "i = " << i << " : " << histo[i] << endl;
     }

}
