

#include "DFMS_Gain_Table_Class.hh"

//
// ---------------------------- Constructor ------------------------------------
//
DFMS_Gain_Table_Class::DFMS_Gain_Table_Class(string p, string f) {

	path = p;
	fileName = f;
	GTfile = path+fileName;
	TableRows = 0;
	TableCols = 0;

}

//
// ----------------------------- Destructor ------------------------------------
//
DFMS_Gain_Table_Class::~DFMS_Gain_Table_Class() {
    
	// Release dynamically allocated memory
	GTobj.clear();
	gTdata.clear();
}

//
// ------------------------------ readGTdata ------------------------------------
//
int DFMS_Gain_Table_Class::readGTdata(fstream &is) {

	char input[100];

	// Now read the HK data into the HKdata string array
 	for (int i=0; i<TableRows; i++) {
         
 		gTdata.push_back(dslice());
         
		for (int j=0; j<4; j++) {
			is.getline(input,100,',');
			gTdata[i].push_back(atof(input));
		}
	}

	return 1;	
}

//
// ------------------------------ getGainTable -----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method reads a specific gain table and stores the data in the
// members of the GTObject
//
// inputs:
//   path      -  path to gain table
//   file   -  file name
//
// outputs:
//   GTObject   -  gain object
//
// returns:
//   1 if success 0 otherwise
// =============================================================================
// History: Written by Mike Rinaldi, Feb 2014
//          March 2018 - Adapted from getPGTable Pixel Gain object method
// =============================================================================
//
int DFMS_Gain_Table_Class::getGainTable() {

    string sFunctionName = "DFMS_Gain_Table_Class::DFMS_get_Gain_Table";
	int nSuccess = 0;
	fstream is;

	// Open GT file
	openFile(GTfile,is,1);

	// Read the Gain Table file PDS header
	nSuccess = readPDSheader(is);
	if (!nSuccess) {
		sErrorMessage = "Unable to read the PDS header for file: ";
		sErrorMessage += GTfile+"\n";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}

	// Read the Gain Table file GT object description
	nSuccess = readObj(is, GTobj, TableRows, TableCols);
	if (!nSuccess) {
		sErrorMessage = "Unable to read the GT Object Descp. for file: ";
		sErrorMessage += GTfile+"\n";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}

	// Read the Gain Table file Data
	nSuccess = readGTdata(is);
	if (!nSuccess) {
		sErrorMessage = "Unable to read the GT Table file: ";
		sErrorMessage += GTfile+"\n";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
	}

	// Close the GT File stream
	closeFile(is);
     
	sInfoMessage = "Completed Reading Pixel Gain Table Files ";
	writeToLog(sInfoMessage, sFunctionName, INFO);

	return 1;
}

//
// ------------------------------ printGTtable ------------------------------------
//
void DFMS_Gain_Table_Class::printGTtable(string type) {

	// First print the PDS Header
	
	mapstr::const_iterator it;

	if (type == "PDSHeader" || type == "All") {
		for (it = PDSheader.begin(); it != PDSheader.end(); ++it) {
			printf("%-32s =     %-s%s",(it->first).c_str(),(it->second).c_str(),EOL);
		}
	}

	// Next print out Object description
	if (type == "GTobj" || type == "All") {
		for (it = GTobj.begin(); it != GTobj.end(); ++it) {
			printf("%32s =     %-s%s",(it->first).c_str(),(it->second).c_str(),EOL);
		}
	}

	// Finally print out the Gain Table data
	if (type == "gTdata" || type == "All") {
 		for (int i=0; i<TableRows; i++) {
			cout << gTdata[i][0] << " , "; 
			cout << gTdata[i][1] << " , ";
			cout << gTdata[i][2] << " , " << endl;
		}	
	}
}

//
// ---------------------- getGTFileInfo -------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method retrieves all the Gain Table File information. File name is of
// from GAIN_TABLE_20140317_FS.TAB
//
// inputs:
//   sGTpath  -  Path to Gain tables
//   GTFileInfo   -  Vector with GT file info
//
// returns:
//   1 if success 0 otherwise
// =============================================================================
// History: Written by Mike Rinaldi, March 2018
// =============================================================================
//
int DFMS_Gain_Table_Class::getGTFileInfo(string sGTpath,
                                               vector<gtFileInfo> &GTFileInfo) {
    
    string sFunctionName="DFMS_Gain_Table_Class::getGTFileInfo";
    
    DIR *topDir;
    string file;
    struct dirent *entry;
    string fileName;
    gtFileInfo G;
    GTFileInfo.clear();
    
    // For sorting purposes
    vector<string> fvec;

    // Look into specified Instrument Model Directory
    //cout << "sGTpath = " << sGTpath << endl;
    
    int numFile=0;
    if ((topDir = opendir(sGTpath.c_str())) == NULL) {
        sErrorMessage="Default Source directory - "+sGTpath+" does not exist";
        writeToLog(sErrorMessage, sFunctionName, FATALERROR);
    } else {
    	while ((entry = readdir(topDir)) != NULL) {
    	            fileName = string(entry->d_name);
    	            // Skip all files/directories shorter than 3 characters.. (like ./, ../)
    	            if (fileName.size() > 10 &&
    	                fileName.substr(0,10).compare("GAIN_TABLE") == 0 &&
    	                util_isNumber(fileName.substr(11,8))) {

    	            	fvec.push_back(fileName);   // This just stores the Gain file names
    	            }
    	}
    	closedir(topDir);

    	// First lets sort the vector of filenames
    	sort(fvec.begin(),fvec.end());

    	// Now fill the structure
        for (int i=0; i<fvec.size(); i++) {
                G.index = numFile++;
                G.dirName = sGTpath;
                G.filename = fvec[i];
                G.dateTime = fvec[i].substr(11,8);
                G.ModelType = fvec[i].substr(20,2);
                G.secSince1970 = rosToSec(G.dateTime, 2);
                G.printHeader();
                GTFileInfo.push_back(G);
        }

    }
    
    if (GTFileInfo.size() == 0) {
        sErrorMessage="Default source directory - " + sGTpath+" does not contain expected files\n";
        sErrorMessage += "Expected Gain Table file name is of the form GAIN_TABLE_YYYYMMDD_FS.TAB";
        writeToLog(sErrorMessage, sFunctionName, FATALERROR);
    } else {
        cout << "Gain File Info was stored for ";
        cout << GTFileInfo.size() << " file(s)" << endl;
    }
    
    return 1;
}
