/*
 * DFMS_GainTable_Interp_Class.cc
 *
 *  Created on: March 5, 2018
 *      Author: marinaldi
 */

#include "DFMS_GainTable_Interp_Class.hh"

//
// ---------------------------- Constructor -------------------------------------
//
DFMS_GainTable_Interp_Class::DFMS_GainTable_Interp_Class(string gtpath) {

	path = gtpath;

	for (int i=0; i< 16; i++) {
	    for (int l=0; l<5; l++) {
	    	gtFit[i][l] = 0.0;
	    }
	}

}

//
// ---------------------------- Destructor -------------------------------------
//
DFMS_GainTable_Interp_Class::~DFMS_GainTable_Interp_Class() {
	// TODO Auto-generated destructor stub
}

//
// ---------------------------- readAllPGT -------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method reads all the Pixel Gain file and stores in the data in the
// vector structure PGTinfo.
//
// inputs:
//   PGTinfo   -  Vector with PGT info
//
// returns:
//   1 if success 0 otherwise
// =============================================================================
// History: Written by Mike Rinaldi, March 5, 2018
// =============================================================================
//
int DFMS_GainTable_Interp_Class::readAllGT(vector<gtFileInfo> &GTinfo) {

	string sFunctionName="DFMS_GainTable_Interp_Class::readAllGT";

	// Now read in all Gain Tables
	for (int i=0; i<GTinfo.size(); i++) {
		string gpath = GTinfo[i].dirName+"/";
		string gfile = GTinfo[i].filename;
		DFMS_Gain_Table_Class *GTObject = new DFMS_Gain_Table_Class(gpath, gfile);
		int nSuccess = GTObject->getGainTable();
		if (!nSuccess) {
			sErrorMessage = "Reading Gain Table: " + gfile+" was unsuccessful ";
			writeToLog(sErrorMessage, sFunctionName, ERROR);
			//cout << "Reading Pixel Gain Table: " << pgfile << " was unsuccessful" << endl;
			exit(EXIT_FAILURE);
		}
		GTinfo[i].readGain(GTObject->gTdata);
        GTinfo[i].printGain();
		if (verbose >=3) GTinfo[i].printHeader();
		if (verbose >=3) GTinfo[i].printGain();
		delete GTObject;
	}

	return 1;
}

//
// ---------------------------- buildPixGTdata -----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method creates the required pixGTdata array using interpolation or
// extrapolation of relevant Pixel Gain tables
//
// inputs:
//   t   -  time when PG Table is required (in seconds since 19700101T00:00:00.00)
//   G   -  Gain Table file info
//
// returns:

//   returns 1 if interpolation was successful, 0 otherwise
// =============================================================================
// History: Written by Mike Rinaldi, March 5, 2018
// =============================================================================
//
int DFMS_GainTable_Interp_Class::buildGTdata(double t, vector<gtFileInfo> &G) {

	// Number of Gain tables
	int num = (int)G.size();
    
    // Maximum j
    int maxj = 4;

	// Calculate the scaled time of the L2 file
	double tL2 = t/JULIANYEAR;

 	// Find time boundaries for these Gain Tables
 	string bTime = G[0].dateTime+"_000000000";
 	double tb = rosToSec(bTime, 1)/JULIANYEAR;
 	string fTime = G[num-1].dateTime+"_000000000";
 	double tf = rosToSec(fTime, 1)/JULIANYEAR;

 	// Find how to interpolate
 	// if tL2 > tf:  Extrapolate Forwards using the last two PG tables for this GS
 	// if tL2 < tb:  Extrapolate backwards using the first two PG tables for this GS
 	// if tb <= tL2 <= tf:  Interpolate using the two time bracketing PG tables for this GS

// 	for (int i=0; i<num; ++i) {
// 		cout << i << " - P date: " << P[i].dateTime << endl;
// 	}
    
    // If only one Gain Table exists then just copy the values into the gFit array.
    if (num == 1) {
        for (int i=0; i<16; ++i) {
            gtFit[i][0] = i+1;      // Just copy the Gain Step number
            //cout << "GS = " << gtFit[i][0] << endl;
            for (int j=1; j<maxj; j++) {
                gtFit[i][j+1] = G[num-1].gain[i][j+1];
                //cout << "gtFit[" << i << "][" << j+1 << "] = " << gtFit[i][j+1] << endl;
            }
        }
        return 1;
    }

 	if (tL2 > tf) {   // ----------------------------  Extrapolate Forwards using the last two Gain tables
 		string sTime = G[num-2].dateTime+"_000000000";
 	 	//cout << "sTime = " << sTime << endl;
 		double ts = rosToSec(sTime, 1)/JULIANYEAR;
 		string eTime = G[num-1].dateTime+"_000000000";
 	 	//cout << "eTime = " << eTime << endl;
 		double te = rosToSec(eTime, 1)/JULIANYEAR;
 		for (int i=0; i<16; ++i) {
            gtFit[i][0] = i+1;      // Just copy the Gain Step number
 			for (int j=1; j<maxj; j++) {
 				//P[num-2].printPG();
 				double Gs = G[num-2].gain[i][j+1];
 				double Ge = G[num-1].gain[i][j+1];
 				double Gt = Gs + (Ge-Gs)/(te-ts)*(tL2-ts);
// 				if (j == 3) {
// 					//cout << "[" << i << "][" << j << "]: Pt = " << Pt << endl;
// 					cout << i+1 << ", " << Ps << ", " << Pe << ", " << Pt << endl;
// 				}
                gtFit[i][j+1] = Gt;
		 	}
		 }
 		//exit(EXIT_SUCCESS);

 	} else if (tL2 < tb) {   // ----------------------  Extrapolate Backwards using the first two Gain tables

 		string sTime = G[0].dateTime+"_000000000";
 		double ts = rosToSec(sTime, 1)/JULIANYEAR;
 		string eTime = G[1].dateTime+"_000000000";
 		double te = rosToSec(eTime, 1)/JULIANYEAR;
 		for (int i=0; i<16; ++i) {
 			gtFit[i][0] = i+1;   // Just copy the Gain Step number
 			for (int j=1; j<maxj; j++) {
 				double Gs = G[num-2].gain[i][j+1];
 				double Ge = G[num-1].gain[i][j+1];
 				double Gt = Gs + (Ge-Gs)/(te-ts)*(tL2-ts);
                gtFit[i][j+1] = Gt;
		 	}
		}


 	} else if (tL2 >= tb && tL2 <= tf) {   // ----------  Interpolate using the time bracketing Gain tables

 		for (int k=0; k<G.size()-1; k++) {
 			string sTime = G[k].dateTime+"_000000000";
 		 	double ts = rosToSec(sTime, 1)/JULIANYEAR;
 		 	string eTime = G[k+1].dateTime+"_000000000";
 		 	double te = rosToSec(eTime, 1)/JULIANYEAR;
 		 	if (tL2 >= ts && tL2 <= te) {
 		 		for (int i=0; i<16; ++i) {
 		 			gtFit[i][0] = i+1;   // Just copy the Gain Step number
 		 			for (int j=1; j<maxj; j++) {
 		 				double Gs = G[k].gain[i][j+1];
 		 				double Ge = G[k+1].gain[i][j+1];
 		 				double Gt = Gs + (Ge-Gs)/(te-ts)*(tL2-ts);
 		 				gtFit[i][j+1] = Gt;
 		 			}
 		 		}
 		 	}
 		}

 	} else {
 		if (L3INFOTOFILE) l3Log << "Unknown error with L2 time during Gain Table interpolation" << dfmsEOL;
 		cout << "Unknown error with L2 time during Gain Table interpolation" << endl;
 		return 0;
 	}

 	// If necessary dump the Gain data and resulting interpolation to
 	// check for accuracy.  Must set DUMPPGINTP == 1 in file DFMS_definedConstants.hh
 	// This is used mainly for interpolation diagnostics and thus will dump
 	// One GS value (set in if block below) then exit.
 	if (DUMPPGINTP) {
 		dumpGTIntpData(t, G);
 	 	exit(EXIT_SUCCESS);
 	}

	return 1;
}

//
// -------------------------- dumpGIntpData -----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method will dump the interpolation results versus input data to check
// for accuracy
//
// inputs:
//  t   -  time when PG Table is required (in seconds since 19700101T00:00:00.00)
//  P   -  The Pixel Gain info structure
//  pgO -  The pixel Gain Object
//
// returns:
//   None
// =============================================================================
// History: Written by Mike Rinaldi, March 5, 2018
// =============================================================================
//
void DFMS_GainTable_Interp_Class::dumpGTIntpData(double t, vector<gtFileInfo> &G) {
    
    double gtout[16][5][5] = {0};
    double gtTime[5] = {0};
    string gtFname[5];
    
    fstream out;
    string line;
    string ofile = "gtTest.out";
    
    int maxj=4;
    
    for (int k=0; k<G.size(); k++) {
        string sTime = G[k].dateTime+"_000000000";
        gtTime[k] = rosToSec(sTime, 1)/JULIANYEAR;
        gtFname[k] = G[k].filename;
        for (int i=0; i<16; ++i) {
            gtout[i][0][k] = i+1;
            for (int j=1; j<maxj; j++) {
                gtout[i][j][k] = G[k].gain[i][j];
            }
        }
    }
    
    // Set up I/O to create dump file
    out.open(ofile.c_str(),ios::out);
    out << "t = " << t/JULIANYEAR << dfmsEOL;
    for (int i=0; i<5; i++) {
        out << "Gain file: " << gtFname[i] << "  -  t[" << i << "]= " << gtTime[i] << dfmsEOL;
        cout << "Gain file: " << gtFname[i] << "  -  t[" << i << "]= " << gtTime[i] << endl;
    }
    
    // Dump interpolated data at at time t and other PG data used for interpolation
    for (int i=0; i<16; ++i) {
        char tmp[200];
        int pix = (int)gtout[i][0][0];
        sprintf(tmp,"%3.3d     %f  %f  %f  %f  %f  %f\n",
                pix,gtFit[i][1],gtout[i][1][0],gtout[i][1][1],gtout[i][1][2],gtout[i][1][3],gtout[i][1][4]);
        line.assign(tmp);
        //cout << line << endl;
        out << line;
    }
    out.close();
    cout << "Complete.... Check for gtTest.out file" << endl;
    exit(EXIT_SUCCESS);
    
}

