/*
 * DFMS_CreatePix0FitFiles_Class.cc
 *
 *  Created on: Jun 26, 2014
 *      Author: marinaldi
 */

#include "DFMS_CreateGCUSLFPix0_Class.hh"

//
// ------------------------- Constructor ---------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This Class will step through the various processes to create pix0 files from
// the set of L2 files in the source directory.
// This method is the constructor
// inputs:
//   mpt - Mass Peak Search Table path
//   ffp - Path of Fit files
//   gA - gain step value for row A
//   gB - gain step value for row B
//   pg - pixel gain array
//   L2 - L2 file object pointer

// returns:
//   None
// =============================================================================
// History: Written by Mike Rinaldi, July 2014
// =============================================================================
//
DFMS_CreateGCUSLFPix0_Class::DFMS_CreateGCUSLFPix0_Class (string mpt, string ffp,
						double gA, double gB, dbl2D &pg, DFMS_PDS_L2_file_Class *L2)
{

	string sFunctionName="DFMS_CreateGCUSLFPix0_Class::Constructor";

	// Initialize various member fields
	mptRows=0;
	zoom = 0.0;
	mptIndx = -1;
	MPSTobjIn = NULL;

	linFitA=NULL;			 // Row A linear fit object
	linFitB=NULL;			 // Row B linear fit object

	iXl = 0;				 // valid starting pixel (left)
	iXr = 0;				 // valid end pixel (right)

	numGoodPks = 0;
	threshold = 0.0;
	foundVerifPeak = false;  // Set verification peak found flag to false

	meanOffset = 0.0;
	meanOffsetStdDev = 0.0;
	meanOffsetA = 0.0;
	meanOffsetB = 0.0;

	sMPSTpath = mpt;         // Mass Peak Table path
	fitFilePath = ffp;       // Path to fit file
	L2Obj = L2;              // L2 file Object
	gainA = gA;              // Gain Step value for Row A
	gainB = gB;              // Gain Step value for Row B
	commandedMass = L2Obj->HKdata[2][2];  // ROSINA_DFMS_SCI_MASS housekeeping value
    massCal = NULL;

	currConvConstant=0.0;
	l3PkMaxIdA=0, l3PkMaxIdB=0;

	slfPix0FitFileName = "None";
	gcuPix0FitFileName = "None";

	// Copy the pixel Gain array pg into the local var pixG
	for (int i=0; i<NUMPIX; i++) {
		// Use vector objects dynamic fill to add values
		pixG.push_back(dslice());
		for (int j=0; j<5; ++j) {
				pixG[i].push_back(0.0);
				pixG[i][j] = pg[i][j];
		}
	}

	offSetFitA = new double[NUMPIX];
	offSetFitB = new double[NUMPIX];
	for(int i=0; i<NUMPIX; ++i) {
		offSetFitA[i] = 0.0;
		offSetFitB[i] = 0.0;
	}

	// Make local L2 data arrays
	Xin = new double[NUMPIX];
	for(int i=0; i<NUMPIX; ++i) {
			Xin[i] = (double)L2Obj->L2Data[i][0];
	}
	YAin = new double[NUMPIX];
	yAFit = new double[NUMPIX];
	for(int i=0; i<NUMPIX; ++i) {
		YAin[i] = (double)L2Obj->L2Data[i][1];
		yAFit[i] = 0.0;
	}
	YBin = new double[NUMPIX];
	yBFit = new double[NUMPIX];
	for(int i=0; i<NUMPIX; ++i) {
		YBin[i] = (double)L2Obj->L2Data[i][2];
		yBFit[i] = 0.0;
	}

	for(int i=0; i<4; ++i) pCoeff[i] = 0.0;

}

//
// -------------------------- Destructor ---------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Release dynamic memory
// inputs:
//
// returns:
//   None
// =============================================================================
// History: Written by Mike Rinaldi, June 2014
// =============================================================================
//
DFMS_CreateGCUSLFPix0_Class::~DFMS_CreateGCUSLFPix0_Class() {

	// Release dynamically allocated memory
	delete[] YAin;  YAin=0;
	delete[] YBin;  YBin=0;
	delete[] yAFit; yAFit=0;
	delete[] yBFit; yBFit=0;
	delete[] Xin;   Xin=0;
	delete[] offSetFitA; offSetFitA=0;
	delete[] offSetFitB; offSetFitB=0;
	delete linFitA; linFitA=0;
	delete linFitB; linFitB=0;
	pixG.clear();
}

//
// ------------------------------ processL2 ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method does the processing of L2 to L3 via calls to various methods
// inputs:
//   L2I - L2 File Info Structure
//   modeInfo - DFMS mode Info Structure
//
// returns:
//   1 if success, 0 otherwise
// =============================================================================
// History: Written by Mike Rinaldi, March 2013
// =============================================================================
//
int DFMS_CreateGCUSLFPix0_Class::processL2forPix0Fit(L2FileInfo &L2I,
						DFMSModeInfo &modeInfo, string type, set<double> &distinctMasses) {

	string sFunctionName="DFMS_CreateGCUSLFPix0_Class::processL2";

	int nSuccess=0;
    verifPeakInfo peak;
	DFMS_PeakFit_Class *mCalFit;
	DFMS_FindAllPeaks_Class *fP;
	DFMS_X0FitFile_Class *x0Fit=NULL;
	double m0 = 0.0;
	string massType, resType;

	// Set the local copy of L2Info using struct overloaded = operator
	L2Info = L2I;

	// Get remaining mode Info
	L2Info.zoom = modeInfo.zoom;

	if (L2Info.lowRes) resType = "L";
	if (L2Info.hiRes) resType = "H";

	// Then 10.x.a.b Step representation below reflects the fact that x runs from 3-10 depending
	// on which of the 8 specific GCU or SLF file types is being processed.
	// See DFMS_PDS_L2_dir_Class::processPix0FitData for more details.

	// STEP 10.x.7.1 - Get MPST data from file
	cout << "Best MPS Table = " << sBestMPST << endl;
	nSuccess = getMPSTdata(sBestMPST, peak);
	if (!nSuccess) {
		sErrorMessage = "Unable to access proper MPS table or Mass for file: ";
		sErrorMessage += sBestMPST+" - Skipping file processing";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		delete MPSTobjIn; MPSTobjIn=0;
		return 0;
	} else {
		//cout << "mptIndx = " << mptIndx << endl;
		if (L2Info.isGCU) {
			// Assume peak is located in between MPS table defined limits
			if (iXl <= 0) iXl = PKLOWPT;
			if (iXr <= 0) iXr = NUMPIX-PKLOWPT;
		} else {
			// Use entire range
			iXl = PKLOWPT;      // Always look near 255 for target peak
			iXr = NUMPIX-PKLOWPT;
		}
	}

	// STEP 10.x.7.2 - Get a list of all the X0FitFiles from the directory
	if (type.compare("SLF") == 0) {
		nSuccess = DFMS_X0FitFile_Class::getFitFilesInDir(sx0FitsDataPath, fitInfo);
		if (!nSuccess) {
			sErrorMessage = "Unable to get X0 Fit File dir info at: ";
			sErrorMessage += sx0FitsDataPath;
			writeToLog(sErrorMessage, sFunctionName, ERROR);
			cout << "Unable to get X0Fit File directory info at: " << sx0FitsDataPath << endl;
			exit(EXIT_FAILURE);
		} else {
			sInfoMessage = "Successfully found: "+util_intToString(fitInfo.size());
			sInfoMessage += " X0 fit files";
			writeToLog(sInfoMessage, sFunctionName, INFO);
		}

	    m0 = L2Info.commandedMass;
	    if (L2Info.lowRes) {
	    	if (m0 <= GCULOWCUTOFF) {
	    		massType = "L";
	    	} else {
	    		massType = "H";
	    	}
	    } else {
	    	if (m0 < GCULOWCUTOFF) {
	    		massType = "L";
	    	} else if (m0 >= GCULOWCUTOFF && m0 < GCUHICUTOFF) {
	    		massType = "M";
	    	} else {
	    		massType = "H";
	    		L2Info.printData();
	    	}
	    }

	   	x0Fit = new DFMS_X0FitFile_Class(sx0FitsDataPath);
	   	nSuccess = x0Fit->readPix0FitFile(1, "GCU", resType, massType, L2Info.dateTime, fitInfo);
	   	if (!nSuccess) {
	   		sErrorMessage="The pix0 File was not found or not read properly";
	   		writeToLog(sErrorMessage, sFunctionName, ERROR);
	   		delete x0Fit; x0Fit=0;
	   		return 0;
	   	}

	}

    // For GCU/SLF with HMHR set zoom = 4.0
    if (L2Info.hiRes && (L2Info.commandedMass >= GCUHICUTOFF) ) {
    	L2Info.zoom = 4.0;
    }

	// STEP 10.x.7.3 - Perform Mass Scale calibration for each of side A and B
	int fitParams = FITPARAMS;
	for (int iRow=1; iRow<3; ++iRow) {

		// STEP 10.x.7.3.1 - Calculate the Offset
		calcOffset(iRow);

		// STEP 10.x.7.3.2 - Define peak threshold based on meanOffset
		threshold = meanOffset+meanOffsetStdDev*NUMSDEV;
		if (verbose == 10) {
			cout << "The meanOffset = " << meanOffset << endl;
			cout << "The threshold = " << threshold << endl;
			//
		}

		if (iRow == 1) {
			fP = new DFMS_FindAllPeaks_Class(YAin, iXl, iXr, meanOffset, meanOffsetStdDev);
		} else {
			fP = new DFMS_FindAllPeaks_Class(YBin, iXl, iXr, meanOffset, meanOffsetStdDev);
		}

		// STEP 10.x.7.3.3 - Get all peaks above Threshold
		numGoodPks = fP->findPeaksAboveThreshold(threshold, meanOffset);
		if (numGoodPks <= 0) {
			sErrorMessage = "No useful Peaks Found for Row: "+sRow[iRow-1];
			writeToLog(sErrorMessage, sFunctionName, WARN);
			pkFI.maxId = fP->maxId;
			if (iRow == 1) sideAexceptionType["peakFinderError"]++;
			if (iRow == 2) sideBexceptionType["peakFinderError"]++;
			delete fP; fP=0;
		} else {
			for (int i=0; i<numGoodPks; ++i) {
				pkFI.peakLoc[i] = fP->peakLoc[i];
				pkFI.peakHght[i] = fP->peakVal[i];
				pkFI.peakWidth[i] = fP->peakWd[i];
				pkFI.pkInBounds[i] = fP->inBounds[i];
				pkFI.pkAboveThresh[i] = fP->pkAboveThresh[i];
			}
			pkFI.maxId = fP->maxId;
			pkFI.maxPkLoc = fP->maxPkLoc;
			pkFI.maxPkVal = fP->maxPkVal;
			// Release fP
			delete fP; fP=0;
		}

		if (verbose == 10) pkFI.printPeaks(numGoodPks);
		//cout << "numGoodPks = "  << numGoodPks << endl;
		//pkFI.printPeaks(numGoodPks);

		// STEP 10.x.7.3.4 - Apply the Offset Correction
		nSuccess = applyOffsetSubtraction(iRow);
		if (!nSuccess) {
			sErrorMessage = "Instrument Model: "+sInstrumentModel+" is not known";
			writeToLog(sErrorMessage, sFunctionName, ERROR);
			cout << "Instrument Model: " << sInstrumentModel << " is not known" << endl;
			delete MPSTobjIn; MPSTobjIn=0;
	   		if (type.compare("SLF") == 0) delete x0Fit; x0Fit=0;
			toLog.close();
			exit(EXIT_FAILURE);
		}

		// STEP 10.x.7.3.5 - Apply Overall and Pixel Gain correction to offset subtracted data
		applyGainCorrection(iRow);

		// STEP 10.x.7.3.6 - Convert corrected Data to Ion Rate
		findIonRate(iRow);

		// Create PeakFit of object to perform curve fitting
		if (iRow == 1) {
			meanOffsetA = meanOffset;
			mCalFit = new DFMS_PeakFit_Class (iRow, YAin, fitParams, MPSTdata, meanOffset,
       		 	 	 	 	 	 	 	 	 	 threshold, mptRows, iXl, iXr);
		} else {
			meanOffsetB = meanOffset;
			mCalFit = new DFMS_PeakFit_Class (iRow, YBin, fitParams, MPSTdata, meanOffset,
 	 	 	 	 	 	 	 	 	 	 	 	 threshold, mptRows, iXl, iXr);
		}

		// STEP 10.x.7.3.7 - Initialize the fit coefficients by histo-gramming Ion Rate data
		nSuccess = mCalFit->initPeakFit(iRow, pkFI.maxPkLoc, pkFI.maxPkVal,
				                   pkFI.peakWidth[pkFI.maxId],pkFI.pkAboveThresh[pkFI.maxId]);
		if (!nSuccess) {
			sErrorMessage = "Unable to find initial peak values";
			writeToLog(sErrorMessage, sFunctionName, ERROR);
			delete MPSTobjIn; MPSTobjIn=0;
			delete mCalFit;  mCalFit=0;
	   		if (type.compare("SLF") == 0) delete x0Fit; x0Fit=0;
			return 0;
		} else if (nSuccess == -1) {
			mCalFit->fitSucceeded = false;
		}

		// STEP 10.x.7.3.8 - Perform a non-linear curve fit if Data warrants it
		if (mCalFit->fitSucceeded) {

			nSuccess = mCalFit->processCurveFit(iRow, 0);
			if (!nSuccess) {
				sErrorMessage = "processCurveFit returned badly: ";
				writeToLog(sErrorMessage, sFunctionName, ERROR);
				mCalFit->fitSucceeded = false;
			} else {

				// 2.11.1 - Use Fisher Matrix to calculate coefficient errors.
				nSuccess = mCalFit->calcParamErrors(iRow);
				if (!nSuccess) {
					sErrorMessage = "calcParamErrors returned badly: ";
					writeToLog(sErrorMessage, sFunctionName, ERROR);
					mCalFit->fitSucceeded = false;
				}

			}

		}

		// STEP 10.x.7.3.9 - Construct the best mass Scale. Calculate Calibration Value pix0.
		// For     GCU modes -    Use GCUpix0Calc()
		// For non-GCU modes - A. Use SLFpix0Calc() if verif. peaks found.
		if (L2Info.isGCU) {

			if (L2Info.lowRes) {
				if (L2Info.commandedMass <= GCULOWCUTOFF) {
					if (mCalFit->fitSucceeded) {
						distinctMasses.insert(L2Info.commandedMass);
						gcuLmLrGoodFit++;
					}
				} else {
					if (mCalFit->fitSucceeded) {
						distinctMasses.insert(L2Info.commandedMass);
						gcuHmLrGoodFit++;
					}
				}
			} else {
				if (L2Info.commandedMass < GCULOWCUTOFF) {
					if (mCalFit->fitSucceeded) {
						distinctMasses.insert(L2Info.commandedMass);
						gcuLmHrGoodFit++;
					}
				} else if (L2Info.commandedMass >= GCULOWCUTOFF && L2Info.commandedMass < GCUHICUTOFF) {
					if (mCalFit->fitSucceeded) {
						distinctMasses.insert(L2Info.commandedMass);
						gcuMmHrGoodFit++;
					}
				} else {
					if (mCalFit->fitSucceeded) {
						distinctMasses.insert(L2Info.commandedMass);
						gcuHmHrGoodFit++;
					}
				}
			}

			nSuccess = GCUpix0Calc(mCalFit, iRow);
			if (!nSuccess) {
				sErrorMessage = "GCUpix0Calc returned badly: ";
				writeToLog(sErrorMessage, sFunctionName, ERROR);
				delete mCalFit;  mCalFit=0;
				delete MPSTobjIn; MPSTobjIn=0;
				return 0;
			}

		} else {

			// STEP 10.x.7.3.10 - In nonGCU mode check iF a verification peak exists.  If so
			// use it to calculate pix0
			//cout << "fileName = " << L2Info.fileName << endl;
			//cout << "comm mass = " << L2Info.commandedMass << endl;
			nSuccess = SLFpix0Calc(mCalFit, iRow, m0, L2Info, x0Fit);
			if (nSuccess == 1) {
				if (L2Info.lowRes) {
					if (L2Info.commandedMass < SLFLOWCUTOFF) {
						if (mCalFit->fitSucceeded) {
							distinctMasses.insert(L2Info.commandedMass);
							slfLmLrGoodFit++;
						}
					} else {
						if (mCalFit->fitSucceeded) {
							distinctMasses.insert(L2Info.commandedMass);
							slfHmLrGoodFit++;
						}
					}
				} else if (L2Info.hiRes) {
					if (L2Info.commandedMass < SLFLOWCUTOFF) {
						if (mCalFit->fitSucceeded) {
							distinctMasses.insert(L2Info.commandedMass);
							slfLmHrGoodFit++;
						}
					} else if (L2Info.commandedMass >= SLFLOWCUTOFF && L2Info.commandedMass < SLFHICUTOFF) {
						if (mCalFit->fitSucceeded) {
							distinctMasses.insert(L2Info.commandedMass);
							slfMmHrGoodFit++;
						}
					} else {
						if (mCalFit->fitSucceeded) {
							distinctMasses.insert(L2Info.commandedMass);
							slfHmHrGoodFit++;
						}
					}
				}
				sInfoMessage = "Verification Peak found and used in Pix0 fit ";
				writeToLog(sInfoMessage, sFunctionName, INFO);
			} else if (nSuccess == -1) {
				sInfoMessage="Fit did not succeed or Bad Data - go to next L2 file";
         		writeToLog(sInfoMessage, sFunctionName, INFO);
			} else if (nSuccess == -2) {
				sInfoMessage="SLF mass not in delta Mass Range - go to next L2 file";
         		writeToLog(sInfoMessage, sFunctionName, INFO);
			} else if (nSuccess == -3) {
				sInfoMessage="SLF mass peak not within pix0 +/- dpix0 Range - go to next L2 file";
         		writeToLog(sInfoMessage, sFunctionName, INFO);
			}
		}

		delete mCalFit;  mCalFit=0;
	}

	// STEP 10.x.7.3.11 - Release Memory
	delete MPSTobjIn; MPSTobjIn=0;
	if (type.compare("SLF") == 0) delete x0Fit; x0Fit=0;

	return 1;
}

//
// -------------------- applyOffsetSubtraction -----------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This routine subtracts the offset from the input Data.
//
// inputs:
//   int  iRow
//
// returns:
//   None
// =============================================================================
// History: Written by Mike Rinaldi, March 2013
// =============================================================================
//
int DFMS_CreateGCUSLFPix0_Class::applyOffsetSubtraction(int iRow) {

	string sFunctionName = "DFMS_CreateGCUSLFPix0_Class::applyOffsetSubtraction";

	if (sInstrumentModel.compare("FM") == 0) {

		if (iRow == 1) {
			for (int i=0; i<NUMPIX; i++) {
				double x = (double)(i+1);
				double offSet = -2.0e-6*pow(x,3) + 0.0017*pow(x,2) - 0.5268*x + meanOffset;
				//double offset = c3*pow(x,3) + c2*pow(x,2) + c1*x + meanOffset;
				YAin[i] -= offSet;
				offSetFitA[i] = offSet;
			}
		} else {
			for (int i=0; i<NUMPIX; i++) {
				double x = (double)(i+1);
				double offSet = -1.0e-6*pow(x,3) + 0.0013*pow(x,2) - 0.5058*x + meanOffset;
				//double offset = c3*pow(x,3) + c2*pow(x,2) + c1*x + meanOffset;
				YBin[i] -= offSet;
				offSetFitB[i] = offSet;
			}
		}

	} else if (sInstrumentModel.compare("FS") == 0) {

		if (iRow == 1) {
			for (int i=0; i<NUMPIX; i++) {
				double x = (double)(i+1);
				double offSet = -4.88e-6*pow(x,3) + 0.00513*pow(x,2) - 1.84*x + meanOffset;
				YAin[i] -= offSet;
				offSetFitA[i] = offSet;
			}
		} else {
			for (int i=0; i<NUMPIX; i++) {
				double x = (double)(i+1);
				double offSet = -1.72e-6*pow(x,3) + 0.00198*pow(x,2) - 0.618*x + meanOffset;
				YBin[i] -= offSet;
				offSetFitB[i] = offSet;
			}
		}

	} else {
		sErrorMessage = "Instrument Model: "+sInstrumentModel+" is not defined";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}

	return 1;

}

//
// ---------------------- applyGainCorrection ----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This routine applies the overall Gain correction the input Data.  This includes
// the overall gain and the individual pixel Gain correction.
//
// inputs:
//   None
// returns:
//   None
// =============================================================================
// History: Written by Mike Rinaldi, March 2013
// =============================================================================
//
void DFMS_CreateGCUSLFPix0_Class::applyGainCorrection(int iRow)
{

	string sFunctionName = "DFMS_CreateGCUSLFPix0_Class::::applyGainCorrection";

	// Apply the overall gain and pixel gain correction
	// factor on side A
	if (iRow == 1) {
		for (int i=0; i<NUMPIX; ++i) {
			//cout << "YAin[" << i << "] = " << YAin[i] << " gainA = " << gainA << "  pixG[" << i << "][3] = " << pixG[i][3];
			if (pixG[i][3] > 0) {
				double yy = YAin[i];
				YAin[i] /= (gainA*pixG[i][3]);
				/*
				if (i == 33) {
					cout << "Ybefore = " << yy << endl;
					cout << "YAin[i] = " << YAin[i] << endl;
					cout << "gainA = " << gainA << endl;
					cout << "PGA = " << pixG[i][3] << endl;
					cout << "GainCorr = " << 1.0/(gainA*pixG[i][3]) << endl;
				}
				*/
			} else {
				YAin[i] = 0.0;
			}
			//cout << "   YAin[" << i << "] = " << YAin[i] << endl;
		}
	} else {
		// Apply the overall gain and pixel gain correction
		// factor on side B
		for (int i=0; i<NUMPIX; ++i) {
			//cout << "YBin[" << i << "] = " << YBin[i] << " gainB = " << gainB << "  pixG[" << i << "][4] = " << pixG[i][4];
			if (pixG[i][4] > 0) {
				double yy = YBin[i];
				YBin[i] /= (gainB*pixG[i][4]);
				/*
				if (i == 33) {
					cout << "Ybefore = " << yy << endl;
					cout << "YBin[i] = " << YBin[i] << endl;
					cout << "gainB = " << gainB << endl;
					cout << "PGB = " << pixG[i][4] << endl;
					cout << "GainCorr = " << 1.0/(gainB*pixG[i][4]) << endl;
				}
				*/
			} else {
				YBin[i] = 0.0;
			}
			//cout << "   YBin[" << i << "] = " << YBin[i] << endl;
		}
	}

}

//
// ------------------------------- calcOffset ----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method will calculate the average offset by histogramming the log of
// the raw counts into 50 bins.  It will then search for the largest bin.  Since
// most of the 512 pixels are just offset the method will use these pixels to
// fid the mean offset and std Dev.
//
// inputs:
//	int iRow  -  Flag to indicate Row A or B
//
// returns:
//   1 for success 0 for failure
// =============================================================================
// History: Written by Mike Rinaldi, March 2014
// =============================================================================
//
int DFMS_CreateGCUSLFPix0_Class::calcOffset(int iRow) {

	string sFunctionName = "DFMS_CreateGCUSLFPix0_Class::calcOffset";

	// 2.3.1 - Create the histogram object
	DFMS_Histogram_Class *hOff;

	// 2.3.2 - Use full data are for offset calculation
	if (iRow == 1) {
		hOff = new DFMS_Histogram_Class(YAin, NUMPIX, HISTONUMBINS, string("offset"));
	} else {
		hOff = new DFMS_Histogram_Class(YBin, NUMPIX, HISTONUMBINS, string("offset"));
	}
	// 2.3.3 - Make the entire data range histogram
	hOff->createHisto(iRow);

	// 2.3.4 - Find the Actual Mean & standard Deviation of the Offset
	double mean,stdDev;
	hOff->meanOffset(&mean,&stdDev);
	meanOffset = mean;
	meanOffsetStdDev = stdDev;

	double pk = hOff->getHmax();

	// 2.3.5 - Release memory
    delete hOff; hOff=0;

    return 1;

}

//
// ------------------------------ findIonRate ----------------------------------
//
//
//  ------------------ Calculate the Ion Rate given counts ---------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method calculates the ion current for each side on a per pixel basis.
//
// inputs:
//   int iRow  -  Row A or B
//
// returns:
//   None
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
void DFMS_CreateGCUSLFPix0_Class::findIonRate(int iRow) {

	string sFunctionName = "DFMS_CreateGCUSLFPix0_Class::findIonRate";

	double ymax = -1.0e99;

	// Integration time
	double intgT = L2Info.intgTime;   // in seconds
	double ys = 1.0;

	// 6.1 - Calculate Dimensional current conversion constant
	// currConvConstant = CONVADC*CLEDA/(Q*ys*intgT);   // Ion Rate
	currConvConstant = CONVADC*CLEDA/(Q*ys);      		// # of Ions

	// 6.2 - Perform Ion Rate calculation
	// Row A
	if (iRow == 1) {
		for (int i=0; i<NUMPIX; ++i) {
			YAin[i] *= currConvConstant;
			if (i >= iXl && i <= iXr) {
				if (YAin[i] > ymax) {
					ymax = YAin[i];
					l3PkMaxIdA = i;    // index of maximum Ion rate
				}
			}
			//cout << "   YAin[" << i << "] = " << YAin[i] << endl;
		}
	} else { 	// Row B
		for (int i=0; i<NUMPIX; ++i) {
			YBin[i] *= currConvConstant;
			if (i >= iXl && i <= iXr) {
				if (YBin[i] > ymax) {
					ymax = YBin[i];
					l3PkMaxIdB = i;    // index of maximum Ion rate
				}
			}
			//cout << "   YBin[" << i << "] = " << YBin[i] << endl;
		}
	}

}

//
// ------------------------------- getMPSTdata ----------------------------------
//
//
//  --------------------- get the relevant MPST data ----------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method retrieves the Mass Peak Table data
//
// inputs:
//   	sBestMPST - the MPST file closest in time to the L2 file
//      peak - verification peak info
//
// returns:
//   None
// =============================================================================
// History:  Written by Mike Rinaldi
// =============================================================================
//
int DFMS_CreateGCUSLFPix0_Class::getMPSTdata(string sBestMPST, verifPeakInfo &peak) {

	string sFunctionName = "DFMS_CreateGCUSLFPix0_Class::getMPSTdata";

	int nSuccess;

    // 1.1 - Create MPST object using sBestMPST file
	MPSTobjIn = new DFMS_MPST_Class(sMPSTpath, sBestMPST);

	// 1.2 - Get the MPST data file
	nSuccess = MPSTobjIn->getMPSTdata();
	if (!nSuccess) {
		sErrorMessage = "Unable to read the MPST file: ";
		sErrorMessage += MPSTobjIn->getfileName();
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}
	if (verbose >= 2) MPSTobjIn->printMPSTobj("All");

	// 1.3 - Create a local (to this class) copy of the MPSTdata, and verifcation peaks info
	mptRows = MPSTobjIn->getRows();
	// 1.4 - Build local MPST data array
	for (int i=0; i<mptRows; i++) {
		// Use vector objects dynamic fill to create local values
		MPSTdata.push_back(sslice());
		for (int j=0; j<7; ++j) {
			MPSTdata[i].push_back(MPSTobjIn->MPSTdata[i][j]);
        }
	}


	// 1.5 - Find peak in MPS Table
	double tMass = (double)atof(commandedMass.c_str());
	if (L2Info.isGCU) {
		nSuccess = findMPSTpeak(tMass, peak, GCUMASSDEL);
	} else {
		nSuccess = findMPSTpeak(tMass, peak, NONGCUMASSDEL);
	}
	mptIndx = peak.index;
	// 1.6 - Set peak search area
	if (L2Info.isGCU) {
		if (mptIndx == -1) {
			skipReas["gcuMassNotInMPST"]++;
			sErrorMessage = "Unable to find a suitable Mass in MPS Table ";
			sErrorMessage += "for Target mass = "+util_doubleToString(tMass);
			writeToLog(sErrorMessage, sFunctionName, WARN);
			if (QLINFOFILE) {
				QuickLookLog << " - Skipped - Unable to find a suitable Mass in MPS Table" << endl;
				qlCount++;
				if (qlCount % QLPPAGE == 0) qlHeader();
			}
			cout << " - Skipped - Unable to find a suitable Mass in MPS Table" << endl;
			return 0;
		} else {
			iXl = util_stringToInt(MPSTdata[mptIndx][4]);
			iXr = util_stringToInt(MPSTdata[mptIndx][5]);
		}
	// nonGCU file
	} else {
		if (mptIndx != -1) {
			iXl = util_stringToInt(MPSTdata[mptIndx][4]);
			iXr = util_stringToInt(MPSTdata[mptIndx][5]);
			//numVerificPeaksFound++;
		}
	}

	return 1;

}

//
// --------------------------- findMPSTpeak -----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Searches the MPS table for a specific peak corresponding to mass inMass
//
// input:
//      inMass - mass to look for
//      massDel - Mass increment to use for peak ID
// output:
//   p - a struct of verifPeakInfo type
//
// returns:
//   1 for success, 0 for failure
// =============================================================================
// History: Written by Mike Rinaldi, August 2013
// =============================================================================
//
int DFMS_CreateGCUSLFPix0_Class::findMPSTpeak(double inMass, verifPeakInfo &p,
													double massDel) {

	string sFunctionName = "DFMS_CreateGCUSLFPix0_Class::findMPSTpeak";

	double mpsTableMass;

	mptIndx = -1;    // Assume no verification peak available

	// Look for target peak data index if in GCU mode
	for (int i=0; i<mptRows; i++) {
		mpsTableMass = atof(MPSTdata[i][1].c_str());
		if (mpsTableMass > inMass-massDel && mpsTableMass < inMass+massDel ) {
			mptIndx = i;
			p.index = mptIndx;
			p.mass = mpsTableMass;
			p.peakNameA = MPSTdata[mptIndx][0];
			break;
		}
	}

	// Set peak struct values for case where no verification peak is present
	if (mptIndx < 0) {
		p.index = mptIndx;
		p.mass = 0.0;
		p.peakNameA = "None Found";
		return 0;
	}

	return 1;
}


//
// ----------------------------- GCUpix0Calc -----------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Calculates/Assigns the various calibration values pix0, ppmDiff,... for GCU
// mode files;
//
// inputs:
//   mC - pointer to Mass Calib object
//
// returns:
//   1 for normal and 0 for error
// =============================================================================
// History: Written by Mike Rinaldi, May 2013
// =============================================================================
//
int DFMS_CreateGCUSLFPix0_Class::GCUpix0Calc(DFMS_PeakFit_Class *mC, int iRow)
{

     string sFunctionName = "DFMS_CreateGCUSLFPix0_Class::GCUpix0Calc";

     int nSuccess=0;

     double pix0=0.0;
     double x=0.0;
     double xUnc=0.0;
     double m = 0.0, m0 = 0.0;

     x0FitInfo xF;

     // 2.11.1 - Assign Target mass from ROSINA_SCI value in L2 HK data
     m0 = atof(commandedMass.c_str());

     // 2.11.2 - Use MPS Table mass as actual 'm'
     m = atof(MPSTdata[mptIndx][1].c_str());

	 x = mC->coeff[1];
	 xUnc = mC->errCoeff[1];

     // 2.11.3 - If the peak curvfit was successful save get the peak center data
     if (mC->fitSucceeded) {

    	 // 2.11.3.1a - if GCU file calculate pix0 using formula
    	 pix0 = massToPix0(L2Info.zoom,x,m,m0);

    	 // Set the required variables needed for the fit and to be output
    	 // to the x0Fit file.
		 xF.m0 = m0;
		 xF.m = m;
		 xF.x = x;
		 xF.pix0 = pix0;
		 xF.pix0Unc = xUnc;
		 xF.file = L2Info.fileName;

    	 // Build up a gcuMass and Pix0 array for later fitting
		 if (L2Info.lowRes) {
			 if (iRow == 1) {
				 if (m0 < GCULOWCUTOFF) {
					 gcuLmLrA.push_back(xF);
				 } else {
					 gcuHmLrA.push_back(xF);
				 }
			 } else {
				 if (m0 < GCULOWCUTOFF) {
					 gcuLmLrB.push_back(xF);
				 } else {
					 gcuHmLrB.push_back(xF);
				 }
			 }
    	 } else {
			 if (iRow == 1) {
				 if (m0 < GCULOWCUTOFF) {
					 gcuLmHrA.push_back(xF);
				 } else if (m0 >= GCULOWCUTOFF && m0 < GCUHICUTOFF){
					 gcuMmHrA.push_back(xF);
				 } else {
					 gcuHmHrA.push_back(xF);
				 }
			 } else {
				 if (m0 < GCULOWCUTOFF) {
					 gcuLmHrB.push_back(xF);
				 } else if (m0 >= GCULOWCUTOFF && m0 < GCUHICUTOFF){
					 gcuMmHrB.push_back(xF);
				 } else {
					 gcuHmHrB.push_back(xF);
				 }
			 }
    	 }
		 xF.reset();


     } else {
    	 // 2.11.4 - mC->fitSucceeded = false - Bad peak fit or No peak found
    	 // Do nada for now
     }

     return 1;

}

//
// ---------------------------- SLFpix0Calc ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Calculates the calibration value pix0 for NON-GCU mode files where a
// verification peak (SLF) is available
//
// inputs:
//   mC - pointer to Mass Calib object
//   iRow - specifies Row A/B
//   m0 - The value of the commanded mass
//   L2Info - L2 File Info structure
//   x0Fit - Pointer to the pix0 fit file object
//
// returns:
//   1 for normal and 0 for error
// =============================================================================
// History: Written by Mike Rinaldi, Feb 2014
// =============================================================================
//
int DFMS_CreateGCUSLFPix0_Class::SLFpix0Calc(DFMS_PeakFit_Class *mC, int iRow, double m0,
												L2FileInfo &L2Info, DFMS_X0FitFile_Class *x0Fit) {

	string sFunctionName = "DFMS_CreateGCUSLFPix0_Class::SLFpix0Calc";

	int nSuccess=0;

	double pix0=0.0;
	double a=0.0, b=0.0;
	double R2coeff=0.0, error=0.0;
	double x=0.0;
	double xUnc=0.0;
	double m = 0.0;

	double gcuPix0 = 0;

    // Check if this mass is one of the special Verification peak masses that
    // will be used to create the nonGCU m0 vs pix0 fit files.
	if (!(slfSet.find(L2Info.commandedMass) != slfSet.end()) ) {
		return 0;
	}

    // 2.12.1 - Define the peak center location and uncertainty
    x = mC->coeff[1];
    xUnc = mC->errCoeff[1];

    string pix0FitFileName = x0Fit->getFileName();		// Latest calibration fit file

    if (iRow == 1) {
    	a = x0Fit->aA;   // Get the fitting parameters for Row A
    	b = x0Fit->bA;
    	R2coeff = x0Fit->R2coeffA;
    	error = x0Fit->sErrorA;
    	gcuP.aA = a;
    	gcuP.bA = b;
    	gcuP.R2coeffA = R2coeff;
    } else {
    	a = x0Fit->aB;   // Get the fitting parameters for Row B
    	b = x0Fit->bB;
    	R2coeff = x0Fit->R2coeffB;
    	error = x0Fit->sErrorB;
    	gcuP.aB = a;
    	gcuP.bB = b;
    	gcuP.R2coeffB = R2coeff;
    }

    // 2.12.5 - Determine gcuPix0 from latest fit file
    gcuPix0 = x0Fit->determinePix0(a, b, m0);

    // Calculate gcu mass
    double mGCU = pixToMass(L2Info.zoom, gcuPix0, x, m0);

    // 2.12.6 - Check if CurveFit succeeded
    if (mC->fitSucceeded) {

        // 2.12.7 - Calculate mass scale borders using GCU pix0
        double mL = pixToMass(L2Info.zoom, gcuPix0, (double)PKLOWPT, m0);
        double mR = pixToMass(L2Info.zoom, gcuPix0, (double)PKHIGHPT, m0);

        // 2.12.8 - Search for MPST mass in between m1 and m2
        double pixfSpecies=0;
        double pix1,pix2;
        for (int i=0; i<mptRows; i++) {
        	m = util_stringToDouble(MPSTdata[i][1]);
        	// Check if   m1 <= mass <= m2
        	if (m >= mL && m <= mR) {
        		pixfSpecies = massToPix(zoom, gcuPix0, m, m0);
        		// Calculate the m1/m2 corresponding pixels
        		pix1 = massToPix(L2Info.zoom, gcuPix0, m-NONGCUMASSDEL, m0);
        		pix2 = massToPix(L2Info.zoom, gcuPix0, m+NONGCUMASSDEL, m0);
        		// Now check if   pix1 <= peakLoc x <= pix2
        		if (pix1 > pix2) {
        			double tp = pix2;
        			pix2 = pix1;
        			pix1 = tp;
        		}
            	if (x >= pix1 && x <= pix2) {
                    double eDisp = calcDispersion(m0);
            		pix0 = x - log(m/m0)/(CONV/(eDisp*L2Info.zoom));
                   	mptIndx = i;
                   	calcVerifPix0FitParams(L2Info, pix0, m0, m, x, xUnc, iRow);
                   	//findMaxPpmDiff(L2Info, pix0, m0, m, x, xUnc, iRow);
                    //cout << "SLFpix0Calc - Success" << endl;
            		break;
            	} else {
            		cout << "SLFpix0Calc - Failure - 3" << endl;
        			return -3;
            	}
        	} else {
            	if (i == mptRows-1) {
            		cout << "SLFpix0Calc - Failure - 2" << endl;
            		return -2;
            	}
            }
        }
   	} else {

   		// Fit did not succeed
   		cout << "SLFpix0Calc - Failure - 1" << endl;
   		return -1;

   	}

	return 1;
}

//
// -------------------------- calcVerifPix0FitParams ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method does a linear fit of verif mass vs verif pix0 (SLF).  It writes the
// results file.  The results will later be accessed to calculate pix0 for non-GCU
// files
//
// inputs:
//   L2Info - L2 file information
//   pix0 - the calculated pix0
//   m0    - commanded mass
//   m     - Calculated mass from massscale
//   x     - peak location
//   xUnc  - uncertainty in peak location
//   iRow  -  Row A/B
//
// returns:
//   None
// =============================================================================
// History: Written by Mike Rinaldi, June 2014
// =============================================================================
//
void DFMS_CreateGCUSLFPix0_Class::calcVerifPix0FitParams(L2FileInfo &L2Info,
							        double pix0, double m0, double m, double x,
							        double xUnc, int iRow) {

	x0FitInfo xF;

	 // Set the required variables needed for the fit and to be output
	 // to the x0Fit file.
	 xF.m0 = m0;
	 xF.m = m;
	 xF.x = x;
	 xF.pix0 = pix0;
	 xF.pix0Unc = xUnc;
	 xF.file = L2Info.fileName;

	// Build up a slfMass and Pix0 array for later fitting
	if (L2Info.lowRes) {
		if (iRow == 1) {
			if (m0 < SLFLOWCUTOFF) {
				slfLmLrA.push_back(xF);
			} else {
				slfHmLrA.push_back(xF);
			}
			xF.reset();
		} else {
			if (m0 <= SLFLOWCUTOFF) {
				slfLmLrB.push_back(xF);
			} else {
				slfHmLrB.push_back(xF);
			}
			xF.reset();
		}
	} else if (L2Info.hiRes) {
		if (iRow == 1) {
			if (m0 < SLFLOWCUTOFF) {
				slfLmHrA.push_back(xF);
			} else if (m0 >= SLFLOWCUTOFF && m0 < SLFHICUTOFF){
				//xF.print();
				slfMmHrA.push_back(xF);
			} else {
				slfHmHrA.push_back(xF);
			}
			xF.reset();
		} else {
			if (m0 < SLFLOWCUTOFF) {
				slfLmHrB.push_back(xF);
			} else if (m0 >= SLFLOWCUTOFF && m0 < SLFHICUTOFF){
				slfMmHrB.push_back(xF);
			} else {
				slfHmHrB.push_back(xF);
			}
			xF.reset();
		}
	}

}

//
// ---------------------------- calcPPMDiff ------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// Calculation of parts per million difference between mass found and target mass
//
// inputs:
//   m - mass found
//   m0 - target mass
//
// returns:
//   ppmdiff - absolute value in ppm
// =============================================================================
// History: Written by Mike Rinaldi, May 2013
// =============================================================================
//
double DFMS_CreateGCUSLFPix0_Class::calcPPMDiff(double m, double m0) {

	return abs((m - m0)/m0*1.0e6);

}
