
#include "DFMS_COPS_Class.hh"

//
// ---------------------------- Constructor ------------------------------------
//
DFMS_COPS_Class::DFMS_COPS_Class(string p, string f, string type) {
     
	pathC = p;
	fileC = f;
	COPSfile = pathC+"/"+fileC;
	COPSHKrows = 0;
	COPSHKcols = 0;
}

//
// ----------------------------- Destructor ------------------------------------
//
DFMS_COPS_Class::~DFMS_COPS_Class() {
    
    // Release dynamically allocated memory
    COPSHKobj.clear();
    COPSHKdata.clear();
    pathC.clear();
    fileC.clear();
    COPSfile.clear();
}

//
// ------------------------- readCOPSHKdata ------------------------------------
//
int DFMS_COPS_Class::readCOPSHKdata(fstream &is) {

	string sFunctionName = "DFMS_COPS_Class::readCOPSHKdata";

	char input[100];
	int cpos0,cpos1,cpos2;   // positions of commas within string/substring
	int len;                 // length of a string  

     // Now read the HK data into the HKdata string array
 	for (int i=0; i<COPSHKrows; i++) {

   		// Use vector objects dynamic fill to create on row
        COPSHKdata.push_back(sslice());
          
		is.getline(input,100); 
		string line = string(input);
		// Look for 1st comma
		cpos0 = line.find_first_of(",",0);
		len = cpos0-2;
		COPSHKdata[i].push_back(util_trim(line.substr(1,len)));
		// Look for second comma
		cpos1 = line.find_first_of(",",cpos0+1);
		len = cpos1-cpos0-3;
		COPSHKdata[i].push_back(util_trim(line.substr(cpos0+2,len)));
 		// Look for third and last comma
		cpos2 = line.find_first_of(",",cpos1+1);
		len = cpos2-cpos1-3;
		COPSHKdata[i].push_back(util_trim(line.substr(cpos1+2,len)));

		len = 5;
		COPSHKdata[i].push_back(util_trim(line.substr(cpos2+2,len)));
	}

	return 1;	

}

//
// ------------------------------ getCOPSdata ------------------------------------
//
int DFMS_COPS_Class::getCOPSdata() {

	string sFunctionName = "DFMS_COPS_Class::getCOPSdata";
	int nSuccess = 0;
	fstream is;

	// Open COPS file
	//cout << "COPSfile = " << COPSfile << endl;

	nSuccess = openFile(COPSfile,is,1);
	if (!nSuccess) {
		sErrorMessage = "Unable to open the PDS header for file: ";
		sErrorMessage += COPSfile+" - Skipping file processing" ;
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}

	// Read the file PDS header
	nSuccess = readPDSheader(is);
	if (!nSuccess) {
		sErrorMessage = "Unable to read the PDS header for file: ";
		sErrorMessage += COPSfile+" - Skipping file processing" ;
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}

	// Read the COPS file object description
	nSuccess = readObj(is, COPSHKobj, COPSHKrows, COPSHKcols);
	if (!nSuccess) {
		sErrorMessage = "Unable to read the Object Descp. for file: ";
		sErrorMessage += COPSfile+" - Skipping file processing" ;
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}

	// Read the COPS file Data
	nSuccess = readCOPSHKdata(is);
	if (!nSuccess) {
		sErrorMessage = "Unable to read the COPS file: ";
		sErrorMessage += COPSfile+" - Skipping file processing" ;
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	}

	// Close the COPS File stream
	closeFile(is);
     
	sInfoMessage = "Completed Reading COPS Data File ";
	writeToLog(sInfoMessage, sFunctionName, INFO);

	return 1;
}


//
// ------------------------------ printCOPSdata ------------------------------------
//
void DFMS_COPS_Class::printCOPSdata(string type) {

	// First print the PDS Header
	
	mapstr::const_iterator it;

	if (type == "PDSHeader" || type == "All") {
		for (it = PDSheader.begin(); it != PDSheader.end(); ++it) {
			printf("%-32s =     %-s%s",(it->first).c_str(),(it->second).c_str(),EOL);
		}
	}

	if (type == "COPSobj" || type == "All") {
		// Next print out Object description
		for (it = COPSHKobj.begin(); it != COPSHKobj.end(); ++it) {
			printf("%32s =     %-s%s",(it->first).c_str(),(it->second).c_str(),EOL);
		}
	}

	if (type == "COPSHKdata" || type == "All") {
		// Finally print out the COPS HK data
 		for (int i=0; i<COPSHKrows; i++) {
			printf("\"%-32s\",\"%-5s\",\"%-15s\",\"%-5s\"%s",
				(COPSHKdata[i][0]).c_str(),
				(COPSHKdata[i][1]).c_str(),
				(COPSHKdata[i][2]).c_str(),
				(COPSHKdata[i][3]).c_str(), EOL);
		}
	}	

}

//
// ---------------------- getCOPSfilesInDir -------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method retrieves all the COPS Data files in the directory
//
// inputs: The COPS Class pointer

// returns:
//   1 if success 0 otherwise
// =============================================================================
// History: Written by Mike Rinaldi, March 2013
// =============================================================================
//
int DFMS_COPS_Class::getCOPSfilesInDir(string type, string sCOPSpath, vector<L2FileInfo> &COPSinfo) {
	
	string sFunctionName="getCOPSfilesInDir";

  	DIR *dir;
  	struct dirent *entry;
	string fileName;
	L2FileInfo C;
	C.init();
	string uCapType = "", lCapType="";
	COPSinfo.clear();

	vector<string> fVec;

	if (type.compare("ng") == 0 || type.compare("NG") == 0) {
		uCapType = "NG";
		lCapType = "ng";
	} else if (type.compare("rg") == 0 || type.compare("RG") == 0) {
		uCapType = "RG";
		lCapType = "rg";
	} else if (type.compare("bg") == 0 || type.compare("BG") == 0) {
		uCapType = "BG";
		lCapType = "bg";
	}

	long good=0L;
	long k=0;
	double timeTag=0;

	// This is if the files are segregated into separate folders based on type.
	// Such as <dir>/NG, <dir>/BG,...
	//string searchPath = sCOPSpath+type+"/";
	// This is if the files are all in one folder
	string searchPath = sCOPSpath;
     
	if ((dir = opendir(searchPath.c_str())) == NULL) {
		sErrorMessage="Default Source directory - "+searchPath+" does not exist";
		writeToLog(sErrorMessage, sFunctionName, FATALERROR);
  	} else {
  		while ((entry = readdir(dir)) != NULL) {
  		  	fileName = string(entry->d_name);
  		  	if (fileName.length() > 27) {
  		  		string prefix = fileName.substr(0,2);
  		  		string firstChar = fileName.substr(0,1);
  				string suffix = fileName.substr(fileName.find_last_of(".") + 1);
  				// Skip all files shorter than 3 characters.. (like ./, ../)
  				if ((prefix.compare(uCapType) == 0 || prefix.compare(lCapType) == 0) &&
  						(suffix == "tab" || suffix == "TAB") &&
  					firstChar.compare(".") != 0) {

  					fVec.push_back(fileName);

  				} else {
  					sErrorMessage="File: "+fileName+" is not a candidate file";
  					writeToLog(sErrorMessage, sFunctionName, WARN);
  				}
  		  	} else {
  				sErrorMessage="File: "+fileName+" is not a candidate file";
  				writeToLog(sErrorMessage, sFunctionName, WARN);
  		  	}
  		}
  		closedir(dir);

  		// sort files
  		sort(fVec.begin(),fVec.end());

  		// Now go through all the sorted files and retain pertinent info
  		for(int i=0; i<fVec.size(); i++) {
			k++;
			string smode = fVec[i].substr(fVec[i].find_last_of(".")-5,5);
			string dateTime = fVec[i].substr(3,18);
			timeTag = rosToSec(dateTime,1);
			// Only keep COPS files that are within the L2 files time range
			if (k %1000 == 0) {
				cout << "Examining COPS file # " << k << endl;
			}
			if (timeTag > L2TimeMin && timeTag <= L2TimeMax) {
				good++;
				C.fileName = fVec[i];
				C.dateTime = dateTime;
				C.mode = util_stringToInt(smode.substr(1,4));
				//C.printData();
				COPSinfo.push_back(C);
				C.init();  // reset struc
			}

  		}

  	}

	//cout << "Of " << k << " COPS files found " << good << " were within L2 time bounds" << endl;

	if (type.compare("ng") == 0 && COPSinfo.size() == 0) {
		sErrorMessage="Default source directory - " + 
                      searchPath+" contains "+util_intToString(COPSinfo.size())+" useful NG files";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	} else if (COPSinfo.size() != 0 && type.compare("ng") == 0) {
		sInfoMessage = "Successfully cataloged "+util_intToString(COPSinfo.size());
		sInfoMessage += " COPS NG Data Files ";
		writeToLog(sInfoMessage, sFunctionName, INFO);
		//cout << "There were " << good << " files found" << endl;
    }
     
	if (type.compare("rg") == 0 && COPSinfo.size() == 0) {
		sErrorMessage="Default source directory - " + 
                      searchPath+" contains "+util_intToString(COPSinfo.size())+" useful RG files";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	} else if (COPSinfo.size() != 0 && type.compare("rg") == 0) {
		sInfoMessage = "Successfully cataloged "+util_intToString(COPSinfo.size());
		sInfoMessage += " COPS RG Data Files ";
		writeToLog(sInfoMessage, sFunctionName, INFO);
		//cout << "There were " << good << " files found" << endl;
	}

	if (type.compare("bg") == 0 && COPSinfo.size() == 0) {
		sErrorMessage="Default source directory - " +
	                     searchPath+" contains "+util_intToString(COPSinfo.size())+" useful BG files";
		writeToLog(sErrorMessage, sFunctionName, ERROR);
		return 0;
	} else if (COPSinfo.size() != 0 && type.compare("bg") == 0) {
		sInfoMessage = "Successfully cataloged "+util_intToString(COPSinfo.size());
		sInfoMessage += " COPS BG Data Files ";
		writeToLog(sInfoMessage, sFunctionName, INFO);
		//cout << "There were " << good << " files found" << endl;
	}
     
	return 1;
}

//
// ----------------------- findBestCOPSfile -------------------------------------
//
// =============================================================================
// Routine Description
// =============================================================================
// This method finds the closest COPS Data file to the current
// L2 file date and time
//
// inputs: 
//   sRefDateAndTime - Input date and time
//   
// returns:
//   sBestCOPS - best file name found
//   1 if success 0 otherwise
// =============================================================================
// History: Written by Mike Rinaldi, March 2013
// =============================================================================
//
int DFMS_COPS_Class::findBestCOPSfile(string sCOPSpath, string sRefDateAndTime, 
                vector<L2FileInfo> &COPSNGinfo, vector<L2FileInfo> &COPSRGinfo,
                vector<L2FileInfo> &COPSBGinfo)
{
	
	string sFunctionName="findBestCOPSfile";

	string sDateAndTime;

	// Initially set to COPS 
	sBestNGCOPS = "N/A";
	sBestRGCOPS = "N/A";
	sBestBGCOPS = "N/A";

	map<double,string> diffNG;
	map<double,string> diffRG;
	map<double,string> diffBG;

	//
	// -------------- Look for best NG files
	//
	if (COPSNGinfo.size() > 0) {
		//cout << "COPSNGsize = " << COPSNGinfo.size() << endl;
		// Filter directory contents for those that match NG COPS Data
		// file info
		for (int i=0; i<COPSNGinfo.size(); i=i+1) {
			sDateAndTime = COPSNGinfo[i].dateTime;
			//cout << i << " - sRefDateAndTime = " << sRefDateAndTime << endl;
			//cout << i << " - sDateAndTime = " << sDateAndTime << endl;
			double diff = timeDiffInSec(sDateAndTime,sRefDateAndTime);
			//cout << "diff = " << diff << endl;
			// Check if times are within one COPSDELTA
			// COPSDELTA is in minutes
			// diff is in seconds
			if (diff <= COPSDELTA*60.0) {
				sBestNGCOPS = COPSNGinfo[i].fileName;
				diffNG[diff] = sBestNGCOPS;
				//cout << "sBestNGCOPS  = " << sBestNGCOPS << endl;
			}
		}
	}
     
	// No matching files found
	if (sBestNGCOPS == "N/A") {
		sErrorMessage = "No suitable NG COPS Data file ";
		sErrorMessage += "was found in supplied path: " + sCOPSpath + "/ng";
		writeToLog(sErrorMessage, sFunctionName, WARN);
	} else {
		// Use C++ map automatic key sort property.
		// First element is the shortest time diff
		//for(map<double,string>::iterator it=diffNG.begin(); it!=diffNG.end(); ++it) {
		//	cout << ":diffNG:diff = " << it->first;
		//	cout << "  - diffNG:file = " << it->second << endl;
		//}
		sBestNGCOPS = diffNG.begin()->second;
		cout << "Found NG file: " << sBestNGCOPS << endl;
	}


	//
	// -------------- Look for best RG files
	//
	if (COPSRGinfo.size() > 0) {
		// Filter directory contents for those that match RG COPS Data
		// file info
		for (int i=0; i<COPSRGinfo.size(); i=i+1) {
			sDateAndTime = COPSRGinfo[i].dateTime;
			double diff = timeDiffInSec(sDateAndTime,sRefDateAndTime);
			// Check if times are within one COPSDELTA
			// COPSDELTA is in minutes
			// diff is in seconds
			if (diff <= COPSDELTA*60.0) {
				sBestRGCOPS = COPSRGinfo[i].fileName;
				diffRG[diff] = sBestRGCOPS;
			}
		}
	}

	// No matching files found
	if (sBestRGCOPS == "N/A") {
		sErrorMessage = "No suitable RG COPS Data file ";
		sErrorMessage += "was found in supplied path: " + sCOPSpath + "/rg";
		writeToLog(sErrorMessage, sFunctionName, WARN);
	} else {
		// Use C++ map automatic key sort property.
		// First element is the shortest time diff
		sBestRGCOPS = diffRG.begin()->second;
		cout << "Found RG file: " << sBestRGCOPS << endl;
	}

	//
	// -------------- Look for best BG files
	//
	if (COPSBGinfo.size() > 0) {
		// Filter directory contents for those that match BG COPS Data
		// file info
		for (int i=0; i<COPSBGinfo.size(); i=i+1) {
			sDateAndTime = COPSBGinfo[i].dateTime;
			//cout << "sRefDateAndTime = " << sRefDateAndTime << endl;
			//cout << "sDateAndTime = " << sDateAndTime << endl;
			double diff = timeDiffInSec(sDateAndTime,sRefDateAndTime);
			//cout << "diff = " << diff << endl;
			// Check if times are within one COPSDELTA
			// COPSDELTA is in minutes
			// diff is in seconds
			if (diff <= COPSDELTA*60.0) {
				sBestBGCOPS = COPSBGinfo[i].fileName;
				diffBG[diff] = sBestBGCOPS;
				//cout << "sBestBGCOPS  = " << sBestBGCOPS << endl;
			}
		}
	}

	// No matching files found
	if (sBestBGCOPS == "N/A") {
		sErrorMessage = "No suitable BG COPS Data file ";
		sErrorMessage += "was found in supplied path: " + sCOPSpath + "/bg";
		writeToLog(sErrorMessage, sFunctionName, WARN);
	} else {
		// Use C++ map automatic key sort property.
		// First element is the shortest time diff
		sBestBGCOPS = diffBG.begin()->second;
		cout << "Found BG file: " << sBestBGCOPS << endl;
	}

	return 1;

}
