#!/usr/bin/perl -w

use Date::Calc qw(Mktime);
use warnings;

# Handle usage issues
$num_args = $#ARGV + 1;
if ($num_args > 2) {
    print "\n";
    print "Usage(0): runOneDay.pl <fileType>\n";
    print "          Process all <MC or CE> files L2 data in $allData \n";
    print "Usage(1): runOneDay.pl <MC or CE> <20141001_204130118> <20141014_214322167>\n";
    print "          Process all <MC or CE> L2 data between the <starting> and <ending> date args \n\n";
}

if ($num_args == 1 && ($ARGV[0] eq "help" || $ARGV[0] eq "?" || $ARGV[0] eq "Help" || $ARGV[0] eq "HELP")) {
    print "\n";
    print "Usage(0): runOneDay.pl <fileType>\n";
    print "          Process all <MC or CE> files L2 data in $allData \n";
    print "Usage(1): runOneDay.pl <MC or CE> <20141001_204130118> <20141014_214322167>\n";
    print "          Process all <MC or CE> L2 data between the <starting> and <ending> date args \n\n";
    exit;
}

# Get DFMS_INSTALL_DIR env var
$installDir = $ENV{'DFMS_INSTALL_DIR'};
if ($installDir eq "") {
        print "\n\nEnvironment Variable DFMS_INSTALL_DIR must be defined\n\n";
        exit;
}

my $last = substr $installDir, -1;
if ($last ne '/') {
	$installDir = $installDir . "/";
}

# Read the perl script configuration file
my $configFile = "runOneDay.dat";
open(CONFIG, $configFile) or die "Could not open file '$configFile' $!";

# Now read the whole file
while (<CONFIG>) {
    chomp;                  # no newline
    s/#.*//;                # no comments
    s/^\s+//;               # no leading white
    s/\s+$//;               # no trailing white
    next unless length;     # anything left?
    my ($var, $value) = split(/\s*=\s*/, $_, 2);
    $config_values{$var} = $value;
}

# Global directory Variables
$allData = $config_values{ALLDATA};
my $oneDayDir = $config_values{ONEDAYDIR};
# File Type Specific directory variables
my $mcpDir = $config_values{MCPDIR};
my $cemDir = $config_values{CEMDIR};
my $copsNgDir = $config_values{COPSNGDIR};
my $copsRgDir = $config_values{COPSRGDIR};
my $copsBgDir = $config_values{COPSBGDIR};

# Set legacy variable
# 	If set to 1 then the code assumes data is coming from a single user created data directories
# 	If set to 0 then code assumes data is coming from separate data directories
$legacyProcessing = $config_values{LEGACYPROCESSING};    

# Set time gap variables
$gapTimeDelta = $config_values{GAPTIMEDELTA};    # Gap time in seconds
$maxTime = $config_values{MAXTIME};    			 # Max Time, 24 hours in seconds

# theis variable is modified depending on command line arguments
$processAllData = 0;   # 0 - means process all data
                       # 1 - means process only data within the bounds
                       #     of the command line arguments.  see usage
                       #     below.

# intialize the array allFiles
my @allFiles = ();

# get start time
$beginTime = time();
$totFilesProcessed=0;

# Set some needed constants and data prefix's
$idx = 3;
$mcPrefix = 'MC_';
$cePrefix = 'CE_';
$bgPrefix = 'BG_';
$ngPrefix = 'NG_';
$rgPrefix = 'RG_';

#print "ARGV = @{ARGV}\n";

# Process CLA's if present.
if ($ARGV[0] eq 'CE' || $ARGV[0] eq 'ce' || $ARGV[0] eq 'CEM' || $ARGV[0] eq 'cem') {
        $l2dataPrefix = $cePrefix;
}
if ($ARGV[0] eq 'MC' || $ARGV[0] eq 'mc' || $ARGV[0] eq 'MCP' || $ARGV[0] eq 'mcp') {
        $l2dataPrefix = $mcPrefix;
}
# Processe start date string and end date string, if any are present
if ($num_args == 3) {
        $start_date = $ARGV[1];
        $end_date   = $ARGV[2];
        $processAllData = 1;
} else {
        $processAllData = 0;
}


print "\n\nFirst remove all existing data in $oneDayDir \n";

#Clean out all leftover files in the $oneDayDir
#opendir DIR, $oneDayDir or die "cannot open dir $oneDayDir: $!";
#my @fd = readdir DIR;
#closedir DIR;
#$cnt=0;
$comm = "find $oneDayDir -name \"*.*\" -delete";
system($comm) == 0 or die "system $comm failed: $?";

$comm = "";

# Step 1 - If run in legacy mode then Find all files in the directory $allData
# 		   If not in legacy mode then read data from specific file type directories
if ($legacyProcessing == 1) {
	
	@allFiles = file_search();
	print "\n\nFind all available files in $allData\n";
	
} else {
	
	# First read all MCP or CE L2 files in user specified directory $mcpDir or $cemDir
	if ($l2dataPrefix eq $mcPrefix) {
		@mcFiles = file_search($mcpDir);
		push(@allFiles, @mcFiles);
	} elsif ($l2dataPrefix eq $cePrefix) {
		@ceFiles = file_search($cemDir);
		push(@allFiles, @ceFiles);
	} else {
		print "$l2dataPrefix does not exist or is neither of MCP or CEM file type";
	}
	
	# Next read all COPS files available
	my @ngCopsFiles = file_search($copsNgDir);
		push(@allFiles, @ngCopsFiles);
	my @rgCopsFiles = file_search($copsRgDir);
		push(@allFiles, @rgCopsFiles);
	my @bgCopsFiles = file_search($copsBgDir);
		push(@allFiles, @bgCopsFiles);
		
}

#print "@allFiles\n";

@allSorted = sort @allFiles;

# Total files available in time period specified
my $numFiles = scalar @allSorted;

print "There are a total of $numFiles ($l2dataPrefix and COPS files) available\n";

print "\nCopy all NG-RG-BG COPS files to $oneDayDir\n";

#STEP 2 - First copy all available COPS files into $oneDayDir
my $ibgcnt=0;
my $irgcnt=0;
my $ingcnt=0;
my $iallcnt = 0;
if ($legacyProcessing == 1) {
	foreach $fi (@allSorted) {
        $prefix = substr($fi,0,3);
        if ($prefix eq $bgPrefix || $prefix eq $ngPrefix || $prefix eq $rgPrefix) {
      		system("cp $allData/$fi $oneDayDir/");
          	$iallcnt++;
          	if (!($iallcnt % 500)) {
          		print "copied the ${iallcnt}th COPS file from $allData to $oneDayDir\n";
          	}
       	}
	}
} else {
	foreach $fi (@allSorted) {
        $prefix = substr($fi,0,3);
        if ($prefix eq $bgPrefix) {
      		system("cp $copsBgDir/$fi $oneDayDir/");
          	$ibgcnt++;
          	if (!($ibgcnt % 500)) {
          		print "copied the ${ibgcnt}th COPS file from $copsBgDir to $oneDayDir\n";
          	}        	
        } elsif ($prefix eq $ngPrefix) {
      		system("cp $copsNgDir/$fi $oneDayDir/");
          	$ingcnt++;
          	if (!($ingcnt % 500)) {
          		print "copied the ${ingcnt}th COPS file from $copsNgDir to $oneDayDir\n";
          	}        	
        } elsif ($prefix eq $rgPrefix) {        	
      		system("cp $copsRgDir/$fi $oneDayDir/");
          	$irgcnt++;
          	if (!($irgcnt % 500)) {
          		print "copied the ${irgcnt}th COPS file from $copsRgDir to $oneDayDir\n";
          	} 
       	}
	}
}

# Now copy a block of files from the $allData or distinc data directory into the
# $oneDayDir directory.  We need to set the ~24hr cycle to start at the
# natural instrument mode change point.  There are normally two of these
# per day.  Therefore skip the first data gap and stop at the second.

# STEP 3 - First analyze MC or CEM files to find natural data gaps
if ($l2dataPrefix eq $mcPrefix) {
    ($strtBlck_ref, $endBlck_ref) = get_indiciesOfGaps(\@mcFiles);
} elsif ($l2dataPrefix eq $cePrefix) {
    ($strtBlck_ref, $endBlck_ref) = get_indiciesOfGaps(\@ceFiles);	
}
my @strtBlck = @$strtBlck_ref;
my @endBlck = @$endBlck_ref;

# STEP 4- Proceed to process one block of data.

for (my $j=0; $j<$#endBlck+1; $j++) {
	$iTot = 0;
    # Step 4a - Copy one block of MC or CEM data to $oneDayDir/
#    for (my $i=$strtBlck[$j]; $i<=$endBlck[$j]; $i++) {
    for (my $i=$ibgcnt+$strtBlck[$j]; $i<=$ibgcnt+$endBlck[$j]; $i++) {

        $fi = $allSorted[$i];
        if (!($iTot % 500)) {
        	print "Copied the ${iTot}th $l2dataPrefix file to $oneDayDir\n";
        }
        $prefix = substr($fi,0,3);
        if ($prefix eq $l2dataPrefix) {  # This is an MCP or CEM file
        	if ($legacyProcessing == 1) {
                system("cp $allData/$fi $oneDayDir/");
                $iTot++;
             } else {
				if ($prefix eq $mcPrefix) {
                	system("cp $mcpDir/$fi $oneDayDir/");
                    $iTot++;
                } elsif ($prefix eq $cePrefix) {
                	system("cp $cemDir/$fi $oneDayDir/");
                   	$iTot++;                		
               	}
			}
        } else {
            print "$fi is not a $l2dataPrefix file"
        }
    }

    print "\nA total of $iTot Level 2 $l2dataPrefix files were copied to $oneDayDir \n\n";
    $totFilesProcessed += $iTot;
    #sleep(5);
    
    my $comm = "";

    # STEP 4b - Now run the DFMS L2->L3 code on the files in the oneDay directory
    if($^O eq 'darwin') {
    	$comm = "${installDir}bin/DFMS_PDS_L2_to_L3";
    } elsif ($^O eq 'cygwin') {
        $comm = "${installDir}bin/DFMS_PDS_L2_to_L3.exe";
    } elsif ($^O eq 'linux') {
        $comm = "${installDir}bin/DFMS_PDS_L2_to_L3";
    } else {
    	print "command $comm not set properly!!\n OS response must be one of (darwin, linux, or cygwin)\n";
        exit;
    }
    system($comm) == 0 or die "system $comm failed: $?";

    print"\n\nDelete processed $l2dataPrefix files from $oneDayDir \n";

    # STEP 4c - Now delete all MCP or CEM files in the $oneDayDir directory
    if ($prefix eq $mcPrefix) {
		$comm = "find $oneDayDir -name \"MC*.*\" -delete";
    } elsif ($prefix eq $cePrefix) {
		$comm = "find $oneDayDir -name \"CE*.*\" -delete";                		
    }
    system($comm) == 0 or die "system $comm failed: $?";
    


    print "\nDeleted $l2dataPrefix files from $oneDayDir\n\n";

    $totNum = $#endBlck+1;
    my $blkCnt = $j+1;
    print "\n--------------------------------------------------------------------------\n";
    print "      ------- Completed L2->L3 processing of block $blkCnt of $totNum ----------\n";
    print "--------------------------------------------------------------------------\n\n";
    #sleep(5);

}

# get end time
$endTime = time();

$totTimeForProcessing = ($endTime-$beginTime)/60.0;   # in minutes
$sTotTimeForProcessing = sprintf("%.3f minutes",$totTimeForProcessing);

print "------------ Processing of $l2dataPrefix files Complete ------------- \n";
print "               Processed: $totFilesProcessed L2 $l2dataPrefix files \n";
print "           Total processing time: $sTotTimeForProcessing \n";
print "----------------------------------------------------------- \n";





# ----------------------------------------------------------------------------
# Look for every 2nd 30-60 minute gap that naturally occurs due
# to mode changes. This gap should happen between 21:30-22:30 UTC if
# we start looking from before 06:00.  Otherwise it can happen between
# 06:30-07:30.  It can also happen at other times if the spacecfrat is
# performing a maneuver.
# ----------------------------------------------------------------------------
sub get_indiciesOfGaps  {
	
	    my ($fiRef) = @_;
	    my @aFiles = @{$fiRef};
	    
        my @sIndx = ();
        my @eIndx = ();
        my @timeArray = ();
        my @fileIndxArray = ();
        my $lastTime=0;
        my $thisTime=0;
        my $nextTime=0;
        my $gapNum=0;
        
        #my $mcF = 0;
        
        @srted = sort @aFiles;
        
        # First gather the time and index info for the MC or CEM files
        for (my $i=0; $i<$#srted+1; $i++) {
                $fi = $srted[$i];
                my $prefix = substr($fi,0,3);
                if ($prefix eq $l2dataPrefix) {     # This is an MC or CEM file
                        $fTime = get_Filetime($fi,$idx);
                        push(@timeArray,$fTime);
                        push(@fileIndxArray,$i);
                        #$mcF++;
                }
        }
        
        #print "Total Files = $#srted \n";
        #print "Total MC files = $mcF \n";

        # Now look for the gap time and break up files into blocks separated by 2 gaps
        # if the time in
        push(@sIndx,$fileIndxArray[0]);
        for (my $i=1; $i<$#fileIndxArray+1; $i++) {
                if ($i+1 < $#fileIndxArray+1) {
                        $lastTime = $timeArray[$i-1];
                        $thisTime = $timeArray[$i];
                        $nextTime = $timeArray[$i+1];
                        #$t1 = sprintf("     last time = %d\n",$lastTime);  print $t1;
                        #$t2 = sprintf("%4.4d this time = %d\n",$i,$thisTime);  print $t2;
                        #$t3 = sprintf("     next time = %d\n",$nextTime);  print $t3;
                        if (($nextTime - $thisTime) >= $maxTime) {
                                push(@sIndx,$fileIndxArray[$i+1]);
                                push(@eIndx,$fileIndxArray[$i]);   # Set previous file as end of block
                                $gapNum=0;
                        } else {
                                if (($nextTime - $thisTime) > $gapTimeDelta && ($nextTime - $thisTime) < $maxTime) {
                                        $gapNum++;
                                        if ($gapNum == 2) {
                                                $gapNum = 0;
                                                push(@eIndx,$fileIndxArray[$i]);
                                                if ($i+1 < $#fileIndxArray) {
                                                        push(@sIndx,$fileIndxArray[$i+1]);
                                                } else {
                                                        last;
                                                }
                                        }
                                }
                        }
                }
        }
        if ($#sIndx > $#eIndx) {
                push(@eIndx,$fileIndxArray[$#fileIndxArray]);
        }

        # Print start stop indices to screen
        #print "Num of start Indicies: {$#sIndx+1}\n";
        #print "Num of end   Indicies: {$#eIndx+1}\n";
        #print "Num of TimeArray     : {$#timeArray+1}\n";
        #print "Num of fileIndxArray : {$#fileIndxArray+1}\n";
        
        print"\n\n";
        print "-------------------------------------------------------------------------------------------------------------\n";
        print "Blck  First Idx   Last Idx       First File in Block                Last File in Block           Delta(min)\n";
        print "-------------------------------------------------------------------------------------------------------------\n";

        for (my $i=0; $i<$#sIndx+1; $i++) {
                $delta = ($timeArray[$eIndx[$i]] - $timeArray[$sIndx[$i]])/60.00;
                $sfi = $srted[$sIndx[$i]];
                $efi = $srted[$eIndx[$i]];
                $text = sprintf(" %2.2d   %5.5d  -->  %5.5d  -- %s   %s -- %5.2f\n", $i+1,$sIndx[$i],$eIndx[$i],$sfi,$efi,$delta);
                print $text;
        }
        print "-------------------------------------------------------------------------------------------------------------\n\n";

        #sleep(2);

        return (\@sIndx, \@eIndx);
}


# ----------------------------------------------------------------------------
# Use Perl module to get # of seconds since 1/1/1970 from date in file name
# ----------------------------------------------------------------------------
sub get_Filetime
{
        my $syr = substr($fi,$idx,4);
        my $smn = substr($fi,$idx+4,2);
        my $sday = substr($fi,$idx+6,2);
        my $shr = substr($fi,$idx+9,2);
        my $smin = substr($fi,$idx+11,2);
        my $sec = substr($fi,$idx+13,2);
        my $ttime = Date::Calc::Mktime($syr, $smn, $sday, $shr, $smin, $sec);
        return $ttime;
}

# ----------------------------------------------------------------------------
#  Find all BG and CEM or MCP files in the $allData directory
# ----------------------------------------------------------------------------
sub file_search
{
		my @fout = ();

        my $numargs = scalar @_;

        # First check if num of args is 0.  If so then use legacy method.  All input files in one dir
        if ($numargs == 0) {

        	opendir DIR, $allData or die "cannot open dir $allData: $!";
        	my @f = readdir DIR;
        	foreach $fi (@f) {
            	my $prefix = substr($fi,0,3);
               	if ($prefix eq $l2dataPrefix) {
                	my $date = substr($fi,$idx,8);
                    if ($processAllData == 1) {
                    	if ($date ge $start_date && $date le $end_date) {
                        	push(@fout, $fi);
                        }
                 	} else {
                 		push(@fout, $fi);
                 	}
                    #print "MCP - $date is >= $start_date and <= $end_date\n";
                } elsif ($prefix eq  $bgPrefix || $prefix eq $ngPrefix || $prefix eq $rgPrefix) {
                	my $date = substr($fi,$idx,8);
                    if ($processAllData == 1) {
                    	if ($date ge $start_date && $date le $end_date) {
                        	push(@fout, $fi);
                        }
                     } else {
                     	push(@fout, $fi);
                     }
                     #print "COPS - $date is >= $start_date and <= $end_date\n";
                }
        	}

        	#   print "dir ::@f\n";
        	closedir DIR;
        	
        # If num of args is > 0.  Then use file specific directory method.
        } else {
        	
					my $dataDir = $_[0];
					if (-d $dataDir) {   # check if directory exists
    
						opendir DIR, $dataDir or die "cannot open dir $dataDir: $!";
        		my @f = readdir DIR;
        		foreach $fi (@f) {
            		my $prefix = substr($fi,0,3);
              	 	if ($prefix eq $l2dataPrefix) {
                		my $date = substr($fi,$idx,8);
                  	  if ($processAllData == 1) {
                    		if ($date ge $start_date && $date le $end_date) {
                      	  	push(@fout, $fi);
                        	}
                 		} else {
                 			push(@fout, $fi);
                 		}
                    	#print "MCP - $date is >= $start_date and <= $end_date\n";
                	} elsif ($prefix eq  $bgPrefix || $prefix eq $ngPrefix || $prefix eq $rgPrefix) {
                		my $date = substr($fi,$idx,8);
                    	if ($processAllData == 1) {
                    		if ($date ge $start_date && $date le $end_date) {
                        		push(@fout, $fi);
                        	}
                     	} else {
                     		push(@fout, $fi);
                     	}
                     	#print "COPS - $date is >= $start_date and <= $end_date\n";
                	}
        		}

        		#   print "dir ::@f\n";
        		closedir DIR;
        	} else {
    				print "Directory $dataDir does not exist.... skipping\n";
					}
        }

    return @fout;
}
