/*
 ================================================================================================================================
 Author     : Sebastien Gasc
 Version    : 1.0
 Date       : 05.11.2018
 Description: ROSINA DFMS PDS L3 mass correction & enhancement
 Compilation: cls & gcc -O2 -Wall -o ROSINA_DFMS_L3_ENHANCEMENT.exe ROSINA_DFMS_L3_ENHANCEMENT.c & ROSINA_DFMS_L3_ENHANCEMENT.exe
 ================================================================================================================================
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <dirent.h>
#include <windows.h>

/// PATHS ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
char* DFMS_L2_path = "D:\\L2_DFMS\\";
char* DFMS_L3_path = "D:\\L3_DFMS\\";
char* DFMS_L3_output_path = "D:\\L3_DFMS_ENHANCED\\";
char* p0_list = "p0_L2.DAT";
char* p0_skipped = "p0_L2_skipped.DAT";
/// PARAMETERS ///////////////////////////////////////////////////////////////////////////////////////////////////////////////
int process_p0 = 0;   // 1 = process p0 from DFMS L2, 0 = read p0_list
int precision = 2;    // number of digits of the output
int MTP_START = 2;    // First MTP with DFMS spectra = MTP2
int MTP_STOP  = 35;   // Last MTP = MTP35
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

struct data
{
    time_t time;
    float res;
    int rowA;
    int rowB;
};

char* concat (const char *s1, const char *s2)
{
    const size_t len1 = strlen(s1);
    const size_t len2 = strlen(s2);
    char *result = malloc(len1+len2+1);    // +1 for the null-terminator

    memcpy(result, s1, len1);
    memcpy(result+len1, s2, len2+1);    // +1 to copy the null-terminator

    return result;
}

int find_closest (void const *data, time_t time, int row)
{
    struct data const *p = data;
    int n = 0;

    while ((p[n].time <= time) && (p[n].time != 0)) {++n;}

    if (row == 0)
    {
        if (abs(p[n].time - time) > abs(p[n-1].time - time))
            return p[n-1].rowA;
        else
            return p[n].rowA;
    }
    else
    {
        if (abs(p[n].time - time) > abs(p[n-1].time - time))
            return p[n-1].rowB;
        else
            return p[n].rowB;
    }
}

float interpolate_p0 (float m, void const **data, time_t time, int resolution, int row)
{
    struct data const **p = data;
    float xa = 0, xb = 0, ya = 0, yb = 0;
    if (resolution == 1)    // HR
    {
        if (m < 18)
        {
            xa = 16;
            xb = 18;
    //        ya = find_closest(p[0], time, row);
            yb = find_closest(p[1], time, row);
            if (time < 1453849200) ya = yb + 1.17;    // before 2016-01-27
            else ya = yb + 3.55;                      // after 2016-01-27
        }
        else if (m < 28)
        {
            xa = 18;
            xb = 28;
            ya = find_closest(p[1], time, row);
            yb = find_closest(p[2], time, row);
        }
        else if (m < 44)
        {
            xa = 28;
            xb = 44;
            ya = find_closest(p[2], time, row);
            yb = find_closest(p[3], time, row);
        }
        else if (m < 70)
        {
            xa = 44;
            xb = 60;
            ya = find_closest(p[3], time, row);
    //        yb = find_closest(p[4], time, row);
            int y_temp = find_closest(p[1], time, row);
            if (time < 1453849200) yb = y_temp + 0.04;    // before 2016-01-27
            else yb = y_temp + 2.37;                      // after 2016-01-27
        }
        else
        {
            int y_temp = find_closest(p[1], time, row);
            if (time < 1453849200) ya = y_temp + 12.79;    // before 2016-01-27
            else ya = y_temp + 32.83;                      // after 2016-01-27
    
            return ya;
        }
        return ya + (m-xa)*(yb-ya)/(xb-xa);
    }
    else    // LR
    {
        if (m < 18)
        {
            xa = 16;
            xb = 18;
            yb = find_closest(p[1], time, row);
            ya = yb - 2;
        }
        else if (m < 28)
        {
            xa = 18;
            xb = 28;
            ya = find_closest(p[1], time, row);
            yb = ya + 1;
        }
        else
        {
            ya = find_closest(p[1], time, row);
            return ya;
        }
        return ya + (m-xa)*(yb-ya)/(xb-xa);
    }
}

float interpolate_D (float m, int resolution)
{
    if ((resolution == 1) && (m >= 70))
    {
        return 382200*pow(m,-0.34);
    }
    else    // LR modes and HR modes with m < 70
    {
        return 127000;
    }
}

float zoom (float m, char* mode)
{
    int md = 0;
    sscanf(mode, "M%d.TAB", &md);

    // LR modes
    if (md == 150    // Gas, MCP, MeEM, LR, medGCU
     || md == 151    // Gas, MCP, MeEM, LR, lowGCU
     || md == 154    // Gas, MCP, LoEM, LR, medGCU
     || md == 155    // Gas, MCP, HiEM, LR, lowGCU
     || md == 200    // Gas, MCP, LoEM, LR, survey
     || md == 205    // Ion, MCP, LR, survey, MG = -10V
     || md == 215    // Ion, MCP, LR, survey, MG = -5V
     || md == 220    // Gas, MCP, HiEM, LR, survey
     || md == 250    // Gas, MCP, HiEM, LR, survey
     || md == 260    // Gas, MCP, HiEM, LR, survey, hypersensitive (PA = -3kV)
     || md == 500    // Gas, MCP, HiEM,LR, m13-50, 20s int
     || md == 505    // Ion, MCP, Energy stepping 10-970eV in 60eV steps, m18, LR
     || md == 515    // Ion, MCP, Energy stepping 10-100eV in 15eV steps,m18+44, LR
     || md == 520    // Gas, MCP, HiEM, LR, early comet, 20s int
     || md == 560    // Gas, MCP, HiEM, LR, 44-150,20s int
     || md == 570    // Gas, MCP, HiEM, LR, early comet, 20s int, MG = -50V
     || md == 580    // Gas, MCP, HiEM, LR, early comet, 20s int, MG = 50V
     || md == 1040   // Gas, MCP, HiEM, LR, m16-22
     || md == 1060   // Gas, MCP, 90eV, 200uA, LR, Kr - LR mass selection
     || md == 1070)  // Gas, MCP, 90eV, 200uA, LR, Xe - LR mass selection
    {
        return 1;
    }
    else    // HR modes
    {
        return 6.4;
    }
}


int main ()
{
    clock_t start_clock = clock();    // Time stamp needed for time calculation

    if (process_p0 == 1)
    {
        /// Get p0 values from DFMS L2 PDS
        struct data *p0_LR[6];    // 0 = 16, 1 = 18, 2 = 28, 3 = 44, 4 = 60, 5 = 76 (, 6 = 78)
        struct data *p0_HR[6];    // 0 = 16, 1 = 18, 2 = 28, 3 = 44, 4 = 60, 5 = 76 (, 6 = 78)

        int mm = 0;
        while (mm < 6)
        {
           p0_LR[mm] = malloc(100000 * sizeof(struct data));
           p0_HR[mm] = malloc(100000 * sizeof(struct data));
            ++mm;
        }
        int DFMS_file_number_m[2][6] = {{0}};

        int MTP = MTP_START;
        while (MTP <= MTP_STOP)
        {
            printf("-------------------- MTP%d --------------------\n", MTP);
            clock_t begin = clock();

            /// Definition of MTP paths
            char MTP_string[3];
            sprintf(MTP_string, "%d", MTP);

            /// Definition of the DFMS L2 directories
            char* r1 = concat(DFMS_L2_path, "MTP");
            char* r2 = concat(r1, MTP_string);
            char* DFMS_L2_directory[2];
            DFMS_L2_directory[0] = concat(r2, "\\DFMS\\CE\\");
            DFMS_L2_directory[1] = concat(r2, "\\DFMS\\MC\\");
            free(r1);
            free(r2);

            DIR* DFMS_dir[1];
            struct dirent* DFMS_in_file;
            FILE* DFMS_L2_file;
            FILE* p0_save = fopen(p0_list, "a");
            FILE* p0_skip = fopen(p0_skipped, "a");
            char buffer[80];

            /// Read DFMS files in the DFMS directories

            int source = 1;
            int DFMS_file_number = 0;

            while (source < 2)    // 1 = MCP
            {
                /// Open the directory
                DFMS_dir[source] = opendir(DFMS_L2_directory[source]);

                /// Create MCP folders
                if (source == 1)
                {
                    printf("Processing DFMS MCP file      ");

                    /// Read all DFMS L2 files in current directory
                    while ((DFMS_in_file = readdir(DFMS_dir[source])))
                    {
                        if (!strcmp (DFMS_in_file->d_name, "."))    // Skip "." (from DFMS_dir)
                            continue;
                        if (!strcmp (DFMS_in_file->d_name, ".."))    // Skip ".." (from DFMS_dir)
                            continue;
                        if (!strncmp (&DFMS_in_file->d_name[22], "M0600", 5)
                         || !strncmp (&DFMS_in_file->d_name[22], "M0601", 5)
                         || !strncmp (&DFMS_in_file->d_name[22], "M0602", 5)
                         || !strncmp (&DFMS_in_file->d_name[22], "M0620", 5)
                         || !strncmp (&DFMS_in_file->d_name[22], "M0621", 5)
                         || !strncmp (&DFMS_in_file->d_name[22], "M0622", 5)
                         || !strncmp (&DFMS_in_file->d_name[22], "M0630", 5)
                         || !strncmp (&DFMS_in_file->d_name[22], "M0631", 5)
                         || !strncmp (&DFMS_in_file->d_name[22], "M0632", 5)
                         || !strncmp (&DFMS_in_file->d_name[22], "M9999", 5))
                            continue;
//if (!strncmp (&DFMS_in_file->d_name[22], "M0152", 5)    // LR only
// || !strncmp (&DFMS_in_file->d_name[22], "M0153", 5)
// || !strncmp (&DFMS_in_file->d_name[22], "M0157", 5)
// || !strncmp (&DFMS_in_file->d_name[22], "M0202", 5)
// || !strncmp (&DFMS_in_file->d_name[22], "M0212", 5)
// || !strncmp (&DFMS_in_file->d_name[22], "M0217", 5)
// || !strncmp (&DFMS_in_file->d_name[22], "M0222", 5)
// || !strncmp (&DFMS_in_file->d_name[22], "M0232", 5)
// || !strncmp (&DFMS_in_file->d_name[22], "M0242", 5)
// || !strncmp (&DFMS_in_file->d_name[22], "M0245", 5)
// || !strncmp (&DFMS_in_file->d_name[22], "M0262", 5)
// || !strncmp (&DFMS_in_file->d_name[22], "M0502", 5)
// || !strncmp (&DFMS_in_file->d_name[22], "M0522", 5)
// || !strncmp (&DFMS_in_file->d_name[22], "M0562", 5)
// || !strncmp (&DFMS_in_file->d_name[22], "M0564", 5)
// || !strncmp (&DFMS_in_file->d_name[22], "M0566", 5)
// || !strncmp (&DFMS_in_file->d_name[22], "M0572", 5)
// || !strncmp (&DFMS_in_file->d_name[22], "M0582", 5)
// || !strncmp (&DFMS_in_file->d_name[22], "M1012", 5)
// || !strncmp (&DFMS_in_file->d_name[22], "M1042", 5)
// || !strncmp (&DFMS_in_file->d_name[22], "M1052", 5)
// || !strncmp (&DFMS_in_file->d_name[22], "M1062", 5)
// || !strncmp (&DFMS_in_file->d_name[22], "M1072", 5)
// || !strncmp (&DFMS_in_file->d_name[22], "M9000", 5))
//    continue;

                        char* s = concat(DFMS_L2_directory[source], DFMS_in_file->d_name);
                        DFMS_L2_file = fopen (s, "r");
                        free(s);

                        if (DFMS_L2_file == NULL)
                        {
                            fprintf(stderr, "\n\nError: failed to open DFMS L3 file \"%s\" - %s\n", DFMS_in_file->d_name, strerror(errno));
                            return 0;
                        }

                        /// Read header
                        int line_number = 1;

                        while (line_number < 33)    // Skip 33 lines
                        {
                            fgets(buffer, 80, DFMS_L2_file);
                            ++line_number;
                        }

                        /// Get start acquisition time
                        float sec = 0;
                        struct tm start;
                        fgets(buffer, 80, DFMS_L2_file);
                        sscanf(buffer, "START_TIME                       =     %4d-%2d-%2dT%2d:%2d:%6f                ", &start.tm_year, &start.tm_mon, &start.tm_mday, &start.tm_hour, &start.tm_min, &sec);
                        start.tm_year = start.tm_year - 1900;    // Conversion needed by struct tm
                        start.tm_mon = start.tm_mon - 1;    // Conversion needed by struct tm
                        start.tm_sec = (int) sec;    // Converts seconds to int (needed by (struct tm))
                        start.tm_isdst = -1;    // Avoids conversion to summer/winter time
                        ++line_number;

                        /// Get stop acquisition time
                        sec = 0;
                        struct tm stop;
                        fgets(buffer, 80, DFMS_L2_file);
                        sscanf(buffer, "STOP_TIME                        =     %4d-%2d-%2dT%2d:%2d:%6f                ", &stop.tm_year, &stop.tm_mon, &stop.tm_mday, &stop.tm_hour, &stop.tm_min, &sec);
                        stop.tm_year = stop.tm_year - 1900;    // Conversion needed by struct tm
                        stop.tm_mon = stop.tm_mon - 1;    // Conversion needed by struct tm
                        stop.tm_sec = (int) sec;    // Converts seconds to int (needed by struct tm)
                        stop.tm_isdst = -1;    // Avoids conversion to summer/winter time
                        ++line_number;

                        /// Calculate mean acquisition time
                        time_t time_start = mktime (&start);    // Converts struct tm to time_t (number of seconds since 1970)
                        time_t time_stop = mktime (&stop);
                        time_t time_mean = time_start/2 + time_stop/2;    // Calculate mean value [(start + stop)/2 exceeds time_t (= int) max value]

                        while (line_number < 98)    // Skip lines
                        {
                            fgets(buffer, 80, DFMS_L2_file);
                            ++line_number;
                        }

                        float m0 = 0;
                        fgets(buffer, 80, DFMS_L2_file);
                        sscanf(buffer, "\"ROSINA_DFMS_SCI_MASS            \",\"     \",\"%f\",\"     \" \n", &m0);
                        ++line_number;

                        int m = 0;
                             if (m0 == 16) {m = 0;}
                        else if (m0 == 18) {m = 1;}
                        else if (m0 == 28) {m = 2;}
                        else if (m0 == 44) {m = 3;}
                        else if (m0 == 60) {m = 4;}
                        else if (m0 == 76) {m = 5;}
                        else
                        {
                            fclose(DFMS_L2_file);
                            continue;
                        }

                        while (line_number < 341)    // Skip header
                        {
                            fgets(buffer, 80, DFMS_L2_file);
                            ++line_number;
                        }
                        
                        line_number = 0;
                        int bin[512] = {0};
                        float ions_rowA[512] = {0};
                        float ions_rowB[512] = {0};
                        float rowA_temp = 0;
                        float rowB_temp = 0;
                        int p0_temp[2] = {0, 0};

                        float zoom_value = zoom(m0, &DFMS_in_file->d_name[22]);
                        int z = 1;    // HR
                        if (zoom_value == 1) {z = 0;}    // LR
                        float D = interpolate_D(m0, z);

                        /// Find peak position (highest value in the whole spectrum)
                        while (fscanf(DFMS_L2_file, "%d,%f,%f,                                                \n",
                                                     &bin[line_number],
                                                            &ions_rowA[line_number],
                                                                  &ions_rowB[line_number]) == 3)    // Read lines
                        {
                            if (ions_rowA[line_number] > rowA_temp)
                            {
                                rowA_temp = ions_rowA[line_number];
                                p0_temp[0] = bin[line_number];
                            }
                            if (ions_rowB[line_number] > rowB_temp)
                            {
                                rowB_temp = ions_rowB[line_number];
                                p0_temp[1] = bin[line_number];
                            }
                            ++line_number;
                        }

                        /// Calculate p0
                        int qA = 0;
                        int qB = 0;
                        if (m0 == 16)
                        {
                            qA = p0_temp[0] - (D*zoom_value*log(15.99436604/m0))/25;
                            qB = p0_temp[1] - (D*zoom_value*log(15.99436604/m0))/25;
                            ++DFMS_file_number_m[z][m];
                        }
                        else if (m0 == 18)
                        {
                            qA = p0_temp[0] - (D*zoom_value*log(18.0100161/m0))/25;
                            qB = p0_temp[1] - (D*zoom_value*log(18.0100161/m0))/25;
                            ++DFMS_file_number_m[z][m];
                        }
                        else if (m0 == 28)
                        {
                            qA = p0_temp[0] - (D*zoom_value*log(27.99436604/m0))/25;
                            qB = p0_temp[1] - (D*zoom_value*log(27.99436604/m0))/25;
                            ++DFMS_file_number_m[z][m];
                        }
                        else if (m0 == 44)
                        {
                            qA = p0_temp[0] - (D*zoom_value*log(43.98928066/m0))/25;
                            qB = p0_temp[1] - (D*zoom_value*log(43.98928066/m0))/25;
                            ++DFMS_file_number_m[z][m];
                        }
                        else if (m0 == 60)
                        {
                            qA = p0_temp[0] - (D*zoom_value*log(59.96643721/m0))/25;
                            qB = p0_temp[1] - (D*zoom_value*log(59.96643721/m0))/25;
                            ++DFMS_file_number_m[z][m];
                        }
                        else if (m0 == 76)
                        {
                            qA = p0_temp[0] - (D*zoom_value*log(75.94359377/m0))/25;
                            qB = p0_temp[1] - (D*zoom_value*log(75.94359377/m0))/25;
                            ++DFMS_file_number_m[z][m];
                        }

                        /// Save values in structure
                        if (z == 0)    // LR
                        {
                            p0_LR[m][DFMS_file_number_m[0][m]-1].time = mktime (&time_mean);
                            p0_LR[m][DFMS_file_number_m[0][m]-1].res = zoom_value;
                            p0_LR[m][DFMS_file_number_m[0][m]-1].rowA = qA;
                            p0_LR[m][DFMS_file_number_m[0][m]-1].rowB = qB;
                        }
                        else    // HR
                        {
                            p0_HR[m][DFMS_file_number_m[1][m]-1].time = mktime (&time_mean);
                            p0_HR[m][DFMS_file_number_m[1][m]-1].res = zoom_value;
                            p0_HR[m][DFMS_file_number_m[1][m]-1].rowA = qA;
                            p0_HR[m][DFMS_file_number_m[1][m]-1].rowB = qB;
                        }

                        struct tm mean =* localtime(&time_mean);

                        /// Sort and save values in a file
                        if (((z == 0) && (qA > 245) && (qB > 245) && (qA < 280) && (qB < 280))
                         || ((z == 1)
                             &&
                                (((mean.tm_year == 114 || (mean.tm_year == 115 && mean.tm_mon < 8))
                                  && ((m0 == 16 && (qA > 265) && (qB > 265) && (qA < 296) && (qB < 298))
                                   || (m0 == 18 && (qA > 265) && (qB > 265) && (qA < 296) && (qB < 298))
                                   || (m0 == 28 && (qA > 265) && (qB > 270) && (qA < 291) && (qB < 296))
                                   || (m0 == 44 && (qA > 265) && (qB > 265) && (qA < 292) && (qB < 293))
                                   || (m0 == 60 && (qA > 265) && (qB > 265) && (qA < 297) && (qB < 297))
                                   || (m0 == 76 && (qA > 280) && (qB > 280) && (qA < 304) && (qB < 304))))
                             
                              || (((mean.tm_year == 115 && mean.tm_mon >= 8) || (mean.tm_year == 116 && mean.tm_mon < 1) || (mean.tm_year == 116 && mean.tm_mon == 0 && mean.tm_mday < 27))
                                  && ((m0 == 16 && (qA > 264) && (qB > 264) && (qA < 309) && (qB < 311))
                                   || (m0 == 18 && (qA > 264) && (qB > 264) && (qA < 309) && (qB < 309))
                                   || (m0 == 28 && (qA > 264) && (qB > 264) && (qA < 309) && (qB < 309))
                                   || (m0 == 44 && (qA > 264) && (qB > 264) && (qA < 304) && (qB < 306))
                                   || (m0 == 60 && (qA > 264) && (qB > 264) && (qA < 304) && (qB < 311))
                                   || (m0 == 76 && (qA > 264) && (qB > 280) && (qA < 315) && (qB < 315))))
                               
                              || (((mean.tm_year == 116 && mean.tm_mon > 0) || (mean.tm_year == 116 && mean.tm_mon == 0 && mean.tm_mday >= 27))
                                  && ((m0 == 16 && (qA > 200) && (qB > 200) && (qA < 230) && (qB < 231))
                                   || (m0 == 18 && (qA > 200) && (qB > 200) && (qA < 230) && (qB < 230))
                                   || (m0 == 28 && (qA > 200) && (qB > 200) && (qA < 230) && (qB < 230))
                                   || (m0 == 44 && (qA > 200) && (qB > 200) && (qA < 230) && (qB < 230))
                                   || (m0 == 60 && (qA > 200) && (qB > 200) && (qA < 230) && (qB < 230))
                                   || (m0 == 76 && (qA > 239) && (qB > 239) && (qA < 260) && (qB < 260)))))))
                        {
                            fprintf(p0_save, "\"%4d-%02d-%02dT%02d:%02d:%02d\"  %d  %d  %d  %d\n", 1900+mean.tm_year, mean.tm_mon+1, mean.tm_mday, mean.tm_hour, mean.tm_min, mean.tm_sec, qA, qB, z, (int) m0);
                        }
                        else fprintf(p0_skip, "\"%4d-%02d-%02dT%02d:%02d:%02d\"  %d  %d  %d  %d\n", 1900+mean.tm_year, mean.tm_mon+1, mean.tm_mday, mean.tm_hour, mean.tm_min, mean.tm_sec, qA, qB, z, (int) m0);

                        fclose(DFMS_L2_file);
                        ++DFMS_file_number;
                        printf("\b\b\b\b\b\b%6d", DFMS_file_number);

                    }
                }
                ++source;
            }
            fclose(p0_save);
            fclose(p0_skip);

            clock_t mid = clock();
            double time_spent = (double)(mid - begin) / CLOCKS_PER_SEC;

            printf("\b\b\b\b\b\bs complete (%d files).\n", DFMS_file_number);
            printf("Computation time = %.2lf sec\n\n", time_spent);

            ++MTP;
        }
    }

    /// Read p0 list

    FILE* p0_row;
    p0_row = fopen(p0_list, "r");
    struct data *p0_LR[6];    // 0 = 16, 1 = 18, 2 = 28, 3 = 44, 4 = 60, 5 = 76 (, 6 = 78)
    struct data *p0_HR[6];    // 0 = 16, 1 = 18, 2 = 28, 3 = 44, 4 = 60, 5 = 76 (, 6 = 78)

    int m = 0;
    while (m < 6)
    {
        p0_LR[m] = malloc(100000 * sizeof(struct data));
        p0_HR[m] = malloc(100000 * sizeof(struct data));
        ++m;
    }

    /// Read files
    int entry_number[2][6] = {{0}};
    struct tm time;
    int p0_temp[2] = {0, 0};
    int m_temp = 0;
    int z = 0;

    while (fscanf(p0_row, "\"%d-%d-%dT%d:%d:%d\"  %d  %d  %d  %d\n", &time.tm_year, &time.tm_mon, &time.tm_mday, &time.tm_hour, &time.tm_min, &time.tm_sec, &p0_temp[0], &p0_temp[1], &z, &m_temp) == 10)
    {
        time.tm_year = time.tm_year - 1900;    // Conversion needed by struct tm
        time.tm_mon = time.tm_mon - 1;    // Conversion needed by struct tm
        time.tm_isdst = -1;    // Avoids conversion to summer/winter time
             if (m_temp == 16) {m = 0;}
        else if (m_temp == 18) {m = 1;}
        else if (m_temp == 28) {m = 2;}
        else if (m_temp == 44) {m = 3;}
        else if (m_temp == 60) {m = 4;}
        else if (m_temp == 76) {m = 5;}
        if (z == 0)    // LR
        {
            p0_LR[m][entry_number[0][m]].time = mktime (&time);
            p0_LR[m][entry_number[0][m]].res = 1;
            p0_LR[m][entry_number[0][m]].rowA = p0_temp[0];
            p0_LR[m][entry_number[0][m]].rowB = p0_temp[1];
            ++entry_number[0][m];
        }
        else    // HR
        {
            p0_HR[m][entry_number[1][m]].time = mktime (&time);
            p0_HR[m][entry_number[1][m]].res = 6.4;
            p0_HR[m][entry_number[1][m]].rowA = p0_temp[0];
            p0_HR[m][entry_number[1][m]].rowB = p0_temp[1];
            ++entry_number[1][m];
        }
        m_temp = 0;
        z = 0;
        p0_temp[0] = 0;
        p0_temp[1] = 0;
    }
    fclose(p0_row);

    printf("------------- p0 process complete -------------\n\n");


    /// Process DFMS L3 PDS

    int MTP = MTP_START;    // First MTP with DFMS spectra = MTP2
    while (MTP <= MTP_STOP)    // Last MTP = MTP35
    {
        printf("-------------------- MTP%d --------------------\n", MTP);
        clock_t begin = clock();

        /// Definition of MTP paths
        char MTP_string[3];
        sprintf(MTP_string, "%d", MTP);

        /// Definition of the DFMS L3 directories
        char* r1 = concat(DFMS_L3_path, "MTP");
        char* r2 = concat(r1, MTP_string);
        char* DFMS_L3_directory[2];
        DFMS_L3_directory[0] = concat(r2, "\\DFMS\\CE\\");
        DFMS_L3_directory[1] = concat(r2, "\\DFMS\\MC\\");
        free(r1);
        free(r2);
        char* r3 = concat(DFMS_L3_output_path, "MTP");
        char* r4 = concat(r3, MTP_string);
        char* DFMS_L3_corrected_directory = concat(r4, "\\DFMS\\MC\\");
        free(r3);
        free(r4);

        DIR* DFMS_dir[1];
        struct dirent* DFMS_in_file;
        FILE* DFMS_L3_file;
        FILE* DFMS_L3_corrected_file;
        char buffer[80];

        /// Read DFMS files in the DFMS directories

        int source = 0;
        int DFMS_file_number = 0;

        while (source < 2)    // 0 = CEM, 1 = MCP
        {
            /// Open the directory
            DFMS_dir[source] = opendir(DFMS_L3_directory[source]);

            /// Copy CEM data
            if (source == 0 && DFMS_dir[source] != NULL)
            {
                printf("Processing DFMS CEM files: ");
                char command[100];
                sprintf(command, "xcopy /S /Q %s* %sMTP%d\\DFMS\\CE\\", DFMS_L3_directory[source], DFMS_L3_output_path, MTP);
                system(command);
            }

            /// Create MCP folders
            if (source == 1)
            {
                char command[100];
                sprintf(command, "mkdir %s", DFMS_L3_corrected_directory);
                system(command);

                printf("Processing DFMS MCP file      ");

                /// Read all DFMS L3 files in current directory
                while ((DFMS_in_file = readdir(DFMS_dir[source])))
                {
                    if (!strcmp (DFMS_in_file->d_name, "."))    // Skip "." (from DFMS_dir)
                        continue;
                    if (!strcmp (DFMS_in_file->d_name, ".."))    // Skip ".." (from DFMS_dir)
                        continue;
                    if (!strncmp (&DFMS_in_file->d_name[21], "M0600", 5)
                     || !strncmp (&DFMS_in_file->d_name[21], "M0601", 5)
                     || !strncmp (&DFMS_in_file->d_name[21], "M0602", 5)
                     || !strncmp (&DFMS_in_file->d_name[21], "M0620", 5)
                     || !strncmp (&DFMS_in_file->d_name[21], "M0621", 5)
                     || !strncmp (&DFMS_in_file->d_name[21], "M0622", 5)
                     || !strncmp (&DFMS_in_file->d_name[21], "M0630", 5)
                     || !strncmp (&DFMS_in_file->d_name[21], "M0631", 5)
                     || !strncmp (&DFMS_in_file->d_name[21], "M0632", 5)
                     || !strncmp (&DFMS_in_file->d_name[21], "M9999", 5))
                        continue;

                    char* s = concat(DFMS_L3_directory[source], DFMS_in_file->d_name);
                    DFMS_L3_file = fopen (s, "r");
                    free(s);

                    if (DFMS_L3_file == NULL)
                    {
                        fprintf(stderr, "\n\nError: failed to open DFMS L3 file \"%s\" - %s\n", DFMS_in_file->d_name, strerror(errno));
                        return 0;
                    }

                    char* s_c = concat(DFMS_L3_corrected_directory, DFMS_in_file->d_name);
                    DFMS_L3_corrected_file = fopen (s_c, "w");
                    free(s_c);

                    /// Read header
                    int line_number = 1;
                    while (line_number < 42)    // Skip 41 lines
                    {
                        fgets(buffer, 80, DFMS_L3_file);
                        fprintf(DFMS_L3_corrected_file, "%s", buffer);
                        ++line_number;
                    }

                    /// Get start acquisition time
                    float sec = 0;
                    struct tm start;
                    fgets(buffer, 80, DFMS_L3_file);
                    fprintf(DFMS_L3_corrected_file, "%s", buffer);
                    sscanf(buffer, "START_TIME                       =     %4d-%2d-%2dT%2d:%2d:%6f                ", &start.tm_year, &start.tm_mon, &start.tm_mday, &start.tm_hour, &start.tm_min, &sec);
                    start.tm_year = start.tm_year - 1900;    // Conversion needed by struct tm
                    start.tm_mon = start.tm_mon - 1;    // Conversion needed by struct tm
                    start.tm_sec = (int) sec;    // Converts seconds to int (needed by (struct tm))
                    start.tm_isdst = -1;    // Avoids conversion to summer/winter time
                    ++line_number;

                    /// Get stop acquisition time
                    sec = 0;  
                    struct tm stop;
                    fgets(buffer, 80, DFMS_L3_file);
                    fprintf(DFMS_L3_corrected_file, "%s", buffer);
                    sscanf(buffer, "STOP_TIME                        =     %4d-%2d-%2dT%2d:%2d:%6f                ", &stop.tm_year, &stop.tm_mon, &stop.tm_mday, &stop.tm_hour, &stop.tm_min, &sec);
                    stop.tm_year = stop.tm_year - 1900;    // Conversion needed by struct tm
                    stop.tm_mon = stop.tm_mon - 1;    // Conversion needed by struct tm
                    stop.tm_sec = (int) sec;    // Converts seconds to int (needed by struct tm)
                    stop.tm_isdst = -1;    // Avoids conversion to summer/winter time
                    ++line_number;

                    /// Calculate mean acquisition time
                    time_t time_start = mktime (&start);    // Converts struct tm to time_t (number of seconds since 1970)
                    time_t time_stop = mktime (&stop);
                    time_t time_mean = time_start/2 + time_stop/2;    // Calculate mean value [(start + stop)/2 exceeds time_t (= int) max value]

                    while (line_number < 115)    // Copy header
                    {
                        fgets(buffer, 80, DFMS_L3_file);
                        fprintf(DFMS_L3_corrected_file, "%s", buffer);
                        ++line_number;
                    }

                    float m0 = 0;
                    fgets(buffer, 80, DFMS_L3_file);
                    fprintf(DFMS_L3_corrected_file, "%s", buffer);
                    sscanf(buffer, "\"ROSINA_DFMS_SCI_MASS            \",\"     \",\"%f\",\"     \" \n", &m0);
                    ++line_number;

                    while (line_number < 148)    // Copy header
                    {
                        fgets(buffer, 80, DFMS_L3_file);
                        fprintf(DFMS_L3_corrected_file, "%s", buffer);
                        ++line_number;
                    }

                    /// Extrapolate p0
                    float zoom_value = zoom(m0, &DFMS_in_file->d_name[21]);
                    int z = 1;    // HR
                    if (zoom_value == 1) {z = 0;}    // LR
                    float fact = (25/(interpolate_D(m0, z)*zoom_value));
                    float interpolated_p0_rowA = 0;
                    float interpolated_p0_rowB = 0;
                    if (z == 0)    // LR
                    {
                        interpolated_p0_rowA = interpolate_p0(m0, p0_LR, time_mean, z, 0);
                        interpolated_p0_rowB = interpolate_p0(m0, p0_LR, time_mean, z, 1);
                    }
                    else    // HR
                    {
                        interpolated_p0_rowA = interpolate_p0(m0, p0_HR, time_mean, z, 0);
                        interpolated_p0_rowB = interpolate_p0(m0, p0_HR, time_mean, z, 1);
                    }

                    /// Write p0 values in the header
                    fprintf(DFMS_L3_corrected_file, "\"ROSINA_DFMS_SCI_SELF_PIXEL0_A   \",\"ON   \",\"%-24.1f\",\"     \" \n", interpolated_p0_rowA);
                    fprintf(DFMS_L3_corrected_file, "\"ROSINA_DFMS_SCI_SELF_PIXEL0_UNCA\",\"ON   \",\"%-24.1f\",\"     \" \n", 10.0);
                    fprintf(DFMS_L3_corrected_file, "\"ROSINA_DFMS_SCI_SELF_PIXEL0_B   \",\"ON   \",\"%-24.1f\",\"     \" \n", interpolated_p0_rowB);
                    fprintf(DFMS_L3_corrected_file, "\"ROSINA_DFMS_SCI_SELF_PIXEL0_UNCB\",\"ON   \",\"%-24.1f\",\"     \" \n", 10.0);

                    while (line_number < 152)
                    {
                        fgets(buffer, 80, DFMS_L3_file);
                        ++line_number;
                    }

                    while (line_number < 190)    // Copy header
                    {
                        fgets(buffer, 80, DFMS_L3_file);
                        fprintf(DFMS_L3_corrected_file, "%s", buffer);
                        ++line_number;
                    }

                    /// Get bin, mass, and ion values from "former" L3 and correct the mass scale in "corrected" L3
                    line_number = 0;
                    int bin[512] = {0};
                    float mass_rowA[512] = {0};
                    float ions_rowA[512] = {0};
                    float mass_rowB[512] = {0};
                    float ions_rowB[512] = {0};

                    if (precision == 4)
                    {
                        while (fscanf(DFMS_L3_file, "%d,%f,%f,%f,%f,    \n",
                                                     &bin[line_number],
                                                         &mass_rowA[line_number],
                                                            &ions_rowA[line_number],
                                                               &mass_rowB[line_number],
                                                                  &ions_rowB[line_number]) == 5)    // Read lines
                        {
                            fprintf(DFMS_L3_corrected_file, "%3d,%17.4f,%16.1f,%17.4f,%16.1f,    \n",
                                                             bin[line_number],
                                                                 m0 * exp((bin[line_number]-interpolated_p0_rowA)*fact),
                                                                        ions_rowA[line_number],
                                                                               m0 * exp((bin[line_number]-interpolated_p0_rowB)*fact),
                                                                                      ions_rowB[line_number]);    // Write corrected lines
                            ++line_number;
                        }
                    }
                    else
                    {
                        while (fscanf(DFMS_L3_file, "%d,%f,%f,%f,%f,    \n",
                                                     &bin[line_number],
                                                         &mass_rowA[line_number],
                                                            &ions_rowA[line_number],
                                                               &mass_rowB[line_number],
                                                                  &ions_rowB[line_number]) == 5)    // Read lines
                        {
                            fprintf(DFMS_L3_corrected_file, "%3d,%17.2f,%16.1f,%17.2f,%16.1f,    \n",
                                                             bin[line_number],
                                                                 m0 * exp((bin[line_number]-interpolated_p0_rowA)*fact),
                                                                        ions_rowA[line_number],
                                                                               m0 * exp((bin[line_number]-interpolated_p0_rowB)*fact),
                                                                                      ions_rowB[line_number]);    // Write corrected lines
                            ++line_number;
                        }
                    }

                    fclose(DFMS_L3_file);
                    fclose(DFMS_L3_corrected_file);
                    ++DFMS_file_number;
                    printf("\b\b\b\b\b\b%6d", DFMS_file_number);
                }
            }
            ++source;
        }
        clock_t mid = clock();
        double time_spent = (double)(mid - begin) / CLOCKS_PER_SEC;

        printf("\b\b\b\b\b\bs complete (%d files).\n", DFMS_file_number);
        printf("Computation time = %.2lf sec\n\n", time_spent);

        ++MTP;
    }

    printf("--------------- Process complete ---------------\n\n");

    /// Calculate total computation time
    clock_t stop_clock = clock();
    double time_spent_tot = (double)(stop_clock - start_clock) / CLOCKS_PER_SEC;

    int hours = time_spent_tot / (60 * 60);
    time_spent_tot -= hours * (60 * 60);
    int minutes = time_spent_tot / 60;
    time_spent_tot -= minutes * 60;

    /// Display final summary
    printf("Total computation time =");
    if (hours) printf(" %d h", hours);
    if (minutes) printf(" %02d min", minutes);
    printf(" %04.2lf sec\n\n", time_spent_tot);

    return 1;
}
