######################################################################
### START:  Python script to convert SDC instrument calibration matrix
###         between FITS and PDS TABLE
"""
sdcfit_pdsascii.py.py

Convert SDC Calibration Matrix from FITS PDU array to PDS ASCII
TABLES for PDS archiving, and reverse

- Input FITS filename is sdc_cal_matrix_base.dat
- Output TABLE filename is sdc_cal_matrix_TYPE_CHnn.{lbl,tab}
  - TYPE is coef, mean, or stdv
  - nn is 01 through 14

Usage:

  1) Convert from FITS to TABLEs:

       python sdcfit_pdsascii.py totable

  2) Convert from TABLES to FITS:

       python sdcfit_pdsascii.py tofit
       python __sdc_cal_matrix_table.py xyz.fit

"""
import os
import sys
import hashlib
import numpy
import pyfits
import pprint
import numpy as np
from pyfits import Column


########################################################################
fieldlen = 24
ndecimals = fieldlen - 8
fieldfmt = '%%%d.%de' % (fieldlen,ndecimals,)
linelen = (4 * fieldlen) + 2
fitmd5 = '7ff25cfe01af09ed9c856296ab7a61f4'
crlf = '\r\n'


########################################################################
def tw3(fmt='',val=()): return (' '*fieldlen + fmt % val)[-fieldlen:]

def pdsifylines(lines,L):
  return crlf.join( [ (i.rstrip()+(' '*L))[:L] for i in lines ] ) + crlf


if __name__=="__main__":


  fitSOCinp = 'sdc_cal_matrix_base.dat'

  vtemps = ['49.9','40.0','34.25', '-7.1']
  vtypes = ['COEF','MEAN','STDV']
  units = ['N/A','DATA NUMBER','DATA NUMBER']

  if not sys.argv[1:]:
    print( "Usage:  python sdcfit_pdsascii.py [totable|xyz.fit]" )

  elif sys.argv[1:] and sys.argv[1]=='totable':
    sdcCalData = pyfits.open(fitSOCinp)[0].data
    ### LABEL
    labelfmt = """
PDS_VERSION_ID       = PDS3
RECORD_TYPE          = FIXED_LENGTH
RECORD_BYTES         = %(linelen)d
FILE_RECORDS         = %%(labellenplus19)d

^TABLE               = %%(labellenplus1)d

MISSION_NAME         = "NEW HORIZONS"
DATA_SET_ID          = "NH-J-SDC-3-JUPITER-V3.0"                 
START_TIME           = 2006-01-17T00:00:00  /* Nominal NH Launch */
STOP_TIME            = "N/A"
PRODUCT_ID           = "SDC_CAL_MATRIX_%(vtype)s_CH%(channel)s"
PRODUCER_INSTITUTION_NAME = "SOUTHWEST RESEARCH INSTITUTE"
PRODUCT_CREATION_TIME = 2013-12-06T00:00:00
INSTRUMENT_NAME      = "
  STUDENT DUST COUNTER
"
INSTRUMENT_HOST_NAME = "NEW HORIZONS"
INSTRUMENT_ID        = "SDC"
TARGET_NAME          = "N/A"

OBJECT                 = TABLE

  DESCRIPTION          = "
%(tabledescription)s

  This TABLE is one of several archiving calibration parameters used in the
  calibration of SDC data.

  Details of the calibration may be found in the SOC ICD document,
  which was delivered with this data set.

Genesis of the calibration data (from the ICD):                               
                                                                              
  In a temperature controlled environment, the electronics from the end of the
  PVDF to the DN in the raw data were calibrated, at each of 4 calibration box
  temperatures and for each of the 14 channels. This was done by injecting 19 
  (actually 21; see the ICD) fixed-amplitude charge pulses 100 times into a     
  channel and recording the DN value each time. From those recorded values,   
  the average DN (DNavg) and its standard deviation (SIG) at each charge pulse
  amplitude, box temperature and channel were calculated. Then, for each box  
  temperature and channel, a 9th order polynomial fit of Q(DNavg) was derived.
  Finally, these 3 sets of values (the polynomial coefficients, DNavg, and    
  SIG) were stored in a matrix. This matrix contains all information required 
  to calculate the charge equivalent to a DN as a function of box temperature 
  and channel (detector), as well as the uncertainty in that calculated charge
  value.                                                                      

That matrix has been broken out into forty-two PDS ASCII TABLEs; this TABLE
is one of those forty-two.
  "

  NAME                 = "%(filenamepfx)s"
  INTERCHANGE_FORMAT   = ASCII
  ROW_BYTES            = %(linelen)d
  ROWS                 = %(maxNRows)d
  COLUMNS              = 4
%(columns)s

END_OBJECT             = TABLE

END
""".lstrip()
    ### COLUMN
    columnfmt = """
  OBJECT               = COLUMN
    NAME                 = "SDC_CAL_%(vtype)s_CH%(channel)s_%(ntemp)s"
    COLUMN_NUMBER        = %(colnum)d
    DATA_TYPE            = ASCII_REAL
    FORMAT               = "E%(fieldlen)d.%(ndecimals)d"
    START_BYTE           = %(startbyte)d
    BYTES                = %(fieldlen)d
    UNIT                 = "%(unit)s"
    DESCRIPTION          = "
%(columndescription)s
    "
    MISSING_CONSTANT     = 0.0
  END_OBJECT           = COLUMN
""".rstrip()
    ### TABLE descriptions
    tdscfmts = [ i[1:].rstrip() for i in
    [ """
  TABLE of polynomial coefficients Ci, for channel %(channel)s of the
  SDC instrument at %(nTemps)d temperatures, fitting the response
  relationship

    log10( Q(DN) )= C0 + C1 x DN + C2 x DN^2 + ... + C18 x DN^%(lastcoeff)d

  where Q is an input charge to the SDC instrument channel, in units of
  'Number of equivalent electrons,' and DN (Data Number) is the 16-bit
  digital output that results from input Q.

  The values in this TABLE apply to one SDC channel (%(channel)s), and each
  COLUMN corresponds to one temperature.

  The first ROW contains coefficients C0, and the last ROW contains
  coefficients C%(lastcoeff)d.  There are typically fewer non-zero
  coefficients than ROWs in the TABLE.
""" , """
  TABLE of mean output Data Number (DN) values from the ground calibration
  of channel %(channel)s of the the SDC instrument at four temperatures.
  Each value in the TABLE is the mean of 100 DN samples for a known input
  charge.

  The values in this TABLE apply to one SDC channel (%(channel)s), and each
  COLUMN corresponds to one temperature.

  The ROW indicates the order in which the samples were taken at each
  temperature; there is no correlation across COLUMNs on each ROW between
  the input charges that generated the values, and values in later ROWs are
  sometimes zero indicating that fewer than %(maxNRows)d calibration runs
  resulted in non-zero values.
""" , """
  TABLE of output Data Number (DN) standard deviation values from the
  ground calibration of channel %(channel)s of the the SDC instrument at
  %(nTemps)d temperatures.  Each value in the TABLE is the standard
  deviation of 100 DN samples for a known input charge.

  The values in this TABLE apply to one SDC channel (%(channel)s), and each
  COLUMN corresponds to one temperature.

  The ROW indicates the order in which the samples were taken at each
  temperature; there is no correlation across COLUMNs on each ROW between
  the input charges that generated the values, and values in later ROWs are
  sometimes zero indicating that fewer than %(maxNRows)d calibration runs
  resulted in non-zero values.
""" ] ]
    ### COLUMN descriptions
    cdscfmts = [ i[1:].rstrip() for i in
    [ """
    Calibration polynomial coefficients for SDC instrument channel %(channel)s
    from the ground calibration at temperature %(vtemp)s Celcius.
""" , """
    Mean Data Number values for SDC instrument channel %(channel)s
    from the ground calibration at temperature %(vtemp)s Celcius.
""" , """
    Standard Deviation Data Number values for SDC instrument channel %(channel)s
    from the ground calibration at temperature %(vtemp)s Celcius.
""" ] ]

    ### (19, 3, 14, 4)
    maxNRows,nTypes,nChans,nTemps = sdcCalData.shape

    dikt = {}
    for itype in range(nTypes):
      dikt = dict( vtype=vtypes[itype]
                 , unit=units[itype]
                 , nTemps=nTemps
                 , maxNRows=maxNRows
                 , lastcoeff=maxNRows-1
                 , linelen=linelen
                 , fieldlen=fieldlen
                 , ndecimals=ndecimals
                 )
      for ichan in range(nChans):
        dikt['channel'] = '%02d' % (ichan+1,)
        dikt['tabledescription'] = tdscfmts[itype] % dikt
        allcolumns = ''
        for itemp in range(nTemps):
          dikt['vtemp'] = vtemps[itemp]
          dikt['ntemp'] = vtemps[itemp].replace('.','_').replace('-','M')[:4]
          dikt['columndescription'] = cdscfmts[itype] % dikt
          dikt['startbyte'] = (itemp * fieldlen) + 1
          dikt['colnum'] = itemp + 1
          dikt['filenamepfx'] = "SDC_CAL_MATRIX_%(vtype)s_CH%(channel)s" % dikt
          allcolumns += '\n' + columnfmt % dikt

        dikt['columns'] = allcolumns % dikt

        label0 = (labelfmt % dikt).strip()
        dikt['labellenplus1'] = len(label0.split('\n')) + 1
        dikt['labellenplus19'] = dikt['labellenplus1'] + 18
        labellines = (label0 % dikt).split('\n')
        for irow in range(maxNRows):
          onerow = ''
          for itemp in range(nTemps):
            onerow += fieldfmt % (sdcCalData[irow,itype,ichan,itemp],)
          labellines.append( onerow )
        labellines = pdsifylines(labellines,linelen-2)
        open(('%(filenamepfx)s.tab'%dikt).lower(),"w").write(labellines)

  else:

    fitout = sys.argv[1]

    if os.path.exists(fitout) or os.path.lexists(fitout):
      print( "### ERROR:  cannot overwrite existing file (or link) '%s'" % (fitout,) )

    else:
      arr = numpy.zeros(4*14*3*19,dtype='>f8').reshape((19,3,14,4,))
      for itype in range(3):
        for ichan in range(14):
          dikt = dict( vtype=vtypes[itype]
                     , channel = '%02d' % (ichan+1,)
                     )
          filename = "SDC_CAL_MATRIX_%(vtype)s_CH%(channel)s.tab" % dikt
          arr[:,itype,ichan,:] = numpy.array( ' '.join(open(filename.lower(),'r').readlines()[-19:]).split(), dtype=numpy.float ).reshape((-1,4,))

      pdu = arr.tostring()
      L = (2880 - (len(pdu) % 2880)) % 2880
      if L: pdu += '\0' * L
      hdr = """
SIMPLE  =                    T / Written by IDL:  Tue Sep 11 12:22:02 2007      BITPIX  =                  -64 / Number of bits per data pixel                  NAXIS   =                    4 / Number of data axes                            NAXIS1  =                    4 /                                                NAXIS2  =                   14 /                                                NAXIS3  =                    3 /                                                NAXIS4  =                   19 /                                                EXTEND  =                    T / FITS data may contain extensions               DATE    = '2007-02-21'         / Creation UTC (CCCC-MM-DD) date of FITS header  COMMENT FITS (Flexible Image Transport System) format is defined in 'Astronomy  COMMENT and Astrophysics', volume 376, page 359; bibcode 2001A&A...376..359H    COMMENT                                                                         COMMENT This file contains information used by the SDC Level 2 pipeline to      COMMENT calibrate SDC data.                                                     END                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             
""".lstrip()[:2880]

      both = hdr + pdu
      h = hashlib.md5()
      h.update(both)
      print( (h.hexdigest(), h.hexdigest()==fitmd5, fitmd5,) )
      open( sys.argv[1],'w' ).write( both )

### END:  Python script to convert SDC instrument calibration matrix
###         between FITS and PDS TABLE
######################################################################
